//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * TEh}l[W
 * @author Junpee
 */

#include "LampBasic.h"
#include "Sound/System/SoundManager.h"
#include "Sound/System/SoundDefinition.h"
#include "Sound/Reader/WaveReader.h"
#include "Sound/Reader/OggVorbisReader.h"
#include "Sound/Stereo/StaticSound.h"
#include "Sound/Stereo/StreamSound.h"
#include "Sound/3D/StaticSound3D.h"
#include "Sound/3D/StreamSound3D.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
SoundManager::SoundManager(DirectSound* directSound) :
	directSound_(directSound), algorithm_(algorithmNormal){
}
//------------------------------------------------------------------------------
// fXgN^
SoundManager::~SoundManager(){
	Assert(sounds_.getCount() == 0);
	Assert(updateSounds_.getCount() == 0);
}
//------------------------------------------------------------------------------
// TEh
//------------------------------------------------------------------------------
// TEh̍폜
void SoundManager::destroy(Sound* sound){
	sounds_.removeByValue(sound);
	delete sound;
}
//------------------------------------------------------------------------------
// STEh̔j
void SoundManager::destroyAll(){
	int soundCount = getCount();
	for(int i = 0; i < soundCount; i++){ delete getSound(i); }
	sounds_.clear();
}
//------------------------------------------------------------------------------
// ÓITEh
//------------------------------------------------------------------------------
// ÓITEh̍쐬
StaticSound* SoundManager::createStaticSound(
	u_int size, int sample, int channel, int bit, Sound::Focus focus){
	// EF[utH[}bg̍\z
	WAVEFORMATEX waveFormat;
	buildWaveFormat(&waveFormat, sample, channel, bit);
	// obt@Lq̍\z
	DSBUFFERDESC description;
	buildBufferDescription(&description, &waveFormat, size, focus);
	// pǉ
	description.dwFlags |= DSBCAPS_CTRLPAN;
	// obt@̍\z
	DirectSoundBuffer* soundBuffer = buildSoundBuffer(&description);
	// TEh̍\z
	StaticSound* sound = new StaticSound(soundBuffer);
	sound->initialize(size, sample, channel, bit, focus);
	sounds_.pushBack(sound);
	return sound;
}
//------------------------------------------------------------------------------
// ÓITEh̍쐬
StaticSound* SoundManager::createStaticSound(
	SoundReader* soundReader, const String& name, Sound::Focus focus){
	if(soundReader == NULL){ return NULL; }
	if(!soundReader->readHeader()){ return NULL; }
	// TEh쐬
	StaticSound* sound = createStaticSound(
		soundReader->getSize(), soundReader->getSample(),
		soundReader->getChannel(), soundReader->getBit(), focus);
	sound->setName(name);
	sound->setComment(soundReader->getComment());
	// TEhǂݍ
	if(!readSoundBuffer(sound, soundReader)){
		destroy(sound);
		sound = NULL;
	}
	return sound;
}
//------------------------------------------------------------------------------
// ÓITEh̍쐬
StaticSound* SoundManager::createStaticSound(
	const String& fileName, Sound::Focus focus){
	FilePath filePath(fileName);
	SoundReader* soundReader = createSoundReader(filePath);
	StaticSound* sound =
		createStaticSound(soundReader, filePath.getFileName(), focus);
	SafeDelete(soundReader);
	return sound;
}
//------------------------------------------------------------------------------
// ÓITEh̕
StaticSound* SoundManager::cloneStaticSound(StaticSound* staticSound){
	DirectSoundBuffer* soundBuffer =
		duplicateSoundBuffer(staticSound->getSoundBuffer());
	// TEh̍\z
	StaticSound* sound = new StaticSound(soundBuffer);
	sounds_.pushBack(sound);
	return sound;
}
//------------------------------------------------------------------------------
// Xg[TEh
//------------------------------------------------------------------------------
// Xg[TEh̍쐬
StreamSound* SoundManager::createStreamSound(
	u_int size, int sample, int channel, int bit, Sound::Focus focus){
	// EF[utH[}bg̍\z
	WAVEFORMATEX waveFormat;
	buildWaveFormat(&waveFormat, sample, channel, bit);
	// obt@Lq̍\z
	DSBUFFERDESC description;
	buildBufferDescription(&description, &waveFormat, size, focus);
	// pAʒmǉ
	description.dwFlags |= (DSBCAPS_CTRLPAN | DSBCAPS_CTRLPOSITIONNOTIFY);
	// obt@̍\z
	DirectSoundBuffer* soundBuffer = buildSoundBuffer(&description);
	// TEh̍\z
	StreamSound* sound = new StreamSound(soundBuffer);
	sound->initialize(size, sample, channel, bit, focus);
	sounds_.pushBack(sound);
	return sound;
}
//------------------------------------------------------------------------------
// Xg[TEh̍쐬
StreamSound* SoundManager::createStreamSound(
	SoundReader* soundReader, const String& name, Sound::Focus focus){
	if(soundReader == NULL){ return NULL; }
	if(!soundReader->readHeader()){ return NULL; }
	// Xg[obt@TCY̎Zo
	u_int size = (u_int)(SoundDefinition::streamBufferLength *
		soundReader->getOneSecondBytes());
	// TEh쐬
	StreamSound* sound = createStreamSound(
		size, soundReader->getSample(),
		soundReader->getChannel(), soundReader->getBit(), focus);
	sound->setName(name);
	sound->setComment(soundReader->getComment());
	// TEh[_̐ݒ
	sound->setSoundReader(soundReader);
	return sound;
}
//------------------------------------------------------------------------------
// Xg[TEh̍쐬
StreamSound* SoundManager::createStreamSound(
	const String& fileName, Sound::Focus focus){
	FilePath filePath(fileName);
	SoundReader* soundReader = createSoundReader(filePath);
	StreamSound* sound =
		createStreamSound(soundReader, filePath.getFileName(), focus);
	// TEh[_̓Xg[TEhɂč폜
	return sound;
}
//------------------------------------------------------------------------------
// ÓI3DTEh
//------------------------------------------------------------------------------
// ÓI3DTEh̍쐬
StaticSound3D* SoundManager::createStaticSound3D(u_int size, int sample,
	int channel, int bit, Sound::Focus focus){
	if(channel != 1){
		ErrorOut("SoundManager::createStaticSound3D() "
			"młȂTEh3DTEhɂł܂B");
	}
	// EF[utH[}bg̍\z
	WAVEFORMATEX waveFormat;
	buildWaveFormat(&waveFormat, sample, channel, bit);
	// obt@Lq̍\z
	DSBUFFERDESC description;
	buildBufferDescription(&description, &waveFormat, size, focus);
	// 3D
	description.dwFlags |= DSBCAPS_CTRL3D;
	// 3DASY
	if(algorithm_ == algorithmNormal){
		description.guid3DAlgorithm = DS3DALG_DEFAULT;
	}else if(algorithm_ == algorithmLightHRTF){
		description.guid3DAlgorithm = DS3DALG_HRTF_LIGHT;
	}else if(algorithm_ == algorithmFullHRTF){
		description.guid3DAlgorithm = DS3DALG_HRTF_FULL;
	}
	// obt@̍\z
	DirectSoundBuffer* soundBuffer = buildSoundBuffer(&description);
	// TEh̍\z
	StaticSound3D* sound = new StaticSound3D(soundBuffer);
	sound->initialize(size, sample, channel, bit, focus);
	sounds_.pushBack(sound);
	return sound;
}
//------------------------------------------------------------------------------
// ÓI3DTEh̍쐬
StaticSound3D* SoundManager::createStaticSound3D(SoundReader* soundReader,
	const String& name, Sound::Focus focus){
	if(soundReader == NULL){ return NULL; }
	if(!soundReader->readHeader()){ return NULL; }
	// TEh쐬
	StaticSound3D* sound = createStaticSound3D(soundReader->getSize(),
		soundReader->getSample(),soundReader->getChannel(),
		soundReader->getBit(), focus);
	sound->setName(name);
	sound->setComment(soundReader->getComment());
	// TEhǂݍ
	if(!readSoundBuffer(sound, soundReader)){
		destroy(sound);
		sound = NULL;
	}
	return sound;
}
//------------------------------------------------------------------------------
// ÓI3DTEh̍쐬
StaticSound3D* SoundManager::createStaticSound3D(
	const String& fileName, Sound::Focus focus){
	FilePath filePath(fileName);
	SoundReader* soundReader = createSoundReader(filePath);
	StaticSound3D* sound = createStaticSound3D(
		soundReader, filePath.getFileName(), focus);
	SafeDelete(soundReader);
	return sound;
}
//------------------------------------------------------------------------------
// ÓI3DTEh̕
StaticSound3D* SoundManager::cloneStaticSound3D(StaticSound3D* staticSound3D){
	DirectSoundBuffer* soundBuffer =
		duplicateSoundBuffer(staticSound3D->getSoundBuffer());
	// TEh̍\z
	StaticSound3D* sound = new StaticSound3D(soundBuffer);
	sounds_.pushBack(sound);
	return sound;
}
//------------------------------------------------------------------------------
// Xg[3DTEh
//------------------------------------------------------------------------------
// Xg[3DTEh̍쐬
StreamSound3D* SoundManager::createStreamSound3D(
	u_int size, int sample, int channel, int bit, Sound::Focus focus){
	// EF[utH[}bg̍\z
	WAVEFORMATEX waveFormat;
	buildWaveFormat(&waveFormat, sample, channel, bit);
	// obt@Lq̍\z
	DSBUFFERDESC description;
	buildBufferDescription(&description, &waveFormat, size, focus);
	// 3DAʒmǉ
	description.dwFlags |= (DSBCAPS_CTRL3D | DSBCAPS_CTRLPOSITIONNOTIFY);
	// 3DASY
	if(algorithm_ == algorithmNormal){
		description.guid3DAlgorithm = DS3DALG_DEFAULT;
	}else if(algorithm_ == algorithmLightHRTF){
		description.guid3DAlgorithm = DS3DALG_HRTF_LIGHT;
	}else if(algorithm_ == algorithmFullHRTF){
		description.guid3DAlgorithm = DS3DALG_HRTF_FULL;
	}
	// obt@̍\z
	DirectSoundBuffer* soundBuffer = buildSoundBuffer(&description);
	// TEh̍\z
	StreamSound3D* sound = new StreamSound3D(soundBuffer);
	sound->initialize(size, sample, channel, bit, focus);
	sounds_.pushBack(sound);
	return sound;
}
//------------------------------------------------------------------------------
// Xg[3DTEh̍쐬
StreamSound3D* SoundManager::createStreamSound3D(
	SoundReader* soundReader, const String& name, Sound::Focus focus){
	if(soundReader == NULL){ return NULL; }
	if(!soundReader->readHeader()){ return NULL; }
	// Xg[obt@TCY̎Zo
	u_int size = (u_int)(SoundDefinition::streamBufferLength *
		soundReader->getOneSecondBytes());
	// TEh쐬
	StreamSound3D* sound = createStreamSound3D(
		size, soundReader->getSample(),
		soundReader->getChannel(), soundReader->getBit(), focus);
	sound->setName(name);
	sound->setComment(soundReader->getComment());
	// TEh[_̐ݒ
	sound->setSoundReader(soundReader);
	return sound;
}
//------------------------------------------------------------------------------
// Xg[3DTEh̍쐬
StreamSound3D* SoundManager::createStreamSound3D(
	const String& fileName, Sound::Focus focus){
	FilePath filePath(fileName);
	SoundReader* soundReader = createSoundReader(filePath);
	StreamSound3D* sound =
		createStreamSound3D(soundReader, filePath.getFileName(), focus);
	// TEh[_̓Xg[3DTEhɂč폜
	return sound;
}
//------------------------------------------------------------------------------
// TEhobt@
//------------------------------------------------------------------------------
// EF[utH[}bg̍\z
void SoundManager::buildWaveFormat(
	WAVEFORMATEX* waveFormat, int sample, int channel, int bit){
	Assert((sample >= DSBFREQUENCY_MIN) && (sample <= 100000));
	Assert((channel >= 1) && (channel <= 2));
	Assert((bit == 16) || (bit == 8));
	::memset(waveFormat, 0, sizeof(WAVEFORMATEX));
	waveFormat->wFormatTag = WAVE_FORMAT_PCM;
	waveFormat->nSamplesPerSec = sample;
	waveFormat->nChannels = channel;
	waveFormat->wBitsPerSample = bit;
	waveFormat->nBlockAlign =
		(waveFormat->nChannels * waveFormat->wBitsPerSample) / 8;
	waveFormat->nAvgBytesPerSec =
		waveFormat->nSamplesPerSec * waveFormat->nBlockAlign;
}
//------------------------------------------------------------------------------
// obt@Lq̍\z
void SoundManager::buildBufferDescription(DSBUFFERDESC* description,
	WAVEFORMATEX* waveFormat, int size, Sound::Focus focus){
	::memset(description, 0, sizeof(DSBUFFERDESC));
	description->dwSize = sizeof(DSBUFFERDESC);
	description->lpwfxFormat = waveFormat;
	description->dwBufferBytes = size;
	// I{CXǗA{[AgAmȍĐJ[\
	description->dwFlags = (DSBCAPS_LOCDEFER | DSBCAPS_CTRLVOLUME |
		DSBCAPS_CTRLFREQUENCY | DSBCAPS_GETCURRENTPOSITION2);
	if(focus == Sound::focusSticky){
		description->dwFlags |= DSBCAPS_STICKYFOCUS;
	}else if(focus == Sound::focusGlobal){
		description->dwFlags |= DSBCAPS_GLOBALFOCUS;
	}
	// ǉׂtO
	// Stereo : DSBCAPS_CTRLPAN
	// 3D     : 3DASY
	// Stream : DSBCAPS_CTRLPOSITIONNOTIFY
}
//------------------------------------------------------------------------------
// TEhobt@̍\z
DirectSoundBuffer* SoundManager::buildSoundBuffer(DSBUFFERDESC* description){
	IDirectSoundBuffer* buffer;
	if(DirectXFailed(directSound_->CreateSoundBuffer(
		description, &buffer, NULL))){
		ErrorOut("SoundManager::buildSoundBuffer() "
			"TEhobt@̍\zɎs܂");
	}
	DirectSoundBuffer* soundBuffer;
	if(DirectXFailed(buffer->QueryInterface(
		DirectSoundBufferInterfaceID, (void**)&soundBuffer))){
		ErrorOut("SoundManager::buildSoundBuffer() "
			"TEhobt@̃C^[tF[X擾Ɏs܂");
	}
	SafeRelease(buffer);
	return soundBuffer;
}
//------------------------------------------------------------------------------
// TEh[_̍쐬
SoundReader* SoundManager::createSoundReader(const FilePath& filePath){
	SoundReader* soundReader = NULL;
	// [_̎
	if(WaveReader::isWaveFileName(filePath)){
		soundReader = new WaveReader(filePath);
	}else if(OggVorbisReader::isOggVorbisFileName(filePath)){
		soundReader = new OggVorbisReader(filePath);
	}
	return soundReader;
}
//------------------------------------------------------------------------------
// TEhobt@̓ǂݍ
bool SoundManager::readSoundBuffer(
	SoundBuffer* soundBuffer, SoundReader* soundReader){
	SoundBuffer::Lock lock = soundBuffer->lock();
	bool result = lock.isValid();
	if(result){
		u_int bufferSize = lock.getSize0();
		Assert(bufferSize == soundReader->getSize());
		int resdSize = soundReader->read(lock.getAddress0(), bufferSize);
		result = (resdSize == bufferSize);
		soundBuffer->unlock();
	}
	return result;
}
//------------------------------------------------------------------------------
// TEhobt@̕
DirectSoundBuffer* SoundManager::duplicateSoundBuffer(
	DirectSoundBuffer* soundBuffer){
	IDirectSoundBuffer* buffer;
	if(DirectXFailed(directSound_->DuplicateSoundBuffer(
		soundBuffer, &buffer))){
		ErrorOut("SoundManager::duplicateSoundBuffer() "
			"TEhobt@̕Ɏs܂");
	}
	DirectSoundBuffer* duplicatedBuffer;
	if(DirectXFailed(buffer->QueryInterface(
		DirectSoundBufferInterfaceID, (void**)&duplicatedBuffer))){
		ErrorOut("SoundManager::duplicateSoundBuffer() "
			"TEhobt@̃C^[tF[X擾Ɏs܂");
	}
	SafeRelease(buffer);
	return duplicatedBuffer;
}
//------------------------------------------------------------------------------
// Abvf[g
//------------------------------------------------------------------------------
// Abvf[g
void SoundManager::update(){
	for(int i = 0; i < updateSounds_.getCount(); i++){
		Sound* sound = updateSounds_[i];
		if(sound->update()){
			updateSounds_.remove(i);
			i--;
		}
	}
}
//------------------------------------------------------------------------------
// Abvf[gTEh̒ǉ
void SoundManager::addUpdateSound(Sound* sound){
	updateSounds_.pushBack(sound);
}
//------------------------------------------------------------------------------
// Abvf[gTEh̍폜
void SoundManager::removeUpdateSound(Sound* sound){
	int result = updateSounds_.removeByValue(sound);
	Assert(result != -1);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
