//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * 
 * @author Junpee
 */

#include "LampBasic.h"
#include "Sound/System/SoundBuffer.h"
#include "Sound/System/LampSound.h"
#include "Sound/System/SoundManager.h"
#include "Sound/System/SoundDefinition.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
SoundBuffer::SoundBuffer(DirectSoundBuffer* soundBuffer) :
	Sound(), soundBuffer_(soundBuffer){
	lock_.clear();
	// f[^̎擾
	long decibelVolume;
	if(DirectXFailed(soundBuffer->GetVolume(&decibelVolume))){
		ErrorOut("SoundBuffer::SoundBuffer() {[̎擾Ɏs܂B");
	}
	volume_ = decibelToVolume(decibelVolume);
	frequency_ = getBufferFrequency();
	fadeStartVolume_ = fadeEndVolume_ = 0.f;
	isFading_ = false;
	setPriority(priorityDefault);
	setOwnership(true);
	setLoop(false);
	isSuspended_ = false;
}
//------------------------------------------------------------------------------
// 
void SoundBuffer::initialize(
	u_int size, int sample, int channel, int bit, Focus focus){
	size_ = size;
	sample_ = sample;
	channel_ = channel;
	bit_ = bit;
	focus_ = focus;
}
//------------------------------------------------------------------------------
// fXgN^
SoundBuffer::~SoundBuffer(){
	// tF[h
	stopFade();
	stop();
	SafeRelease(soundBuffer_);
}
//------------------------------------------------------------------------------
// TEhobt@f[^̃Rs[
void SoundBuffer::copySoundBufferData(SoundBuffer* destination){
	destination->setName(getName());
	destination->initialize(
		getBufferSize(), getSample(), getChannel(), getBit(), getFocus());
	destination->setPriority(getPriority());
	destination->setLoop(isLoop());
}
//------------------------------------------------------------------------------
// Đ
//------------------------------------------------------------------------------
// Đ
bool SoundBuffer::play(){
	// ĐsetCursor(0)ƃXg[ǂ݂Ƃ̑Ȃ
	isSuspended_ = false;
	return playBuffer();
}
//------------------------------------------------------------------------------
// ~
void SoundBuffer::stop(){
	isSuspended_ = false;
	stopBuffer();
	setCursor(0);
}
//------------------------------------------------------------------------------
// Đ̈ꎞ~
void SoundBuffer::suspend(){
	State state = getState();
	// ~Ȃ~܂
	if(state == stateStop){ return; }
	isSuspended_ = true;
	stopBuffer();
}
//------------------------------------------------------------------------------
// ĐĊJ
bool SoundBuffer::resume(){
	State state = getState();
	// ~Ȃ~܂
	if(state == stateStop){ return true; }
	// obt@ĐĂ璆ftONA
	bool result = playBuffer();
	isSuspended_ = false;
	return result;
}
//------------------------------------------------------------------------------
// Ԃ̎擾
SoundBuffer::State SoundBuffer::getState() const{
	u_long status;
	if(DirectXFailed(soundBuffer_->GetStatus(&status))){
		ErrorOut("SoundBuffer::getState() Xe[^X̎擾Ɏs܂B");
	}
	if((status & DSBSTATUS_BUFFERLOST) != 0){
		// XgȂ烊XgAčēxXe[g擾
		if(DirectXFailed(soundBuffer_->Restore())){
			ErrorOut("SoundBuffer::getState() obt@̃XgAɎs܂B");
		}
		if(DirectXFailed(soundBuffer_->GetStatus(&status))){
			ErrorOut("SoundBuffer::getState() Xe[^X̎擾Ɏs܂B");
		}
		if((status & DSBSTATUS_BUFFERLOST) != 0){ return stateLost; }
	}
	if((status & DSBSTATUS_PLAYING) != 0){ return statePlay; }
	if(isSuspended_){ return stateSuspend; }
	return stateStop;
}
//------------------------------------------------------------------------------
// obt@̍Đ
bool SoundBuffer::playBuffer(){
	u_int priority = (u_int)(getPriority() + 32768);
	u_int flag = getPlayFlag();
	// Đs
	HRESULT result = soundBuffer_->Play(0, priority, flag);
	if(DirectXSucceeded(result)){ return true; }
	// ȂATEh񓯎ƋN邱Ƃ
	if(result == E_OUTOFMEMORY){ Assert(false); return false; }
	// obt@XgȂ烊XgAĂ݂
	if(result != DSERR_BUFFERLOST){
		ErrorOut("SoundBuffer::playBuffer() obt@̍ĐɎs܂B");
		return false;
	}
	result = soundBuffer_->Restore();
	if(DirectXFailed(result)){ return false; }
	// ēxĐs
	result = soundBuffer_->Play(0, priority, flag);
	if(DirectXSucceeded(result)){ return true; }
	return false;
}
//------------------------------------------------------------------------------
// obt@̒~
void SoundBuffer::stopBuffer(){
	if(DirectXFailed(soundBuffer_->Stop())){
		ErrorOut("SoundBuffer::stopBuffer() TEh̒~Ɏs܂B");
	}
}
//------------------------------------------------------------------------------
// Đʒu
//------------------------------------------------------------------------------
// Đʒuݒ
void SoundBuffer::setCursor(u_int cursor){
	Assert((cursor >= 0) && (cursor < getSize()));
	if(DirectXFailed(soundBuffer_->SetCurrentPosition(cursor))){
		ErrorOut("SoundBuffer::setCursor() Đʒu̐ݒɎs܂B");
	}
}
//------------------------------------------------------------------------------
// Đʒu擾
u_int SoundBuffer::getCursor() const{
	u_long result;
	if(DirectXFailed(soundBuffer_->GetCurrentPosition(&result, NULL))){
		ErrorOut("SoundBuffer::getCursor() Đʒu̎擾Ɏs܂B");
	}
	return result;
}
//------------------------------------------------------------------------------
// {[
//------------------------------------------------------------------------------
// {[̐ݒ
void SoundBuffer::setVolume(float volume){
	if(volume_ == volume){ return; }
	int db = volumeToDecibel(volume);
	if(DirectXFailed(soundBuffer_->SetVolume(db))){
		ErrorOut("SoundBuffer::setVolume() {[̐ݒɎs܂B");
	}
	volume_ = volume;
}
//------------------------------------------------------------------------------
// g
//------------------------------------------------------------------------------
// g̐ݒ
void SoundBuffer::setFrequency(int frequency){
	if(frequency_ == frequency){ return; }
	Assert(frequency >= DSBFREQUENCY_MIN);
	Assert(frequency <= 100000);
	if(DirectXFailed(soundBuffer_->SetFrequency(frequency))){
		ErrorOut("SoundBuffer::setFrequency() g̐ݒɎs܂B");
	}
	frequency_ = frequency;
}
//------------------------------------------------------------------------------
// IWig̐ݒ
void SoundBuffer::setOriginalFrequency(){
	if(DirectXFailed(soundBuffer_->SetFrequency(DSBFREQUENCY_ORIGINAL))){
		ErrorOut("SoundBuffer::resetFrequency() g̐ݒɎs܂B");
	}
	frequency_ = getBufferFrequency();
}
//------------------------------------------------------------------------------
// obt@g̎擾
int SoundBuffer::getBufferFrequency() const{
	u_long frequency;
	if(DirectXFailed(soundBuffer_->GetFrequency(&frequency))){
		ErrorOut("SoundBuffer::getBufferFrequency() "
			"obt@g̎擾Ɏs܂B");
	}
	Assert(frequency >= DSBFREQUENCY_MIN);
	Assert(frequency <= 100000);
	return frequency;
}
//------------------------------------------------------------------------------
// tF[h
//------------------------------------------------------------------------------
// tF[h
void SoundBuffer::fade(float millisecond, float startVolume, float endVolume){
	Assert(millisecond >= 0.f);
	if(isFading_){
		LampSound::getSoundManager()->removeUpdateSound(this);
		isFading_ = false;
	}
	if(millisecond <= Math::epsilon){ return; }
	fadeStartTime_ = Timer::getTick();
	fadePeriod_ = millisecond;
	fadeStartVolume_ = startVolume;
	fadeEndVolume_ = endVolume;
	isFading_ = true;
	setVolume(fadeStartVolume_);
	play();
	LampSound::getSoundManager()->addUpdateSound(this);
}
//------------------------------------------------------------------------------
// Abvf[g
bool SoundBuffer::update(){
	Assert(isFading());
	float rate = Timer::getInterval(fadeStartTime_) / fadePeriod_;
	// tF[hI
	if(rate >= 1.f){
		setVolume(fadeEndVolume_);
		if(fadeEndVolume_ <= Math::epsilon){ stop(); }
		isFading_ = false;
		return true;
	}
	float volume =
		(fadeEndVolume_ - fadeStartVolume_) * rate + fadeStartVolume_;
	setVolume(volume);
	return false;
}
//------------------------------------------------------------------------------
// bN
//------------------------------------------------------------------------------
// bN
SoundBuffer::Lock& SoundBuffer::lock(){
	Assert(!lock_.isValid());
	HRESULT result = soundBuffer_->Lock(0, 0,
		&lock_.address0_, &lock_.size0_, NULL, NULL, DSBLOCK_ENTIREBUFFER);
	if(DirectXSucceeded(result)){ return lock_; }
	// obt@XgȂ烊XgAĂ݂
	lock_.clear();
	if(result != DSERR_BUFFERLOST){
		ErrorOut("SoundBuffer::lock() obt@̃bNɎs܂B");
		return lock_;
	}
	result = soundBuffer_->Restore();
	if(DirectXFailed(result)){ return lock_; }
	// ēxbNs
	result = soundBuffer_->Lock(0, 0,
		&lock_.address0_, &lock_.size0_, NULL, NULL, DSBLOCK_ENTIREBUFFER);
	if(DirectXFailed(result)){ lock_.clear(); }
	return lock_;
}
//------------------------------------------------------------------------------
// bN
SoundBuffer::Lock& SoundBuffer::lock(u_int offset, u_int bytes){
	Assert(!lock_.isValid());
	HRESULT result = soundBuffer_->Lock(offset, bytes,
		&lock_.address0_, &lock_.size0_, &lock_.address1_, &lock_.size1_, 0);
	if(DirectXSucceeded(result)){ return lock_; }
	// obt@XgȂ烊XgAĂ݂
	lock_.clear();
	if(result != DSERR_BUFFERLOST){
		ErrorOut("SoundBuffer::lock() obt@̃bNɎs܂B");
		return lock_;
	}
	result = soundBuffer_->Restore();
	if(DirectXFailed(result)){ return lock_; }
	// ēxbNs
	result = soundBuffer_->Lock(0, 0,
		&lock_.address0_, &lock_.size0_, &lock_.address1_, &lock_.size1_, 0);
	if(DirectXFailed(result)){ lock_.clear(); }
	return lock_;
}
//------------------------------------------------------------------------------
// AbN
void SoundBuffer::unlock(){
	// LȃbNsĂȂΉȂ
	if(!lock_.isValid()){ return; }
	if(DirectXFailed(soundBuffer_->Unlock(
		lock_.address0_, lock_.size0_, lock_.address1_, lock_.size1_))){
		ErrorOut("SoundBuffer::unlock() obt@̃AbNɎs܂B");
	}
	lock_.clear();
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
