//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * Lamp͎
 * @author Junpee
 */

#include "LampBasic.h"
#include "Input/System/LampInput.h"
#include "Input/System/BufferedInput.h"
#include "Input/Keyboard/KeyboardKey.h"
#include "Input/Keyboard/Keyboard.h"
#include "Input/Keyboard/KeyboardDevice.h"
#include "Input/Mouse/Mouse.h"
#include "Input/Mouse/MouseDevice.h"
#include "Input/Joystick/Joystick.h"
#include "Input/Joystick/JoystickDevice.h"
#include "Core/InputOutput/BinaryFileWriter.h"
#include "Core/InputOutput/BinaryFileReader.h"

namespace Lamp{

// EBhEnh
HWND LampInput::windowHandle_ = NULL;
// DirectInput
DirectInput* LampInput::directInput_ = NULL;
// ̓[h
LampInput::InputMode LampInput::inputMode_ = LampInput::modePolling;
// obt@
BufferedInput* LampInput::bufferedInput_ = NULL;

// OC^
BinaryWriter* LampInput::logWriter_ = NULL;
// oCit@CC^
BinaryWriter* LampInput::binaryFileWriter_ = NULL;
// O[_
BinaryReader* LampInput::logReader_ = NULL;
// oCit@C[_
BinaryReader* LampInput::binaryFileReader_ = NULL;

// L[{[h
Keyboard* LampInput::keyboard_ = NULL;
// L[{[hfoCX
KeyboardDevice* LampInput::keyboardDevice_ = NULL;
// }EX
Mouse* LampInput::mouse_ = NULL;
// }EXfoCX
MouseDevice* LampInput::mouseDevice_ = NULL;
// WCXeBbN
ArrayList<Joystick*> LampInput::joysticks_;
// WCXeBbNfoCX
ArrayList<JoystickDevice*> LampInput::joystickDevices_;

// tO
bool LampInput::isInitialized_ = false;
// O擾tO
bool LampInput::isLogging_ = false;
// OĐtO
bool LampInput::isLogPlaying_ = false;

//------------------------------------------------------------------------------
// An
//------------------------------------------------------------------------------
// 
bool LampInput::initialize(
	HINSTANCE instanceHandle, HWND windowHandle, InputMode inputMode){
	if(isInitialized_){ return true; }
	Assert(inputMode_ == modePolling);
	windowHandle_ = windowHandle;
	LampCore::initialize();
	KeyboardKey::initializeKeyString();
	// DirectInput쐬
	if(DirectXFailed(DirectInputCreate(instanceHandle, DIRECTINPUT_VERSION,
		DirectInputInterfaceID, (void**)&directInput_, NULL))){
		ErrorOut("LampInput::initialize() "
			"DirectInput̍쐬Ɏs܂");
		return false;
	}

	// L[{[hfoCX̍쐬
	DirectInputDevice* device;
	if(DirectXFailed(directInput_->CreateDevice(
		GUID_SysKeyboard, &device, NULL))){
		ErrorOut("LampInput::initialize() "
			"L[{[hfoCX̏Ɏs܂");
		return false;
	}
	keyboardDevice_ = new KeyboardDevice();
	if(!keyboardDevice_->initialize(device, windowHandle_)){ return false; }
	keyboard_ = new Keyboard(keyboardDevice_);

	// }EXfoCX̍쐬
	if(DirectXFailed(directInput_->CreateDevice(
		GUID_SysMouse, &device, NULL))){
		ErrorOut("LampInput::initialize() "
			"}EXfoCX̏Ɏs܂");
		return false;
	}
	mouseDevice_ = new MouseDevice();
	if(!mouseDevice_->initialize(device, windowHandle_)){ return false; }
	mouse_ = new Mouse(mouseDevice_);

	// WCXeBbNfoCX̍쐬
	if(DirectXFailed(directInput_->EnumDevices(
		DirectInputDeviceClass_GameController, joystickEnumeration,
		NULL, DIEDFL_ATTACHEDONLY))){
		ErrorOut("LampInput::initialize() "
			"WCXeBbNfoCX̗񋓂Ɏs܂");
		return false;
	}
	// tOĂ
	isInitialized_ = true;
	setInputMode(inputMode);
	return true;
}
//------------------------------------------------------------------------------
// WCXeBbN̗񋓃R[obN
int __stdcall LampInput::joystickEnumeration(
	const DIDEVICEINSTANCE* instance, void* userData){
	// foCX쐬
	DirectInputDevice* device;
	if(DirectXFailed(directInput_->CreateDevice(
		instance->guidInstance, &device, NULL))){
		ErrorOut("LampInput::joystickEnumeration() "
			"WCXeBbNfoCX̍쐬Ɏs܂");
		return DIENUM_STOP;
	}
	// WCXeBbN
	JoystickDevice* joystickDevice = new JoystickDevice();
	if(!joystickDevice->initialize(device, windowHandle_)){
		ErrorOut("LampInput::joystickEnumeration() "
			"WCXeBbNfoCX̏Ɏs܂");
		return DIENUM_STOP;
	}
	joystickDevices_.add(joystickDevice);
	joysticks_.add(new Joystick(joystickDevice));
	return DIENUM_CONTINUE;
}
//------------------------------------------------------------------------------
// n
void LampInput::finalize(){
	// OĐ~
	if(isLogPlaying()){ stopLog(); }
	// O擾~
	if(isLogging()){ endLogging(); }
	// obt@͂̌n
	if(inputMode_ == modeBuffering){ finalizeBufferInput(); }
	// foCX
	for(int i = joysticks_.getCount() - 1; i >= 0; i--){ delete joysticks_[i]; }
	joysticks_.clear();
	for(int i = joystickDevices_.getCount() - 1; i >= 0; i--){
		delete joystickDevices_[i];
	}
	joystickDevices_.clear();
	SafeDelete(mouse_);
	SafeDelete(mouseDevice_);
	SafeDelete(keyboard_);
	SafeDelete(keyboardDevice_);
	// DirectInput
	SafeRelease(directInput_);
	// tONA
	isInitialized_ = false;
}
//------------------------------------------------------------------------------
// NA
void LampInput::clear(){
	Assert(isInitialized_);
	keyboard_->clear();
	mouse_->clear();
	int joystickCount = joysticks_.getCount();
	for(int i = 0; i < joystickCount; i++){ joysticks_[i]->clear(); }
}
//------------------------------------------------------------------------------
// obt@NA
void LampInput::bufferClear(){
	Assert(isInitialized_);
	Assert(inputMode_ == modeBuffering);
	while(bufferedInput_->hasMoreInput()){ bufferedInput_->nextInput(); }
	// obt@NAĂ猻݂̓͂NA
	clear();
}
//------------------------------------------------------------------------------
// ̓[h
//------------------------------------------------------------------------------
// ̓[h̐ݒ
void LampInput::setInputMode(InputMode inputMode){
	Assert(isInitialized_);
	Assert(!isLogPlaying());
	Assert(!isLogging());
	if(inputMode_ == inputMode){ return; }
	// obt@͂̌n
	if(inputMode_ == modeBuffering){ finalizeBufferInput(); }
	inputMode_ = inputMode;
	// obt@͂̏
	if(inputMode_ == modeBuffering){ initializeBufferInput(); }
}
//------------------------------------------------------------------------------
// |[O
//------------------------------------------------------------------------------
// |[O
bool LampInput::polling(){
	Assert(isInitialized_);
	Assert(inputMode_ == modePolling);
	bool result = true;
	if(!isLogPlaying_){
		// foCX|[O
		if(!devicePolling()){ result = false; }
	}else{
		// O|[O
		logPolling();
	}
	// Ȍo
	writeLog();
	return result;
}
//------------------------------------------------------------------------------
// foCX|[O
bool LampInput::devicePolling(){
	bool result = true;
	// L[{[h̃|[O
	if(!keyboardDevice_->polling()){ result = false; }
	keyboard_->setNextState(keyboardDevice_->getKeyboardState());

	// }EX̃|[O
	if(!mouseDevice_->polling()){ result = false; }
	mouse_->setNextState(mouseDevice_->getMouseState());

	// WCXeBbÑ|[O
	int joystickDeviceCount = getJoystickDeviceCount();
	for(int i = 0; i < joystickDeviceCount; i++){
		JoystickDevice* joystickDevice = getJoystickDevice(i);
		if(!joystickDevice->polling()){ result = false; }
		getJoystick(i)->setNextState(joystickDevice->getJoystickState());
	}
	return result;
}
//------------------------------------------------------------------------------
// O|[O
void LampInput::logPolling(){
	// L[{[h̃Oǂݍ
	KeyboardState keyboardState;
	keyboardState.readBinary(logReader_);
	keyboard_->setNextState(keyboardState);

	// }EX̃Oǂݍ
	MouseState mouseState;
	mouseState.readBinary(logReader_);
	mouse_->setNextState(mouseState);

	// WCXeBbÑOǂݍ
	int joystickCount = getJoystickCount();
	for(int i = 0; i < joystickCount; i++){
		JoystickState joystickState;
		joystickState.readBinary(logReader_);
		getJoystick(i)->setNextState(joystickState);
	}

	// OI[`FbN
	if(logReader_->isEnd()){ stopLog(); }
}
//------------------------------------------------------------------------------
// Ȍo
void LampInput::writeLog(){
	if(!isLogging()){ return; }
	keyboard_->getState().writeBinary(logWriter_);
	mouse_->getState().writeBinary(logWriter_);
	int joystickCount = getJoystickCount();
	for(int i = 0; i < joystickCount; i++){
		getJoystick(i)->getState().writeBinary(logWriter_);
	}
}
//------------------------------------------------------------------------------
// obt@
//------------------------------------------------------------------------------
// obt@͏
void LampInput::initializeBufferInput(){
	Assert(bufferedInput_ == NULL);
	// obt@͏
	bufferedInput_ = new BufferedInput();
	bufferedInput_->setKeyboard(keyboard_, keyboardDevice_);
	bufferedInput_->setMouse(mouse_, mouseDevice_);
	int joystickCount = joysticks_.getCount();
	for(int i = 0; i < joystickCount; i++){
		bufferedInput_->addJoystick(joysticks_[i], joystickDevices_[i]);
	}
	// obt@͊Jn
	bufferedInput_->start();
}
//------------------------------------------------------------------------------
// obt@͌n
void LampInput::finalizeBufferInput(){
	Assert(bufferedInput_ != NULL);
	// obt@͏I
	bufferedInput_->stop();
	// obt@͌n
	SafeDelete(bufferedInput_);
}
//------------------------------------------------------------------------------
// ͂邩
bool LampInput::hasMoreInput(){
	Assert(isInitialized_);
	Assert(inputMode_ == modeBuffering);
	return bufferedInput_->hasMoreInput();
}
//------------------------------------------------------------------------------
// ͂҂
void LampInput::waitForInput(){
	while(true){
		if(hasMoreInput()){ return; }
		Thread::sleep(1);
	}
}
//------------------------------------------------------------------------------
// ̓
void LampInput::nextInput(){
	Assert(isInitialized_);
	Assert(inputMode_ == modeBuffering);
	bufferedInput_->nextInput();
}
//------------------------------------------------------------------------------
// ͐̎擾
int LampInput::getInputCount(){
	Assert(isInitialized_);
	Assert(inputMode_ == modeBuffering);
	return bufferedInput_->getInputCount();
}
//------------------------------------------------------------------------------
// O擾
//------------------------------------------------------------------------------
// O擾̊Jn
void LampInput::startLogging(const String& filePath){
	Assert(binaryFileWriter_ == NULL);
	binaryFileWriter_ = new BinaryFileWriter(filePath);
	startLogging(binaryFileWriter_);
}
//------------------------------------------------------------------------------
// O擾̊Jn
void LampInput::startLogging(BinaryWriter* binaryWriter){
	Assert(!isLogging_);
	Assert(logWriter_ == NULL);
	logWriter_ = binaryWriter;
	// wb_̏o
	logWriter_->writeString("LampInputLog");
	logWriter_->writeInt(getJoystickCount());
	InputMode inputMode = getInputMode();
	logWriter_->writeInt(inputMode);
	if(inputMode == modeBuffering){ bufferedInput_->startLogging(logWriter_); }
	isLogging_ = true;
}
//------------------------------------------------------------------------------
// O擾̏I
void LampInput::endLogging(){
	Assert(isLogging_);
	Assert(logWriter_ != NULL);
	isLogging_ = false;
	if(getInputMode() == modeBuffering){ bufferedInput_->endLogging(); }
	logWriter_->flush();
	logWriter_ = NULL;
	SafeDelete(binaryFileWriter_);
}
//------------------------------------------------------------------------------
// OĐ
//------------------------------------------------------------------------------
// OĐ̊Jn
void LampInput::playLog(const String& filePath){
	Assert(binaryFileReader_ == NULL);
	binaryFileReader_ = new BinaryFileReader(filePath);
	playLog(binaryFileReader_);
}
//------------------------------------------------------------------------------
// OĐ̊Jn
void LampInput::playLog(BinaryReader* binaryReader){
	Assert(!isLogPlaying_);
	Assert(logReader_ == NULL);
	logReader_ = binaryReader;
	// wb_̓ǂݍ
	String headerString = logReader_->readString();
	if(!headerString.equals("LampInputLog")){
		ErrorOut("LampInput::playLog LampInputLog`̃t@Cł͂܂");
	}
	int joystickCount = logReader_->readInt();
	if(joystickCount != getJoystickCount()){
		ErrorOut("LampInput::playLog WCXeBbN̐Ⴂ܂");
	}
	// t[ɓ̓[ho͂ΓI؂ւɂΉ\
	// LampInputgpR[hpollingnextInputgKv
	InputMode inputMode = (InputMode)logReader_->readInt();
	setInputMode(inputMode);
	if(inputMode == modeBuffering){ bufferedInput_->playLog(binaryReader); }
	isLogPlaying_ = true;
}
//------------------------------------------------------------------------------
// OĐ̒~
void LampInput::stopLog(){
	Assert(isLogPlaying_);
	Assert(logReader_ != NULL);
	isLogPlaying_ = false;
	if(getInputMode() == modeBuffering){ bufferedInput_->stopLog(); }
	logReader_ = NULL;
	SafeDelete(binaryFileReader_);
}
//------------------------------------------------------------------------------
// EBhEvV[W
LRESULT LampInput::windowProcedure(
	HWND windowHandle, u_int message, WPARAM wParam, LPARAM lParam){
/*
	switch(message){
	// EBhẼANeBuAANeBuɕύX
	case WM_ACTIVATEAPP:
bool minimized = ((int)HIWORD(wParam) != 0);
//		if(mouse_->isExclusive()){
			if((LOWORD(wParam) != WA_INACTIVE)){
				// EBhEANeBuɂȂ̂ŃJ[\
				while(::ShowCursor(false) >= 0){}
			}else{
				// EBhEANeBułȂȂ̂ŃJ[\
				while(::ShowCursor(true) < 0){}
			}
//		}
		break;
	}
*/
	return 0;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
