//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * XvCg
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics2D/Sprite/Sprite.h"
#include "Graphics2D/Renderer/SpriteRenderState.h"
#include "Graphics2D/Sprite/SpritePictureRGBA8.h"
#include "Graphics2D/Sprite/SpritePictureRGB8.h"
#include "Core/InputOutput/FilePath.h"
#include "Core/Codec/Tga/TargaLoader.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
Sprite::Sprite() : SpriteRequest(), picture_(NULL),
	rectangle_(RectangleF::zero), imageRectangle_(RectangleF::unit),
	align_(alignNone), fit_(fitNone), enabled_(true){
}
//------------------------------------------------------------------------------
// Rs[RXgN^
Sprite::Sprite(const Sprite& copy) : SpriteRequest(copy), picture_(NULL),
	rectangle_(copy.rectangle_), imageRectangle_(copy.imageRectangle_),
	align_(copy.align_), fit_(copy.fit_), enabled_(copy.enabled_){
	// sN`̃Rs[
	setPicture(copy.picture_);
}
//------------------------------------------------------------------------------
// Rs[
Sprite& Sprite::operator =(const Sprite& copy){
	// gȂ烊^[
	if(this == &copy){ return *this; }
	// eNX̃Rs[
	SpriteRequest::operator =(copy);
	// sN`̃Rs[
	setPicture(copy.picture_);
	// oRs[
	rectangle_ = copy.rectangle_;
	imageRectangle_ = copy.imageRectangle_;
	align_ = copy.align_;
	fit_ = copy.fit_;
	enabled_ = copy.enabled_;
	return *this;
}
//------------------------------------------------------------------------------
// fXgN^
Sprite::~Sprite(){
	// sN`
	setPicture(NULL);
}
//------------------------------------------------------------------------------
// _O
//------------------------------------------------------------------------------
// _O
void Sprite::render(SpriteRenderState* renderState){
	// `FbN
	if(!enabled_){ return; }

	// `̎Zo
	RectangleF rectangle(rectangle_);
	const DimensionF& renderTargetSize = renderState->getRenderTargetSize();
	// tBbg̓Kp
	if(fit_ != fitNone){
		rectangle = applyFit(rectangle, renderTargetSize);
	}
	// AC̓Kp
	if(align_ != alignNone){
		rectangle = applyAlign(rectangle, renderTargetSize);
	}

	// ʒu̎Zo
	Point2f minPosition(rectangle.x, rectangle.y);
	Point2f maxPosition(rectangle.x + rectangle.width,
		rectangle.y + rectangle.height);

	// UV̎Zo
	TexCoord2 minUV(imageRectangle_.x, imageRectangle_.y);
	TexCoord2 maxUV(imageRectangle_.x + imageRectangle_.width,
		imageRectangle_.y + imageRectangle_.height);

	// sNZ炵
	// sNZoɂUV͈́AXvCgTCYKv
	// ~bv}bv̏ꍇAeNX`TCY͊֌WȂ
	TexCoord2 halfPixel(
		(imageRectangle_.width) * 0.5f / rectangle.width,
		(imageRectangle_.height) * 0.5f / rectangle.height);
	minUV += halfPixel;
	maxUV += halfPixel;

	// `
	renderState->request(picture_, minPosition, maxPosition, minUV, maxUV);
}
//------------------------------------------------------------------------------
// tBbg̓Kp
RectangleF Sprite::applyFit(const RectangleF& rectangle,
	const DimensionF& renderTargetSize){
	RectangleF result(rectangle);
	if((fit_ == fitScreen) || (fit_ == fitScreenWidth)){
		result.width = renderTargetSize.width;
	}
	if((fit_ == fitScreen) || (fit_ == fitScreenHeight)){
		result.height = renderTargetSize.height;
	}
// 摜̃AXyNg̕ێȂǂ
	return result;
}
//------------------------------------------------------------------------------
// AC̓Kp
RectangleF Sprite::applyAlign(const RectangleF& rectangle,
	const DimensionF& renderTargetSize){
	RectangleF result(rectangle);
	// XAC
	if((align_ == alignTop) || (align_ == alignCenter) ||
		(align_ == alignBottom)){
		result.x += (renderTargetSize.width - rectangle.width) * 0.5f;
	}else if((align_ == alignTopRight) || (align_ == alignRight) ||
		(align_ == alignBottomRight)){
		result.x += renderTargetSize.width - rectangle.width;
	}

	// YAC
	if((align_ == alignLeft) || (align_ == alignCenter) ||
		(align_ == alignRight)){
		result.y += (renderTargetSize.height - rectangle.height) * 0.5f;
	}else if((align_ == alignBottomLeft) || (align_ == alignBottom) ||
		(align_ == alignBottomRight)){
		result.y += renderTargetSize.height - rectangle.height;
	}
	return result;
}
//------------------------------------------------------------------------------
// sN`
//------------------------------------------------------------------------------
// sN`̃[h
bool Sprite::loadPicture(const String& fileName){
	// sN`̃[h
	SpritePicture* picture;
	// t@C̗L`FbN
	FilePath filePath(fileName);
	if(!filePath.existFile()){ return false; }
	// gq`FbN
	String extension = filePath.getExtension();
	if(extension == "tga"){
		TargaLoader loader(filePath.getPath());
		loader.loadHeader();
		if(loader.hasAlpha()){
			SpritePictureRGBA8* pictureRGBA8 = new SpritePictureRGBA8();
			pictureRGBA8->setSize(loader.getSize());
			loader.loadImage(pictureRGBA8->getImageBuffer());
			picture = pictureRGBA8;
		}else{
			SpritePictureRGB8* pictureRGB8 = new SpritePictureRGB8();
			pictureRGB8->setSize(loader.getSize());
			loader.loadImage(pictureRGB8->getImageBuffer());
			picture = pictureRGB8;
		}
	}else{
		return false;
	}
	// sN`̐ݒ
	setPicture(picture);
	return true;
}
//------------------------------------------------------------------------------
// sN`̐ݒ
void Sprite::setPicture(SpritePicture* picture){
	// sN`
	if(picture_ != NULL){
		if(picture_->removeReference() == 0){ SafeDelete(picture_); }
	}
	// sN`ݒ
	picture_ = picture;
	if(picture != NULL){ picture_->addReference(); }
}
//------------------------------------------------------------------------------
// C[W`
//------------------------------------------------------------------------------
// Aj[V̐ݒ
void Sprite::setAnimation(const DimensionI& animationDivision,
	int animation, const RectangleF& imageRectangle){
	int animationCount = animationDivision.width * animationDivision.height;
	Assert((animation >= 0) && (animation < animationCount));
	int xIndex = animation % animationDivision.width;
	int yIndex = animation / animationDivision.width;
	float xScale = imageRectangle.width / animationDivision.width;
	float yScale = imageRectangle.height / animationDivision.height;

	RectangleF result;
	result.x = imageRectangle.x + xIndex * xScale;
	result.y = imageRectangle.y + yIndex * yScale;
	result.width = xScale;
	result.height = yScale;
	setImageRectangle(result);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
