//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * XvCg_
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics2D/Renderer/SpriteRenderer.h"
#include "Graphics2D/Renderer/SpriteGraphicsBuffer.h"
#include "Graphics2D/Renderer/SpriteRenderState.h"
#include "Graphics2D/Renderer/SpriteRequest.h"
#include "Graphics/Renderer/RenderingDevice.h"

#include "Graphics2D/SpriteState/ColorSpriteState.h"
#include "Graphics2D/SpriteState/AddressModeSpriteState.h"
#include "Graphics2D/SpriteState/BlendSpriteState.h"
#include "Graphics2D/SpriteState/ViewportSpriteState.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
SpriteRenderer::SpriteRenderer(int maxPriority, int defaultPriority) :
	maxPriority_(maxPriority){
	Assert(maxPriority_ > 0);
	setDefaultPriority(defaultPriority);
	requests_ = new ArrayList<SpriteRequest*>[maxPriority_];
	graphicsBuffer_ = new SpriteGraphicsBuffer();
}
//------------------------------------------------------------------------------
// fXgN^
SpriteRenderer::~SpriteRenderer(){
	SafeDelete(graphicsBuffer_);
	SafeArrayDelete(requests_);
}
//------------------------------------------------------------------------------
// NGXg
//------------------------------------------------------------------------------
// NGXg
void SpriteRenderer::request(SpriteRequest* spriteRequest, int priority){
	Assert((priority >= 0) && (priority < maxPriority_));
	requests_[priority].add(spriteRequest);
}
//------------------------------------------------------------------------------
// ftHgXe[g̃NGXg
void SpriteRenderer::requestDefaultState(int priority){
	request(&ColorSpriteState::defaultState, priority);
	request(&AddressModeSpriteState::defaultState, priority);
	request(&BlendSpriteState::defaultState, priority);
	request(&ViewportSpriteState::defaultState, priority);
}
//------------------------------------------------------------------------------
// ftHgXe[g̓Kp
void SpriteRenderer::applyDefaultState(SpriteRenderState* renderState){
	((ColorSpriteState*)(&ColorSpriteState::defaultState))->
		render(renderState);
	((AddressModeSpriteState*)(&AddressModeSpriteState::defaultState))->
		render(renderState);
	((BlendSpriteState*)(&BlendSpriteState::defaultState))->
		render(renderState);
	((ViewportSpriteState*)(&ViewportSpriteState::defaultState))->
		render(renderState);
}
//------------------------------------------------------------------------------
// _O
//------------------------------------------------------------------------------
// _O
void SpriteRenderer::render(int startPriority, int endPriority){
	Assert(startPriority <= endPriority);
	Assert((startPriority >= 0) && (endPriority < maxPriority_));
	RenderingDevice* device = RenderingDevice::getInstance();
	SpriteRenderState renderState(graphicsBuffer_);

	// ftHgXe[gubN̓Kp
	device->applyDefaultStateBlock();

	// OtBbNXobt@̃ZbgAbv
	graphicsBuffer_->setup();

	// _Xe[gKp
	applyRendererState();

	// XvCgftHgXe[g̓Kp
	applyDefaultState(&renderState);

	// `Jn
	if(device->beginScene()){
		// endPriority܂ރ[v
		for(int i = startPriority; i <= endPriority; i++){
			// _O
			int count = requests_[i].getCount();
			for(int j = 0; j < count; j++){
				SpriteRequest* request = requests_[i].get(j);
				request->render(&renderState);
			}
			// NGXgNA
			requests_[i].clear();
		}
		// `n
		graphicsBuffer_->render();
		device->endScene();
	}

	// ftHgXe[gubN̓Kp
	device->applyDefaultStateBlock();
}
//------------------------------------------------------------------------------
// _Xe[g̓Kp
void SpriteRenderer::applyRendererState(){
	// _Oɕsς̃Xe[gݒ肷
	RenderingDevice* device = RenderingDevice::getInstance();

	//--------------------------------------------------------------------------
	// ÓIݒ
	//--------------------------------------------------------------------------
	// CeBO
	device->setRenderState(D3DRS_LIGHTING, false);
	// JO
	device->setRenderState(D3DRS_CULLMODE, D3DCULL_NONE);
	// Zݖ
	device->setRenderState(D3DRS_ZWRITEENABLE, false);
	// ZeXg
	device->setZTest(false);

	// J[Xe[W̐ݒ
	device->setTextureState(0, D3DTSS_COLOROP, D3DTOP_MODULATE4X);
	device->setTextureState(0, D3DTSS_COLORARG1, D3DTA_TFACTOR);
	device->setTextureState(0, D3DTSS_COLORARG2, D3DTA_TEXTURE);
	// At@Xe[W̐ݒ
	device->setTextureState(0, D3DTSS_ALPHAOP, D3DTOP_MODULATE);
	device->setTextureState(0, D3DTSS_ALPHAARG1, D3DTA_TFACTOR);
	device->setTextureState(0, D3DTSS_ALPHAARG2, D3DTA_TEXTURE);

	// eNX`TvɃ~bv}bvƐ`Ԃݒ肷
	device->setSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR);
	device->setSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_LINEAR);
	device->setSamplerState(0, D3DSAMP_MIPFILTER, D3DTEXF_LINEAR);

	// At@eXg̐ݒ
	device->setRenderState(D3DRS_ALPHATESTENABLE, true);
	device->setRenderState(D3DRS_ALPHAREF, 1);
	device->setRenderState(D3DRS_ALPHAFUNC, D3DCMP_GREATEREQUAL);

	//--------------------------------------------------------------------------
	// Iݒ
	//--------------------------------------------------------------------------
	// _[^[QbgTCY擾
	DimensionF renderTargetSize;
	renderTargetSize.set(device->getRenderTargetSize());

	// ˉes
	Matrix44 projectionMatrix;
	float nearZ = 1.f;
	float farZ = 10001.f;
	projectionMatrix.set(
		2.f / renderTargetSize.width, 0.f, 0.f, 0.f,
		0.f, 2.f / renderTargetSize.height, 0.f, 0.f,
		0.f, 0.f, 1.f / (nearZ - farZ), nearZ / (nearZ - farZ),
		0.f, 0.f, 0.f, 1.f);
	device->setProjectionMatrix(projectionMatrix);

	// r[s
	Matrix44 viewMatrix;
	Vector3 viewPosition(renderTargetSize.width * 0.5f,
		renderTargetSize.height * 0.5f, (farZ - nearZ) * 0.5f + nearZ);
	viewMatrix.setTranslation(-viewPosition);
	viewMatrix.addScale(Vector3(1.f, -1.f, 1.f));
	device->setViewMatrix(viewMatrix);

}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
