//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * V[
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/Scene/Scene.h"
#include "Core/Renamer/CountRenamer.h"
#include "Graphics/Camera/CameraManager.h"
#include "Graphics/Fog/Fog.h"
#include "Graphics/SceneNode/SceneNodeManager.h"
#include "Graphics/Light/LightManager.h"
#include "Graphics/Model/ModelManager.h"
#include "Graphics/Mesh/MeshManager.h"
#include "Graphics/MeshData/MeshDataManager.h"
#include "Graphics/Material/MaterialManager.h"
#include "Graphics/Texture/TextureManager.h"
#include "Graphics/Picture/PictureManager.h"
#include "Graphics/Renderer/DrawRequest.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
Scene::Scene(const String& name) :
	name_(name), currentCamera_(NULL), tick_(0){
	// l[}̏
	renamer_ = new CountRenamer();
	// e}l[W
	cameraManager_ = new CameraManager(this);
	sceneNodeManager_ = new SceneNodeManager(this);
	lightManager_ = new LightManager(this);
	modelManager_ = new ModelManager(this);
	meshManager_ = new MeshManager(this);
	meshDataManager_ = new MeshDataManager(this);
	materialManager_ = new MaterialManager(this);
	textureManager_ = new TextureManager(this);
	pictureManager_ = new PictureManager(this);
	// [gm[h
	rootNode_ = sceneNodeManager_->createSceneNode("RootNode");
	// tHO
	fog_ = new Fog();
}
//------------------------------------------------------------------------------
// fXgN^
Scene::~Scene(){
	// tHǑn
	SafeDelete(fog_);
	// [gm[ȟn
	sceneNodeManager_->destroy(rootNode_);
	// e}l[W̌n
	SafeDelete(pictureManager_);
	SafeDelete(textureManager_);
	SafeDelete(materialManager_);
	SafeDelete(meshDataManager_);
	SafeDelete(meshManager_);
	SafeDelete(modelManager_);
	SafeDelete(lightManager_);
	SafeDelete(sceneNodeManager_);
	SafeDelete(cameraManager_);
	// l[}̌n
	SafeDelete(renamer_);
}
//------------------------------------------------------------------------------
// NA
int Scene::clear(){
	int result = 0;
	result += pictureManager_->clear();
	result += textureManager_->clear();
	result += materialManager_->clear();
	result += meshDataManager_->clear();
	result += meshManager_->clear();
	// RootNodec߁ASceneLeafSceneNodeNAKv
	result += sceneNodeManager_->clear();
	result += modelManager_->clear();
	result += lightManager_->clear();
	result += cameraManager_->clear();
	return result;
}
//------------------------------------------------------------------------------
// 
void Scene::traverse(){
	// `bÑCNg
	tick_++;
	rootNode_->traverse();
}
//------------------------------------------------------------------------------
// bVXg̎擾
void Scene::getMeshList(ArrayList<Mesh*>* meshList, Camera* camera){
	// V[ǗVXeɂ鍂Ȋ荞݂ɓIɐ؂ւ悤ɂ
	int meshCount = meshManager_->getCount();
	for(int i = 0; i < meshCount; i++){
		Mesh* mesh = meshManager_->get(i);
		if(!mesh->isGlobalEnabled()){ continue; }
		// }eAAbVf[^ĂȂbV͋֎~
		Assert(mesh->getMaterial() != NULL);
		Assert(mesh->getMeshData() != NULL);
		// XtBA`FbNɂ茩ȂbV͂
		Clipping::State clippingState =
			camera->clipping(mesh->getWorldBoundingSphere());
		if(clippingState == Clipping::invisible){ continue; }
		// Ăꍇ̓{bNX`FbNs
		if(clippingState == Clipping::intersect){
			clippingState = camera->clipping(mesh->getWorldBoundingBox());
			if(clippingState == Clipping::invisible){ continue; }
		}
		// Xgɒǉ
		meshList->add(mesh);
	}
}
//------------------------------------------------------------------------------
// [JCgXg̎擾
void Scene::getLocalLightList(Mesh* mesh, DrawRequest* drawRequest){
	// V[ǗVXeɂ鍂Ȋ荞݂ɓIɐ؂ւ悤ɂ
	const Sphere& meshSphere = mesh->getWorldBoundingSphere();
	const AxisAlignedBox& meshBox = mesh->getWorldBoundingBox();
	u_int lightMask = drawRequest->getMaterial()->getLightMask();
	int lightCount = lightManager_->getCount();
	for(int i = 0; i < lightCount; i++){
		Light* light = lightManager_->get(i);
		if(!light->isLocalLight()){ continue; }
		if(!light->isGlobalEnabled()){ continue; }
		// Cg}XN
		if((light->getLightMask() & lightMask) == 0){ continue; }
		if(light->isPointLight()){
			// |CgCg
			PointLight* pointLight = light->castPointLight();
			// XtBA`FbN
			float limitSquaredDistance =
				pointLight->getRange() + meshSphere.getRadius();
			limitSquaredDistance *= limitSquaredDistance;
			float checkValue = limitSquaredDistance -
				(pointLight->getWorldPosition() -
				meshSphere.getCenter()).getSquaredLength();
			if(checkValue <= 0.f){ continue; }
			// {bNX`FbN
			Sphere lightSphere(
				pointLight->getWorldPosition(), pointLight->getRange());
			if(!meshBox.intersect(lightSphere)){ continue; }
			// `惊NGXgɒǉ
			drawRequest->addLocalLight(pointLight);
		}else{
			ErrorOut("Scene::getLocalLightList() T|[g̃Cgł");
		}
	}
}
//------------------------------------------------------------------------------
// l[}̐ݒ
void Scene::setRenamer(Renamer* renamer){
	Assert(renamer != NULL);
	SafeDelete(renamer_);
	renamer_ = renamer;
}
//------------------------------------------------------------------------------
// foCXIuWFNg̏
bool Scene::initializeGraphicsDeviceObjects(){
	if(!meshManager_->initializeGraphicsDeviceObjects()){ return false; }
	if(!meshDataManager_->initializeGraphicsDeviceObjects()){ return false; }
	if(!materialManager_->initializeGraphicsDeviceObjects()){ return false; }
	if(!pictureManager_->initializeGraphicsDeviceObjects()){ return false; }
	return true;
}
//------------------------------------------------------------------------------
// foCXIuWFNg̍폜
void Scene::deleteGraphicsDeviceObjects(){
	pictureManager_->deleteGraphicsDeviceObjects();
	materialManager_->deleteGraphicsDeviceObjects();
	meshDataManager_->deleteGraphicsDeviceObjects();
	meshManager_->deleteGraphicsDeviceObjects();
}
//------------------------------------------------------------------------------
// foCXIuWFNg̃XgA
bool Scene::restoreGraphicsDeviceObjects(){
	if(!meshManager_->restoreGraphicsDeviceObjects()){ return false; }
	if(!meshDataManager_->restoreGraphicsDeviceObjects()){ return false; }
	if(!materialManager_->restoreGraphicsDeviceObjects()){ return false; }
	if(!pictureManager_->restoreGraphicsDeviceObjects()){ return false; }
	return true;
}
//------------------------------------------------------------------------------
// foCXIuWFNg̖
void Scene::invalidateGraphicsDeviceObjects(){
	pictureManager_->invalidateGraphicsDeviceObjects();
	materialManager_->invalidateGraphicsDeviceObjects();
	meshDataManager_->invalidateGraphicsDeviceObjects();
	meshManager_->invalidateGraphicsDeviceObjects();
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
