//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * _OfoCX
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/Renderer/RenderingDevice.h"
#include "Graphics/Renderer/Renderer.h"
#include "Graphics/System/GraphicsDeviceCapacity.h"
#include "Graphics/Fog/Fog.h"
#include "Graphics/Light/DirectionalLight.h"
#include "Graphics/Light/PointLight.h"
#include "Graphics/Material/Material.h"
#include "Graphics/Texture/Texture.h"

namespace Lamp{

// CX^X
RenderingDevice* RenderingDevice::instance_ = NULL;
// őANeBuCg
const int RenderingDevice::maxActiveLightCount_ =
	Renderer::maxActiveLightCount_;

//------------------------------------------------------------------------------
// VXe
//------------------------------------------------------------------------------
// RXgN^
RenderingDevice::RenderingDevice(GraphicsDeviceCapacity* capacity) :
	capacity_(capacity), device_(NULL), defaultStateBlock_(NULL){
	Assert(instance_ == NULL);
	instance_ = this;
	resetCounter();
}
//------------------------------------------------------------------------------
// fXgN^
RenderingDevice::~RenderingDevice(){
	// ftHgXe[gubN
	invalidateDefaultStateBlock();
	Assert(instance_ == this);
	instance_ = NULL;
}
//------------------------------------------------------------------------------
// Xe[gݒ
//------------------------------------------------------------------------------
// _[Xe[g̐ݒ
void RenderingDevice::setRenderState(D3DRENDERSTATETYPE type, u_int state){
	DirectXCheck(device_->SetRenderState(type, state));
}
//------------------------------------------------------------------------------
// eNX`Xe[g̐ݒ
void RenderingDevice::setTextureState(
	u_int stage, D3DTEXTURESTAGESTATETYPE type, u_int state){
	DirectXCheck(device_->SetTextureStageState(stage, type, state));
}
//------------------------------------------------------------------------------
// TvXe[g̐ݒ
void RenderingDevice::setSamplerState(
	u_int stage, D3DSAMPLERSTATETYPE type, u_int state){
	DirectXCheck(device_->SetSamplerState(stage, type, state));
}
//------------------------------------------------------------------------------
// Xe[gubN֘A
//------------------------------------------------------------------------------
// Xe[gubN̊Jn
void RenderingDevice::beginStateBlock(){
	DirectXCheck(device_->BeginStateBlock());
}
//------------------------------------------------------------------------------
// Xe[gubN̏I
Direct3DStateBlock* RenderingDevice::endStateBlock(){
	Direct3DStateBlock* stateBlock;
	DirectXCheck(device_->EndStateBlock(&stateBlock));
	return stateBlock;
}
//------------------------------------------------------------------------------
// Xe[gubN̓Kp
void RenderingDevice::applyStateBlock(Direct3DStateBlock* stateBlock){
	Assert(stateBlock != NULL);
	DirectXCheck(stateBlock->Apply());
}
//------------------------------------------------------------------------------
// ftHgXe[gubÑXgA
void RenderingDevice::restoreDefaultStateBlock(Direct3DDevice* direct3DDevice){
	device_ = direct3DDevice;
	// ftHgXe[gubN̍\z
	DirectXCheck(device_->CreateStateBlock(D3DSBT_ALL, &defaultStateBlock_));
}
//------------------------------------------------------------------------------
// _[^[Qbg
//------------------------------------------------------------------------------
// _[^[QbgTCY̎擾
DimensionI RenderingDevice::getRenderTargetSize(){
	Direct3DSurface* surface;
	DirectXCheck(device_->GetRenderTarget(0, &surface));
	D3DSurfaceDescription description;
	DirectXCheck(surface->GetDesc(&description));
	DimensionI result(description.Width, description.Height);
	SafeRelease(surface);
	return result;
}
//------------------------------------------------------------------------------
// V[֘A
//------------------------------------------------------------------------------
// V[̊Jn
bool RenderingDevice::beginScene(){
	// V[Jn
	if(DirectXFailed(device_->BeginScene())){ return false; }
	return true;
}
//------------------------------------------------------------------------------
// V[̏I
void RenderingDevice::endScene(){
	DirectXCheck(device_->EndScene());
}
//------------------------------------------------------------------------------
// s֘A
//------------------------------------------------------------------------------
// es̐ݒ
void RenderingDevice::setProjectionMatrix(const Matrix44& projectionMatrix){
	Matrix44 transposeMatrix = projectionMatrix;
	transposeMatrix.transpose();
	DirectXCheck(device_->SetTransform(
		D3DTS_PROJECTION, (const D3DMATRIX*)transposeMatrix.array));
}
//------------------------------------------------------------------------------
// r[s̐ݒ
void RenderingDevice::setViewMatrix(const Matrix44& viewMatrix){
	Matrix44 transposeMatrix = viewMatrix;
	transposeMatrix.transpose();
	DirectXCheck(device_->SetTransform(
		D3DTS_VIEW, (const D3DMATRIX*)transposeMatrix.array));
}
//------------------------------------------------------------------------------
// [hs̐ݒ
void RenderingDevice::setWorldMatrix(const Matrix34& worldMatrix){
	Matrix44 worldMatrix44(
		worldMatrix.m00, worldMatrix.m10, worldMatrix.m20, 0.f,
		worldMatrix.m01, worldMatrix.m11, worldMatrix.m21, 0.f,
		worldMatrix.m02, worldMatrix.m12, worldMatrix.m22, 0.f,
		worldMatrix.m03, worldMatrix.m13, worldMatrix.m23, 1.f);
	DirectXCheck(device_->SetTransform(
		D3DTS_WORLD, (D3DMATRIX*)worldMatrix44.array));
}
//------------------------------------------------------------------------------
// eNX`gXtH[ݒ
void RenderingDevice::setTextureTransform2(u_int stage,
	const TexCoord2& repeat, const TexCoord2& offset){
	if((repeat == TexCoord2(1.f, 1.f)) &&
		(offset == TexCoord2(0.f, 0.f))){
		setTextureState(stage, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_DISABLE);
		return;
	}
	Matrix44 textureMatrix(
		repeat.u, 0.f, 0.f, 0.f,
		0.f, repeat.v, 0.f, 0.f,
		offset.u, offset.v, 1.f, 0.f,
		0.f, 0.f, 0.f, 1.f);
	DirectXCheck(device_->SetTransform(
		(D3DTRANSFORMSTATETYPE)(D3DTS_TEXTURE0 + stage),
		(const D3DMATRIX*)textureMatrix.array));
	setTextureState(stage, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2);
}
//------------------------------------------------------------------------------
// r[|[g
//------------------------------------------------------------------------------
// r[|[g̐ݒ
void RenderingDevice::setViewport(const RectangleI& rectangle){
	D3DViewport viewport;
	viewport.X = rectangle.x;
	viewport.Y = rectangle.y;
	viewport.Width = rectangle.width;
	viewport.Height = rectangle.height;
	viewport.MinZ = 0.f;
	viewport.MaxZ = 1.f;
	DirectXCheck(device_->SetViewport(&viewport));
}
//------------------------------------------------------------------------------
// r[|[g̃Zbg
void RenderingDevice::resetViewport(){
	RectangleI rectangle;
	DimensionI size = getRenderTargetSize();
	rectangle.set(0, 0, size.width, size.height);
	setViewport(rectangle);
}
//------------------------------------------------------------------------------
// CeBO֘A
//------------------------------------------------------------------------------
// CeBO̐ݒ
void RenderingDevice::setLighting(bool lighting){
	setRenderState(D3DRS_LIGHTING, lighting);
}
//------------------------------------------------------------------------------
// ArGgJ[̐ݒ
void RenderingDevice::setAmbientColor(const Color3f& ambientColor){
	Color4c color(ambientColor);
	setRenderState(D3DRS_AMBIENT, color.getARGB());
}
//------------------------------------------------------------------------------
// fBNViCgLɂ
void RenderingDevice::enableDirectionalLight(
	int lightIndex, DirectionalLight* directionalLight){
	Assert((lightIndex >= 0) && (lightIndex < maxActiveLightCount_));
	D3DLight light;
	::memset(&light, 0, sizeof(D3DLight));
	light.Type = D3DLIGHT_DIRECTIONAL;
	setD3DColor(light.Diffuse, directionalLight->getDiffuseColor());
	setD3DColor(light.Specular, directionalLight->getSpecularColor());
	setD3DVector(light.Direction, directionalLight->getWorldDirection());
	DirectXCheck(device_->LightEnable(lightIndex, true));
	DirectXCheck(device_->SetLight(lightIndex, &light));
}
//------------------------------------------------------------------------------
// |CgCgLɂ
void RenderingDevice::enablePointLight(
	int lightIndex, PointLight* pointLight){
	Assert((lightIndex >= 0) && (lightIndex < maxActiveLightCount_));
	D3DLight light;
	::memset(&light, 0, sizeof(D3DLight));
	light.Type = D3DLIGHT_POINT;
	setD3DColor(light.Diffuse, pointLight->getDiffuseColor());
	setD3DColor(light.Specular, pointLight->getSpecularColor());
	setD3DVector(light.Position, pointLight->getWorldPosition());
	light.Range = pointLight->getGlobalRange();
	light.Attenuation0 = pointLight->getAttenuation0();
	light.Attenuation1 = pointLight->getAttenuation1();
	light.Attenuation2 = pointLight->getAttenuation2();
	DirectXCheck(device_->LightEnable(lightIndex, true));
	DirectXCheck(device_->SetLight(lightIndex, &light));
}
//------------------------------------------------------------------------------
// Cg
void RenderingDevice::closeLight(int lightIndex){
	for(int i = lightIndex; i < maxActiveLightCount_; i++){
		DirectXCheck(device_->LightEnable(i, false));
	}
}
//------------------------------------------------------------------------------
// }eA֘A
//------------------------------------------------------------------------------
// }eA̐ݒ
void RenderingDevice::setMaterial(
	const Color3f& diffuseColor, const Color3f& specularColor,
	const Color3f& ambientColor, const Color3f& emissiveColor,
	float specularPower, float alpha){
	// XyL̗Lݒ
	if(specularColor == Color3f::black){
		setRenderState(D3DRS_SPECULARENABLE, false);
	}else{
		setRenderState(D3DRS_SPECULARENABLE, true);
	}
	// }eAݒ
	D3DMaterial material;
	setD3DColor(material.Diffuse, diffuseColor);
	material.Diffuse.a = alpha;
	setD3DColor(material.Specular, specularColor);
	material.Power = specularPower;
	setD3DColor(material.Ambient, ambientColor);
	setD3DColor(material.Emissive, emissiveColor);
	DirectXCheck(device_->SetMaterial(&material));
}
//------------------------------------------------------------------------------
// ufBO̐ݒ
void RenderingDevice::setBlending(bool blendingFlag){
	setRenderState(D3DRS_ALPHABLENDENABLE, blendingFlag);
}
//------------------------------------------------------------------------------
// uh[h̐ݒ
void RenderingDevice::setBlendMode(int mode, int source, int destination){
	Assert((mode > 0) && (mode < Material::blendModeMax));
	Assert((source >= 0) && (source < Material::blendStateMax));
 	Assert((destination >= 0) && (destination < Material::blendStateMax));
	// uh[h
	D3DBLENDOP blendModeTable[] = {
		(D3DBLENDOP)0,
		D3DBLENDOP_ADD,
		D3DBLENDOP_SUBTRACT,
		D3DBLENDOP_REVSUBTRACT,
		D3DBLENDOP_MIN,
		D3DBLENDOP_MAX,
	};
	setRenderState(D3DRS_BLENDOP, blendModeTable[mode]);
	// uhXe[g
	D3DBLEND blendStateTable[] = {
		D3DBLEND_ZERO,
		D3DBLEND_ONE,
		D3DBLEND_SRCCOLOR,
		D3DBLEND_INVSRCCOLOR,
		D3DBLEND_SRCALPHA,
		D3DBLEND_INVSRCALPHA,
		D3DBLEND_SRCALPHASAT,
		D3DBLEND_DESTCOLOR,
		D3DBLEND_INVDESTCOLOR,
		D3DBLEND_DESTALPHA,
		D3DBLEND_INVDESTALPHA,
	};
	setRenderState(D3DRS_SRCBLEND, blendStateTable[source]);
	setRenderState(D3DRS_DESTBLEND, blendStateTable[destination]);
}
//------------------------------------------------------------------------------
// eNX`֘A
//------------------------------------------------------------------------------
// eNX`̐ݒ
void RenderingDevice::setTexture(int textureStage, Texture* texture){
	DirectXCheck(device_->SetTexture(textureStage, texture->getD3DTexture()));
}
//------------------------------------------------------------------------------
// eNX`̐ݒ
void RenderingDevice::setTexture(int textureStage, Direct3DTexture* texture){
	DirectXCheck(device_->SetTexture(textureStage, texture));
}
//------------------------------------------------------------------------------
// AhX[h̐ݒ
void RenderingDevice::setTextureAddressMode2(int textureStage,
	int addressModeU, int addressModeV){
	// AhX[hU
	if(addressModeU == Texture::addressModeWrap){
		setSamplerState(textureStage, D3DSAMP_ADDRESSU, D3DTADDRESS_WRAP);
	}else if(addressModeU == Texture::addressModeClamp){
		setSamplerState(textureStage, D3DSAMP_ADDRESSU, D3DTADDRESS_CLAMP);
	}else{
		Assert(addressModeU == Texture::addressModeMirror);
		setSamplerState(textureStage, D3DSAMP_ADDRESSU, D3DTADDRESS_MIRROR);
	}
	// AhX[hV
	if(addressModeV == Texture::addressModeWrap){
		setSamplerState(textureStage, D3DSAMP_ADDRESSV, D3DTADDRESS_WRAP);
	}else if(addressModeV == Texture::addressModeClamp){
		setSamplerState(textureStage, D3DSAMP_ADDRESSV, D3DTADDRESS_CLAMP);
	}else{
		Assert(addressModeV == Texture::addressModeMirror);
		setSamplerState(textureStage, D3DSAMP_ADDRESSV, D3DTADDRESS_MIRROR);
	}
}
//------------------------------------------------------------------------------
// eNX`̍쐬
Direct3DTexture* RenderingDevice::createTexture(
	D3DFORMAT format, int width, int height){
	Direct3DTexture* texture;
	DirectXCheck(device_->CreateTexture(width, height, 0, 0,
		format, D3DPOOL_MANAGED, &texture, NULL));
	return texture;
}
//------------------------------------------------------------------------------
// eNX`̃bN
D3DLOCKED_RECT RenderingDevice::lockTexture(
	Direct3DTexture* texture, int mipmapLevel){
	D3DLOCKED_RECT lockedRect;
	DirectXCheck(texture->LockRect(mipmapLevel, &lockedRect, NULL, 0));
	return lockedRect;
}
//------------------------------------------------------------------------------
// eNX`̃AbN
void RenderingDevice::unlockTexture(Direct3DTexture* texture, int mipmapLevel){
	DirectXCheck(texture->UnlockRect(mipmapLevel));
}
//------------------------------------------------------------------------------
// eNX`Xe[W֘A
//------------------------------------------------------------------------------
// J[eNX`Xe[Wݒ
void RenderingDevice::setColorTextureStage(int colorStage,
	D3DTEXTUREOP operation, u_int arg1, u_int arg2, int uvIndex){
	setTextureState(colorStage, D3DTSS_COLOROP, operation);
	setTextureState(colorStage, D3DTSS_COLORARG1, arg1);
	setTextureState(colorStage, D3DTSS_COLORARG2, arg2);
	setTextureState(colorStage, D3DTSS_TEXCOORDINDEX, uvIndex);
}
//------------------------------------------------------------------------------
// J[eNX`Xe[W
void RenderingDevice::closeColorTextureStage(int colorStage){
	if(colorStage == 0){
		// JgJ[o͂
		setColorTextureStage(colorStage,
			D3DTOP_SELECTARG1, D3DTA_CURRENT, D3DTA_CURRENT, 0);
		colorStage++;
	}
	// J[Xe[W
	setTextureState(colorStage, D3DTSS_COLOROP, D3DTOP_DISABLE);
}
//------------------------------------------------------------------------------
// At@eNX`Xe[Wݒ
void RenderingDevice::setAlphaTextureStage(int alphaTextureStage){
	if(alphaTextureStage == -1){
		// At@eNX`ꍇ
		setTextureState(0, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1);
		setTextureState(0, D3DTSS_ALPHAARG1, D3DTA_CURRENT);
		// At@Xe[W
		setTextureState(1, D3DTSS_ALPHAOP, D3DTOP_DISABLE);
		return;
	}
	// At@eNX`Xe[W܂ŃJgo͂s
	for(int i = 0; i < alphaTextureStage; i++){
		setTextureState(i, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1);
		setTextureState(i, D3DTSS_ALPHAARG1, D3DTA_CURRENT);
	}
	// Current * alphaTexture
	setTextureState(alphaTextureStage, D3DTSS_ALPHAOP, D3DTOP_MODULATE);
	setTextureState(alphaTextureStage, D3DTSS_ALPHAARG1, D3DTA_CURRENT);
	setTextureState(alphaTextureStage, D3DTSS_ALPHAARG2, D3DTA_TEXTURE);
	// At@Xe[W
	alphaTextureStage++;
	setTextureState(alphaTextureStage, D3DTSS_ALPHAOP, D3DTOP_DISABLE);
}
//------------------------------------------------------------------------------
// tHO
//------------------------------------------------------------------------------
// tHO̐ݒ
void RenderingDevice::setFog(Fog* fog){
	if(fog->isEnabled()){
		setRenderState(D3DRS_FOGENABLE, true);
		setRenderState(D3DRS_FOGCOLOR, fog->getColor().getARGB());
		// ͈̓tHOLɂ
		setRenderState(D3DRS_RANGEFOGENABLE, true);
		// tHO[hɂ킹ݒ
		Fog::Mode fogMode = fog->getMode();
		if(fogMode == Fog::modeLinear){
			// `tHO
			setRenderState(D3DRS_FOGVERTEXMODE, D3DFOG_LINEAR);
			float fogNear = fog->getNear();
			setRenderState(D3DRS_FOGSTART, *(u_int*)&fogNear);
			float fogFar = fog->getFar();
			setRenderState(D3DRS_FOGEND, *(u_int*)&fogFar);
		}else if(fogMode == Fog::modeExponent){
			// wtHO
			setRenderState(D3DRS_FOGVERTEXMODE, D3DFOG_EXP);
			float fogDensity = fog->getDensity();
			setRenderState(D3DRS_FOGDENSITY , *(u_int*)&fogDensity);
		}else if(fogMode == Fog::modeExponent2){
			// ̎wtHO
			setRenderState(D3DRS_FOGVERTEXMODE, D3DFOG_EXP2);
			float fogDensity = fog->getDensity();
			setRenderState(D3DRS_FOGDENSITY , *(u_int*)&fogDensity);
		}
	}else{
		setRenderState(D3DRS_FOGENABLE, false);
	}
}
//------------------------------------------------------------------------------
// ZeXg
//------------------------------------------------------------------------------
// ZeXg̐ݒ
void RenderingDevice::setZTest(bool zTest){
	if(zTest){ setRenderState(D3DRS_ZENABLE, D3DZB_TRUE); }
	else{ setRenderState(D3DRS_ZENABLE, D3DZB_FALSE); }
}
//------------------------------------------------------------------------------
// CfbNXobt@
//------------------------------------------------------------------------------
// ÓICfbNXobt@̍\z
Direct3DIndexBuffer* RenderingDevice::createStaticIndexBuffer(int bufferSize){
	Assert(bufferSize > 0);
	Direct3DIndexBuffer* indexBuffer;
	DirectXCheck(device_->CreateIndexBuffer(
		bufferSize, D3DUSAGE_WRITEONLY, D3DFMT_INDEX16,
		D3DPOOL_MANAGED, &indexBuffer, NULL));
	Assert(indexBuffer != NULL);
	return indexBuffer;
}
//------------------------------------------------------------------------------
// ÓICfbNXobt@̃bN
u_char* RenderingDevice::lockStaticIndexBuffer(
	Direct3DIndexBuffer* indexBuffer, int offset, int size){
	Assert((indexBuffer != NULL) && (offset >= 0) && (size > 0));
	u_char* lockAddress;
	DirectXCheck(indexBuffer->Lock(offset, size, (void**)&lockAddress, 0));
	return lockAddress;
}
//------------------------------------------------------------------------------
// ÓICfbNXobt@̃AbN
void RenderingDevice::unlockStaticIndexBuffer(
	Direct3DIndexBuffer* indexBuffer){
	Assert(indexBuffer != NULL);
	DirectXCheck(indexBuffer->Unlock());
}
//------------------------------------------------------------------------------
// ÓICfbNXobt@̏
void RenderingDevice::writeStaticIndexBuffer(
	Direct3DIndexBuffer* indexBuffer, const void* data, int dataSize){
	Assert(data != NULL);
	u_char* lockAddress = lockStaticIndexBuffer(indexBuffer, 0, dataSize);
	::memcpy(lockAddress, data, dataSize);
	unlockStaticIndexBuffer(indexBuffer);
}
//------------------------------------------------------------------------------
// ICfbNXobt@̍\z
Direct3DIndexBuffer* RenderingDevice::createDynamicIndexBuffer(int bufferSize){
	Assert(bufferSize > 0);
	Direct3DIndexBuffer* indexBuffer;
	DirectXCheck(device_->CreateIndexBuffer(
		bufferSize, (D3DUSAGE_DYNAMIC | D3DUSAGE_WRITEONLY), D3DFMT_INDEX16,
		D3DPOOL_DEFAULT, &indexBuffer, NULL));
	Assert(indexBuffer != NULL);
	return indexBuffer;
}
//------------------------------------------------------------------------------
// ICfbNXobt@̏
void RenderingDevice::writeDynamicIndexBuffer(
	Direct3DIndexBuffer* indexBuffer, const void* data, int dataSize){
	Assert(indexBuffer != NULL);
	Assert(data != NULL);
	Assert(dataSize > 0);
	u_char* lockAddress;
	DirectXCheck(indexBuffer->Lock(
		0, dataSize, (void**)&lockAddress, D3DLOCK_DISCARD));
	::memcpy(lockAddress, data, dataSize);
	DirectXCheck(indexBuffer->Unlock());
}
//------------------------------------------------------------------------------
// CfbNXobt@̐ݒ
void RenderingDevice::setIndexBuffer(Direct3DIndexBuffer* indexBuffer){
	DirectXCheck(device_->SetIndices(indexBuffer));
}
//------------------------------------------------------------------------------
// _Lq
//------------------------------------------------------------------------------
// _Lq̍\z
int RenderingDevice::createVertexDeclaration(
	Direct3DVertexDeclaration** vertexDeclaration, bool hasPosition,
	int weightsPerVertex, int bonesPerVertex, bool hasNormal,
	bool hasColor, int texCoordSetCount,
	const TexCoord::Type* texCoordTypeArray){
	// Lqz
	D3DVertexElement declarationArray[] = {
		D3DDECL_END(), // ʒu
		D3DDECL_END(), // EFCg
		D3DDECL_END(), // {[CfbNX
		D3DDECL_END(), // @
		D3DDECL_END(), // J[
		D3DDECL_END(), // eNX`W0
		D3DDECL_END(), // eNX`W1
		D3DDECL_END(), // eNX`W2
		D3DDECL_END(), // eNX`W3
		D3DDECL_END(), // eNX`W4
		D3DDECL_END(), // eNX`W5
		D3DDECL_END(), // eNX`W6
		D3DDECL_END(), // eNX`W7
		D3DDECL_END()
	};
	int vertexSize = 0;
	int declarationIndex = 0;

	// ʒu̒`
	if(hasPosition){
		D3DVertexElement element = { 0, vertexSize, D3DDECLTYPE_FLOAT3,
			D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_POSITION, 0};
		declarationArray[declarationIndex] = element;
		declarationIndex++;
		vertexSize += sizeof(float) * 3;
	}

	// EFCg̒`
	if(weightsPerVertex != 0){
		Assert(false);
	}

	// {[CfbNX̒`
	if(bonesPerVertex != 0){
		Assert(false);
	}

	// @̒`
	if(hasNormal){
		D3DVertexElement element = {
			0, vertexSize, D3DDECLTYPE_FLOAT3,
				D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_NORMAL, 0};
		declarationArray[declarationIndex] = element;
		declarationIndex++;
		vertexSize += sizeof(float) * 3;
	}

	// J[̒`
	if(hasColor){
		D3DVertexElement element = {
			0, vertexSize, D3DDECLTYPE_D3DCOLOR,
				D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_COLOR, 0};
		declarationArray[declarationIndex] = element;
		declarationIndex++;
		vertexSize += sizeof(u_int);
	}

	// eNX`W̒`
	D3DDECLTYPE texCoordTypeTable[] = {
		D3DDECLTYPE_FLOAT1,// gpȂ
		D3DDECLTYPE_FLOAT1,
		D3DDECLTYPE_FLOAT2,
		D3DDECLTYPE_FLOAT3,
		D3DDECLTYPE_FLOAT4
	};
	for(int i = 0; i < texCoordSetCount; i++){
		int texCoordCount = texCoordTypeArray[i];
		Assert((texCoordCount > 0) && (texCoordCount <= 4));
		D3DVertexElement element = {
			0, vertexSize, texCoordTypeTable[texCoordCount],
			D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_TEXCOORD, i};
		declarationArray[declarationIndex] = element;
		declarationIndex++;
		vertexSize += sizeof(float) * texCoordCount;
	}
	// _Lq̍\z
	DirectXCheck(device_->CreateVertexDeclaration(
		declarationArray, vertexDeclaration));
	Assert((*vertexDeclaration) != NULL);
	Assert(vertexSize != 0);
	return vertexSize;
}
//------------------------------------------------------------------------------
// _Lq̐ݒ
void RenderingDevice::setVertexDeclaration(
	Direct3DVertexDeclaration* vertexDeclaration){
	DirectXCheck(device_->SetVertexDeclaration(vertexDeclaration));
}
//------------------------------------------------------------------------------
// _obt@
//------------------------------------------------------------------------------
// ÓI_obt@̍\z
Direct3DVertexBuffer* RenderingDevice::createStaticVertexBuffer(int bufferSize){
	Assert(bufferSize > 0);
	Direct3DVertexBuffer* vertexBuffer;
	DirectXCheck(device_->CreateVertexBuffer(
		bufferSize, D3DUSAGE_WRITEONLY, 0,
		D3DPOOL_MANAGED, &vertexBuffer, NULL));
	Assert(vertexBuffer != NULL);
	return vertexBuffer;
}
//------------------------------------------------------------------------------
// ÓI_obt@̏
void RenderingDevice::writeStaticVertexBuffer(
	Direct3DVertexBuffer* vertexBuffer, int bufferSize, int vertexCount,
	const Vector3* positions, int weightsPerVertex, const float* weights,
	int bonesBerVertex, const u_char* boneIndices, const Vector3* normals,
	const Color4c* colors, int texCoordSetCount,
	const TexCoord::Type* texCoordTypeArray, const float* const* texCoords){
	u_char* lockAddress;
	DirectXCheck(vertexBuffer->Lock(0, bufferSize, (void**)&lockAddress, 0));
	writeVertices(lockAddress, bufferSize, vertexCount, positions,
		weightsPerVertex, weights, bonesBerVertex, boneIndices, normals,
		colors, texCoordSetCount, texCoordTypeArray, texCoords);
	DirectXCheck(vertexBuffer->Unlock());
}
//------------------------------------------------------------------------------
// I_obt@̍\z
Direct3DVertexBuffer* RenderingDevice::createDynamicVertexBuffer(
	int bufferSize){
	Assert(bufferSize > 0);
	Direct3DVertexBuffer* vertexBuffer;
	DirectXCheck(device_->CreateVertexBuffer(
		bufferSize, (D3DUSAGE_DYNAMIC | D3DUSAGE_WRITEONLY), 0,
		D3DPOOL_DEFAULT, &vertexBuffer, NULL));
	Assert(vertexBuffer != NULL);
	return vertexBuffer;
}
//------------------------------------------------------------------------------
// I_obt@̃bN
u_char* RenderingDevice::lockDynamicVertexBuffer(
	Direct3DVertexBuffer* vertexBuffer, int offset, int size){
	u_int flag;
	if(offset == 0){ flag = D3DLOCK_DISCARD; }
	else{ flag = D3DLOCK_NOOVERWRITE; }
	u_char* lockAddress;
	DirectXCheck(vertexBuffer->Lock(offset, size, (void**)&lockAddress, flag));
	return lockAddress;
}
//------------------------------------------------------------------------------
// I_obt@̃AbN
void RenderingDevice::unlockDynamicVertexBuffer(
	Direct3DVertexBuffer* vertexBuffer){
	DirectXCheck(vertexBuffer->Unlock());
}
//------------------------------------------------------------------------------
// I_obt@̏
void RenderingDevice::writeDynamicVertexBuffer(
	Direct3DVertexBuffer* vertexBuffer, int bufferSize, int vertexCount,
	const Vector3* positions, int weightsPerVertex, const float* weights,
	int bonesBerVertex, const u_char* boneIndices, const Vector3* normals,
	const Color4c* colors, int texCoordSetCount,
	const TexCoord::Type* texCoordTypeArray, const float* const* texCoords){
	u_char* lockAddress = lockDynamicVertexBuffer(vertexBuffer, 0, bufferSize);
	writeVertices(lockAddress, bufferSize, vertexCount, positions,
		weightsPerVertex, weights, bonesBerVertex, boneIndices, normals,
		colors, texCoordSetCount, texCoordTypeArray, texCoords);
	unlockDynamicVertexBuffer(vertexBuffer);
}
//------------------------------------------------------------------------------
// _obt@̐ݒ
void RenderingDevice::setVertexBuffer(
	Direct3DVertexBuffer* vertexBuffer, int vertexSize){
	DirectXCheck(device_->SetStreamSource(0, vertexBuffer, 0, vertexSize));
}
//------------------------------------------------------------------------------
// `
//------------------------------------------------------------------------------
// OpXg`
void RenderingDevice::drawTriangleList(int primitiveCount){
	DirectXCheck(device_->DrawPrimitive(D3DPT_TRIANGLELIST, 0, primitiveCount));
}
//------------------------------------------------------------------------------
// CfbNXOpXg`
void RenderingDevice::drawIndexedTriangleList(
	int vertexCount, int primitiveCount){
	DirectXCheck(device_->DrawIndexedPrimitive(
		D3DPT_TRIANGLELIST, 0, 0, vertexCount, 0, primitiveCount));
}
//------------------------------------------------------------------------------
// CfbNXOpXg`
void RenderingDevice::drawIndexedTriangleList(int baseVertexIndex,
	int minIndex, int vertexCount, int startIndex, int primitiveCount){
	DirectXCheck(device_->DrawIndexedPrimitive(D3DPT_TRIANGLELIST,
		baseVertexIndex, minIndex, vertexCount, startIndex, primitiveCount));
}
//------------------------------------------------------------------------------
// CXg`
void RenderingDevice::drawLineList(int primitiveCount){
	DirectXCheck(device_->DrawPrimitive(D3DPT_LINELIST, 0, primitiveCount));
}
//------------------------------------------------------------------------------
// CfbNXCXg`
void RenderingDevice::drawIndexedLineList(
	int vertexCount, int primitiveCount){
	DirectXCheck(device_->DrawIndexedPrimitive(
		D3DPT_LINELIST, 0, 0, vertexCount, 0, primitiveCount));
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// _f[^̏
void RenderingDevice::writeVertices(
	u_char* buffer, int bufferSize, int vertexCount,
	const Vector3* positions, int weightsPerVertex, const float* weights,
	int bonesBerVertex, const u_char* boneIndices, const Vector3* normals,
	const Color4c* colors, int texCoordSetCount,
	const TexCoord::Type* texCoordTypeArray, const float* const* texCoords){
	Assert(buffer != NULL);
	Assert(bufferSize != 0);
	Assert(vertexCount != 0);
	bool hasPosition = (positions != NULL);
	bool hasWeights = (weightsPerVertex > 0);
	Assert(!hasWeights);
	bool hasBoneIndices = (bonesBerVertex > 0);
	Assert(!hasBoneIndices);
	bool hasNormal = (normals != NULL);
	bool hasColors = (colors != NULL);
	u_char* startAddress = buffer;
	for(int i = 0; i < vertexCount; i++){
		// ʒȕ
		if(hasPosition){
			(*(Vector3*)buffer) = (*positions);
			buffer += sizeof(Vector3);
			positions++;
		}
		// EFCg̏
		if(hasWeights){
		}
		// {[CfbNX̏
		if(hasBoneIndices){
		}
		// @̏
		if(hasNormal){
			(*(Vector3*)buffer) = (*normals);
			buffer += sizeof(Vector3);
			normals++;
		}
		// _J[̏
		if(hasColors){
			(*(u_int*)buffer) = (*colors).getARGB();
			colors++;
			buffer += sizeof(u_int);
		}

		// eNX`W̏
		for(int j = 0; j < texCoordSetCount; j++){
			int numFloat = texCoordTypeArray[j];
			int copySize = sizeof(float) * numFloat;
			::memcpy(buffer, texCoords[j] + (numFloat * i), copySize);
			buffer += copySize;
		}

	}
	Assert((buffer - startAddress) == bufferSize);
}
//------------------------------------------------------------------------------
// D3DxNg̐ݒ
void RenderingDevice::setD3DVector(
	D3DVECTOR& destination, const Vector3& source){
	destination.x = source.x;
	destination.y = source.y;
	destination.z = source.z;
}
//------------------------------------------------------------------------------
// D3DJ[̐ݒ
void RenderingDevice::setD3DColor(
	D3DCOLORVALUE& destination, const Color4f& source){
	destination.r = source.r;
	destination.g = source.g;
	destination.b = source.b;
	destination.a = source.a;
}
//------------------------------------------------------------------------------
// D3DJ[̐ݒ
void RenderingDevice::setD3DColor(
	D3DCOLORVALUE& destination, const Color3f& source){
	destination.r = source.r;
	destination.g = source.g;
	destination.b = source.b;
	destination.a = 0.f;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
