//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * _
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/Renderer/Renderer.h"
#include "Graphics/Renderer/DrawRequest.h"
#include "Graphics/System/LampGraphics.h"
#include "Graphics/Renderer/RenderingDevice.h"
#include "Graphics/System/GraphicsDeviceCapacity.h"
#include "Graphics/Camera/Camera.h"
#include "Graphics/Light/LightManager.h"
#include "Graphics/Mesh/Mesh.h"
#include "Graphics/Material/Material.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
Renderer::Renderer() : meshList_(initialMeshListSize){
	// foCXIuWFNgz_o^
	LampGraphics::addDeviceObjectHolder(this);
	buildRendererStateBlock();
	drawRequest_ = new DrawRequest();
}
//------------------------------------------------------------------------------
// fXgN^
Renderer::~Renderer(){
	SafeDelete(drawRequest_);
	SafeRelease(rendererStateBlock_);
	// foCXIuWFNgz_o^
	LampGraphics::removeDeviceObjectHolder(this);
}
//------------------------------------------------------------------------------
// _O
//------------------------------------------------------------------------------
// _Os
void Renderer::renderingSetup(Scene* scene){
	RenderingDevice* device = RenderingDevice::getInstance();
	scene_ = scene;
	// bVXg̍\z
	buildMeshList();
	// `惊NGXg̏
	initializeDrawRequest();
}
//------------------------------------------------------------------------------
// _Os
void Renderer::rendering(){
	RenderingDevice* device = RenderingDevice::getInstance();
	// ftHgXe[gubNKp
	device->applyDefaultStateBlock();
	// _Xe[gubNKp
	device->applyStateBlock(rendererStateBlock_);
	// O[oݒ̏
	initializeGlobalSettings();
	// `
	if(device->beginScene()){
		int meshCount = meshList_.getCount();
		for(int i = 0; i < meshCount; i++){
			Mesh* mesh = meshList_.get(i);
			Material* material = mesh->getMaterial();
			drawRequest_->setMesh(mesh);
			// ufBOLɂ
			if(drawRequest_->isBlendEnabled()){
				device->setBlending(true);
				device->setRenderState(
					D3DRS_ALPHAREF, blendingAlphaTestBorder_);
			}
			// [JCgXg̍\z
			if(material->useLight()){
				drawRequest_->clearLocalLights();
				scene_->getLocalLightList(mesh, drawRequest_);
			}
			// `
			material->draw(drawRequest_);
		}
		// `n
		device->endScene();
	}
	// ftHgXe[gubNKp
	device->applyDefaultStateBlock();
}
//------------------------------------------------------------------------------
// bVXg̍\z
void Renderer::buildMeshList(){
	meshList_.clear();
	Camera* camera = scene_->getCurrentCamera();
	scene_->getMeshList(&meshList_, camera);
	int meshCount = meshList_.getCount();
	// JƂ̋̓e|f[^ɐݒ
	Vector3 cameraPosition = camera->getPosition();
	for(int i = 0; i < meshCount; i++){
		Mesh* mesh = meshList_.get(i);
		mesh->setRenderingTemporaryData(
			(cameraPosition - mesh->getWorldCenter()).getSquaredLength());
	}
	// bVXg̃\[g
	meshList_.sort(sortMeshList);
/* \[gsĂ邩printf`FbN
	for(int i = 0; i < meshCount; i++){
		Mesh* mesh = meshList_.get(i);
		Material* material = mesh->getMaterial();
		DebugOut("%3d %d %2d 0x%08x %6.1f %s\n",
			i, material->getBlendMode(), material->getPriority(),
			material,
			mesh->getRenderingTemporaryData(),
			material->getName().getBytes());
	}
	DebugOut("\n");// Ƀu[N|Cgd|
//*/
}
//------------------------------------------------------------------------------
// bVXg̃\[g
int Renderer::sortMeshList(Mesh* const* left, Mesh* const* right){
	const Mesh* leftMesh = (*left);
	const Mesh* rightMesh = (*right);
	const Material* leftMaterial = leftMesh->getMaterial();
	const Material* rightMaterial = rightMesh->getMaterial();
	if(!leftMaterial->isBlendEnabled()){
		// At@\[g
		if(rightMaterial->isBlendEnabled()){ return -1; }
		// At@łȂbṼ\[g
		// }eADxŔr
		int priority =
			rightMaterial->getPriority() - leftMaterial->getPriority();
		if(priority != 0){ return priority; }
		// }eÃAhXŃ\[g
		int leftAddress = (int)((*(u_int*)&leftMaterial) & 0x8fffffff);
		int rightAddress = (int)((*(u_int*)&rightMaterial) & 0x8fffffff);
		int address = leftAddress - rightAddress;
		if(address != 0){ return address; }
		// O牜Z\[g
		float cameraSquaredDistance = rightMesh->getRenderingTemporaryData() -
			leftMesh->getRenderingTemporaryData();
		if(cameraSquaredDistance > 0.f){ return -1; }
		else{ return 1; }
	}else{
		// At@\[g
		if(!rightMaterial->isBlendEnabled()){ return 1; }
		// At@bṼ\[g
		// OZ\[g
		float cameraSquaredDistance = leftMesh->getRenderingTemporaryData() -
			rightMesh->getRenderingTemporaryData();
		if(cameraSquaredDistance > 0.f){ return -1; }
		else{ return 1; }
	}
	return 0;
}
//------------------------------------------------------------------------------
// `惊NGXg̏
void Renderer::initializeDrawRequest(){
	drawRequest_->clear();
	drawRequest_->setFog(scene_->getFog());
	drawRequest_->setCamera(scene_->getCurrentCamera());
	// O[oCg̏
	LightManager* lightManager = scene_->getLightManager();
	int lightCount = lightManager->getCount();
	for(int i = 0; i < lightCount; i++){
		Light* light = lightManager->get(i);
		// ȃCg𖳎
		if(!light->isGlobalEnabled()){ continue; }
		if(light->isAmbientLight()){
			drawRequest_->addAmbientLight(light->castAmbientLight());
		}else if(light->isDirectionalLight()){
			drawRequest_->addDirectionalLight(light->castDirectionalLight());
		}
	}
}
//------------------------------------------------------------------------------
// O[oݒ̏
void Renderer::initializeGlobalSettings(){
	// ̃_Oɂ͕ωȂݒs
	RenderingDevice* device = RenderingDevice::getInstance();
	// Js̐ݒ
	Camera* camera = drawRequest_->getCamera();
	device->setProjectionMatrix(camera->getProjectionMatrix());
	device->setViewMatrix(camera->getViewMatrix());
	// tHO̐ݒ
	device->setFog(drawRequest_->getFog());
}
//------------------------------------------------------------------------------
// _Xe[gubN̍\z
void Renderer::buildRendererStateBlock(){
	RenderingDevice* device = RenderingDevice::getInstance();
	GraphicsDeviceCapacity* capacity = GraphicsDeviceCapacity::getInstance();
	// ftHgXe[gubNKp
	device->applyDefaultStateBlock();
	// _Xe[gubN̍\zJn
	device->beginStateBlock();

	// JOtɂ
	device->setRenderState(D3DRS_CULLMODE, D3DCULL_CW);

	// eNX`TvɃ~bv}bvƐ`Ԃݒ肷
	int textureStageCount = capacity->getMaxTextureBlendStages();
	for(int i = 0; i < textureStageCount; i++){
		device->setSamplerState(i, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR);
		device->setSamplerState(i, D3DSAMP_MINFILTER, D3DTEXF_LINEAR);
		device->setSamplerState(i, D3DSAMP_MIPFILTER, D3DTEXF_LINEAR);
	}

	// _J[G~bVuƂĎgp
	device->setRenderState(D3DRS_DIFFUSEMATERIALSOURCE, D3DMCS_MATERIAL);
	device->setRenderState(D3DRS_SPECULARMATERIALSOURCE, D3DMCS_MATERIAL);
	device->setRenderState(D3DRS_EMISSIVEMATERIALSOURCE, D3DMCS_COLOR1);

	// At@eXg̐ݒ
	// At@ufBOsȂĂAt@eXg͍s
	device->setRenderState(D3DRS_ALPHATESTENABLE, true);
	device->setRenderState(D3DRS_ALPHAREF, alphaTestBorder_);
	device->setRenderState(D3DRS_ALPHAFUNC, D3DCMP_GREATEREQUAL);

	// _Xe[gubN̍\z
	rendererStateBlock_ = device->endStateBlock();
	// ftHgXe[gubNKp
	device->applyDefaultStateBlock();
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
