//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * v~eBu_
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/PrimitiveRenderer/PrimitiveRenderer.h"
#include "Graphics/PrimitiveRenderer/PrimitiveDrawRequestBuilder.h"
#include "Graphics/System/LampGraphics.h"
#include "Graphics/Renderer/RenderingDevice.h"
#include "Graphics/Camera/Camera.h"
#include "Graphics/Material/Material.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
PrimitiveRenderer::PrimitiveRenderer() :
	positionDeclaration_(NULL), positionColorDeclaration_(NULL){
	LampGraphics::addDeviceObjectHolder(this);
	// ev~eBu̍\z
	PrimitiveDrawRequestBuilder::buildPoint(&point_);	
	PrimitiveDrawRequestBuilder::buildAxisPoint(&axisAxisPoint_);	
	PrimitiveDrawRequestBuilder::buildAxis(&axis_);	
	PrimitiveDrawRequestBuilder::buildArrow(&arrow_);	
	PrimitiveDrawRequestBuilder::buildGrid(&grid_, 10, 10);	
	PrimitiveDrawRequestBuilder::buildPlane(&plane_);	
	PrimitiveDrawRequestBuilder::buildSphere(&sphere_, 1.f, 12, 8);	
	PrimitiveDrawRequestBuilder::buildBox(&box_, 1.f, 1.f, 1.f);	
	PrimitiveDrawRequestBuilder::buildCylinder(&cylinder_, 1.f, 1.f, 12);	
	PrimitiveDrawRequestBuilder::buildCone(&cone_, 1.f, 1.f, 12);	
}
//------------------------------------------------------------------------------
// fXgN^
PrimitiveRenderer::~PrimitiveRenderer(){
	LampGraphics::removeDeviceObjectHolder(this);
	// OtBbNXIuWFNg
	invalidateGraphicsDeviceObjects();
}
//------------------------------------------------------------------------------
// _O
//------------------------------------------------------------------------------
// _O
void PrimitiveRenderer::render(const Matrix44& viewMatrix,
	const Matrix44& projectionMatrix){
	if(requests_.getCount() == 0){ return; }
	//--------------------------------------------------------------------------
	// 
	RenderingDevice* device = RenderingDevice::getInstance();
	// ftHgXe[gubNKp
	device->applyDefaultStateBlock();
	// r[s̐ݒ
	device->setViewMatrix(viewMatrix);
	// es̐ݒ
	device->setProjectionMatrix(projectionMatrix);
	// JO𖳌ɂ
	device->setRenderState(D3DRS_CULLMODE, D3DCULL_NONE);

	// CeBO𖳌ɂ
	device->setRenderState(D3DRS_LIGHTING, false);
	// ftHg̃}eAݒ肷A_J[ꍇfBt[Y̔ɂȂ
	device->setMaterial(Color3f::white, Color3f::black, Color3f::black,
		Color3f::black, 0.f, 1.f);
	// J[Xe[W̐ݒiDiffuse * TextureFactorj
	device->setTextureState(0, D3DTSS_COLOROP, D3DTOP_MODULATE);
	device->setTextureState(0, D3DTSS_COLORARG1, D3DTA_DIFFUSE);
	device->setTextureState(0, D3DTSS_COLORARG2, D3DTA_TFACTOR);
	// At@Xe[W̐ݒiDiffuse * TextureFactorj
	device->setTextureState(0, D3DTSS_ALPHAOP, D3DTOP_MODULATE);
	device->setTextureState(0, D3DTSS_ALPHAARG1, D3DTA_DIFFUSE);
	device->setTextureState(0, D3DTSS_ALPHAARG2, D3DTA_TFACTOR);
	// ufBOLɂ
	device->setBlending(true);
	device->setBlendMode(Material::blendModeAdd,
		Material::blendStateSourceAlpha,
		Material::blendStateInverseSourceAlpha);

	//--------------------------------------------------------------------------
	// `
	if(device->beginScene()){
		Request request;
		int requestCount = requests_.getCount();
		for(int i = 0; i < requestCount; i++){
			renderRequest(requests_[i]);
			// NGXgNAĂ
			requests_.set(i, request);
		}
		device->endScene();
	}

	//--------------------------------------------------------------------------
	// n
	// ftHgXe[gubNKp
	device->applyDefaultStateBlock();
	// NGXg̃NA
	requests_.clear();
}
//------------------------------------------------------------------------------
// _O
void PrimitiveRenderer::render(Camera* camera){
	render(camera->getViewMatrix(), camera->getProjectionMatrix());
}
//------------------------------------------------------------------------------
// NGXg̕`
void PrimitiveRenderer::renderRequest(Request& request){
	RenderingDevice* device = RenderingDevice::getInstance();
	// s̐ݒ
	device->setWorldMatrix(request.matrix_);
	// FeNX`萔ɐݒ
	device->setRenderState(D3DRS_TEXTUREFACTOR, request.color_.getARGB());
	// ZeXg̐ݒ
	device->setZTest(request.zTest_);

	// v~eBu̐ݒ
	PrimitiveDrawRequest& primitive = request.primitive_;
	// _`̐ݒ
	int vertexSize = primitive.getVertexSize();
	if(!primitive.hasColor()){
		setPositionVertexDeclaration();
	}else{
		setPositionColorVertexDeclaration();
	}
	// _obt@̐ݒ
	device->setVertexBuffer(primitive.getVertexBuffer(), vertexSize);

	// `
	bool hasIndices = primitive.hasVertexIndices();
	int vertexCount = primitive.getVertexCount();
	if(hasIndices){
		device->setIndexBuffer(primitive.getIndexBuffer());
		int indexCount = primitive.getVertexIndexCount();
		Assert((indexCount % 2) == 0);
		device->drawIndexedLineList(vertexCount, indexCount / 2);
	}else{
		Assert((vertexCount % 2) == 0);
		device->drawLineList(vertexCount / 2);
	}
}
//------------------------------------------------------------------------------
// NGXg
//------------------------------------------------------------------------------
// NGXg
void PrimitiveRenderer::request(const PrimitiveDrawRequest& primitive,
	const Matrix34& matrix, Color4c color, bool zTest){
	Request request;
	request.primitive_ = primitive;
	request.matrix_ = matrix;
	request.color_ = color;
	request.zTest_ = zTest;
	requests_.add(request);
}
//------------------------------------------------------------------------------
// ̃NGXg
void PrimitiveRenderer::requestLine(int vertexCount, Vector3* positions,
	const Matrix34& matrix, Color4c color, bool zTest){
	Assert((vertexCount > 1) && ((vertexCount % 2) == 0));
	Assert(positions != NULL);
	PrimitiveDrawRequest primitive;
	primitive.setVertexCount(vertexCount);
	for(int i = 0; i < vertexCount; i++){
		primitive.setPosition(i, positions[i]);
	}
	request(primitive, matrix, color, zTest);
}
//------------------------------------------------------------------------------
// ̃NGXg
void PrimitiveRenderer::requestLine(int vertexCount, Vector3* positions,
	Color4c* colors, const Matrix34& matrix, Color4c color, bool zTest){
	Assert((vertexCount > 1) && ((vertexCount % 2) == 0));
	Assert(positions != NULL);
	Assert(colors != NULL);
	PrimitiveDrawRequest primitive;
	primitive.setVertexCount(vertexCount);
	primitive.enableColor(true);
	for(int i = 0; i < vertexCount; i++){
		primitive.setPosition(i, positions[i]);
		primitive.setColor(i, colors[i]);
	}
	request(primitive, matrix, color, zTest);
}
//------------------------------------------------------------------------------
// foCXIuWFNg
//------------------------------------------------------------------------------
// foCXIuWFNg̖
void PrimitiveRenderer::invalidateGraphicsDeviceObjects(){
	// OtBbNXIuWFNg
	SafeRelease(positionDeclaration_);
	SafeRelease(positionColorDeclaration_);
}
//------------------------------------------------------------------------------
// _Lq
//------------------------------------------------------------------------------
// ʒu̒_Lqݒ
void PrimitiveRenderer::setPositionVertexDeclaration(){
	RenderingDevice* device = RenderingDevice::getInstance();
	if(positionDeclaration_ == NULL){
		device->createVertexDeclaration(&positionDeclaration_,
			true, 0, 0, false, false, 0, NULL);
	}
	device->setVertexDeclaration(positionDeclaration_);
}
//------------------------------------------------------------------------------
// ʒuƃJ[̒_Lqݒ
void PrimitiveRenderer::setPositionColorVertexDeclaration(){
	RenderingDevice* device = RenderingDevice::getInstance();
	if(positionColorDeclaration_ == NULL){
		device->createVertexDeclaration(&positionColorDeclaration_,
			true, 0, 0, false, true, 0, NULL);
	}
	device->setVertexDeclaration(positionColorDeclaration_);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
