//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * LN^f
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/Model/CharacterModel.h"
#include "Graphics/Scene/Scene.h"
#include "Graphics/Model/ModelManager.h"
#include "Graphics/Mesh/Mesh.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
CharacterModel::CharacterModel(const String& name, Scene* scene) :
	Model(name, scene), boneHash_(maxBoneCount, 1.f), boneArray_(maxBoneCount),
	positionDeformMatrixArray_(NULL), normalDeformMatrixArray_(NULL),
	deformMatrixArraySize_(0), boneScaled_(false){
	buildBoneMatrixTick_ = scene_->getTick() - 1;
	deformMatrixArrayTick_ = scene_->getTick() - 1;
}
//------------------------------------------------------------------------------
// fXgN^
CharacterModel::~CharacterModel(){
	SafeArrayDelete(normalDeformMatrixArray_);
	SafeArrayDelete(positionDeformMatrixArray_);
	clearBone();
}
//------------------------------------------------------------------------------
// LN^f̃Rs[
CharacterModel* CharacterModel::copyCharacterModel(u_int copyMask) const{
	ModelManager* manager = scene_->getModelManager();
	CharacterModel* copyModel =
		manager->createCharacterModel(manager->rename(name_));
	// f̒lRs[
	copyModelValue(copyModel, copyMask);
	// {[̃Rs[
	int boneCount = getBoneCount();
	for(int i = 0; i < boneCount; i++){
		Bone* sourceBone = getBone(i);
		Bone* destinationBone = copyModel->createBone(sourceBone->getName());
		sourceBone->copyBoneValue(destinationBone);
	}
	// {[̃NRs[
	for(int i = 0; i < boneCount; i++){
		Bone* sourceBone = getBone(i);
		Bone* destinationBone = copyModel->getBone(i);
		int childCount = sourceBone->getBoneCount();
		for(int j = 0; j < childCount; j++){
			Bone* childBone = copyModel->searchBone(
				sourceBone->getBone(j)->getName());
			destinationBone->addBone(childBone);
		}
	}
	return copyModel;
}
//------------------------------------------------------------------------------
// {[s̍\z
void CharacterModel::buildBoneMatrix(bool forceCalculation){
	// vZtOĂȂAV[`bNłΌvZȂ
	u_int sceneTick = scene_->getTick();
	if((!forceCalculation) && (buildBoneMatrixTick_ == sceneTick)){ return; }
	buildBoneMatrixTick_ = sceneTick;
	boneScaled_ = getBone(0)->buildBoneMatrix(Matrix34::unit);
}
//------------------------------------------------------------------------------
// ʒuό`sz̎擾
const Matrix34* CharacterModel::getPositionDeformMatrixArray(
	bool forceCalculation){
	// vZtOĂȂAV[`bNłΌvZȂ
	if((!forceCalculation) && (deformMatrixArrayTick_ == scene_->getTick())){
		return positionDeformMatrixArray_;
	}
	// ό`sz̍\z
	buildDeformMatrixArray();
	return positionDeformMatrixArray_;
}
//------------------------------------------------------------------------------
// @ό`sz̎擾
const Matrix33* CharacterModel::getNormalDeformMatrixArray(
	bool forceCalculation){
	Assert(boneScaled_);
	// vZtOĂȂAV[`bNłΌvZȂ
	if((!forceCalculation) && (deformMatrixArrayTick_ == scene_->getTick())){
		return normalDeformMatrixArray_;
	}
	// ό`sz̍\z
	buildDeformMatrixArray();
	return normalDeformMatrixArray_;
}
//------------------------------------------------------------------------------
// ό`sz̍\z
void CharacterModel::buildDeformMatrixArray(){
	// z̊m
	int boneCount = getBoneCount();
	if((positionDeformMatrixArray_ == NULL) ||
		(deformMatrixArraySize_ != boneCount)){
		SafeArrayDelete(positionDeformMatrixArray_);
		deformMatrixArraySize_ = boneCount;
		positionDeformMatrixArray_ = new Matrix34[deformMatrixArraySize_];
		if(boneScaled_){
			normalDeformMatrixArray_ = new Matrix33[deformMatrixArraySize_];
		}
	}
	// f[^ݒ
	for(int i = 0; i < boneCount; i++){
		positionDeformMatrixArray_[i] = getBone(i)->getDeformMatrix();
	}
	if(boneScaled_){
		for(int i = 0; i < boneCount; i++){
			normalDeformMatrixArray_[i].set(positionDeformMatrixArray_[i]);
			normalDeformMatrixArray_[i].invert();
			normalDeformMatrixArray_[i].transpose();
		}
	}
	deformMatrixArrayTick_ = scene_->getTick();
}
//------------------------------------------------------------------------------
// {[̍쐬
Bone* CharacterModel::createBone(const String& boneName){
	Assert(boneArray_.getCount() < maxBoneCount);
	// ÕTCY`FbN
	if(boneName.getSize() == 0){
		ErrorOut("CharacterModel::createBone() boneName.getSize() == 0");
	}
	Bone* bone = new Bone(boneName);
	if(!boneHash_.put(boneName, bone)){
		// Oɏd
		ErrorOut("CharacterModel::createBone() repetition name %s",
			boneName.getBytes());
		delete bone;
		return NULL;
	}
	boneArray_.add(bone);
	return bone;
}
//------------------------------------------------------------------------------
// {[̔j
void CharacterModel::destroyBone(Bone* bone){
	if(boneHash_.remove(bone->getName()) == NULL){
		ErrorOut("CharacterModel::destroyBone() not found %s",
			bone->getName().getBytes());
	}
	boneArray_.removeByValue(bone);
	delete bone;
}
//------------------------------------------------------------------------------
// {[̃NA
int CharacterModel::clearBone(){
	int result = getBoneCount();
	for(int i = 0; i < result; i++){ delete getBone(i); }
	boneHash_.clear();
	boneArray_.clear();
	return result;
}
//------------------------------------------------------------------------------
// bV̒ǉ
void CharacterModel::addMesh(Mesh* mesh){
	Assert(mesh->isCharacterMesh());
	Model::addMesh(mesh);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
