//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * bV
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/Mesh/Mesh.h"
#include "Graphics/Mesh/MeshManager.h"
#include "Graphics/Model/Model.h"
#include "Graphics/MeshData/MeshData.h"
#include "Graphics/Material/Material.h"

namespace Lamp{

// v~eBu^Cve[u
const String Mesh::primitiveTypeStringTable[] = {
	"TriangleList",
	"IndexedTriangleList",
};

//------------------------------------------------------------------------------
// RXgN^
Mesh::Mesh(const String& name, Scene* scene) : SceneObject(name, scene),
	parent_(NULL), meshData_(NULL), material_(NULL),
	worldBoundingSphere_(Sphere::zero), worldBoundingBox_(AxisAlignedBox::zero),
	renderingTemporaryData_(0.f), enabled_(true), globalEnabled_(false){
}
//------------------------------------------------------------------------------
// fXgN^
Mesh::~Mesh(){
}
//------------------------------------------------------------------------------
// bV̒lRs[
void Mesh::copyMeshValue(Mesh* destination, u_int copyMask) const{
	// õRs[
	destination->setEnabled(enabled_);
	// bVf[^
	if(meshData_ != NULL){
		if((copyMask & copyMeshData) == 0){
			// bVf[^L
			destination->setMeshData(meshData_);
		}else{
			// bVf[^Rs[
			destination->setMeshData(meshData_->copy());
		}
	}
	// }eA
	if(material_ != NULL){
		if((copyMask & copyMaterial) == 0){
			// }eAL
			destination->setMaterial(material_);
		}else{
			// }eARs[
			destination->setMaterial(material_->copy(copyMask));
		}
	}
}
//------------------------------------------------------------------------------
// ċAIj
int Mesh::recursiveDestroy(Mesh* mesh){
	Assert(mesh != NULL);
	int result = 0;
	// bVf[^̔j
	MeshData* meshData = mesh->getMeshData();
	if(meshData != NULL){
		mesh->setMeshData(NULL);
		result += MeshData::destroy(meshData);
	}
	// }eA̔j
	Material* material = mesh->getMaterial();
	if(material != NULL){
		mesh->setMaterial(NULL);
		result += Material::recursiveDestroy(material);
	}
	// e폜
	Model* parent = mesh->getParent();
	if(parent != NULL){ parent->removeMesh(mesh); }
	// ̔j
	MeshManager* manager = mesh->getScene()->getMeshManager();
	if(manager->destroy(mesh) == 0){ result++; }
	return result;
}
//------------------------------------------------------------------------------
// 
void Mesh::traverse(const Matrix34& parentMatrix,
	bool parentEnabled, bool parentScaled, bool parentChanged){
	globalEnabled_ = parentEnabled && isEnabled();
	// parentEnabled(false) && isEnabled(true)̏ꍇ̓oEfBOvZ
	if(!isEnabled()){ return; }
	globalScaled_ = parentScaled;
	// eɕύXAoEfBOɕύX΍ČvZ
	if(parentChanged || meshData_->isBoundingChanged()){
		// oEfBO{bNXZo
		worldBoundingBox_ = getBoundingBox().transform(parentMatrix);
		if(globalScaled_){
			worldBoundingSphere_ =
				getBoundingSphere().scaledTransform(parentMatrix);
		}else{
			worldBoundingSphere_ = getBoundingSphere().transform(parentMatrix);
		}
		meshData_->clearBoundingChanged();
	}
}
//------------------------------------------------------------------------------
// oEfBO{bNX
//------------------------------------------------------------------------------
// oEfBO{bNX̐ݒ
void Mesh::setBoundingBox(const AxisAlignedBox& boundingBox){
	Assert(meshData_ != NULL);
	meshData_->setBoundingBox(boundingBox);
}
//------------------------------------------------------------------------------
// oEfBO{bNX̎擾
const AxisAlignedBox& Mesh::getBoundingBox() const{
	Assert(meshData_ != NULL);
	return meshData_->getBoundingBox();
}
//------------------------------------------------------------------------------
// oEfBOXtBA
//------------------------------------------------------------------------------
// oEfBOXtBA̐ݒ
void Mesh::setBoundingSphere(const Sphere& boundingSphere){
	Assert(meshData_ != NULL);
	meshData_->setBoundingSphere(boundingSphere);
}
//------------------------------------------------------------------------------
// oEfBOXtBA̎擾
const Sphere& Mesh::getBoundingSphere() const{
	Assert(meshData_ != NULL);
	return meshData_->getBoundingSphere();
}
//------------------------------------------------------------------------------
// S̎擾
const Vector3& Mesh::getCenter() const{
	Assert(meshData_ != NULL);
	return meshData_->getBoundingSphere().getCenter();
}
//------------------------------------------------------------------------------
// bVf[^C^[tF[X
//------------------------------------------------------------------------------
// bVf[^̐ݒ
void Mesh::setMeshData(MeshData* meshData){
	if(meshData_ != NULL){ meshData_->removeReference(this); }
	meshData_ = meshData;
	if(meshData_ != NULL){ meshData_->addReference(this); }
}
//------------------------------------------------------------------------------
// }eAC^[tF[X
//------------------------------------------------------------------------------
// }eA̐ݒ
void Mesh::setMaterial(Material* material){
	if(material_ != NULL){ material_->removeReference(this); }
	material_ = material;
	if(material_ != NULL){ material_->addReference(this); }
}
//------------------------------------------------------------------------------
// v~eBu^Cv
//------------------------------------------------------------------------------
// v~eBu^CvCfbNXǂ
bool Mesh::primitiveTypeHasIndex(PrimitiveType primitiveType){
	if(primitiveType == Mesh::indexedTriangleList){
		return true;
	}
	return false;
}
//------------------------------------------------------------------------------
// v~eBu^Cv當ւ̕ϊ
String Mesh::primitiveTypeToString(PrimitiveType primitiveType){
	Assert(primitiveType >= 0);
	Assert(primitiveType < ptMax);
	return primitiveTypeStringTable[primitiveType];
}
//------------------------------------------------------------------------------
// 񂩂v~eBu^Cvւ̕ϊ
Mesh::PrimitiveType Mesh::primitiveTypeFromString(
	const String& primitiveTypeString){
	for(int i = 0; i < ptMax; i++){
		if(primitiveTypeStringTable[i].equals(primitiveTypeString)){
			return PrimitiveType(i);
		}
	}
	ErrorOut("Mesh::primitiveTypeFromString() " + primitiveTypeString);
	return ptMax;
}
//-----------------------------------------------------------------------------
// bVC^[tF[X
//-----------------------------------------------------------------------------
// v~eBu^Cv̐ݒ
void Mesh::setPrimitiveType(Mesh::PrimitiveType primitiveType){
	getMeshData()->setPrimitiveType(primitiveType);
}
//-----------------------------------------------------------------------------
// v~eBu^Cv̎擾
Mesh::PrimitiveType Mesh::getPrimitiveType() const{
	return getMeshData()->getPrimitiveType();
}
//-----------------------------------------------------------------------------
// v~eBuJEg̎擾
int Mesh::getPrimitiveCount() const{
	return getMeshData()->getPrimitiveCount();
}
//-----------------------------------------------------------------------------
// Op̎擾
Triangle Mesh::getTriangle(int index) const{
	return getMeshData()->getTriangle(index);
}
//-----------------------------------------------------------------------------
// _CfbNXǂ
bool Mesh::hasVertexIndices() const{
	return getMeshData()->hasVertexIndices();
}
//-----------------------------------------------------------------------------
// _CfbNX̐ݒ
void Mesh::setVertexIndexCount(int vertexIndexCount){
	getMeshData()->setVertexIndexCount(vertexIndexCount);
}
//-----------------------------------------------------------------------------
// _CfbNX̎擾
int Mesh::getVertexIndexCount() const{
	return getMeshData()->getVertexIndexCount();
}
//-----------------------------------------------------------------------------
// _CfbNX̐ݒ
void Mesh::setVertexIndex(int index, int vertexIndex){
	getMeshData()->setVertexIndex(index, vertexIndex);
}
//-----------------------------------------------------------------------------
// _CfbNX̎擾
int Mesh::getVertexIndex(int index) const{
	return getMeshData()->getVertexIndex(index);
}
//-----------------------------------------------------------------------------
// _CfbNXz̎擾
const u_short* Mesh::getVertexIndexArray(){
	return getMeshData()->getVertexIndexArray();
}
//-----------------------------------------------------------------------------
// _̐ݒ
void Mesh::setVertexCount(int vertexCount){
	getMeshData()->setVertexCount(vertexCount);
}
//-----------------------------------------------------------------------------
// _̎擾
int Mesh::getVertexCount() const{ return getMeshData()->getVertexCount(); }
//-----------------------------------------------------------------------------
// ʒu̐ݒ
void Mesh::setPosition(int index, const Vector3& position){
	getMeshData()->setPosition(index, position);
}
//-----------------------------------------------------------------------------
// ʒu̎擾
const Vector3& Mesh::getPosition(int index) const{
	return getMeshData()->getPosition(index);
}
//-----------------------------------------------------------------------------
// ʒuz̎擾
const Vector3* Mesh::getPositionArray() const{
	return getMeshData()->getPositionArray();
}
//-----------------------------------------------------------------------------
// @Lɂ邩ǂ
void Mesh::enableNormal(bool normalFlag){
	getMeshData()->enableNormal(normalFlag);
}
//-----------------------------------------------------------------------------
// @Lǂ
bool Mesh::hasNormal() const{ return getMeshData()->hasNormal(); }
//-----------------------------------------------------------------------------
// @̐ݒ
void Mesh::setNormal(int index, const Vector3& normal){
	getMeshData()->setNormal(index, normal);
}
//-----------------------------------------------------------------------------
// @̎擾
const Vector3& Mesh::getNormal(int index) const{
	return getMeshData()->getNormal(index);
}
//-----------------------------------------------------------------------------
// @z̎擾
const Vector3* Mesh::getNormalArray() const{
	return getMeshData()->getNormalArray();
}
//-----------------------------------------------------------------------------
// J[Lɂ邩ǂ
void Mesh::enableColor(bool colorFlag){
	getMeshData()->enableColor(colorFlag);
}
//-----------------------------------------------------------------------------
// J[Lǂ
bool Mesh::hasColor() const{ return getMeshData()->hasColor(); }
//-----------------------------------------------------------------------------
// J[̐ݒ
void Mesh::setColor(int index, const Color4c& color){
	getMeshData()->setColor(index, color);
}
//-----------------------------------------------------------------------------
// J[̎擾
const Color4c& Mesh::getColor(int index) const{
	return getMeshData()->getColor(index);
}
//-----------------------------------------------------------------------------
// J[z̎擾
const Color4c* Mesh::getColorArray() const{
	return getMeshData()->getColorArray();
}
//-----------------------------------------------------------------------------
// eNX`WZbg̐ݒ
void Mesh::setTexCoordSetCount(int texCoordSetCount){
	getMeshData()->setTexCoordSetCount(texCoordSetCount);
}
//-----------------------------------------------------------------------------
// eNX`WZbg̐ݒ
int Mesh::getTexCoordSetCount() const{
	return getMeshData()->getTexCoordSetCount();
}
//-----------------------------------------------------------------------------
// eNX`W^Cv̐ݒ
void Mesh::setTexCoordType(int texCoordSet, TexCoord::Type texCoordType){
	getMeshData()->setTexCoordType(texCoordSet, texCoordType);
}
//-----------------------------------------------------------------------------
// eNX`W^Cv̎擾
TexCoord::Type Mesh::getTexCoordType(int texCoordSet) const{
	return getMeshData()->getTexCoordType(texCoordSet);
}
//-----------------------------------------------------------------------------
// eNX`W^Cvz̎擾
const TexCoord::Type* Mesh::getTexCoordTypeArray() const{
	return getMeshData()->getTexCoordTypeArray();
}
//-----------------------------------------------------------------------------
// eNX`W̐ݒ
void Mesh::setTexCoord(
	int index, int texCoordSet, const float* texCoord, int numTexCoord){
	getMeshData()->setTexCoord(index, texCoordSet, texCoord, numTexCoord);
}
//-----------------------------------------------------------------------------
// eNX`Wz̎擾
const float* const* Mesh::getTexCoordArray() const{
	return getMeshData()->getTexCoordArray();
}
//-----------------------------------------------------------------------------
// eNX`Wz̎擾
const float* Mesh::getTexCoordArray(int texCoordSet) const{
	return getMeshData()->getTexCoordArray(texCoordSet);
}
//-----------------------------------------------------------------------------
// eNX`WzTCY̎擾
int Mesh::getTexCoordArraySize(int texCoordSet) const{
	return getMeshData()->getTexCoordArraySize(texCoordSet);
}
//-----------------------------------------------------------------------------
// ꎟeNX`W̐ݒ
void Mesh::setTexCoord1(int index, int texCoordSet, const TexCoord1& texCoord){
	getMeshData()->setTexCoord1(index, texCoordSet, texCoord);
}
//-----------------------------------------------------------------------------
// ꎟeNX`W̎擾
const TexCoord1& Mesh::getTexCoord1(int index, int texCoordSet) const{
	return getMeshData()->getTexCoord1(index, texCoordSet);
}
//-----------------------------------------------------------------------------
// ꎟeNX`Wz̎擾
const TexCoord1* Mesh::getTexCoord1Array(int texCoordSet) const{
	return getMeshData()->getTexCoord1Array(texCoordSet);
}
//-----------------------------------------------------------------------------
// 񎟌eNX`W̐ݒ
void Mesh::setTexCoord2(int index, int texCoordSet, const TexCoord2& texCoord){
	getMeshData()->setTexCoord2(index, texCoordSet, texCoord);
}
//-----------------------------------------------------------------------------
// 񎟌eNX`W̎擾
const TexCoord2& Mesh::getTexCoord2(int index, int texCoordSet) const{
	return getMeshData()->getTexCoord2(index, texCoordSet);
}
//-----------------------------------------------------------------------------
// 񎟌eNX`Wz̎擾
const TexCoord2* Mesh::getTexCoord2Array(int texCoordSet) const{
	return getMeshData()->getTexCoord2Array(texCoordSet);
}
//-----------------------------------------------------------------------------
// OeNX`W̐ݒ
void Mesh::setTexCoord3(int index, int texCoordSet, const TexCoord3& texCoord){
	getMeshData()->setTexCoord3(index, texCoordSet, texCoord);
}
//-----------------------------------------------------------------------------
// OeNX`W̎擾
const TexCoord3& Mesh::getTexCoord3(int index, int texCoordSet) const{
	return getMeshData()->getTexCoord3(index, texCoordSet);
}
//-----------------------------------------------------------------------------
// OeNX`Wz̎擾
const TexCoord3* Mesh::getTexCoord3Array(int texCoordSet) const{
	return getMeshData()->getTexCoord3Array(texCoordSet);
}
//-----------------------------------------------------------------------------
// leNX`W̐ݒ
void Mesh::setTexCoord4(int index, int texCoordSet, const TexCoord4& texCoord){
	getMeshData()->setTexCoord4(index, texCoordSet, texCoord);
}
//-----------------------------------------------------------------------------
// leNX`W̎擾
const TexCoord4& Mesh::getTexCoord4(int index, int texCoordSet) const{
	return getMeshData()->getTexCoord4(index, texCoordSet);
}
//-----------------------------------------------------------------------------
// leNX`Wz̎擾
const TexCoord4* Mesh::getTexCoord4Array(int texCoordSet) const{
	return getMeshData()->getTexCoord4Array(texCoordSet);
}
//-----------------------------------------------------------------------------
// _{[̐ݒ
void Mesh::setBonesPerVertex(int bonesPerVertex){
	getMeshData()->setBonesPerVertex(bonesPerVertex);
}
//-----------------------------------------------------------------------------
// _{[̎擾
int Mesh::getBonesPerVertex() const{
	return getMeshData()->getBonesPerVertex();
}
//-----------------------------------------------------------------------------
// {[CfbNXLǂ
bool Mesh::hasBoneIndex() const{ return getMeshData()->hasBoneIndex(); }
//-----------------------------------------------------------------------------
// {[CfbNX̐ݒ
void Mesh::setBoneIndex(
	int vertexIndex, int weightIndex, u_char boneIndex){
	getMeshData()->setBoneIndex(
		vertexIndex, weightIndex, boneIndex);
}
//-----------------------------------------------------------------------------
// {[CfbNX̐ݒ
void Mesh::setBoneIndex(int vertexIndex, u_char boneIndex){
	getMeshData()->setBoneIndex(vertexIndex, boneIndex);
}
//-----------------------------------------------------------------------------
// {[CfbNX̎擾
u_char Mesh::getBoneIndex(int vertexIndex, int weightIndex) const{
	return getMeshData()->getBoneIndex(vertexIndex, weightIndex);
}
//-----------------------------------------------------------------------------
// {[CfbNX̎擾
u_char Mesh::getBoneIndex(int vertexIndex) const{
	return getMeshData()->getBoneIndex(vertexIndex);
}
//-----------------------------------------------------------------------------
// {[CfbNXz̎擾
const u_char* Mesh::getBoneIndexArray() const{
	return getMeshData()->getBoneIndexArray();
}
//-----------------------------------------------------------------------------
// _EFCg̎擾
int Mesh::getWeightsPerVertex() const{
	return getMeshData()->getWeightsPerVertex();
}
//-----------------------------------------------------------------------------
// EFCgLǂ
bool Mesh::hasWeight() const{ return getMeshData()->hasWeight(); }
//-----------------------------------------------------------------------------
// EFCg̐ݒ
void Mesh::setWeight(int vertexIndex, int weightIndex, float weight){
	getMeshData()->setWeight(vertexIndex, weightIndex, weight);
}
//-----------------------------------------------------------------------------
// EFCg̎擾
float Mesh::getWeight(int vertexIndex, int weightIndex) const{
	return getMeshData()->getWeight(vertexIndex, weightIndex);
}
//-----------------------------------------------------------------------------
// EFCgz̎擾
const float* Mesh::getWeightArray() const{
	return getMeshData()->getWeightArray();
}
//-----------------------------------------------------------------------------
// obt@ANZX
//-----------------------------------------------------------------------------
// CfbNXobt@̎擾
Direct3DIndexBuffer* Mesh::getIndexBuffer(){
	return getMeshData()->getIndexBuffer();
}
//-----------------------------------------------------------------------------
// _Lq̎擾
Direct3DVertexDeclaration* Mesh::getVertexDeclaration(){
	return getMeshData()->getVertexDeclaration();
}
//-----------------------------------------------------------------------------
// _TCY̎擾
int Mesh::getVertexSize(){ return getMeshData()->getVertexSize(); }
//-----------------------------------------------------------------------------
// _obt@̍\z
Direct3DVertexBuffer* Mesh::getVertexBuffer(){
	return getMeshData()->getVertexBuffer();
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
