//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * LN^bV
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/Mesh/CharacterMesh.h"
#include "Graphics/Renderer/RenderingDevice.h"
#include "Graphics/Scene/Scene.h"
#include "Graphics/Mesh/MeshManager.h"
#include "Graphics/Model/CharacterModel.h"
#include "Graphics/MeshData/MeshData.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
CharacterMesh::CharacterMesh(const String& name, Scene* scene) :
	Mesh(name, scene), vertexBuffer_(NULL), vertexDeclaration_(NULL),
	vertexSize_(0), deformedVertexCount_(0), deformedPosition_(NULL),
	deformedNormal_(NULL){
}
//------------------------------------------------------------------------------
// fXgN^
CharacterMesh::~CharacterMesh(){
	SafeArrayDelete(deformedNormal_);
	SafeArrayDelete(deformedPosition_);
	SafeRelease(vertexBuffer_);
	SafeRelease(vertexDeclaration_);
}
//------------------------------------------------------------------------------
// LN^bṼRs[
CharacterMesh* CharacterMesh::copyCharacterMesh(u_int copyMask) const{
	MeshManager* manager = scene_->getMeshManager();
	CharacterMesh* copyMesh =
		manager->createCharacterMesh(manager->rename(name_));
	// bV̒lRs[
	copyMeshValue(copyMesh, copyMask);
	return copyMesh;
}
//------------------------------------------------------------------------------
// 
void CharacterMesh::traverse(const Matrix34& parentMatrix,
	bool parentEnabled, bool parentScaled, bool parentChanged){
	Mesh::traverse(parentMatrix, parentEnabled, parentScaled, parentChanged);
	if(!isGlobalEnabled()){ SafeRelease(vertexBuffer_); }
}
//------------------------------------------------------------------------------
// LN^ό`
//------------------------------------------------------------------------------
// LN^ό`
bool CharacterMesh::characterDeform(){
	// eobt@쐬
// xVertexBufferꂽMeshData̒_tH[}bgɕύXƂȂ
// Kv͊ȂAȂ񂩑΍􂤂قAbVf[^eǂH
// ȂȂ
	if(vertexDeclaration_ == NULL){ createVertexDeclaration(); }
	if(vertexBuffer_ == NULL){ createVertexBuffer(); }
	// ό`
	deform();
	// _obt@ZbgAbv
	setupVertexBuffer();
	return true;
}
//------------------------------------------------------------------------------
// ό`
void CharacterMesh::deform(){
	// ό`sz擾
	CharacterModel* characterModel = getParent()->castCharacterModel();
	Assert(characterModel != NULL);
	// {[s̍\zB\zς݂ł΍č\zȂ
	characterModel->buildBoneMatrix();
	const Matrix34* positionDeformMatrixArray =
		characterModel->getPositionDeformMatrixArray();

	// ό`f[^obt@쐬
	int vertexCount = getVertexCount();
	if((deformedPosition_ == NULL) || (deformedVertexCount_ != vertexCount)){
		SafeArrayDelete(deformedNormal_);
		SafeArrayDelete(deformedPosition_);
		deformedVertexCount_ = vertexCount;
		deformedPosition_ = new Vector3[deformedVertexCount_];
		if(hasNormal()){ deformedNormal_ = new Vector3[deformedVertexCount_]; }
	}

	// ό`
	if(!hasNormal()){
		// ʒû
		if(getBonesPerVertex() == 1){
			stitchingDeformP(positionDeformMatrixArray);
		}else{
			skinningDeformP(positionDeformMatrixArray);
		}
	}else if(!characterModel->isBoneScaled()){
		// ʒuA@
		if(getBonesPerVertex() == 1){
			stitchingDeformPN(positionDeformMatrixArray);
		}else{
			skinningDeformPN(positionDeformMatrixArray);
		}
	}else{
		// ʒuA@AXP[
		const Matrix33* normalDeformMatrixArray =
			characterModel->getNormalDeformMatrixArray();
		if(getBonesPerVertex() == 1){
			stitchingDeformPN(
				positionDeformMatrixArray, normalDeformMatrixArray);
		}else{
			skinningDeformPN(
				positionDeformMatrixArray, normalDeformMatrixArray);
		}
	}
}
//------------------------------------------------------------------------------
// XLjOό`
//------------------------------------------------------------------------------
// XLjOό`
void CharacterMesh::skinningDeformP(const Matrix34* positionDeformMatrixArray){
	int vertexCount = getVertexCount();
	int weightsPerVertex = getWeightsPerVertex();
	int bonesPerVertex = getBonesPerVertex();
	const float* weightArray = getWeightArray();
	const u_char* boneIndexArray = getBoneIndexArray();
	const Vector3* readPosition = getPositionArray();
	Vector3* writePosition = deformedPosition_;
	for(int i = 0; i < vertexCount; i++){
		int weightOffset = i * weightsPerVertex;
		int boneOffset = i * bonesPerVertex;
		Vector3 accumulatePosition(Vector3::zero);
		Vector3 accumulateNormal(Vector3::zero);
		float lastWeight = 1.f;
		for(int j = 0; j < weightsPerVertex; j++){
			// ό`O
			float weight = weightArray[weightOffset + j];
			lastWeight -= weight;
			if(weight <= Math::epsilon){ continue; }
			const Matrix34& positionDeformMatrix =
				positionDeformMatrixArray[boneIndexArray[boneOffset + j]];
			// ʒu̕ό`
			accumulatePosition +=
				positionDeformMatrix * (*readPosition) * weight;
		}
		// Ō̃{[̕ό`
		if(lastWeight > Math::epsilon){
			const Matrix34& positionDeformMatrix = positionDeformMatrixArray[
				boneIndexArray[boneOffset + weightsPerVertex]];
			// ʒu̕ό`
			accumulatePosition +=
				positionDeformMatrix * (*readPosition) * lastWeight;
		}
		(*writePosition) = accumulatePosition;
		readPosition++;
		writePosition++;
	}
}
//------------------------------------------------------------------------------
// XLjOό`
void CharacterMesh::skinningDeformPN(const Matrix34* positionDeformMatrixArray){
	int vertexCount = getVertexCount();
	int weightsPerVertex = getWeightsPerVertex();
	int bonesPerVertex = getBonesPerVertex();
	const float* weightArray = getWeightArray();
	const u_char* boneIndexArray = getBoneIndexArray();
	const Vector3* readPosition = getPositionArray();
	const Vector3* readNormal = getNormalArray();
	Vector3* writePosition = deformedPosition_;
	Vector3* writeNormal = deformedNormal_;
	for(int i = 0; i < vertexCount; i++){
		int weightOffset = i * weightsPerVertex;
		int boneOffset = i * bonesPerVertex;
		Vector3 accumulatePosition(Vector3::zero);
		Vector3 accumulateNormal(Vector3::zero);
		float lastWeight = 1.f;
		for(int j = 0; j < weightsPerVertex; j++){
			// ό`O
			float weight = weightArray[weightOffset + j];
			lastWeight -= weight;
			if(weight <= Math::epsilon){ continue; }
			const Matrix34& positionDeformMatrix =
				positionDeformMatrixArray[boneIndexArray[boneOffset + j]];
			// ʒu̕ό`
			accumulatePosition +=
				positionDeformMatrix * (*readPosition) * weight;
			// @̕ό`
			accumulateNormal +=
				positionDeformMatrix.multiply33(*readNormal) * weight;
		}
		// Ō̃{[̕ό`
		if(lastWeight > Math::epsilon){
			const Matrix34& positionDeformMatrix = positionDeformMatrixArray[
				boneIndexArray[boneOffset + weightsPerVertex]];
			// ʒu̕ό`
			accumulatePosition +=
				positionDeformMatrix * (*readPosition) * lastWeight;
			// @̕ό`
			accumulateNormal +=
				positionDeformMatrix.multiply33(*readNormal) * lastWeight;
		}
		(*writePosition) = accumulatePosition;
		readPosition++;
		writePosition++;
		(*writeNormal) = accumulateNormal;
		readNormal++;
		writeNormal++;
	}
}
//------------------------------------------------------------------------------
// XLjOό`
void CharacterMesh::skinningDeformPN(const Matrix34* positionDeformMatrixArray,
	const Matrix33* normalDeformMatrixArray){
	int vertexCount = getVertexCount();
	int weightsPerVertex = getWeightsPerVertex();
	int bonesPerVertex = getBonesPerVertex();
	const float* weightArray = getWeightArray();
	const u_char* boneIndexArray = getBoneIndexArray();
	const Vector3* readPosition = getPositionArray();
	const Vector3* readNormal = getNormalArray();
	bool calcNormal = hasNormal();
	Vector3* writePosition = deformedPosition_;
	Vector3* writeNormal = deformedNormal_;
	for(int i = 0; i < vertexCount; i++){
		int weightOffset = i * weightsPerVertex;
		int boneOffset = i * bonesPerVertex;
		Vector3 accumulatePosition(Vector3::zero);
		Vector3 accumulateNormal(Vector3::zero);
		float lastWeight = 1.f;
		for(int j = 0; j < weightsPerVertex; j++){
			// ό`O
			float weight = weightArray[weightOffset + j];
			lastWeight -= weight;
			if(weight <= Math::epsilon){ continue; }
			// ʒu̕ό`
			const Matrix34& positionDeformMatrix =
				positionDeformMatrixArray[boneIndexArray[boneOffset + j]];
			accumulatePosition +=
				positionDeformMatrix * (*readPosition) * weight;
			// @̕ό`
			const Matrix33& normalDeformMatrix =
				normalDeformMatrixArray[boneIndexArray[boneOffset + j]];
			accumulateNormal += normalDeformMatrix * (*readNormal) * weight;
		}
		// Ō̃{[̕ό`
		if(lastWeight > Math::epsilon){
			// ʒu̕ό`
			const Matrix34& positionDeformMatrix = positionDeformMatrixArray[
				boneIndexArray[boneOffset + weightsPerVertex]];
			accumulatePosition +=
				positionDeformMatrix * (*readPosition) * lastWeight;
			// @̕ό`
			const Matrix33& normalDeformMatrix = normalDeformMatrixArray[
				boneIndexArray[boneOffset + weightsPerVertex]];
			accumulateNormal +=
				normalDeformMatrix * (*readNormal) * lastWeight;
		}
		(*writePosition) = accumulatePosition;
		readPosition++;
		writePosition++;
		(*writeNormal) = accumulateNormal;
		readNormal++;
		writeNormal++;
	}
}
//------------------------------------------------------------------------------
// XeBb`Oό`
//------------------------------------------------------------------------------
// XeBb`Oό`
void CharacterMesh::stitchingDeformP(const Matrix34* positionDeformMatrixArray){
	int vertexCount = getVertexCount();
	const u_char* boneIndexArray = getBoneIndexArray();
	const Vector3* readPosition = getPositionArray();
	Vector3* writePosition = deformedPosition_;
	for(int i = 0; i < vertexCount; i++){
		const Matrix34& positionDeformMatrix =
			positionDeformMatrixArray[boneIndexArray[i]];
		// ʒu̕ό`
		(*writePosition) = positionDeformMatrix * (*readPosition);
		readPosition++;
		writePosition++;
	}
}
//------------------------------------------------------------------------------
// XeBb`Oό`
void CharacterMesh::stitchingDeformPN(
	const Matrix34* positionDeformMatrixArray){
	int vertexCount = getVertexCount();
	const u_char* boneIndexArray = getBoneIndexArray();
	const Vector3* readPosition = getPositionArray();
	const Vector3* readNormal = getNormalArray();
	Vector3* writePosition = deformedPosition_;
	Vector3* writeNormal = deformedNormal_;
	for(int i = 0; i < vertexCount; i++){
		const Matrix34& positionDeformMatrix =
			positionDeformMatrixArray[boneIndexArray[i]];
		// ʒu̕ό`
		(*writePosition) = positionDeformMatrix * (*readPosition);
		readPosition++;
		writePosition++;
		// @̕ό`
		(*writeNormal) = positionDeformMatrix.multiply33(*readNormal);
		readNormal++;
		writeNormal++;
	}
}
//------------------------------------------------------------------------------
// XeBb`Oό`
void CharacterMesh::stitchingDeformPN(const Matrix34* positionDeformMatrixArray,
	const Matrix33* normalDeformMatrixArray){
	int vertexCount = getVertexCount();
	const u_char* boneIndexArray = getBoneIndexArray();
	const Vector3* readPosition = getPositionArray();
	const Vector3* readNormal = getNormalArray();
	Vector3* writePosition = deformedPosition_;
	Vector3* writeNormal = deformedNormal_;
	for(int i = 0; i < vertexCount; i++){
		const Matrix34& positionDeformMatrix =
			positionDeformMatrixArray[boneIndexArray[i]];
		const Matrix33& normalDeformMatrix =
			normalDeformMatrixArray[boneIndexArray[i]];
		// ʒu̕ό`
		(*writePosition) = positionDeformMatrix * (*readPosition);
		readPosition++;
		writePosition++;
		// @̕ό`
		(*writeNormal) = normalDeformMatrix * (*readNormal);
		readNormal++;
		writeNormal++;
	}
}
//------------------------------------------------------------------------------
// _obt@ZbgAbv
//------------------------------------------------------------------------------
// _obt@ZbgAbv
bool CharacterMesh::setupVertexBuffer(){
	// I_obt@̏
	Assert(vertexDeclaration_ != NULL);
	Assert(vertexBuffer_ != NULL);
	RenderingDevice* device = RenderingDevice::getInstance();
	int vertexCount = getVertexCount();
	// ό`ʒuƖ@gpAWeightBoneIndexgpȂ
	device->writeDynamicVertexBuffer(vertexBuffer_, vertexSize_ * vertexCount,
		vertexCount, deformedPosition_, 0, NULL, 0, NULL, deformedNormal_,
		getColorArray(), getTexCoordSetCount(), getTexCoordTypeArray(),
		getTexCoordArray());
	return true;
}
//------------------------------------------------------------------------------
// OtBbNXobt@
//------------------------------------------------------------------------------
// _Lq̍\z
bool CharacterMesh::createVertexDeclaration(){
	Assert(vertexDeclaration_ == NULL);
	Assert(vertexSize_ == 0);
	// \tgEFAŃLN^ό`ŝWeightBoneIndex0ɂ
	RenderingDevice* device = RenderingDevice::getInstance();
	vertexSize_ = device->createVertexDeclaration(
		&vertexDeclaration_, true, 0, 0, hasNormal(), hasColor(),
		getTexCoordSetCount(), getTexCoordTypeArray());
	return true;
}
//------------------------------------------------------------------------------
// _obt@̍\z
bool CharacterMesh::createVertexBuffer(){
	Assert(vertexBuffer_ == NULL);
	// ɒ_Lq\zĂKv
	if(vertexDeclaration_ == NULL){ createVertexDeclaration(); }
	Assert(vertexDeclaration_ != NULL);
	// I_obt@̍쐬
	RenderingDevice* device = RenderingDevice::getInstance();
	int vertexCount = getVertexCount();
	u_int bufferSize = vertexSize_ * vertexCount;
	vertexBuffer_ = device->createDynamicVertexBuffer(bufferSize);
	return true;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
