//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * }eA
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/Material/Material.h"
#include "Graphics/Material/MaterialManager.h"
#include "Graphics/Texture/Texture.h"
#include "Graphics/Renderer/RenderingDevice.h"
#include "Graphics/Renderer/DrawRequest.h"

namespace Lamp{

// uh[he[u
const String Material::blendModeStringTable[] = {
	"Disable",
	"Add",
	"Subtract",
	"InverseSubtract",
	"Minimum",
	"Maximum",
};

// uhXe[ge[u
const String Material::blendStateStringTable[] = {
	"Zero",
	"One",
	"SourceColor",
	"InverseSourceColor",
	"SourceAlpha",
	"InverseSourceAlpha",
	"SourceAlphaSaturate",
	"DestinationColor",
	"InverseDestinationColor",
	"DestinationAlpha",
	"InverseDestinationAlpha",
};

// tHOIvVe[u
const String Material::fogOptionStringTable[] = {
	"None",
	"Disable",
	"Black",
};

//------------------------------------------------------------------------------
// RXgN^
Material::Material(const String& name, Scene* scene) :
	SceneObject(name, scene), startStateBlock_(NULL), endStateBlock_(NULL),
	priority_(0), blendMode_(blendModeDisable),
	blendSource_(blendStateSourceAlpha),
	blendDestination_(blendStateInverseSourceAlpha), alpha_(1.f),
	fogOption_(fogOptionNone), lightMask_(1), pipelineMode_(pipelineModeNone),
	alphaBlend_(false), zWrite_(true), zTest_(true), hasStateChanged_(true){
}
//------------------------------------------------------------------------------
// fXgN^
Material::~Material(){
	releaseStateBlock();
}
//------------------------------------------------------------------------------
// }eA̒lRs[
void Material::copyMaterialValue(Material* destination) const{
	destination->setPriority(priority_);
	destination->setBlendMode(blendMode_);
	destination->setBlendSource(blendSource_);
	destination->setBlendDestination(blendDestination_);
	destination->setAlpha(alpha_);
	destination->setFogOption(fogOption_);
	destination->setLightMask(lightMask_);
	destination->setZWrite(zWrite_);
	destination->setZTest(zTest_);
}
//------------------------------------------------------------------------------
// ċAIj
int Material::recursiveDestroy(Material* material){
	Assert(material != NULL);
	int result = 0;
	// q̔j
	if(material->getReferenceCount() == 0){
		result += material->destroyChildren();
	}
	// ̔j
	MaterialManager* manager = material->getScene()->getMaterialManager();
	if(manager->destroy(material) == 0){
		result++;
	}
	return result;
}
//------------------------------------------------------------------------------
// `̊Jn
void Material::drawStart(){
	// Xe[gύXĂ΃Xe[gubNč\z
	if(hasStateChanged_){
		releaseStateBlock();
		buildStateBlock(&startStateBlock_, &endStateBlock_);
		hasStateChanged_ = false;
	}
	// JnXe[gubNKp
	if(startStateBlock_ != NULL){
		RenderingDevice::getInstance()->applyStateBlock(startStateBlock_);
	}
}
//------------------------------------------------------------------------------
// `̏I
void Material::drawEnd(){
	// Xe[gύXĂ΃Xe[gubNč\z
	if(hasStateChanged_){
		releaseStateBlock();
		buildStateBlock(&startStateBlock_, &endStateBlock_);
		hasStateChanged_ = false;
	}
	// IXe[gubNKp
	if(endStateBlock_ != NULL){
		RenderingDevice::getInstance()->applyStateBlock(endStateBlock_);
	}
}
//------------------------------------------------------------------------------
// `̃ZbgAbv
void Material::drawSetup(DrawRequest* request){
	// }eAȂύXȂ
	if(request->isMaterialChanged()){
		// O̕`I
		Material* preMaterial = request->getPreMaterial();
		if(preMaterial != NULL){ preMaterial->drawEnd(); }
		// `Jn
		drawStart();
	}
}
//------------------------------------------------------------------------------
// ufBO
//------------------------------------------------------------------------------
// uh[h當ւ̕ϊ
const String& Material::blendModeToString(BlendMode blendMode){
	Assert(blendMode >= 0);
	Assert(blendMode < blendModeMax);
	return blendModeStringTable[blendMode];
}
//------------------------------------------------------------------------------
// 񂩂uh[hւ̕ϊ
Material::BlendMode Material::blendModeFromString(
	const String& blendModeString){
	for(int i = 0; i < blendModeMax; i++){
		if(blendModeStringTable[i].equals(blendModeString)){
			return BlendMode(i);
		}
	}
	ErrorOut("Material::blendModeFromString() " + blendModeString);
	return blendModeMax;
}
//------------------------------------------------------------------------------
// uhXe[g當ւ̕ϊ
const String& Material::blendStateToString(BlendState blendState){
	Assert(blendState >= 0);
	Assert(blendState < blendStateMax);
	return blendStateStringTable[blendState];
}
//------------------------------------------------------------------------------
// 񂩂uhXe[gւ̕ϊ
Material::BlendState Material::blendStateFromString(
	const String& blendStateString){
	for(int i = 0; i < blendStateMax; i++){
		if(blendStateStringTable[i].equals(blendStateString)){
			return BlendState(i);
		}
	}
	ErrorOut("Material::blendStateFromString() " + blendStateString);
	return blendStateMax;
}
//------------------------------------------------------------------------------
// tHOIvV當ւ̕ϊ
const String& Material::fogOptionToString(FogOption fogOption){
	Assert(fogOption >= 0);
	Assert(fogOption < fogOptionMax);
	return fogOptionStringTable[fogOption];
}
//------------------------------------------------------------------------------
// 񂩂tHOIvVւ̕ϊ
Material::FogOption Material::fogOptionFromString(
	const String& fogOptionString){
	for(int i = 0; i < fogOptionMax; i++){
		if(fogOptionStringTable[i].equals(fogOptionString)){
			return FogOption(i);
		}
	}
	ErrorOut("Material::fogOptionFromString() " + fogOptionString);
	return fogOptionMax;
}
//------------------------------------------------------------------------------
// eNX`t@X
//------------------------------------------------------------------------------
// eNX`t@X̐ݒ
Texture* Material::setTextureReferense(
	Texture* nowTexture, Texture* newTexture){
	if(nowTexture != NULL){ nowTexture->removeReference(this); }
	if(newTexture != NULL){ newTexture->addReference(this); }
	stateChanged();
	return newTexture;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
