//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * eLXgV[Z[o
 * @author Junpee
 */

#include "LampBasic.h"
#include <direct.h>
#include "Graphics/InputOutput/TextSceneSaver.h"
#include "Core/InputOutput/TextFileWriter.h"
#include "Core/InputOutput/FilePath.h"
#include "Core/Codec/Tga/TargaSaver.h"

#include "Graphics/Scene/Scene.h"
#include "Graphics/Fog/Fog.h"
#include "Graphics/Light/LightManager.h"
#include "Graphics/SceneNode/SceneNodeManager.h"
#include "Graphics/Model/ModelManager.h"
#include "Graphics/Mesh/MeshManager.h"
#include "Graphics/MeshData/MeshDataManager.h"
#include "Graphics/Material/MaterialManager.h"
#include "Graphics/Texture/TextureManager.h"
#include "Graphics/Picture/PictureManager.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
TextSceneSaver::TextSceneSaver(){
}
//------------------------------------------------------------------------------
// fXgN^
TextSceneSaver::~TextSceneSaver(){
}
//------------------------------------------------------------------------------
// Z[u
void TextSceneSaver::save(const String& filePath, Scene* scene){
	FilePath path(filePath);
	TextFileWriter* textFileWriter = new TextFileWriter(filePath);
	save(textFileWriter, scene, path.getFolderPath());
	delete textFileWriter;
}
//------------------------------------------------------------------------------
// Z[u
void TextSceneSaver::save(TextWriter* textWriter, Scene* scene,
	const String& basePath){
	writer_ = textWriter;
	basePath_ = basePath;
	scene_ = scene;
	sceneNodeManager_ = scene->getSceneNodeManager();
	lightManager_ = scene->getLightManager();
	modelManager_ = scene->getModelManager();
	meshManager_ = scene->getMeshManager();
	meshDataManager_ = scene->getMeshDataManager();
	materialManager_ = scene->getMaterialManager();
	textureManager_ = scene->getTextureManager();
	pictureManager_ = scene->getPictureManager();

	// wb_̏o
	writeHeader();

	// IuWFNgXg̏o
	int objectCount = writeObjectList();

	// N̏o
	writeLink(objectCount);
}
//------------------------------------------------------------------------------
// wb_̏o
void TextSceneSaver::writeHeader(){
	writeBlockComment("Header");
	writer_->writeText("Header {\n");
	writer_->writeText("\ttype LampTextSceneFormat\n");
	writer_->writeText("\tversion 0_9_0\n");
	writer_->writeText("}\n\n");
}
//------------------------------------------------------------------------------
// IuWFNgXg̏o
int TextSceneSaver::writeObjectList(){
	int objectCount = 0;

	// tHȌo
	writeFog(scene_->getFog());

	// V[m[hXg̏o
	int sceneNodeCount = sceneNodeManager_->getCount();
	if(sceneNodeCount > 1){// [gm[h͏o͂Ȃ̂2ȏ
		writeBlockComment("SceneNode");
		writer_->writeText("SceneNode {\n");
		writeSceneNodeList();
		writer_->writeText("}\n\n");
	}
	objectCount += sceneNodeCount;

	// CgXg̏o
	int lightCount = lightManager_->getCount();
	if(lightCount > 0){
		writeBlockComment("Light");
		writer_->writeText("Light {\n");
		writeLightList();
		writer_->writeText("}\n\n");
	}
	// ݃IuWFNgJEg̓N̂߂ɎgpĂ̂
	// Cg̓IuWFNgɃJEgȂ

	// fXg̏o
	int modelCount = modelManager_->getCount();
	if(modelCount > 0){
		writeBlockComment("Model");
		writer_->writeText("Model {\n");
		writeModelList();
		writer_->writeText("}\n\n");
	}
	objectCount += modelCount;

	// bVXg̏o
	int meshCount = meshManager_->getCount();
	if(meshCount > 0){
		writeBlockComment("Mesh");
		writer_->writeText("Mesh {\n");
		writeMeshList();
		writer_->writeText("}\n\n");
	}
	objectCount += meshCount;

	// bVf[^Xg̏o
	int meshDataCount = meshDataManager_->getCount();
	if(meshDataCount > 0){
		writeBlockComment("MeshData");
		writer_->writeText("MeshData {\n");
		writeMeshDataList();
		writer_->writeText("}\n\n");
	}
	objectCount += meshDataCount;

	// }eAXg̏o
	int materialCount = materialManager_->getCount();
	if(materialCount > 0){
		writeBlockComment("Material");
		writer_->writeText("Material {\n");
		writeMaterialList();
		writer_->writeText("}\n\n");
	}
	objectCount += materialCount;

	// eNX`Xg̏o
	int textureCount = textureManager_->getCount();
	if(textureCount > 0){
		writeBlockComment("Texture");
		writer_->writeText("Texture {\n");
		writeTextureList();
		writer_->writeText("}\n\n");
	}
	objectCount += textureCount;

	// sN`Xg̏o
	int pictureCount = pictureManager_->getCount();
	if(pictureCount > 0){
		writeBlockComment("Picture");
		writer_->writeText("Picture {\n");
		for(int i = 0; i < pictureCount; i++){
			writePicture(pictureManager_->get(i));
		}
		writer_->writeText("}\n\n");
	}
	objectCount += pictureCount;

	return objectCount;
}
//------------------------------------------------------------------------------
// N̏o
void TextSceneSaver::writeLink(int objectCount){
	// IuWFNg2ȏȂ烊N݂
	if(objectCount > 1){ writeBlockComment("Link"); }

	// V[m[hN̏o
	int sceneNodeCount = sceneNodeManager_->getCount();
	if(sceneNodeCount > 1){
		writer_->writeText("SceneNodeLink {\n");
		for(int i = 0; i < sceneNodeCount; i++){
			writeSceneNodeLink(sceneNodeManager_->get(i));
		}
		writer_->writeText("}\n\n");
		writeLineComment();
	}

	// fN̏o
	int modelCount = modelManager_->getCount();
	if(modelCount > 0){
		writer_->writeText("ModelLink {\n");
		for(int i = 0; i < modelCount; i++){
			writeModelLink(modelManager_->get(i));
		}
		writer_->writeText("}\n\n");
		writeLineComment();
	}

	// bVN̏o
	int meshCount = meshManager_->getCount();
	if(meshCount > 0){
		writer_->writeText("MeshLink {\n");
		for(int i = 0; i < meshCount; i++){
			writeMeshLink(meshManager_->get(i));
		}
		writer_->writeText("}\n\n");
		writeLineComment();
	}

	// }eAeNX`N̏o
	// eNX`݂΃}eÃN
	int textureCount = textureManager_->getCount();
	if(textureCount > 0){
		writeMaterialLinkList();
		writeLineComment();
	}

	// eNX`N̏o
	if(textureCount > 0){
		writer_->writeText("TextureLink {\n");
		for(int i = 0; i < textureCount; i++){
			writeTextureLink(textureManager_->get(i));
		}
		writer_->writeText("}\n\n");
		writeLineComment();
	}
}
//------------------------------------------------------------------------------
// tHO
//------------------------------------------------------------------------------
// tHȌo
void TextSceneSaver::writeFog(Fog* fog){
	writeBlockComment("Fog");
	writer_->writeText("Fog {\n");
	// J[
	Color4c color = fog->getColor();
	writer_->writeFormat("\tcolor { %d %d %d %d }\n",
		color.r, color.g, color.b, color.a);
	// [h
	writer_->writeFormat("\tmode %s\n", fog->getModeString());
	// Zx
	writer_->writeFormat("\tdensity %.8f\n", fog->getDensity());
	// jA
	writer_->writeFormat("\tnear %.8f\n", fog->getNear());
	// t@[
	writer_->writeFormat("\tfar %.8f\n", fog->getFar());
	// LA
	writer_->writeText("\tenabled ");
	writeBool(fog->isEnabled());
	writer_->writeText("\n}\n\n");
}
//------------------------------------------------------------------------------
// V[m[h
//------------------------------------------------------------------------------
// V[m[hXg̏o
void TextSceneSaver::writeSceneNodeList(){
	int sceneNodeCount = sceneNodeManager_->getCount();
	// [gm[h͏o͂Ȃ
	for(int i = 1; i < sceneNodeCount; i++){
		SceneNode* sceneNode = sceneNodeManager_->get(i);
		if(sceneNode->isLODSceneNode()){
			writeLODSceneNode(sceneNode->castLODSceneNode());
		}else{
			writeSceneNode(sceneNode);
		}
	}
}
//------------------------------------------------------------------------------
// V[m[h̏o
void TextSceneSaver::writeSceneNode(SceneNode* sceneNode){
	writer_->writeText("\t");
	writer_->writeText(sceneNode->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Standard\n");
	// XP[
	Vector3 scale = sceneNode->getScale();
	writer_->writeFormat("\t\tscale { %.8f %.8f %.8f }\n",
		scale.x, scale.y, scale.z);
	// ]
	Vector3 rotation = sceneNode->getRotationXYZ();
	writer_->writeFormat("\t\trotation { %.8f %.8f %.8f }\n",
		rotation.x, rotation.y, rotation.z);
	// ړ
	Vector3 translation = sceneNode->getTranslation();
	writer_->writeFormat("\t\ttranslation { %.8f %.8f %.8f }\n",
		translation.x, translation.y, translation.z);
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(sceneNode->isEnabled());
	writer_->writeText("\n\t}\n\n");
}
//------------------------------------------------------------------------------
// xIufBeB[V[m[h̏o
void TextSceneSaver::writeLODSceneNode(LODSceneNode* sceneNode){
	writer_->writeText("\t");
	writer_->writeText(sceneNode->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype LOD\n");
	// XP[
	Vector3 scale = sceneNode->getScale();
	writer_->writeFormat("\t\tscale { %.8f %.8f %.8f }\n",
		scale.x, scale.y, scale.z);
	// ]
	Vector3 rotation = sceneNode->getRotationXYZ();
	writer_->writeFormat("\t\trotation { %.8f %.8f %.8f }\n",
		rotation.x, rotation.y, rotation.z);
	// ړ
	Vector3 translation = sceneNode->getTranslation();
	writer_->writeFormat("\t\ttranslation { %.8f %.8f %.8f }\n",
		translation.x, translation.y, translation.z);
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(sceneNode->isEnabled());
	writer_->writeText("\n");
	// LOD
	int lodThresholdCount = sceneNode->getLODThresholdCount();
	writer_->writeFormat("\t\tlodThresholdCount %d {\n", lodThresholdCount);
	// LODl
	for(int i = 0; i < lodThresholdCount; i++){
		writer_->writeFormat("\t\t\t%.8f\n", sceneNode->getLODThreshold(i));
	}
	writer_->writeText("\t\t}\n");
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// Cg
//------------------------------------------------------------------------------
// CgXg̏o
void TextSceneSaver::writeLightList(){
	int lightCount = lightManager_->getCount();
	for(int i = 0; i < lightCount; i++){
		Light* light = lightManager_->get(i);
		if(light->isAmbientLight()){
			writeAmbientLight(light->castAmbientLight());
		}else if(light->isDirectionalLight()){
			writeDirectionalLight(light->castDirectionalLight());
		}else if(light->isPointLight()){
			writePointLight(light->castPointLight());
		}else{
			ErrorOut("TextSceneSaver::writeLightList() unsupported light %s",
				light->getName().getBytes());
		}
	}
}
//------------------------------------------------------------------------------
// Cg̏o
void TextSceneSaver::writeLight(Light* light){
	writer_->writeText("\t\tLight {\n");
	// Cg}XN
	writer_->writeFormat("\t\t\tlightMask %u\n", light->getLightMask());
	// LA
	writer_->writeText("\t\t\tenabled ");
	writeBool(light->isEnabled());
	writer_->writeText("\n\t\t}\n");
}
//------------------------------------------------------------------------------
// ArGgCg̏o
void TextSceneSaver::writeAmbientLight(AmbientLight* light){
	writer_->writeText("\t");
	writer_->writeText(light->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Ambient\n");
	// J[
	Color3f color = light->getColor();
	writer_->writeFormat("\t\tcolor { %.8f %.8f %.8f }\n",
		color.r, color.g, color.b);
	// Cg̏o
	writeLight(light);
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// fBNViCg̏o
void TextSceneSaver::writeDirectionalLight(DirectionalLight* light){
	writer_->writeText("\t");
	writer_->writeText(light->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Directional\n");
	// fBt[YJ[
	Color3f diffuse = light->getDiffuseColor();
	writer_->writeFormat("\t\tdiffuse { %.8f %.8f %.8f }\n",
		diffuse.r, diffuse.g, diffuse.b);
	// XyLJ[
	Color3f	specular = light->getSpecularColor();
	writer_->writeFormat("\t\tspecular { %.8f %.8f %.8f }\n",
		specular.r, specular.g, specular.b);
	// 
	Vector3 direction = light->getDirection();
	writer_->writeFormat("\t\tdirection { %.8f %.8f %.8f }\n",
		direction.x, direction.y, direction.z);
	// Cg̏o
	writeLight(light);
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// |CgCg̏o
void TextSceneSaver::writePointLight(PointLight* light){
	writer_->writeText("\t");
	writer_->writeText(light->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Point\n");
	// fBt[YJ[
	Color3f diffuse = light->getDiffuseColor();
	writer_->writeFormat("\t\tdiffuse { %.8f %.8f %.8f }\n",
		diffuse.r, diffuse.g, diffuse.b);
	// XyLJ[
	Color3f	specular = light->getSpecularColor();
	writer_->writeFormat("\t\tspecular { %.8f %.8f %.8f }\n",
		specular.r, specular.g, specular.b);
	// ʒu
	Vector3 position = light->getPosition();
	writer_->writeFormat("\t\tposition { %.8f %.8f %.8f }\n",
		position.x, position.y, position.z);
	// W
	writer_->writeFormat("\t\trange %.8f\n", light->getRange());
	// W
	writer_->writeFormat("\t\tattenuation { %.8f %.8f %.8f }\n",
		light->getAttenuation0(), light->getAttenuation1(),
		light->getAttenuation2());
	// Cg̏o
	writeLight(light);
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// f
//------------------------------------------------------------------------------
// fXg̏o
void TextSceneSaver::writeModelList(){
	int modelCount = modelManager_->getCount();
	for(int i = 0; i < modelCount; i++){
		Model* model = modelManager_->get(i);
		if(model->isStandardModel()){
			writeStandardModel(model->castStandardModel());
		}else if(model->isCharacterModel()){
			writeCharacterModel(model->castCharacterModel());
		}else{
			ErrorOut("TextSceneSaver::writeModelList() unsupported model %s",
				model->getName().getBytes());
		}
	}
}
//------------------------------------------------------------------------------
// Wf̏o
void TextSceneSaver::writeStandardModel(StandardModel* model){
	writer_->writeText("\t");
	writer_->writeText(model->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Standard\n");
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(model->isEnabled());
	writer_->writeText("\n\t}\n\n");
}
//------------------------------------------------------------------------------
// LN^f̏o
void TextSceneSaver::writeCharacterModel(CharacterModel* model){
	writer_->writeText("\t");
	writer_->writeText(model->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Character\n");
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(model->isEnabled());
	// {[̏o
	writer_->writeText("\n\t\tbone {\n");
	int boneCount = model->getBoneCount();
	for(int i = 0; i < boneCount; i++){
		writeBone(model->getBone(i));
	}
	writer_->writeText("\t\t}\n");
	// {[N̏o
	writer_->writeText("\t\tboneLink {\n");
	for(int i = 0; i < boneCount; i++){
		writeBoneLink(model->getBone(i));
	}
	writer_->writeText("\t\t}\n");
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// {[̏o
void TextSceneSaver::writeBone(Bone* bone){
	writer_->writeText("\t\t\t");
	writer_->writeText(bone->getName());
	writer_->writeText(" {\n");
	// |[Yts
	Matrix34 invPose = bone->getInversePoseMatrix();
	writer_->writeFormat("\t\t\t\tinversePoseMatrix Matrix34 {\n"
		"\t\t\t\t\t %.8f %.8f %.8f %.8f\n\t\t\t\t\t %.8f %.8f %.8f %.8f\n"
		"\t\t\t\t\t %.8f %.8f %.8f %.8f\n\t\t\t\t}\n",
		invPose.m00, invPose.m01, invPose.m02, invPose.m03,
		invPose.m10, invPose.m11, invPose.m12, invPose.m13,
		invPose.m20, invPose.m21, invPose.m22, invPose.m23);
	// XP[
	Vector3 scale = bone->getScale();
	writer_->writeFormat("\t\t\t\tscale { %.8f %.8f %.8f }\n",
		scale.x, scale.y, scale.z);
	// ]
	Vector3 rotation = bone->getRotationXYZ();
	writer_->writeFormat("\t\t\t\trotation { %.8f %.8f %.8f }\n",
		rotation.x, rotation.y, rotation.z);
	// ړ
	Vector3 translation = bone->getTranslation();
	writer_->writeFormat("\t\t\t\ttranslation { %.8f %.8f %.8f }\n",
		translation.x, translation.y, translation.z);

	writer_->writeText("\t\t\t}\n");
}
//------------------------------------------------------------------------------
// {[N̏o
void TextSceneSaver::writeBoneLink(Bone* bone){
	int boneCount = bone->getBoneCount();
	if(boneCount == 0){ return; }
	writer_->writeText("\t\t\t");
	writer_->writeText(bone->getName());
	writer_->writeText(" {\n");
	for(int i = 0; i < boneCount; i++){
		Bone* child = bone->getBone(i);
		writer_->writeText("\t\t\t\t");
		writer_->writeText(child->getName());
		writer_->writeText("\n");
	}
	writer_->writeText("\t\t\t}\n");
}
//------------------------------------------------------------------------------
// bV
//------------------------------------------------------------------------------
// bVXg̏o
void TextSceneSaver::writeMeshList(){
	int meshCount = meshManager_->getCount();
	for(int i = 0; i < meshCount; i++){
		Mesh* mesh = meshManager_->get(i);
		if(mesh->isRigidMesh()){
			writeRigidMesh(mesh->castRigidMesh());
		}else if(mesh->isCharacterMesh()){
			writeCharacterMesh(mesh->castCharacterMesh());
		}else{
			ErrorOut("TextSceneSaver::writeMeshList() unsupported mesh %s",
				mesh->getName().getBytes());
		}
	}
}
//------------------------------------------------------------------------------
// ̃bV̏o
void TextSceneSaver::writeRigidMesh(RigidMesh* mesh){
	writer_->writeText("\t");
	writer_->writeText(mesh->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Rigid\n");
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(mesh->isEnabled());
	writer_->writeText("\n\t}\n\n");
}
//------------------------------------------------------------------------------
// LN^bV̏o
void TextSceneSaver::writeCharacterMesh(CharacterMesh* mesh){
	writer_->writeText("\t");
	writer_->writeText(mesh->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Character\n");
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(mesh->isEnabled());
	writer_->writeText("\n\t}\n\n");
}
//------------------------------------------------------------------------------
// bVf[^
//------------------------------------------------------------------------------
// bVf[^Xg̏o
void TextSceneSaver::writeMeshDataList(){
	int meshDataCount = meshDataManager_->getCount();
	for(int i = 0; i < meshDataCount; i++){
		writeMeshData(meshDataManager_->get(i));
	}
}
//------------------------------------------------------------------------------
// bVf[^̏o
void TextSceneSaver::writeMeshData(MeshData* meshData){
	writer_->writeText("\t");
	writer_->writeText(meshData->getName());
	writer_->writeText(" {\n");
	// oEfBO{bNX
	const AxisAlignedBox& box = meshData->getBoundingBox();
	writer_->writeFormat(
		"\t\tboundingBox { %.8f %.8f %.8f } { %.8f %.8f %.8f }\n",
		box.getMinimum().x, box.getMinimum().y, box.getMinimum().z,
		box.getMaximum().x, box.getMaximum().y, box.getMaximum().z);
	// oEfBOXtBA
	const Sphere& sphere = meshData->getBoundingSphere();
	writer_->writeFormat(
		"\t\tboundingSphere { %.8f %.8f %.8f %.8f }\n",
		sphere.getCenter().x, sphere.getCenter().y, sphere.getCenter().z,
		sphere.getRadius());
	// v~eBu^Cv
	writer_->writeText("\t\tprimitiveType ");
	writer_->writeText(Mesh::primitiveTypeToString(
		meshData->getPrimitiveType()));
	writer_->writeText("\n");
	// gCAOXg̃CfbNXo
	if(meshData->getPrimitiveType() == Mesh::indexedTriangleList){
		int vertexIndexCount = meshData->getVertexIndexCount();
		writer_->writeFormat("\t\tindices %d {\n", vertexIndexCount);
		vertexIndexCount /= 3;
		for(int i = 0; i < vertexIndexCount; i++){
			int index = i * 3;
			writer_->writeFormat("\t\t\t%d %d %d\n",
				meshData->getVertexIndex(index),
				meshData->getVertexIndex(index + 1),
				meshData->getVertexIndex(index + 2));
		}
		writer_->writeFormat("\t\t}\n");
	}
	// @Ă邩
	bool hasNormal = meshData->hasNormal();
	writer_->writeText("\t\thasNormal ");
	writeBool(hasNormal);
	// _J[Ă邩
	bool hasColor = meshData->hasColor();
	writer_->writeText("\n\t\thasColor ");
	writeBool(hasColor);
	// eNX`WZbg
	int texCoordSetCount = meshData->getTexCoordSetCount();
	writer_->writeFormat("\n\t\ttexCoordSetCount %d\n", texCoordSetCount);
	// eNX`W^Cv
	writer_->writeText("\t\ttexCoordType");
	int texCoordType[TexCoord::maxSetCount];
	for(int i = 0; i < texCoordSetCount; i++){
		texCoordType[i] = meshData->getTexCoordType(i);
		writer_->writeFormat(" %d", texCoordType[i]);
	}
	writer_->writeText("\n");
	// _{[
	int bonesPerVertex = meshData->getBonesPerVertex();
	writer_->writeFormat("\t\tbonesPerVertex %d\n", bonesPerVertex);
	// _EFCg
	int weightsPerVertex = meshData->getWeightsPerVertex();
	// _Xg
	int vertexCount = meshData->getVertexCount();
	writer_->writeFormat("\t\tvertices %d {\n", vertexCount);
	for(int i = 0; i < vertexCount; i++){
		// ʒu
		Vector3 pos = meshData->getPosition(i);
		writer_->writeFormat("\t\t\t%d { %.8f %.8f %.8f }",
			i, pos.x, pos.y, pos.z);
		// @
		if(hasNormal){
			Vector3 nom = meshData->getNormal(i);
			writer_->writeFormat(" { %.8f %.8f %.8f }",
				nom.x, nom.y, nom.z);
		}
		// _J[
		if(hasColor){
			Color4c color = meshData->getColor(i);
			writer_->writeFormat(" { %d %d %d %d }",
				color.r, color.g, color.b, color.a);
		}
		// eNX`W
		for(int j = 0; j < texCoordSetCount; j++){
			writer_->writeText(" { ");
			for(int k = 0; k < texCoordType[j]; k++){
				writer_->writeFormat("%.8f ",
					*(meshData->getTexCoordArray(j) + (i * texCoordType[j]) + k));
			}
			writer_->writeText("}");
		}
		// {[CfbNX
		if(bonesPerVertex != 0){
			writer_->writeText(" { ");
			for(int j = 0; j < bonesPerVertex; j++){
				writer_->writeFormat("%d ",
					(int)meshData->getBoneIndex(i, j));
			}
			writer_->writeText("}");
		}
		// EFCg
		if(weightsPerVertex != 0){
			writer_->writeText(" { ");
			for(int j = 0; j < weightsPerVertex; j++){
				writer_->writeFormat("%.8f ", meshData->getWeight(i, j));
			}
			writer_->writeText("}");
		}
		writer_->writeText("\n");
	}
	writer_->writeText("\t\t}\n");
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// }eA
//------------------------------------------------------------------------------
// }eAXg̏o
void TextSceneSaver::writeMaterialList(){
	int materialCount = materialManager_->getCount();
	for(int i = 0; i < materialCount; i++){
		Material* material = materialManager_->get(i);
		if(material->isBasicMaterial()){
			writeBasicMaterial(material->castBasicMaterial());
		}else{
			ErrorOut("TextSceneSaver::writeMaterialList() "
				"unsupported material %s", material->getName().getBytes());
		}
	}
}
//------------------------------------------------------------------------------
// }eȀo
void TextSceneSaver::writeMaterial(const Material* material){
	writer_->writeText("\t\tMaterial {\n");
	writer_->writeFormat("\t\t\tpriority %d\n", material->getPriority());
	writer_->writeText("\t\t\tblendMode ");
	writer_->writeText(Material::blendModeToString(material->getBlendMode()));
	writer_->writeFormat("\n\t\t\talpha %.8f\n", material->getAlpha());
	writer_->writeText("\t\t\tblendSource ");
	writer_->writeText(
		Material::blendStateToString(material->getBlendSource()));
	writer_->writeText("\n\t\t\tblendDestination ");
	writer_->writeText(
		Material::blendStateToString(material->getBlendDestination()));
	writer_->writeText("\n\t\t\tzWrite ");
	writeBool(material->useZWrite());
	writer_->writeText("\n\t\t\tzTest ");
	writeBool(material->useZTest());
	writer_->writeText("\n\t\t\tfogOption ");
	writer_->writeText(Material::fogOptionToString(material->getFogOption()));
	writer_->writeFormat("\n\t\t\tlightMask %u\n", material->getLightMask());
	writer_->writeText("\t\t}\n");
}
//------------------------------------------------------------------------------
// {}eȀo
void TextSceneSaver::writeBasicMaterial(const BasicMaterial* material){
	writer_->writeText("\t");
	writer_->writeText(material->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Basic\n");
	writer_->writeFormat("\t\tbaseUVIndex %d\n", material->getBaseUVIndex());
	writer_->writeFormat("\t\tglossUVIndex %d\n", material->getGlossUVIndex());
	writer_->writeFormat("\t\tlightUVIndex %d\n", material->getLightUVIndex());
	writer_->writeFormat("\t\tstainUVIndex %d\n", material->getStainUVIndex());
	Color3f diffuse = material->getDiffuseColor();
	writer_->writeFormat("\t\tdiffuseColor { %.8f %.8f %.8f }\n",
		diffuse.r, diffuse.g, diffuse.b);
	Color3f specular = material->getSpecularColor();
	writer_->writeFormat("\t\tspecularColor { %.8f %.8f %.8f }\n",
		specular.r, specular.g, specular.b);
	writer_->writeFormat("\t\tspecularPower %.8f\n",
		material->getSpecularPower());
	Color3f ambient = material->getAmbientColor();
	writer_->writeFormat("\t\tambientColor { %.8f %.8f %.8f }\n",
		ambient.r, ambient.g, ambient.b);
	Color3f emissive = material->getEmissiveColor();
	writer_->writeFormat("\t\temissiveColor { %.8f %.8f %.8f }\n",
		emissive.r, emissive.g, emissive.b);
	// }eȀo
	writeMaterial(material);
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// eNX`
//------------------------------------------------------------------------------
// eNX`Xg̏o
void TextSceneSaver::writeTextureList(){
	int textureCount = textureManager_->getCount();
	for(int i = 0; i < textureCount; i++){
		Texture* texture = textureManager_->get(i);
		if(texture->isSurfaceTexture()){
			writeSurfaceTexture(texture->castSurfaceTexture());
		}else{
			ErrorOut("TextSceneSaver::writeTextureList() unsupported texture %s",
				texture->getName().getBytes());
		}
	}
}
//------------------------------------------------------------------------------
// T[tF[XeNX`̏o
void TextSceneSaver::writeSurfaceTexture(const SurfaceTexture* texture){
	writer_->writeText("\t");
	writer_->writeText(texture->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Surface\n");
	writer_->writeText("\t\taddressModeU ");
	writer_->writeText(Texture::addressModeToString(
		texture->getAddressModeU()));
	writer_->writeText("\n\t\taddressModeV ");
	writer_->writeText(Texture::addressModeToString(
		texture->getAddressModeV()));
	TexCoord2 repeat = texture->getRepeatUV();
	writer_->writeFormat("\n\t\trepeatUV { %.8f %.8f }", repeat.u, repeat.v);
	TexCoord2 offset = texture->getOffsetUV();
	writer_->writeFormat("\n\t\toffsetUV { %.8f %.8f }", offset.u, offset.v);
	writer_->writeText("\n\t}\n\n");
}
//------------------------------------------------------------------------------
// sN`
//------------------------------------------------------------------------------
// sN`̏o
void TextSceneSaver::writePicture(const Picture* picture){
	writer_->writeText("\t");
	writer_->writeText(picture->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\tpath ");
	String path = picture->getPath();
	writer_->writeText(path);
	writer_->writeText("\n\t}\n\n");
	// sN`o̓fBNg̍쐬
	FilePath filePath(path);
	path = filePath.getFolderPath();
	if(path != ""){
		path = basePath_ + path;
		mkdir(path.getBytes());
	}
	// sN`t@C̕ۑ
	TargaSaver saver;
	if(picture->isPictureRGB8()){
		// RGB824bitTargaŕۑ
		PictureRGB8* rgb8 = picture->castPictureRGB8();
		saver.save(basePath_ + rgb8->getPath(),
			rgb8->getSize(), rgb8->getImage());
	}else if(picture->isPictureRGBA8()){
		// RGBA832bitTargaŕۑ
		PictureRGBA8* rgba8 = picture->castPictureRGBA8();
		saver.save(basePath_ + rgba8->getPath(),
			rgba8->getSize(), rgba8->getImage());
	}else{
		ErrorOut("TextSceneSaver::writePictureList() "
			"unsupported picture %s", picture->getName().getBytes());
	}
}
//------------------------------------------------------------------------------
// N
//------------------------------------------------------------------------------
// V[m[hN̏o
void TextSceneSaver::writeSceneNodeLink(SceneNode* sceneNode){
	// N0ǂ`FbN
	int sceneNodeCount = sceneNode->getSceneNodeCount();
	int sceneLeafCount = sceneNode->getSceneLeafCount();
	if(sceneNodeCount + sceneLeafCount == 0){ return; }
	writer_->writeText("\t");
	writer_->writeText(sceneNode->getName());
	writer_->writeText(" {\n");
	if(sceneNodeCount > 0){
		writer_->writeText("\t\tSceneNode {\n");
		for(int i = 0; i < sceneNodeCount; i++){
			SceneNode* linkNode = sceneNode->getSceneNode(i);
			writer_->writeText("\t\t\t");
			writer_->writeText(linkNode->getName());
			writer_->writeText("\n");
		}
		writer_->writeText("\t\t}\n");
	}
	if(sceneLeafCount > 0){
		writer_->writeText("\t\tSceneLeaf {\n");
		for(int i = 0; i < sceneLeafCount; i++){
			SceneLeaf* sceneLeaf = sceneNode->getSceneLeaf(i);
			if(sceneLeaf->isModel()){
				writer_->writeText("\t\t\tModel ");
				writer_->writeText(sceneLeaf->getName());
				writer_->writeText("\n");
			}else if(sceneLeaf->isLight()){
				writer_->writeText("\t\t\tLight ");
				writer_->writeText(sceneLeaf->getName());
				writer_->writeText("\n");
			}else{
				Assert(false);
			}
		}
		writer_->writeText("\t\t}\n");
	}
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// fN̏o
void TextSceneSaver::writeModelLink(const Model* model){
	int meshCount = model->getMeshCount();
	if(meshCount == 0){ return; }
	writer_->writeText("\t");
	writer_->writeText(model->getName());
	writer_->writeText(" {\n");
	for(int i = 0; i < meshCount; i++){
		Mesh* mesh = model->getMesh(i);
		writer_->writeText("\t\t");
		writer_->writeText(mesh->getName());
		writer_->writeText("\n");
	}
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// bVN̏o
void TextSceneSaver::writeMeshLink(const Mesh* mesh){
	if((mesh->getMeshData() == NULL) &&
		(mesh->getMaterial() == NULL)){ return; }
	writer_->writeText("\t");
	writer_->writeText(mesh->getName());
	writer_->writeText(" {\n");
	// bVf[^
	if(mesh->getMeshData() != NULL){
		writer_->writeText("\t\tMeshData ");
		writer_->writeText(mesh->getMeshData()->getName());
		writer_->writeText("\n");
	}
	// }eA
	if(mesh->getMaterial() != NULL){
		writer_->writeText("\t\tMaterial ");
		writer_->writeText(mesh->getMaterial()->getName());
		writer_->writeText("\n");
	}
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// }eANXg̏o
void TextSceneSaver::writeMaterialLinkList(){
	writer_->writeText("MaterialLink {\n");
	int materialCount = materialManager_->getCount();
	for(int i = 0; i < materialCount; i++){
		Material* material = materialManager_->get(i);
		if(material->isBasicMaterial()){
			writeBasicMaterialLink(material->castBasicMaterial());
		}else{
			ErrorOut("TextSceneSaver::writeMaterialLinkList() "
				"unsupported material %s", material->getName().getBytes());
		}
	}
	writer_->writeText("}\n\n");
}
//------------------------------------------------------------------------------
// {}eAN̏o
void TextSceneSaver::writeBasicMaterialLink(const BasicMaterial* material){
	Texture* baseTexture = material->getBaseTexture();
	Texture* glossTexture = material->getGlossTexture();
	Texture* lightTexture = material->getLightTexture();
	Texture* stainTexture = material->getStainTexture();
	// eNX`̗L`FbN
	if((baseTexture == NULL) && (glossTexture == NULL) &&
		(lightTexture == NULL) && (stainTexture == NULL)){ return; }
	writer_->writeText("\t");
	writer_->writeText(material->getName());
	writer_->writeText(" {\n");
	writer_->writeText("\t\ttype Basic\n");
	if(baseTexture != NULL){
		writer_->writeText("\t\tbaseTexture ");
		writer_->writeText(baseTexture->getName());
		writer_->writeText("\n");
	}
	if(glossTexture != NULL){
		writer_->writeText("\t\tglossTexture ");
		writer_->writeText(glossTexture->getName());
		writer_->writeText("\n");
	}
	if(lightTexture != NULL){
		writer_->writeText("\t\tlightTexture ");
		writer_->writeText(lightTexture->getName());
		writer_->writeText("\n");
	}
	if(stainTexture != NULL){
		writer_->writeText("\t\tstainTexture ");
		writer_->writeText(stainTexture->getName());
		writer_->writeText("\n");
	}
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// eNX`N̏o
void TextSceneSaver::writeTextureLink(const Texture* texture){
	int pictureCount = texture->getPictureCount();
	if(pictureCount == 0){ return; }
	writer_->writeText("\t");
	writer_->writeText(texture->getName());
	writer_->writeText(" {\n");
	for(int i = 0; i < pictureCount; i++){
		Picture* picture = texture->getPicture(i);
		writer_->writeText("\t\t");
		writer_->writeText(picture->getName());
		writer_->writeText("\n");
	}
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// bool̏o
void TextSceneSaver::writeBool(bool flag){
	if(flag){
		writer_->writeText("true");
	}else{
		writer_->writeText("false");
	}
}
//------------------------------------------------------------------------------
// Rg̏o
void TextSceneSaver::writeLineComment(){
	writer_->writeText("//----------------------------"
		"--------------------------------------------------\n");
}
//------------------------------------------------------------------------------
// ubNRg̏o
void TextSceneSaver::writeBlockComment(const String& blockName){
	writeLineComment();
	writer_->writeText("// ");
	writer_->writeText(blockName);
	writer_->writeText(" Block\n");
	writeLineComment();
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
