//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * eLXgV[[_
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/InputOutput/TextSceneLoader.h"
#include "Core/InputOutput/TextFileReader.h"
#include "Core/InputOutput/StreamTokenizer.h"
#include "Core/InputOutput/FilePath.h"
#include "Core/Codec/Tga/TargaLoader.h"

#include "Graphics/Scene/Scene.h"
#include "Graphics/Fog/Fog.h"
#include "Graphics/Light/LightManager.h"
#include "Graphics/SceneNode/SceneNodeManager.h"
#include "Graphics/Model/ModelManager.h"
#include "Graphics/Mesh/MeshManager.h"
#include "Graphics/MeshData/MeshDataManager.h"
#include "Graphics/Material/MaterialManager.h"
#include "Graphics/Texture/TextureManager.h"
#include "Graphics/Picture/PictureManager.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
TextSceneLoader::TextSceneLoader(){
}
//------------------------------------------------------------------------------
// fXgN^
TextSceneLoader::~TextSceneLoader(){
}
//------------------------------------------------------------------------------
// [h
void TextSceneLoader::load(const String& filePath, Scene* scene){
	load(filePath, scene, scene->getRootNode());
}
//------------------------------------------------------------------------------
// [h
void TextSceneLoader::load(const String& filePath, Scene* scene,
	SceneNode* rootNode){
	FilePath path(filePath);
	Assert(path.existFile());
	TextFileReader* textFileReader = new TextFileReader(filePath);
	load(textFileReader, scene, rootNode, path.getFolderPath());
	delete textFileReader;
}
//------------------------------------------------------------------------------
// [h
void TextSceneLoader::load(TextReader* textReader, Scene* scene,
	const String& basePath){
	load(textReader, scene, scene->getRootNode(), basePath);
}
//------------------------------------------------------------------------------
// [h
void TextSceneLoader::load(TextReader* textReader, Scene* scene,
	SceneNode* rootNode, const String& basePath){
	// 
	basePath_ = basePath;
	tokenizer_ = new StreamTokenizer(textReader);
	scene_ = scene;
	rootNode_ = rootNode;
	sceneNodeManager_ = scene->getSceneNodeManager();
	lightManager_ = scene->getLightManager();
	modelManager_ = scene->getModelManager();
	meshManager_ = scene->getMeshManager();
	meshDataManager_ = scene->getMeshDataManager();
	materialManager_ = scene->getMaterialManager();
	textureManager_ = scene->getTextureManager();
	pictureManager_ = scene->getPictureManager();

	// [gm[h𖳌ɂĂ
	bool rootNodeEnabled = rootNode_->isEnabled();
	rootNode_->setEnabled(false);

	// wb_̓ǂݍ
	readHeader();

	while(true){
		// g[NǂݏI
		if(!tokenizer_->nextToken()){ break; }
		String chunkName = tokenizer_->getToken();
		if(chunkName == "Fog"){
			readFog();
		}else if(chunkName == "SceneNode"){
			readSceneNodeList();
		}else if(chunkName == "Light"){
			readLightList();
		}else if(chunkName == "Model"){
			readModelList();
		}else if(chunkName == "Mesh"){
			readMeshList();
		}else if(chunkName == "MeshData"){
			readMeshDataList();
		}else if(chunkName == "Material"){
			readMaterialList();
		}else if(chunkName == "Texture"){
			readTextureList();
		}else if(chunkName == "Picture"){
			readPictureList();
		}else if(chunkName == "SceneNodeLink"){
			readSceneNodeLinkList();
		}else if(chunkName == "ModelLink"){
			readModelLinkList();
		}else if(chunkName == "MeshLink"){
			readMeshLinkList();
		}else if(chunkName == "MaterialLink"){
			readMaterialLinkList();
		}else if(chunkName == "TextureLink"){
			readTextureLinkList();
		}else{
			ErrorOut("TextSceneLoader::load() "
				"invalid chunk %s (line %d)",
				chunkName.getBytes(), tokenizer_->getLineNumber());
		}
	}
	delete tokenizer_;

	rootNode_->setEnabled(rootNodeEnabled);
}
//------------------------------------------------------------------------------
// wb_̓ǂݍ
void TextSceneLoader::readHeader(){
	skipWord("Header");
	openChunk();
	skipWord("type");
	skipWord("LampTextSceneFormat");
	skipWord("version");
	skipWord("0_9_0");
	closeChunk();
}
//------------------------------------------------------------------------------
// tHO
//------------------------------------------------------------------------------
// tHO̓ǂݍ
void TextSceneLoader::readFog(){
	Fog* fog = scene_->getFog();
	openChunk();
	// J[
	skipWord("color");
	fog->setColor(readColor4c());
	// [h
	skipWord("mode");
	fog->setModeString(readToken("TextSceneLoader::readFog()"));
	// Zx
	skipWord("density");
	fog->setDensity(readFloat());
	// jA
	skipWord("near");
	fog->setNear(readFloat());
	// t@[
	skipWord("far");
	fog->setFar(readFloat());
	// LA
	skipWord("enabled");
	fog->setEnabled(readBool());
	closeChunk();
}
//------------------------------------------------------------------------------
// V[m[h
//------------------------------------------------------------------------------
// V[m[hXg̓ǂݍ
void TextSceneLoader::readSceneNodeList(){
	openChunk();
	while(true){
		String token = readToken("readSceneNodeList()");
		if(token == "}"){ break; }
		// V[m[h̎
		openChunk();
		skipWord("type");
		String type = readToken("readSceneNodeList()");
		if(type == "Standard"){
			readSceneNode(token);
		}else if(type == "LOD"){
			readLODSceneNode(token);
		}else{
			ErrorOut("TextSceneLoader::readSceneNodeList() "
				"invalid type %s (line %d)",
				type.getBytes(), tokenizer_->getLineNumber());
		}
		closeChunk();
	}
}
//------------------------------------------------------------------------------
// V[m[h̓ǂݍ
void TextSceneLoader::readSceneNode(const String& name){
	SceneNode* sceneNode = sceneNodeManager_->createSceneNode(name);
	// XP[
	skipWord("scale");
	sceneNode->setScale(readVector3());
	// ]
	skipWord("rotation");
	sceneNode->setRotationXYZ(readVector3());
	// ړ
	skipWord("translation");
	sceneNode->setTranslation(readVector3());
	// LAtO
	skipWord("enabled");
	sceneNode->setEnabled(readBool());
}
//------------------------------------------------------------------------------
// xIufBeB[V[m[h̓ǂݍ
void TextSceneLoader::readLODSceneNode(const String& name){
	LODSceneNode* sceneNode = sceneNodeManager_->createLODSceneNode(name);
	// XP[
	skipWord("scale");
	sceneNode->setScale(readVector3());
	// ]
	skipWord("rotation");
	sceneNode->setRotationXYZ(readVector3());
	// ړ
	skipWord("translation");
	sceneNode->setTranslation(readVector3());
	// LAtO
	skipWord("enabled");
	sceneNode->setEnabled(readBool());
	// LOD
	skipWord("lodThresholdCount");
	int lodThresholdCount = readInt();
	sceneNode->setLODThresholdCount(lodThresholdCount);
	// LODl
	openChunk();
	for(int i = 0; i < lodThresholdCount; i++){
		sceneNode->setLODThreshold(i, readFloat());
	}
	closeChunk();
}
//------------------------------------------------------------------------------
// Cg
//------------------------------------------------------------------------------
// CgXg̓ǂݍ
void TextSceneLoader::readLightList(){
	openChunk();
	while(true){
		String token = readToken("readLightList()");
		if(token == "}"){ break; }
		// Cg̎
		openChunk();
		skipWord("type");
		String type = readToken("readLightList()");
		if(type == "Ambient"){
			readAmbientLight(token);
		}else if(type == "Directional"){
			readDirectionalLight(token);
		}else if(type == "Point"){
			readPointLight(token);
		}else{
			ErrorOut("TextSceneLoader::readLightList() "
				"invalid type %s (line %d)",
				type.getBytes(), tokenizer_->getLineNumber());
		}
		closeChunk();
	}
}
//------------------------------------------------------------------------------
// Cg̓ǂݍ
void TextSceneLoader::readLight(Light* light){
	skipWord("Light");
	openChunk();
	// Cg}XN
	skipWord("lightMask");
	light->setLightMask(readUInt());
	// LAtO
	skipWord("enabled");
	light->setEnabled(readBool());
	closeChunk();
}
//------------------------------------------------------------------------------
// ArGgCg̓ǂݍ
void TextSceneLoader::readAmbientLight(const String& name){
	AmbientLight* light = lightManager_->createAmbientLight(name);
	// CgJ[
	skipWord("color");
	light->setColor(readColor3f());
	// Cg̓ǂݍ
	readLight(light);
}
//------------------------------------------------------------------------------
// fBNViCg̓ǂݍ
void TextSceneLoader::readDirectionalLight(const String& name){
	DirectionalLight* light = lightManager_->createDirectionalLight(name);
	// fBt[YJ[
	skipWord("diffuse");
	light->setDiffuseColor(readColor3f());
	// XyLJ[
	skipWord("specular");
	light->setSpecularColor(readColor3f());
	// 
	skipWord("direction");
	light->setDirection(readVector3());
	// Cg̓ǂݍ
	readLight(light);
}
//------------------------------------------------------------------------------
// |CgCg̓ǂݍ
void TextSceneLoader::readPointLight(const String& name){
	PointLight* light = lightManager_->createPointLight(name);
	// fBt[YJ[
	skipWord("diffuse");
	light->setDiffuseColor(readColor3f());
	// XyLJ[
	skipWord("specular");
	light->setSpecularColor(readColor3f());
	// ʒu
	skipWord("position");
	light->setPosition(readVector3());
	// W
	skipWord("range");
	light->setRange(readFloat());
	// W
	skipWord("attenuation");
	Vector3 attenuation = readVector3();
	light->setAttenuation(attenuation.x, attenuation.y, attenuation.z);
	// Cg̓ǂݍ
	readLight(light);
}
//------------------------------------------------------------------------------
// f
//------------------------------------------------------------------------------
// fXg̓ǂݍ
void TextSceneLoader::readModelList(){
	openChunk();
	while(true){
		String token = readToken("readModelList()");
		if(token == "}"){ break; }
		// f̎
		openChunk();
		skipWord("type");
		String type = readToken("readModelList()");
		if(type == "Standard"){
			readStandardModel(token);
		}else if(type == "Character"){
			readCharacterModel(token);
		}else{
			ErrorOut("TextSceneLoader::readModelList() "
				"invalid type %s (line %d)",
				type.getBytes(), tokenizer_->getLineNumber());
		}
		closeChunk();
	}
}
//------------------------------------------------------------------------------
// Wf̓ǂݍ
void TextSceneLoader::readStandardModel(const String& name){
	StandardModel* model = modelManager_->createStandardModel(name);
	// LAtO
	skipWord("enabled");
	model->setEnabled(readBool());
}
//------------------------------------------------------------------------------
// LN^f̓ǂݍ
void TextSceneLoader::readCharacterModel(const String& name){
	CharacterModel* model = modelManager_->createCharacterModel(name);
	// LAtO
	skipWord("enabled");
	model->setEnabled(readBool());
	// {[
	skipWord("bone");
	openChunk();
	while(true){
		String boneName = readToken("TextSceneLoader::readCharacterModel()");
		if(boneName == "}"){ break; }
		readBone(model, boneName);
	}
	// {[N
	skipWord("boneLink");
	openChunk();
	while(true){
		String boneName = readToken("TextSceneLoader::readCharacterModel()");
		if(boneName == "}"){ break; }
		Bone* bone = model->searchBone(boneName);
		openChunk();
		while(true){
			String childName = readToken("TextSceneLoader::readCharacterModel()");
			if(childName == "}"){ break; }
			Bone* child = model->searchBone(childName);
			bone->addBone(child);
		}
	}
}
//------------------------------------------------------------------------------
// {[̓ǂݍ
void TextSceneLoader::readBone(CharacterModel* model, const String& name){
	Bone* bone = model->createBone(name);
	openChunk();
	// |[Yts
	skipWord("inversePoseMatrix");
	skipWord("Matrix34");
	bone->setInversePoseMatrix(readMatrix34());
	// XP[
	skipWord("scale");
	bone->setScale(readVector3());
	// ]
	skipWord("rotation");
	bone->setRotationXYZ(readVector3());
	// ړ
	skipWord("translation");
	bone->setTranslation(readVector3());
	closeChunk();
}
//------------------------------------------------------------------------------
// bV
//------------------------------------------------------------------------------
// bVXg̓ǂݍ
void TextSceneLoader::readMeshList(){
	openChunk();
	while(true){
		String token = readToken("readMeshList()");
		if(token == "}"){ break; }
		// bV̎
		openChunk();
		skipWord("type");
		String type = readToken("readMeshList()");
		if(type == "Rigid"){
			readRigidMesh(token);
		}else if(type == "Character"){
			readCharacterMesh(token);
		}else{
			ErrorOut("TextSceneLoader::readMeshList() "
				"invalid type %s (line %d)",
				type.getBytes(), tokenizer_->getLineNumber());
		}
		closeChunk();
	}
}
//------------------------------------------------------------------------------
// ̃bV̓ǂݍ
void TextSceneLoader::readRigidMesh(const String& name){
	RigidMesh* mesh = meshManager_->createRigidMesh(name);
	// LAtO
	skipWord("enabled");
	mesh->setEnabled(readBool());
}
//------------------------------------------------------------------------------
// LN^bV̓ǂݍ
void TextSceneLoader::readCharacterMesh(const String& name){
	CharacterMesh* mesh = meshManager_->createCharacterMesh(name);
	// LAtO
	skipWord("enabled");
	mesh->setEnabled(readBool());
}
//------------------------------------------------------------------------------
// bVf[^
//------------------------------------------------------------------------------
// bVf[^Xg̓ǂݍ
void TextSceneLoader::readMeshDataList(){
	openChunk();
	while(true){
		String token = readToken("readMeshDataList()");
		if(token == "}"){ break; }
		// bVf[^̓ǂݍ
		readMeshData(token);
	}
}
//------------------------------------------------------------------------------
// bVf[^̓ǂݍ
void TextSceneLoader::readMeshData(const String& name){
	MeshData* meshData = meshDataManager_->createMeshData(name);
	openChunk();
	// oEfBO{bNX
	skipWord("boundingBox");
	meshData->setBoundingBox(readAxisAlignedBox());
	// oEfBOXtBA
	skipWord("boundingSphere");
	meshData->setBoundingSphere(readSphere());
	// v~eBu^Cv
	skipWord("primitiveType");
	String primitiveType = readToken("readMeshData()");
	meshData->setPrimitiveType(Mesh::primitiveTypeFromString(primitiveType));
	// _CfbNX̓ǂݏo
	if(meshData->hasVertexIndices()){
		skipWord("indices");
		int indexCount = readInt();
		meshData->setVertexIndexCount(indexCount);
		openChunk();
		for(int i = 0; i < indexCount; i++){
			meshData->setVertexIndex(i, readUShort());
		}
		closeChunk();
	}
	// @Ă邩
	skipWord("hasNormal");
	bool hasNormal = readBool();
	meshData->enableNormal(hasNormal);
	// _J[Ă邩
	skipWord("hasColor");
	bool hasColor = readBool();
	meshData->enableColor(hasColor);
	// eNX`WZbg
	skipWord("texCoordSetCount");
	int texCoordSetCount = readInt();
	meshData->setTexCoordSetCount(texCoordSetCount);
	// eNX`W^Cv
	skipWord("texCoordType");
	int texCoordType[TexCoord::maxSetCount];
	for(int i = 0; i < texCoordSetCount; i++){
		texCoordType[i] = readInt();
		meshData->setTexCoordType(i, (TexCoord::Type)texCoordType[i]);
	}
	// _{[
	skipWord("bonesPerVertex");
	int bonesPerVertex = readInt();
	meshData->setBonesPerVertex(bonesPerVertex);
	int weightsPerVertex = meshData->getWeightsPerVertex();
	// _f[^
	skipWord("vertices");
	int vertexCount = readInt();
	meshData->setVertexCount(vertexCount);
	openChunk();
	for(int i = 0; i < vertexCount; i++){
		int index = readInt();
		if(index != i){
			ErrorOut("TextSceneLoader::readMeshData() "
				"invalid index %d (line %d)",
				index, tokenizer_->getLineNumber());
		}
		// ʒu̓ǂݍ
		meshData->setPosition(i, readVector3());
		// @̓ǂݍ
		if(hasNormal){ meshData->setNormal(i, readVector3()); }
		// _J[̓ǂݍ
		if(hasColor){ meshData->setColor(i, readColor4c()); }
		// eNX`W̓ǂݍ
		float texCoord[4];
		for(int j = 0; j < texCoordSetCount; j++){
			openChunk();
			for(int k = 0; k < texCoordType[j]; k++){
				texCoord[k] = readFloat();
			}
			closeChunk();
			meshData->setTexCoord(i, j, texCoord, texCoordType[j]);
		}
		// {[CfbNX̓ǂݍ
		if(bonesPerVertex != 0){
			openChunk();
			for(int j = 0; j < bonesPerVertex; j++){
				meshData->setBoneIndex(i, j, readUChar());
			}
			closeChunk();
		}
		// EFCg̓ǂݍ
		if(weightsPerVertex != 0){
			openChunk();
			for(int j = 0; j < weightsPerVertex; j++){
				meshData->setWeight(i, j, readFloat());
			}
			closeChunk();
		}
	}
	closeChunk();
	closeChunk();
}
//------------------------------------------------------------------------------
// }eA
//------------------------------------------------------------------------------
// }eAXg̓ǂݍ
void TextSceneLoader::readMaterialList(){
	openChunk();
	while(true){
		String token = readToken("readMaterialList()");
		if(token == "}"){ break; }
		// }eA̎
		openChunk();
		skipWord("type");
		String type = readToken("readMaterialList()");
		if(type == "Basic"){
			readBasicMaterial(token);
		}else{
			ErrorOut("TextSceneLoader::readMaterialList() "
				"invalid type %s (line %d)",
				type.getBytes(), tokenizer_->getLineNumber());
		}
		closeChunk();
	}
}
//------------------------------------------------------------------------------
// }eA̓ǂݍ
void TextSceneLoader::readMaterial(Material* material){
	skipWord("Material");
	openChunk();
	// Dx
	skipWord("priority");
	material->setPriority(readInt());
	// uh[h
	skipWord("blendMode");
	String blendMode = readToken("readMaterial()");
	material->setBlendMode(Material::blendModeFromString(blendMode));
	// At@l
	skipWord("alpha");
	material->setAlpha(readFloat());
	// uh\[X
	skipWord("blendSource");
	String blendSource = readToken("readMaterial()");
	material->setBlendSource(Material::blendStateFromString(blendSource));
	// uhfXeBl[V
	skipWord("blendDestination");
	String blendDestination = readToken("readMaterial()");
	material->setBlendDestination(
		Material::blendStateFromString(blendDestination));
	// Z
	skipWord("zWrite");
	material->setZWrite(readBool());
	// ZeXg
	skipWord("zTest");
	material->setZTest(readBool());
	// tHOIvV
	skipWord("fogOption");
	String fogOption = readToken("readMaterial()");
	material->setFogOption(Material::fogOptionFromString(fogOption));
	// Cg}XN
	skipWord("lightMask");
	material->setLightMask(readUInt());
	closeChunk();
}
//------------------------------------------------------------------------------
// {}eA̓ǂݍ
void TextSceneLoader::readBasicMaterial(const String& name){
	BasicMaterial* material = materialManager_->createBasicMaterial(name);
	// x[XUVCfbNX
	skipWord("baseUVIndex");
	material->setBaseUVIndex(readInt());
	// UVCfbNX
	skipWord("glossUVIndex");
	material->setGlossUVIndex(readInt());
	// CgUVCfbNX
	skipWord("lightUVIndex");
	material->setLightUVIndex(readInt());
	// UVCfbNX
	skipWord("stainUVIndex");
	material->setStainUVIndex(readInt());

	// fBt[YJ[
	skipWord("diffuseColor");
	material->setDiffuseColor(readColor3f());
	// XyLJ[
	skipWord("specularColor");
	material->setSpecularColor(readColor3f());
	// XyLp[
	skipWord("specularPower");
	material->setSpecularPower(readFloat());
	// ArGgJ[
	skipWord("ambientColor");
	material->setAmbientColor(readColor3f());
	// G~bVuJ[
	skipWord("emissiveColor");
	material->setEmissiveColor(readColor3f());
	// }eA̓ǂݍ
	readMaterial(material);
}
//------------------------------------------------------------------------------
// eNX`
//------------------------------------------------------------------------------
// eNX`Xg̓ǂݍ
void TextSceneLoader::readTextureList(){
	openChunk();
	while(true){
		String token = readToken("readTextureList()");
		if(token == "}"){ break; }
		// eNX`̎
		openChunk();
		skipWord("type");
		String type = readToken("readTextureList()");
		if(type == "Surface"){
			readSurfaceTexture(token);
		}else{
			ErrorOut("TextSceneLoader::readTextureList() "
				"invalid type %s (line %d)",
				type.getBytes(), tokenizer_->getLineNumber());
		}
		closeChunk();
	}
}
//------------------------------------------------------------------------------
// T[tF[XeNX`̓ǂݍ
void TextSceneLoader::readSurfaceTexture(const String& name){
	SurfaceTexture* texture = textureManager_->createSurfaceTexture(name);
	// AhX[hU
	skipWord("addressModeU");
	String addressModeU = readToken("readSurfaceTexture()");
	texture->setAddressModeU(Texture::addressModeFromString(addressModeU));
	// AhX[hV
	skipWord("addressModeV");
	String addressModeV = readToken("readSurfaceTexture()");
	texture->setAddressModeV(Texture::addressModeFromString(addressModeV));
	// s[gUV
	skipWord("repeatUV");
	texture->setRepeatUV(readTexCoord2());
	// ItZbgUV
	skipWord("offsetUV");
	texture->setOffsetUV(readTexCoord2());
}
//------------------------------------------------------------------------------
// sN`
//------------------------------------------------------------------------------
// sN`Xg̓ǂݍ
void TextSceneLoader::readPictureList(){
	openChunk();
	while(true){
		String token = readToken("readPictureList()");
		if(token == "}"){ break; }
		openChunk();
		skipWord("path");
		String path = readToken("readPictureList()");
		closeChunk();
		// gq擾
		FilePath filePath(path);
		String extension = filePath.getExtension();
		// gqɂ킹ăsN`ǂݕ
		Picture* picture;
		if(extension == "tga"){
			TargaLoader loader(basePath_ + path);
			loader.loadHeader();
			// At@̗L
			if(loader.hasAlpha()){
				picture = readPictureRGBA8(token, loader);
			}else{
				picture = readPictureRGB8(token, loader);
			}
		}else{
			ErrorOut("TextSceneLoader::readPictureList() "
				"Unsupported picture format %s (line %d)",
				path.getBytes(), tokenizer_->getLineNumber());
			return;
		}
		// pXݒ肷
		picture->setPath(path);
	}
}
//------------------------------------------------------------------------------
// RGB8rbgsN`̓ǂݍ
Picture* TextSceneLoader::readPictureRGB8(
	const String& name, TargaLoader& loader){
	PictureRGB8* picture = pictureManager_->createPictureRGB8(name);
	picture->setSize(loader.getSize());
	loader.loadImage(picture->getImageBuffer());
	return picture;
}
//------------------------------------------------------------------------------
// RGBA8rbgsN`̓ǂݍ
Picture* TextSceneLoader::readPictureRGBA8(
	const String& name, TargaLoader& loader){
	PictureRGBA8* picture = pictureManager_->createPictureRGBA8(name);
	picture->setSize(loader.getSize());
	loader.loadImage(picture->getImageBuffer());
	return picture;
}
//------------------------------------------------------------------------------
// V[m[hN
//------------------------------------------------------------------------------
// V[m[hNXg̓ǂݍ
void TextSceneLoader::readSceneNodeLinkList(){
	openChunk();
	while(true){
		String token = readToken("readSceneNodeLinkList()");
		if(token == "}"){ break; }
		readSceneNodeLink(token);
	}
}
//------------------------------------------------------------------------------
// V[m[hN̓ǂݍ
void TextSceneLoader::readSceneNodeLink(const String& name){
	SceneNode* sceneNode;
	if(name.equals("RootNode")){
		sceneNode = rootNode_;
	}else{
		sceneNode = sceneNodeManager_->search(name);
		if(sceneNode == NULL){
			ErrorOut("TextSceneLoader::readSceneNodeLink() "
				"not found sceneNode %s (line %d)",
				name.getBytes(), tokenizer_->getLineNumber());
		}
	}

	openChunk();
	while(true){
		String token = readToken("readSceneNodeLink()");
		if(token == "}"){ break; }
		if(token == "SceneNode"){
			// V[m[hN
			openChunk();
			while(true){
				String subToken = readToken("readSceneNodeLink()");
				if(subToken == "}"){ break; }
				SceneNode* childNode = sceneNodeManager_->search(subToken);
				if(childNode == NULL){
					ErrorOut("TextSceneLoader::readSceneNodeLink() "
						"not found sceneNode %s (line %d)",
						subToken.getBytes(), tokenizer_->getLineNumber());
				}
				sceneNode->addSceneNode(childNode);
			}
		}else if(token == "SceneLeaf"){
			// V[[tN
			openChunk();
			while(true){
				String subToken = readToken("readSceneNodeLink()");
				if(subToken == "}"){ break; }
				SceneLeaf* sceneLeaf = NULL;
				String sceneLeafName = readToken("readSceneNodeLink()");
				if(subToken.equals("Model")){
					sceneLeaf = modelManager_->search(sceneLeafName);
				}else if(subToken.equals("Light")){
					sceneLeaf = lightManager_->search(sceneLeafName);
				}else{
					ErrorOut("TextSceneLoader::readSceneNodeLink() "
						"unsupported type %s (line %d)",
						subToken.getBytes(), tokenizer_->getLineNumber());
				}
				if(sceneLeaf == NULL){
					ErrorOut("TextSceneLoader::readSceneNodeLink() "
						"not found sceneLeaf %s (line %d)",
						sceneLeafName.getBytes(), tokenizer_->getLineNumber());
				}
				sceneNode->addSceneLeaf(sceneLeaf);
			}
		}else{
			ErrorOut("TextSceneLoader::readSceneNodeLink() "
				"invalid token %s (line %d)",
				token.getBytes(), tokenizer_->getLineNumber());
		}
	}
}
//------------------------------------------------------------------------------
// fN
//------------------------------------------------------------------------------
// fNXg̓ǂݍ
void TextSceneLoader::readModelLinkList(){
	openChunk();
	while(true){
		String token = readToken("readModelLinkList()");
		if(token == "}"){ break; }
		readModelLink(token);
	}
}
//------------------------------------------------------------------------------
// fN̓ǂݍ
void TextSceneLoader::readModelLink(const String& name){
	Model* model = modelManager_->search(name);
	if(model == NULL){
		ErrorOut("TextSceneLoader::readModelLink() "
			"not found model %s (line %d)",
			name.getBytes(), tokenizer_->getLineNumber());
	}
	openChunk();
	while(true){
		String token = readToken("readModelLink()");
		if(token == "}"){ break; }
		// bVƂ̃N
		Mesh* mesh = meshManager_->search(token);
		if(mesh == NULL){
			ErrorOut("TextSceneLoader::readModelLink() "
				"not found mesh %s (line %d)",
				token.getBytes(), tokenizer_->getLineNumber());
		}
		model->addMesh(mesh);
	}
}
//------------------------------------------------------------------------------
// bVN
//------------------------------------------------------------------------------
// bVNXg̓ǂݍ
void TextSceneLoader::readMeshLinkList(){
	openChunk();
	while(true){
		String token = readToken("readMeshLinkList()");
		if(token == "}"){ break; }
		readMeshLink(token);
	}
}
//------------------------------------------------------------------------------
// bVN̓ǂݍ
void TextSceneLoader::readMeshLink(const String& name){
	Mesh* mesh = meshManager_->search(name);
	if(mesh == NULL){
		ErrorOut("TextSceneLoader::readMeshLink() "
			"not found mesh %s (line %d)",
			name.getBytes(), tokenizer_->getLineNumber());
	}
	openChunk();
	while(true){
		String token = readToken("readMeshLink()");
		if(token == "}"){ break; }
		if(token == "MeshData"){
			// bVf[^Ƃ̃N
			String subToken = readToken("readMeshLink()");
			MeshData* meshData = meshDataManager_->search(subToken);
			if(meshData == NULL){
				ErrorOut("TextSceneLoader::readMeshLink() "
					"not found meshData %s (line %d)",
					subToken.getBytes(), tokenizer_->getLineNumber());
			}
			mesh->setMeshData(meshData);
		}else if(token == "Material"){
			// }eAƂ̃N
			String subToken = readToken("readMeshLink()");
			Material* material = materialManager_->search(subToken);
			if(material == NULL){
				ErrorOut("TextSceneLoader::readMeshLink() "
					"not found material %s (line %d)",
					subToken.getBytes(), tokenizer_->getLineNumber());
			}
			mesh->setMaterial(material);
		}else{
			ErrorOut("TextSceneLoader::readMeshLink() "
				"invalid token %s (line %d)",
				token.getBytes(), tokenizer_->getLineNumber());
		}
	}
}
//------------------------------------------------------------------------------
// }eAN
//------------------------------------------------------------------------------
// }eANXg̓ǂݍ
void TextSceneLoader::readMaterialLinkList(){
	openChunk();
	while(true){
		String token = readToken("readMaterialLinkList()");
		if(token == "}"){ break; }
		openChunk();
		skipWord("type");
		String type = readToken("readMaterialLinkList()");
		if(type == "Basic"){
			readBasicMaterialLink(token);
		}else{
			ErrorOut("TextSceneLoader::readMaterialLinkList() "
				"invalid type %s (line %d)",
				type.getBytes(), tokenizer_->getLineNumber());
		}
	}
}
//------------------------------------------------------------------------------
// }eAN̓ǂݍ
void TextSceneLoader::readBasicMaterialLink(const String& name){
	BasicMaterial* material =
		materialManager_->search(name)->castBasicMaterial();
	if(material == NULL){
		ErrorOut("TextSceneLoader::readBasicMaterialLink() "
			"not found material %s (line %d)",
			name.getBytes(), tokenizer_->getLineNumber());
	}
	while(true){
		String token = readToken("readMaterialLink()");
		if(token == "}"){ break; }
		if(token == "baseTexture"){
			// x[XeNX`Ƃ̃N
			String textureName = readToken("readBasicMaterialLink()");
			Texture* texture = textureManager_->search(textureName);
			if(texture == NULL){
				ErrorOut("TextSceneLoader::readBasicMaterialLink() "
					"not found baseTexture %s (line %d)",
					textureName.getBytes(), tokenizer_->getLineNumber());
			}
			material->setBaseTexture(texture);
		}else if(token == "glossTexture"){
			// eNX`Ƃ̃N
			String textureName = readToken("readBasicMaterialLink()");
			Texture* texture = textureManager_->search(textureName);
			if(texture == NULL){
				ErrorOut("TextSceneLoader::readBasicMaterialLink() "
					"not found glossTexture %s (line %d)",
					textureName.getBytes(), tokenizer_->getLineNumber());
			}
			material->setGlossTexture(texture);
		}else if(token == "lightTexture"){
			// CgeNX`Ƃ̃N
			String textureName = readToken("readBasicMaterialLink()");
			Texture* texture = textureManager_->search(textureName);
			if(texture == NULL){
				ErrorOut("TextSceneLoader::readBasicMaterialLink() "
					"not found lightTexture %s (line %d)",
					textureName.getBytes(), tokenizer_->getLineNumber());
			}
			material->setLightTexture(texture);
		}else if(token == "stainTexture"){
			// eNX`Ƃ̃N
			String textureName = readToken("readBasicMaterialLink()");
			Texture* texture = textureManager_->search(textureName);
			if(texture == NULL){
				ErrorOut("TextSceneLoader::readBasicMaterialLink() "
					"not found stainTexture %s (line %d)",
					textureName.getBytes(), tokenizer_->getLineNumber());
			}
			material->setStainTexture(texture);
		}else{
			ErrorOut("TextSceneLoader::readBasicMaterialLink() "
				"invalid token %s (line %d)",
				token.getBytes(), tokenizer_->getLineNumber());
		}
	}
}
//------------------------------------------------------------------------------
// eNX`N
//------------------------------------------------------------------------------
// eNX`NXg̓ǂݍ
void TextSceneLoader::readTextureLinkList(){
	openChunk();
	while(true){
		String token = readToken("readTextureLinkList()");
		if(token == "}"){ break; }
		readTextureLink(token);
	}
}
//------------------------------------------------------------------------------
// eNX`N̓ǂݍ
void TextSceneLoader::readTextureLink(const String& name){
	Texture* texture = textureManager_->search(name);
	if(texture == NULL){
		ErrorOut("TextSceneLoader::readTextureLink() "
			"not found texture %s (line %d)",
			name.getBytes(), tokenizer_->getLineNumber());
	}
	openChunk();
	while(true){
		String token = readToken("readTextureLink()");
		if(token == "}"){ break; }
		// sN`Ƃ̃N
		Picture* picture = pictureManager_->search(token);
		if(picture == NULL){
			ErrorOut("TextSceneLoader::readTextureLink() "
				"not found picture %s (line %d)",
				token.getBytes(), tokenizer_->getLineNumber());
		}
		texture->addPicture(picture);
	}
}
//------------------------------------------------------------------------------
// l̓ǂݍ
//------------------------------------------------------------------------------
// booll̓ǂݍ
bool TextSceneLoader::readBool(){
	String token = readToken("readBool()");
	if(token == "true"){ return true; }
	else if(token == "false"){ return false; }
	ErrorOut("TextSceneLoader::readBool() invalid token %s (line %d)",
		token.getBytes(), tokenizer_->getLineNumber());
	return false;
}
//------------------------------------------------------------------------------
// u_charl̓ǂݍ
u_char TextSceneLoader::readUChar(){
	String token = readToken("readUChar()");
	u_char value = 0;
	bool result = token.parseUChar(&value);
	if(!result){
		ErrorOut("TextSceneLoader::readUChar() invalid token %s (line %d)",
			token.getBytes(), tokenizer_->getLineNumber());
	}
	return value;
}
//------------------------------------------------------------------------------
// u_shortl̓ǂݍ
u_short TextSceneLoader::readUShort(){
	String token = readToken("readUShort()");
	u_short value = 0;
	bool result = token.parseUShort(&value);
	if(!result){
		ErrorOut("TextSceneLoader::readUShort() invalid token %s (line %d)",
			token.getBytes(), tokenizer_->getLineNumber());
	}
	return value;
}
//------------------------------------------------------------------------------
// intl̓ǂݍ
int TextSceneLoader::readInt(){
	String token = readToken("readInt()");
	int value = 0;
	bool result = token.parseInt(&value);
	if(!result){
		ErrorOut("TextSceneLoader::readInt() invalid token %s (line %d)",
			token.getBytes(), tokenizer_->getLineNumber());
	}
	return value;
}
//------------------------------------------------------------------------------
// u_intl̓ǂݍ
u_int TextSceneLoader::readUInt(){
	String token = readToken("readInt()");
	u_int value = 0;
	bool result = token.parseUInt(&value);
	if(!result){
		ErrorOut("TextSceneLoader::readUInt() invalid token %s (line %d)",
			token.getBytes(), tokenizer_->getLineNumber());
	}
	return value;
}
//------------------------------------------------------------------------------
// floatl̓ǂݍ
float TextSceneLoader::readFloat(){
	String token = readToken("readFloat()");
	float value = 0.f;
	bool result = token.parseFloat(&value);
	if(!result){
		ErrorOut("TextSceneLoader::readFloat() invalid token %s (line %d)",
			token.getBytes(), tokenizer_->getLineNumber());
	}
	return value;
}
//------------------------------------------------------------------------------
// Vector3l̓ǂݍ
Vector3 TextSceneLoader::readVector3(){
	Vector3 result;
	openChunk();
	result.x = readFloat();
	result.y = readFloat();
	result.z = readFloat();
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// Matrix34l̓ǂݍ
Matrix34 TextSceneLoader::readMatrix34(){
	Matrix34 result;
	openChunk();
	for(int i = 0; i < 12; i++){
		result.array[i] = readFloat();
	}
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// OvfJ[l̓ǂݍ
Color3c TextSceneLoader::readColor3c(){
	Color3c result;
	openChunk();
	result.r = readUChar();
	result.g = readUChar();
	result.b = readUChar();
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// lvfJ[l̓ǂݍ
Color4c TextSceneLoader::readColor4c(){
	Color4c result;
	openChunk();
	result.r = readUChar();
	result.g = readUChar();
	result.b = readUChar();
	result.a = readUChar();
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// OvfJ[l̓ǂݍ
Color3f TextSceneLoader::readColor3f(){
	Color3f result;
	openChunk();
	result.r = readFloat();
	result.g = readFloat();
	result.b = readFloat();
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// lvfJ[l̓ǂݍ
Color4f TextSceneLoader::readColor4f(){
	Color4f result;
	openChunk();
	result.r = readFloat();
	result.g = readFloat();
	result.b = readFloat();
	result.a = readFloat();
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// 񎟌eNX`Wl̓ǂݍ
TexCoord2 TextSceneLoader::readTexCoord2(){
	TexCoord2 result;
	openChunk();
	result.u = readFloat();
	result.v = readFloat();
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// {bNX̓ǂݍ
AxisAlignedBox TextSceneLoader::readAxisAlignedBox(){
	AxisAlignedBox result;
	Vector3 minimum, maximum;
	openChunk();
	minimum.x = readFloat();
	minimum.y = readFloat();
	minimum.z = readFloat();
	closeChunk();
	openChunk();
	maximum.x = readFloat();
	maximum.y = readFloat();
	maximum.z = readFloat();
	closeChunk();
	result.set(minimum, maximum);
	return result;
}
//------------------------------------------------------------------------------
// ̓ǂݍ
Sphere TextSceneLoader::readSphere(){
	Sphere result;
	Vector3 vector;
	openChunk();
	vector.x = readFloat();
	vector.y = readFloat();
	vector.z = readFloat();
	result.setCenter(vector);
	result.setRadius(readFloat());
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// g[N̓ǂݍ
String TextSceneLoader::readToken(const String& caller){
	bool hasNext = tokenizer_->nextToken();
	if(!hasNext){
		ErrorOut("TextSceneLoader::%s nothing token (line %d)",
			caller.getBytes(), tokenizer_->getLineNumber());
	}
	return tokenizer_->getToken();
}
//------------------------------------------------------------------------------
// w胏[h̓ǂݔ΂
void TextSceneLoader::skipWord(const String& word){
	String token = readToken("skipWord()");
	if(token != word){
		ErrorOut("TextSceneLoader::skipWord(%s) invalid token %s (line %d)",
			word.getBytes(), token.getBytes(), tokenizer_->getLineNumber());
	}
}
//------------------------------------------------------------------------------
// `N̓ǂݔ΂
void TextSceneLoader::skipChunk(){
	int chunkLine = tokenizer_->getLineNumber();
	int chunkCounter = 0;
	while(true){
		bool hasNext = tokenizer_->nextToken();
		if(!hasNext){
			ErrorOut("TextSceneLoader::skipChunk(%d) nothing token (line %d)",
				chunkLine, tokenizer_->getLineNumber());
		}
		String token = tokenizer_->getToken();
		if(token == "{"){
			chunkCounter++;
		}else if(token == "}"){
			chunkCounter--;
			// fNgɃ[v𔲂`FbNs
			if(chunkCounter == 0){ break; }
			if(chunkCounter < 0){
				ErrorOut("TextSceneLoader::skipChunk(%d) "
					"invalid chunk (line %d)",
					chunkLine, tokenizer_->getLineNumber());
			}
		}
	}
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
