//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * oCiV[Z[o
 * @author Junpee
 */

#include "LampBasic.h"
#include <direct.h>
#include "Graphics/InputOutput/BinarySceneSaver.h"
#include "Core/InputOutput/BinaryFileWriter.h"
#include "Core/InputOutput/FilePath.h"
#include "Core/Codec/Tga/TargaSaver.h"
#include "Graphics/Scene/Scene.h"
#include "Graphics/Fog/Fog.h"
#include "Graphics/Light/LightManager.h"
#include "Graphics/SceneNode/SceneNodeManager.h"
#include "Graphics/Model/ModelManager.h"
#include "Graphics/Mesh/MeshManager.h"
#include "Graphics/MeshData/MeshDataManager.h"
#include "Graphics/Material/MaterialManager.h"
#include "Graphics/Texture/TextureManager.h"
#include "Graphics/Picture/PictureManager.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
BinarySceneSaver::BinarySceneSaver(){
}
//------------------------------------------------------------------------------
// fXgN^
BinarySceneSaver::~BinarySceneSaver(){
}
//------------------------------------------------------------------------------
// Z[u
void BinarySceneSaver::save(const String& filePath, Scene* scene){
	FilePath path(filePath);
	BinaryFileWriter* binaryFileWriter = new BinaryFileWriter(filePath);
	save(binaryFileWriter, scene, path.getFolderPath());
	delete binaryFileWriter;
}
//------------------------------------------------------------------------------
// Z[u
void BinarySceneSaver::save(BinaryWriter* binaryWriter, Scene* scene,
	const String& basePath){
	// 
	writer_ = binaryWriter;
	basePath_ = basePath;
	scene_ = scene;
	sceneNodeManager_ = scene->getSceneNodeManager();
	lightManager_ = scene->getLightManager();
	modelManager_ = scene->getModelManager();
	meshManager_ = scene->getMeshManager();
	meshDataManager_ = scene->getMeshDataManager();
	materialManager_ = scene->getMaterialManager();
	textureManager_ = scene->getTextureManager();
	pictureManager_ = scene->getPictureManager();

	// wb_̏o
	writeHeader();

	// IuWFNgXg̏o
	int linkCount = writeObjectList();

	// N̏o
	writeLink(linkCount);
}
//------------------------------------------------------------------------------
// wb_̏o
void BinarySceneSaver::writeHeader(){
	// tH[}bgID̏o
	writeID("LBScene");
	// o[Wԍ̏o
	u_int binaryVersion = 0x00000900;
	writer_->writeUInt(binaryVersion);
	align();
}
//------------------------------------------------------------------------------
// IuWFNgXg̏o
int BinarySceneSaver::writeObjectList(){
	int linkCount = 0;
	// tHȌo
	writeFog(scene_->getFog());
	// tHO̓NȂ̂ŃJEgȂ

	// V[m[hXg̏o
	int sceneNodeCount = sceneNodeManager_->getCount();
	// [gm[h͏o͂Ȃ̂2ȏ
	if(sceneNodeCount > 1){ writeSceneNodeList(); }
	linkCount += sceneNodeCount;

	// CgXg̏o
	int lightCount = lightManager_->getCount();
	if(lightCount > 0){	writeLightList(); }
	// Cg̓NȂ̂ŃJEgȂ

	// fXg̏o
	int modelCount = modelManager_->getCount();
	if(modelCount > 0){ writeModelList(); }
	linkCount += modelCount;

	// bVXg̏o
	int meshCount = meshManager_->getCount();
	if(meshCount > 0){ writeMeshList(); }
	linkCount += meshCount;

	// bVf[^Xg̏o
	int meshDataCount = meshDataManager_->getCount();
	if(meshDataCount > 0){ writeMeshDataList(); }
	linkCount += meshDataCount;

	// }eAXg̏o
	int materialCount = materialManager_->getCount();
	if(materialCount > 0){ writeMaterialList(); }
	linkCount += materialCount;

	// eNX`Xg̏o
	int textureCount = textureManager_->getCount();
	if(textureCount > 0){ writeTextureList(); }
	linkCount += textureCount;

	// sN`Xg̏o
	int pictureCount = pictureManager_->getCount();
	if(pictureCount > 0){
		int startAddress = startBlock("Picture");
		for(int i = 0; i < pictureCount; i++){
			writePicture(pictureManager_->get(i));
		}
		endBlock(startAddress);
	}
	linkCount += pictureCount;

	return linkCount;
}
//------------------------------------------------------------------------------
// N̏o
void BinarySceneSaver::writeLink(int linkCount){
	// IuWFNg1ȉȂ烊N݂͑Ȃ
	if(linkCount <= 1){ return; }

	{ // V[m[hN̏o
		int startAddress = startBlock("SceneNodeLin");
		int sceneNodeCount = sceneNodeManager_->getCount();
		for(int i = 0; i < sceneNodeCount; i++){
			writeSceneNodeLink(sceneNodeManager_->get(i));
		}
		endBlock(startAddress);
	}

	// fN̏o
	int modelCount = modelManager_->getCount();
	if(modelCount > 0){
		int startAddress = startBlock("ModelLink");
		for(int i = 0; i < modelCount; i++){
			writeModelLink(modelManager_->get(i));
		}
		endBlock(startAddress);
	}

	// bVN̏o
	int meshCount = meshManager_->getCount();
	if(meshCount > 0){
		int startAddress = startBlock("MeshLink");
		for(int i = 0; i < meshCount; i++){
			writeMeshLink(meshManager_->get(i));
		}
		endBlock(startAddress);
	}

	// }eAeNX`N̏o
	// eNX`݂΃}eÃN
	int textureCount = textureManager_->getCount();
	if(textureCount > 0){ writeMaterialLinkList(); }

	// eNX`N̏o
	if(textureCount > 0){
		int startAddress = startBlock("TextureLink");
		for(int i = 0; i < textureCount; i++){
			writeTextureLink(textureManager_->get(i));
		}
		endBlock(startAddress);
	}
}
//------------------------------------------------------------------------------
// tHO
//------------------------------------------------------------------------------
// tHȌo
void BinarySceneSaver::writeFog(Fog* fog){
	int startAddress = startBlock("Fog");
	// J[
	writeColor4c(fog->getColor());
	// [h
	writer_->writeInt(fog->getMode());
	// Zx
	writer_->writeFloat(fog->getDensity());
	// jA
	writer_->writeFloat(fog->getNear());
	// t@[
	writer_->writeFloat(fog->getFar());
	// LA
	writer_->writeBool(fog->isEnabled());
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// V[m[h
//------------------------------------------------------------------------------
// V[m[hXg̏o
void BinarySceneSaver::writeSceneNodeList(){
	int startAddress = startBlock("SceneNode");
	int sceneNodeCount = sceneNodeManager_->getCount();
	// [gm[h͏o͂Ȃ
	for(int i = 1; i < sceneNodeCount; i++){
		SceneNode* sceneNode = sceneNodeManager_->get(i);
		if(sceneNode->isLODSceneNode()){
			writeLODSceneNode(sceneNode->castLODSceneNode());
		}else{
			writeSceneNode(sceneNode);
		}
	}
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// V[m[h̏o
void BinarySceneSaver::writeSceneNode(SceneNode* sceneNode){
	int startAddress = startBlock("Standard");
	// O
	writeString(sceneNode->getName());
	// XP[
	writeVector3(sceneNode->getScale());
	// ]
	writeVector3(sceneNode->getRotationXYZ());
	// ړ
	writeVector3(sceneNode->getTranslation());
	// LA
	writer_->writeBool(sceneNode->isEnabled());
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// xIufBeB[V[m[h̏o
void BinarySceneSaver::writeLODSceneNode(LODSceneNode* sceneNode){
	int startAddress = startBlock("LOD");
	// O
	writeString(sceneNode->getName());
	// XP[
	writeVector3(sceneNode->getScale());
	// ]
	writeVector3(sceneNode->getRotationXYZ());
	// ړ
	writeVector3(sceneNode->getTranslation());
	// LOD
	int lodThresholdCount = sceneNode->getLODThresholdCount();
	writer_->writeInt(lodThresholdCount);
	for(int i = 0; i < lodThresholdCount; i++){
		writer_->writeFloat(sceneNode->getLODThreshold(i));
	}
	// LA
	writer_->writeBool(sceneNode->isEnabled());
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// Cg
//------------------------------------------------------------------------------
// CgXg̏o
void BinarySceneSaver::writeLightList(){
	int startAddress = startBlock("Light");
	int lightCount = lightManager_->getCount();
	for(int i = 0; i < lightCount; i++){
		Light* light = lightManager_->get(i);
		if(light->isAmbientLight()){
			writeAmbientLight(light->castAmbientLight());
		}else if(light->isDirectionalLight()){
			writeDirectionalLight(light->castDirectionalLight());
		}else if(light->isPointLight()){
			writePointLight(light->castPointLight());
		}else{
			ErrorOut("BinarySceneSaver::writeLightList() unsupported light %s",
				light->getName().getBytes());
		}
	}
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// Cg̏o
void BinarySceneSaver::writeLight(Light* light){
	// Cg}XN
	writer_->writeUInt(light->getLightMask());
	// LA
	writer_->writeBool(light->isEnabled());
}
//------------------------------------------------------------------------------
// ArGgCg̏o
void BinarySceneSaver::writeAmbientLight(AmbientLight* light){
	int startAddress = startBlock("Ambient");
	// O
	writeString(light->getName());
	// J[
	writeColor3f(light->getColor());
	// Cg̏o
	writeLight(light);
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// fBNViCg̏o
void BinarySceneSaver::writeDirectionalLight(DirectionalLight* light){
	int startAddress = startBlock("Directional");
	// O
	writeString(light->getName());
	// fBt[YJ[
	writeColor3f(light->getDiffuseColor());
	// XyLJ[
	writeColor3f(light->getSpecularColor());
	// 
	writeVector3(light->getDirection());
	// Cg̏o
	writeLight(light);
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// |CgCg̏o
void BinarySceneSaver::writePointLight(PointLight* light){
	int startAddress = startBlock("Point");
	// O
	writeString(light->getName());
	// fBt[YJ[
	writeColor3f(light->getDiffuseColor());
	// XyLJ[
	writeColor3f(light->getSpecularColor());
	// ʒu
	writeVector3(light->getPosition());
	// W
	writer_->writeFloat(light->getRange());
	// W
	writer_->writeFloat(light->getAttenuation0());
	writer_->writeFloat(light->getAttenuation1());
	writer_->writeFloat(light->getAttenuation2());
	// Cg̏o
	writeLight(light);
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// f
//------------------------------------------------------------------------------
// fXg̏o
void BinarySceneSaver::writeModelList(){
	int startAddress = startBlock("Model");
	int modelCount = modelManager_->getCount();
	for(int i = 0; i < modelCount; i++){
		Model* model = modelManager_->get(i);
		if(model->isStandardModel()){
			writeStandardModel(model->castStandardModel());
		}else if(model->isCharacterModel()){
			writeCharacterModel(model->castCharacterModel());
		}else{
			ErrorOut("BinarySceneSaver::writeModelList() "
				"unsupported model %s", model->getName().getBytes());
		}
	}
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// Wf̏o
void BinarySceneSaver::writeStandardModel(StandardModel* model){
	int startAddress = startBlock("Standard");
	// O
	writeString(model->getName());
	// LA
	writer_->writeBool(model->isEnabled());
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// LN^f̏o
void BinarySceneSaver::writeCharacterModel(CharacterModel* model){
	int startAddress = startBlock("Character");
	// O
	writeString(model->getName());
	// LA
	writer_->writeBool(model->isEnabled());
	// ACg
	align();
	// {[̏o
	int boneCount = model->getBoneCount();
	writer_->writeInt(boneCount);
	for(int i = 0; i < boneCount; i++){ writeBone(model->getBone(i)); }
	// {[N̏o
	for(int i = 0; i < boneCount; i++){
		writeBoneLink(model, model->getBone(i));
	}
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// {[̏o
void BinarySceneSaver::writeBone(Bone* bone){
	// O
	writeString(bone->getName());
	// |[Yts
	writeMatrix34(bone->getInversePoseMatrix());
	// XP[
	writeVector3(bone->getScale());
	// ]
	writeVector3(bone->getRotationXYZ());
	// ړ
	writeVector3(bone->getTranslation());
}
//------------------------------------------------------------------------------
// {[N̏o
void BinarySceneSaver::writeBoneLink(CharacterModel* model, Bone* bone){
	int boneCount = bone->getBoneCount();
	writer_->writeInt(boneCount);
	for(int i = 0; i < boneCount; i++){
		writer_->writeInt(model->getBoneIndex(bone->getBone(i)));
	}
}
//------------------------------------------------------------------------------
// bV
//------------------------------------------------------------------------------
// bVXg̏o
void BinarySceneSaver::writeMeshList(){
	int startAddress = startBlock("Mesh");
	int meshCount = meshManager_->getCount();
	for(int i = 0; i < meshCount; i++){
		Mesh* mesh = meshManager_->get(i);
		if(mesh->isRigidMesh()){
			writeRigidMesh(mesh->castRigidMesh());
		}else if(mesh->isCharacterMesh()){
			writeCharacterMesh(mesh->castCharacterMesh());
		}else{
			ErrorOut("BinarySceneSaver::writeMeshList() unsupported mesh %s",
				mesh->getName().getBytes());
		}
	}
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// ̃bV̏o
void BinarySceneSaver::writeRigidMesh(RigidMesh* mesh){
	int startAddress = startBlock("Rigid");
	// O
	writeString(mesh->getName());
	// LA
	writer_->writeBool(mesh->isEnabled());
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// LN^bV̏o
void BinarySceneSaver::writeCharacterMesh(CharacterMesh* mesh){
	int startAddress = startBlock("Character");
	// O
	writeString(mesh->getName());
	// LA
	writer_->writeBool(mesh->isEnabled());
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// bVf[^
//------------------------------------------------------------------------------
// bVf[^Xg̏o
void BinarySceneSaver::writeMeshDataList(){
	int startAddress = startBlock("MeshData");
	int meshDataCount = meshDataManager_->getCount();
	for(int i = 0; i < meshDataCount; i++){
		writeMeshData(meshDataManager_->get(i));
	}
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// bVf[^̏o
void BinarySceneSaver::writeMeshData(MeshData* meshData){
	// O
	writeString(meshData->getName());
	// oEfBO{bNX
	writeAxisAlignedBox(meshData->getBoundingBox());
	// oEfBOXtBA
	writeSphere(meshData->getBoundingSphere());
	// v~eBu^Cv
	writer_->writeInt(meshData->getPrimitiveType());
	// CfbNXo
	int vertexIndexCount = meshData->getVertexIndexCount();
	writer_->writeInt(vertexIndexCount);
	if(vertexIndexCount > 0){
		writer_->writeBytes(meshData->getVertexIndexArray(),
			sizeof(u_short) * vertexIndexCount);
	}
	// @Ă邩
	bool hasNormal = meshData->hasNormal();
	writer_->writeBool(hasNormal);
	// _J[Ă邩
	bool hasColor = meshData->hasColor();
	writer_->writeBool(hasColor);
	// ACg
	align();
	// _
	int vertexCount = meshData->getVertexCount();
	writer_->writeInt(vertexCount);
	// eNX`WZbg
	int texCoordSetCount = meshData->getTexCoordSetCount();
	writer_->writeInt(texCoordSetCount);
	// eNX`W^Cv
	for(int i = 0; i < texCoordSetCount; i++){
		int texCoordType = meshData->getTexCoordType(i);
		writer_->writeInt(texCoordType);
	}
	// ʒu
	writer_->writeBytes(meshData->getPositionArray(),
		sizeof(Vector3) * vertexCount);
	// @
	if(hasNormal){
		writer_->writeBytes(meshData->getNormalArray(),
			sizeof(Vector3) * vertexCount);
	}
	// _J[
	if(hasColor){
		writer_->writeBytes(meshData->getColorArray(),
			sizeof(Color4c) * vertexCount);
	}
	// eNX`W
	for(int i = 0; i < texCoordSetCount; i++){
		writer_->writeBytes(meshData->getTexCoordArray(i),
			meshData->getTexCoordArraySize(i));
	}
	// _{[
	int bonesPerVertex = meshData->getBonesPerVertex();
	writer_->writeInt(bonesPerVertex);
	if(bonesPerVertex != 0){
		// {[CfbNX
		writer_->writeBytes(meshData->getBoneIndexArray(),
			sizeof(u_char) * vertexCount * bonesPerVertex);
	}
	// _EFCg
	int weightsPerVertex = meshData->getWeightsPerVertex();
	if(weightsPerVertex != 0){
		// EFCg
		writer_->writeBytes(meshData->getWeightArray(),
			sizeof(float) * vertexCount * weightsPerVertex);
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// }eA
//------------------------------------------------------------------------------
// }eAXg̏o
void BinarySceneSaver::writeMaterialList(){
	int startAddress = startBlock("Material");
	int materialCount = materialManager_->getCount();
	for(int i = 0; i < materialCount; i++){
		Material* material = materialManager_->get(i);
		if(material->isBasicMaterial()){
			writeBasicMaterial(material->castBasicMaterial());
		}else{
			ErrorOut("BinarySceneSaver::writeMaterialList() "
				"unsupported material %s", material->getName().getBytes());
		}
	}
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// }eȀo
void BinarySceneSaver::writeMaterial(const Material* material){
	// O
	writeString(material->getName());
	// Dx
	writer_->writeInt(material->getPriority());
	// uh[h
	writer_->writeInt(material->getBlendMode());
	// At@l
	writer_->writeFloat(material->getAlpha());
	// uh\[X
	writer_->writeInt(material->getBlendSource());
	// uhfXeBl[V
	writer_->writeInt(material->getBlendDestination());
	// tHOIvV
	writer_->writeInt(material->getFogOption());
	// Cg}XN
	writer_->writeUInt(material->getLightMask());
	// Z
	writer_->writeBool(material->useZWrite());
	// ZeXg
	writer_->writeBool(material->useZTest());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// {}eȀo
void BinarySceneSaver::writeBasicMaterial(const BasicMaterial* material){
	int startAddress = startBlock("Basic");
	// }eȀo
	writeMaterial(material);
	// x[XUVCfbNX
	writer_->writeInt(material->getBaseUVIndex());
	// UVCfbNX
	writer_->writeInt(material->getGlossUVIndex());
	// CgUVCfbNX
	writer_->writeInt(material->getLightUVIndex());
	// UVCfbNX
	writer_->writeInt(material->getStainUVIndex());
	// fBt[YF
	writeColor3f(material->getDiffuseColor());
	// XyLF
	writeColor3f(material->getSpecularColor());
	// XyLp[
	writer_->writeFloat(material->getSpecularPower());
	// ArGgF
	writeColor3f(material->getAmbientColor());
	// G~bVu
	writeColor3f(material->getEmissiveColor());
	// ACg
	align();
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// eNX`
//------------------------------------------------------------------------------
// eNX`Xg̏o
void BinarySceneSaver::writeTextureList(){
	int startAddress = startBlock("Texture");
	int textureCount = textureManager_->getCount();
	for(int i = 0; i < textureCount; i++){
		Texture* texture = textureManager_->get(i);
		if(texture->isSurfaceTexture()){
			writeSurfaceTexture(texture->castSurfaceTexture());
		}else{
			ErrorOut("BinarySceneSaver::writeTextureList() "
				"unsupported texture %s", texture->getName().getBytes());
		}
	}
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// T[tF[XeNX`̏o
void BinarySceneSaver::writeSurfaceTexture(const SurfaceTexture* texture){
	int startAddress = startBlock("Surface");
	// O
	writeString(texture->getName());
	// AhX[hU
	writer_->writeInt(texture->getAddressModeU());
	// AhX[hV
	writer_->writeInt(texture->getAddressModeV());
	// s[gUV
	writeTexCoord2(texture->getRepeatUV());
	// ItZbgUV
	writeTexCoord2(texture->getOffsetUV());
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// sN`
//------------------------------------------------------------------------------
// sN`̏o
void BinarySceneSaver::writePicture(const Picture* picture){
	// O
	writeString(picture->getName());
	// pX
	String path = picture->getPath();
	writeString(path);
	// sN`o̓fBNg̍쐬
	FilePath filePath(path);
	path = filePath.getFolderPath();
	if(path != ""){
		path = basePath_ + path;
		mkdir(path.getBytes());
	}
	// sN`t@C̕ۑ
	TargaSaver saver;
	if(picture->isPictureRGB8()){
		// RGB824bitTargaŕۑ
		PictureRGB8* rgb8 = picture->castPictureRGB8();
		saver.save(basePath_ + rgb8->getPath(),
			rgb8->getSize(), rgb8->getImage());
	}else if(picture->isPictureRGBA8()){
		// RGBA832bitTargaŕۑ
		PictureRGBA8* rgba8 = picture->castPictureRGBA8();
		saver.save(basePath_ + rgba8->getPath(),
			rgba8->getSize(), rgba8->getImage());
	}else{
		ErrorOut("BinarySceneSaver::writePictureList() "
			"unsupported picture %s", picture->getName().getBytes());
	}
}
//------------------------------------------------------------------------------
// N
//------------------------------------------------------------------------------
// V[m[hN̏o
void BinarySceneSaver::writeSceneNodeLink(SceneNode* sceneNode){
	// N0ǂ`FbN
	int sceneNodeCount = sceneNode->getSceneNodeCount();
	int sceneLeafCount = sceneNode->getSceneLeafCount();
	if(sceneNodeCount + sceneLeafCount == 0){ return; }
	// O
	writeString(sceneNode->getName());
	// V[m[hN
	writer_->writeInt(sceneNodeCount);
	// V[m[hN
	for(int i = 0; i < sceneNodeCount; i++){
		writeString(sceneNode->getSceneNode(i)->getName());
	}
	// V[[tN
	writer_->writeInt(sceneLeafCount);
	// V[[tN
	for(int i = 0; i < sceneLeafCount; i++){
		SceneLeaf* sceneLeaf = sceneNode->getSceneLeaf(i);
		if(sceneLeaf->isModel()){
			writer_->writeInt(0);
		}else if(sceneLeaf->isLight()){
			writer_->writeInt(1);
		}else{
			Assert(false);
		}
		writeString(sceneLeaf->getName());
	}
}
//------------------------------------------------------------------------------
// fN̏o
void BinarySceneSaver::writeModelLink(const Model* model){
	int meshCount = model->getMeshCount();
	if(meshCount == 0){ return; }
	// O
	writeString(model->getName());
	// bVN
	writer_->writeInt(meshCount);
	// bVN
	for(int i = 0; i < meshCount; i++){
		writeString(model->getMesh(i)->getName());
	}
}
//------------------------------------------------------------------------------
// bVN̏o
void BinarySceneSaver::writeMeshLink(const Mesh* mesh){
	bool hasMeshData = (mesh->getMeshData() != NULL);
	bool hasMaterial = (mesh->getMaterial() != NULL);
	if((!hasMeshData) && (!hasMaterial)){ return; }
	// O
	writeString(mesh->getName());
	// bVf[^
	writer_->writeBool(hasMeshData);
	// }eA
	writer_->writeBool(hasMaterial);
	// ACg
	align();
	// bVf[^
	if(hasMeshData){ writeString(mesh->getMeshData()->getName()); }
	// }eA
	if(hasMaterial){ writeString(mesh->getMaterial()->getName()); }
}
//------------------------------------------------------------------------------
// }eANXg̏o
void BinarySceneSaver::writeMaterialLinkList(){
	int startAddress = startBlock("MaterialLink");
	int materialCount = materialManager_->getCount();
	for(int i = 0; i < materialCount; i++){
		Material* material = materialManager_->get(i);
		if(material->isBasicMaterial()){
			writeBasicMaterialLink(material->castBasicMaterial());
		}else{
			ErrorOut("BinarySceneSaver::writeMaterialLinkList() "
				"unsupported material %s", material->getName().getBytes());
		}
	}
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// {}eAN̏o
void BinarySceneSaver::writeBasicMaterialLink(const BasicMaterial* material){
	Texture* baseTexture = material->getBaseTexture();
	Texture* glossTexture = material->getGlossTexture();
	Texture* lightTexture = material->getLightTexture();
	Texture* stainTexture = material->getStainTexture();
	bool hasBaseTexture = (baseTexture != NULL);
	bool hasGlossTexture = (glossTexture != NULL);
	bool hasLightTexture = (lightTexture != NULL);
	bool hasStainTexture = (stainTexture != NULL);
	// eNX`̗L`FbN
	if((!hasBaseTexture) && (!hasGlossTexture) &&
		(!hasLightTexture) && (!hasStainTexture)){ return; }
	int startAddress = startBlock("Basic");
	// O
	writeString(material->getName());
	// eNX`̗L
	writer_->writeBool(hasBaseTexture);
	writer_->writeBool(hasGlossTexture);
	writer_->writeBool(hasLightTexture);
	writer_->writeBool(hasStainTexture);
	// ACg
	align();
	// eNX`
	if(hasBaseTexture){ writeString(baseTexture->getName()); }
	if(hasGlossTexture){ writeString(glossTexture->getName()); }
	if(hasLightTexture){ writeString(lightTexture->getName()); }
	if(hasStainTexture){ writeString(stainTexture->getName()); }
	endBlock(startAddress);
}
//------------------------------------------------------------------------------
// eNX`N̏o
void BinarySceneSaver::writeTextureLink(const Texture* texture){
	int pictureCount = texture->getPictureCount();
	if(pictureCount == 0){ return; }
	// O
	writeString(texture->getName());
	// sN`
	writer_->writeInt(pictureCount);
	// sN`N
	for(int i = 0; i < pictureCount; i++){
		writeString(texture->getPicture(i)->getName());
	}
}
//------------------------------------------------------------------------------
// l̏o
//------------------------------------------------------------------------------
// ̏o
void BinarySceneSaver::writeString(const String& string){
	int writeSize = string.getSize() + 1;
	writer_->writeInt(writeSize);
	writer_->writeBytes(string.getBytes(), writeSize);
	align();
}
//------------------------------------------------------------------------------
// OxNg̏o
void BinarySceneSaver::writeVector3(const Vector3& vector){
	writer_->writeBytes(vector.array, sizeof(Vector3));
}
//------------------------------------------------------------------------------
// 3~4s̏o
void BinarySceneSaver::writeMatrix34(const Matrix34& matrix){
	writer_->writeBytes(matrix.array, sizeof(Matrix34));
}
//------------------------------------------------------------------------------
// lvfJ[l̏o
void BinarySceneSaver::writeColor4c(const Color4c& color){
	writer_->writeBytes(color.array, sizeof(Color4c));
}
//------------------------------------------------------------------------------
// OvfJ[l̏o
void BinarySceneSaver::writeColor3f(const Color3f& color){
	writer_->writeBytes(color.array, sizeof(Color3f));
}
//------------------------------------------------------------------------------
// lvfJ[l̏o
void BinarySceneSaver::writeColor4f(const Color4f& color){
	writer_->writeBytes(color.array, sizeof(Color4f));
}
//------------------------------------------------------------------------------
// 񎟌eNX`Wl̏o
void BinarySceneSaver::writeTexCoord2(const TexCoord2& uv){
	writer_->writeBytes(uv.array, sizeof(TexCoord2));
}
//------------------------------------------------------------------------------
// {bNX̏o
void BinarySceneSaver::writeAxisAlignedBox(const AxisAlignedBox& box){
	writer_->writeBytes(&box, sizeof(AxisAlignedBox));
}
//------------------------------------------------------------------------------
// ̏o
void BinarySceneSaver::writeSphere(const Sphere& sphere){
	writer_->writeBytes(&sphere, sizeof(Sphere));
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// ACg
void BinarySceneSaver::align(){
	writer_->align(16);
}
//------------------------------------------------------------------------------
// ID̏o
void BinarySceneSaver::writeID(const String& id){
	u_char buffer[12];
	int size = id.getSize();
	Assert(size <= 12);
	for(int i = 0; i < size; i++){ buffer[i] = id.charAt(i); }
	for(int i = size; i < 12; i++){ buffer[i] = 0; }
	writer_->writeBytes(buffer, 12);
}
//------------------------------------------------------------------------------
// ubN̊Jn
int BinarySceneSaver::startBlock(const String& blockName){
	writeID(blockName);
	u_int blockAddress = writer_->getPosition();
	// TCY̗\
	writer_->writeInt(0);
	align();
	return blockAddress;
}
//------------------------------------------------------------------------------
// ubN̏I
void BinarySceneSaver::endBlock(int blockStartAddress){
	align();
	int nowPosition = writer_->getPosition();
	// TCYubNTCY
	int blockSize = nowPosition - blockStartAddress - 4;// ACgˑ
	writer_->setPosition(blockStartAddress);
	writer_->writeInt(blockSize);
	writer_->setPosition(nowPosition);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
