//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * oCiV[[_
 * @author Junpee
 */

#include "LampBasic.h"
#include "Graphics/InputOutput/BinarySceneLoader.h"
#include "Core/InputOutput/BinaryFileReader.h"
#include "Core/InputOutput/FilePath.h"
#include "Core/Codec/Tga/TargaLoader.h"
#include "Graphics/Scene/Scene.h"
#include "Graphics/Fog/Fog.h"
#include "Graphics/SceneNode/SceneNodeManager.h"
#include "Graphics/Light/LightManager.h"
#include "Graphics/Model/ModelManager.h"
#include "Graphics/Mesh/MeshManager.h"
#include "Graphics/MeshData/MeshDataManager.h"
#include "Graphics/Material/MaterialManager.h"
#include "Graphics/Texture/TextureManager.h"
#include "Graphics/Picture/PictureManager.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
BinarySceneLoader::BinarySceneLoader(){
}
//------------------------------------------------------------------------------
// fXgN^
BinarySceneLoader::~BinarySceneLoader(){
}
//------------------------------------------------------------------------------
// [h
void BinarySceneLoader::load(const String& filePath, Scene* scene){
	load(filePath, scene, scene->getRootNode());
}
//------------------------------------------------------------------------------
void BinarySceneLoader::load(const String& filePath, Scene* scene,
	SceneNode* rootNode){
	FilePath path(filePath);
	Assert(path.existFile());
	BinaryFileReader* binaryFileReader = new BinaryFileReader(filePath);
	load(binaryFileReader, scene, rootNode, path.getFolderPath());
	delete binaryFileReader;
}
//------------------------------------------------------------------------------
// [h
void BinarySceneLoader::load(BinaryReader* binaryReader, Scene* scene,
	const String& basePath){
	load(binaryReader, scene, scene->getRootNode(), basePath);
}
//------------------------------------------------------------------------------
// [h
void BinarySceneLoader::load(BinaryReader* binaryReader, Scene* scene,
	SceneNode* rootNode, const String& basePath){
	// 
	reader_ = binaryReader;
	basePath_ = basePath;
	scene_ = scene;
	rootNode_ = rootNode;
	sceneNodeManager_ = scene->getSceneNodeManager();
	lightManager_ = scene->getLightManager();
	modelManager_ = scene->getModelManager();
	meshManager_ = scene->getMeshManager();
	meshDataManager_ = scene->getMeshDataManager();
	materialManager_ = scene->getMaterialManager();
	textureManager_ = scene->getTextureManager();
	pictureManager_ = scene->getPictureManager();

	// [gm[h𖳌ɂĂ
	bool rootNodeEnabled = rootNode_->isEnabled();
	rootNode_->setEnabled(false);

	// wb_̓ǂݍ
	readHeader();

	// eubN̓ǂݍ
	while(true){
		if(reader_->isEnd()){ break; }
		String blockName = readID();
		int blockSize = reader_->readInt();
		// ubNJnɃACgƂ
		align();
		int blockAddress = reader_->getPosition();
		if(blockName == "Fog"){
			readFog();
		}else if(blockName == "SceneNode"){
			readSceneNodeList(blockSize);
		}else if(blockName == "Light"){
			readLightList(blockSize);
		}else if(blockName == "Model"){
			readModelList(blockSize);
		}else if(blockName == "Mesh"){
			readMeshList(blockSize);
		}else if(blockName == "MeshData"){
			readMeshDataList(blockSize);
		}else if(blockName == "Material"){
			readMaterialList(blockSize);
		}else if(blockName == "Texture"){
			readTextureList(blockSize);
		}else if(blockName == "Picture"){
			readPictureList(blockSize);
		}else if(blockName == "SceneNodeLin"){
			readSceneNodeLinkList(blockSize);
		}else if(blockName == "ModelLink"){
			readModelLinkList(blockSize);
		}else if(blockName == "MeshLink"){
			readMeshLinkList(blockSize);
		}else if(blockName == "MaterialLink"){
			readMaterialLinkList(blockSize);
		}else if(blockName == "TextureLink"){
			readTextureLinkList(blockSize);
		}else{
// ubNJnACg0Ȃ̂ővǁAƂ܂
// Writerł̓ACgÕTCYubNTCYɂȂĂ
			// mȂubNǂݔ΂
			DebugOut("BinarySceneLoader::load() "
				"Skip unknown block %s (%d)\n",
				blockName.getBytes(), reader_->getPosition());
			reader_->skip(blockSize);
		}
		// SẴubN̓ACgƂ
		align();
		Assert((reader_->getPosition() - blockAddress) == blockSize);
	}

	Assert(reader_->getPosition() == reader_->getSize());
	rootNode_->setEnabled(rootNodeEnabled);
}
//------------------------------------------------------------------------------
// wb_̓ǂݍ
void BinarySceneLoader::readHeader(){
	// tH[}bgID̓ǂݍ
	if(readID() != "LBScene"){
		ErrorOut("BinarySceneLoader::readHeader() "
			"Invalid formatID (%d)", reader_->getPosition());
	}
	// o[Wԍ̓ǂݍ
	u_int binaryVersion = 0x00000900;
	u_int version = reader_->readUInt();
	if(binaryVersion != version){
		ErrorOut("BinarySceneLoader::readHeader() "
			"Invalid version (%d)", reader_->getPosition());
	}
	align();
}
//------------------------------------------------------------------------------
// tHO
//------------------------------------------------------------------------------
// tHO̓ǂݍ
void BinarySceneLoader::readFog(){
	Fog* fog = scene_->getFog();
	// J[
	fog->setColor(readColor4c());
	// [h
	fog->setMode((Fog::Mode)reader_->readInt());
	// Zx
	fog->setDensity(reader_->readFloat());
	// jA
	fog->setNear(reader_->readFloat());
	// t@[
	fog->setFar(reader_->readFloat());
	// LA
	fog->setEnabled(reader_->readBool());
}
//------------------------------------------------------------------------------
// V[m[h
//------------------------------------------------------------------------------
// V[m[hXg̓ǂݍ
void BinarySceneLoader::readSceneNodeList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		String blockName = readID();
		int blockSize = reader_->readInt();
		// ubNJnɃACgƂ
		align();
		int blockAddress = reader_->getPosition();
		if(blockName == "Standard"){
			readSceneNode();
		}else if(blockName == "LOD"){
			readLODSceneNode();
		}else{
			// mȂubNǂݔ΂
			DebugOut("BinarySceneLoader::readSceneNodeList() "
				"Skip unknown sceneNode %s (%d)\n",
				blockName.getBytes(), reader_->getPosition());
			reader_->skip(blockSize);
		}
		// SẴubN̓ACgƂ
		align();
		Assert((reader_->getPosition() - blockAddress) == blockSize);
	}
}
//------------------------------------------------------------------------------
// V[m[h̓ǂݍ
void BinarySceneLoader::readSceneNode(){
	SceneNode* sceneNode = sceneNodeManager_->createSceneNode(readString());
	// XP[
	sceneNode->setScale(readVector3());
	// ]
	sceneNode->setRotationXYZ(readVector3());
	// ړ
	sceneNode->setTranslation(readVector3());
	// LAtO
	sceneNode->setEnabled(reader_->readBool());
}
//------------------------------------------------------------------------------
// xIufBeB[V[m[h̓ǂݍ
void BinarySceneLoader::readLODSceneNode(){
	LODSceneNode* sceneNode =
		sceneNodeManager_->createLODSceneNode(readString());
	// XP[
	sceneNode->setScale(readVector3());
	// ]
	sceneNode->setRotationXYZ(readVector3());
	// ړ
	sceneNode->setTranslation(readVector3());
	// LOD
	int lodThresholdCount = reader_->readInt();
	sceneNode->setLODThresholdCount(lodThresholdCount);
	// LODl
	for(int i = 0; i < lodThresholdCount; i++){
		sceneNode->setLODThreshold(i, reader_->readFloat());
	}
	// LAtO
	sceneNode->setEnabled(reader_->readBool());
}
//------------------------------------------------------------------------------
// Cg
//------------------------------------------------------------------------------
// CgXg̓ǂݍ
void BinarySceneLoader::readLightList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		String blockName = readID();
		int blockSize = reader_->readInt();
		// ubNJnɃACgƂ
		align();
		int blockAddress = reader_->getPosition();
		if(blockName == "Ambient"){
			readAmbientLight();
		}else if(blockName == "Directional"){
			readDirectionalLight();
		}else if(blockName == "Point"){
			readPointLight();
		}else{
			// mȂubNǂݔ΂
			DebugOut("BinarySceneLoader::readLightList() "
				"Skip unknown light %s (%d)\n",
				blockName.getBytes(), reader_->getPosition());
			reader_->skip(blockSize);
		}
		// SẴubN̓ACgƂ
		align();
		Assert((reader_->getPosition() - blockAddress) == blockSize);
	}
}
//------------------------------------------------------------------------------
// Cg̓ǂݍ
void BinarySceneLoader::readLight(Light* light){
	// Cg}XN
	light->setLightMask(reader_->readUInt());
	// LAtO
	light->setEnabled(reader_->readBool());
}
//------------------------------------------------------------------------------
// ArGgCg̓ǂݍ
void BinarySceneLoader::readAmbientLight(){
	AmbientLight* light = lightManager_->createAmbientLight(readString());
	// CgJ[
	light->setColor(readColor3f());
	// Cg̓ǂݍ
	readLight(light);
}
//------------------------------------------------------------------------------
// fBNViCg̓ǂݍ
void BinarySceneLoader::readDirectionalLight(){
	DirectionalLight* light =
		lightManager_->createDirectionalLight(readString());
	// fBt[YJ[
	light->setDiffuseColor(readColor3f());
	// XyLJ[
	light->setSpecularColor(readColor3f());
	// 
	light->setDirection(readVector3());
	// Cg̓ǂݍ
	readLight(light);
}
//------------------------------------------------------------------------------
// |CgCg̓ǂݍ
void BinarySceneLoader::readPointLight(){
	PointLight* light = lightManager_->createPointLight(readString());
	// fBt[YJ[
	light->setDiffuseColor(readColor3f());
	// XyLJ[
	light->setSpecularColor(readColor3f());
	// ʒu
	light->setPosition(readVector3());
	// W
	light->setRange(reader_->readFloat());
	// W
	float attenuation0 = reader_->readFloat();
	float attenuation1 = reader_->readFloat();
	float attenuation2 = reader_->readFloat();
	light->setAttenuation(attenuation0, attenuation1, attenuation2);
	// Cg̓ǂݍ
	readLight(light);
}
//------------------------------------------------------------------------------
// f
//------------------------------------------------------------------------------
// fXg̓ǂݍ
void BinarySceneLoader::readModelList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		String blockName = readID();
		int blockSize = reader_->readInt();
		// ubNJnɃACgƂ
		align();
		int blockAddress = reader_->getPosition();
		if(blockName == "Standard"){
			readStandardModel();
		}else if(blockName == "Character"){
			readCharacterModel();
		}else{
			// mȂubNǂݔ΂
			DebugOut("BinarySceneLoader::readModelList() "
				"Skip unknown model %s (%d)\n",
				blockName.getBytes(), reader_->getPosition());
			reader_->skip(blockSize);
		}
		// SẴubN̓ACgƂ
		align();
		Assert((reader_->getPosition() - blockAddress) == blockSize);
	}
}
//------------------------------------------------------------------------------
// Wf̓ǂݍ
void BinarySceneLoader::readStandardModel(){
	StandardModel* model = modelManager_->createStandardModel(readString());
	// LAtO
	model->setEnabled(reader_->readBool());
}
//------------------------------------------------------------------------------
// LN^f̓ǂݍ
void BinarySceneLoader::readCharacterModel(){
	CharacterModel* model = modelManager_->createCharacterModel(readString());
	// LAtO
	model->setEnabled(reader_->readBool());
	// ACg
	align();
	// {[
	int boneCount = reader_->readInt();
	// {[̓ǂݍ
	for(int i = 0; i < boneCount; i++){ readBone(model); }
	// {[N
	for(int i = 0; i < boneCount; i++){
		Bone* parent = model->getBone(i);
		int childCount = reader_->readInt();
		for(int j = 0; j < childCount; j++){
			parent->addBone(model->getBone(reader_->readInt()));
		}
	}
}
//------------------------------------------------------------------------------
// {[̓ǂݍ
void BinarySceneLoader::readBone(CharacterModel* model){
	Bone* bone = model->createBone(readString());
	// |[Yts
	bone->setInversePoseMatrix(readMatrix34());
	// XP[
	bone->setScale(readVector3());
	// ]
	bone->setRotationXYZ(readVector3());
	// ړ
	bone->setTranslation(readVector3());
}
//------------------------------------------------------------------------------
// bV
//------------------------------------------------------------------------------
// bVXg̓ǂݍ
void BinarySceneLoader::readMeshList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		String blockName = readID();
		int blockSize = reader_->readInt();
		// ubNJnɃACgƂ
		align();
		int blockAddress = reader_->getPosition();
		if(blockName == "Rigid"){
			readRigidMesh();
		}else if(blockName == "Character"){
			readCharacterMesh();
		}else{
			// mȂubNǂݔ΂
			DebugOut("BinarySceneLoader::readMeshList() "
				"Skip unknown mesh %s (%d)\n",
				blockName.getBytes(), reader_->getPosition());
			reader_->skip(blockSize);
		}
		// SẴubN̓ACgƂ
		align();
		Assert((reader_->getPosition() - blockAddress) == blockSize);
	}
}
//------------------------------------------------------------------------------
// ̃bV̓ǂݍ
void BinarySceneLoader::readRigidMesh(){
	RigidMesh* mesh = meshManager_->createRigidMesh(readString());
	// LAtO
	mesh->setEnabled(reader_->readBool());
}
//------------------------------------------------------------------------------
// LN^bV̓ǂݍ
void BinarySceneLoader::readCharacterMesh(){
	CharacterMesh* mesh = meshManager_->createCharacterMesh(readString());
	// LAtO
	mesh->setEnabled(reader_->readBool());
}
//------------------------------------------------------------------------------
// bVf[^
//------------------------------------------------------------------------------
// bVf[^Xg̓ǂݍ
void BinarySceneLoader::readMeshDataList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		readMeshData();
	}
}
//------------------------------------------------------------------------------
// bVf[^̓ǂݍ
void BinarySceneLoader::readMeshData(){
	MeshData* meshData = meshDataManager_->createMeshData(readString());
	// oEfBO{bNX
	meshData->setBoundingBox(readAxisAlignedBox());
	// oEfBOXtBA
	meshData->setBoundingSphere(readSphere());
	// v~eBu^Cv
	meshData->setPrimitiveType((Mesh::PrimitiveType)reader_->readInt());
	// _CfbNX
	int vertexIndexCount = reader_->readInt();
	if(vertexIndexCount > 0){
		meshData->setVertexIndexCount(vertexIndexCount);
		for(int i = 0; i < vertexIndexCount; i++){
			meshData->setVertexIndex(i, reader_->readUShort());
		}
	}
	// @Ă邩
	bool hasNormal = reader_->readBool();
	meshData->enableNormal(hasNormal);
	// _J[Ă邩
	bool hasColor = reader_->readBool();
	meshData->enableColor(hasColor);
	// ACg
	align();
	// _
	int vertexCount = reader_->readInt();
	meshData->setVertexCount(vertexCount);
	// eNX`WZbg
	int texCoordSetCount = reader_->readInt();
	meshData->setTexCoordSetCount(texCoordSetCount);
	// eNX`W^Cv
	int texCoordType[TexCoord::maxSetCount];
	for(int i = 0; i < texCoordSetCount; i++){
		texCoordType[i] = reader_->readInt();
		meshData->setTexCoordType(i, (TexCoord::Type)texCoordType[i]);
	}
	// ʒu
	for(int i = 0; i < vertexCount; i++){
		meshData->setPosition(i, readVector3());
	}
	// @
	if(hasNormal){
		for(int i = 0; i < vertexCount; i++){
			meshData->setNormal(i, readVector3());
		}
	}
	// _J[
	if(hasColor){
		for(int i = 0; i < vertexCount; i++){
			meshData->setColor(i, readColor4c());
		}
	}
	// eNX`W
	float texCoord[4];
	for(int i = 0; i < texCoordSetCount; i++){
		int texCoordSize = sizeof(float) * texCoordType[i];
		for(int j = 0; j < vertexCount; j++){
			reader_->readBytes(texCoord, texCoordSize);
			meshData->setTexCoord(j, i, texCoord, texCoordType[i]);
		}
	}
	// _{[
	int bonesPerVertex = reader_->readInt();
	meshData->setBonesPerVertex(bonesPerVertex);
	if(bonesPerVertex > 0){
		for(int i = 0; i < vertexCount; i++){
			for(int j = 0; j < bonesPerVertex; j++){
				meshData->setBoneIndex(i, j, reader_->readUChar());
			}
		}
	}
	// _EFCg
	int weightsPerVertex = meshData->getWeightsPerVertex();
	if(weightsPerVertex > 0){
		for(int i = 0; i < vertexCount; i++){
			for(int j = 0; j < weightsPerVertex; j++){
				meshData->setWeight(i, j, reader_->readFloat());
			}
		}
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// }eA
//------------------------------------------------------------------------------
// }eAXg̓ǂݍ
void BinarySceneLoader::readMaterialList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		String blockName = readID();
		int blockSize = reader_->readInt();
		// ubNJnɃACgƂ
		align();
		int blockAddress = reader_->getPosition();
		if(blockName == "Basic"){
			readBasicMaterial();
		}else{
			// mȂubNǂݔ΂
			DebugOut("BinarySceneLoader::readMaterialList() "
				"Skip unknown material %s (%d)\n",
				blockName.getBytes(), reader_->getPosition());
			reader_->skip(blockSize);
		}
		// SẴubN̓ACgƂ
		align();
		Assert((reader_->getPosition() - blockAddress) == blockSize);
	}
}
//------------------------------------------------------------------------------
// }eA̓ǂݍ
void BinarySceneLoader::readMaterial(Material* material){
	// Dx
	material->setPriority(reader_->readInt());
	// uh[h
	material->setBlendMode((Material::BlendMode)reader_->readInt());
	// At@l
	material->setAlpha(reader_->readFloat());
	// uh\[X
	material->setBlendSource((Material::BlendState)reader_->readInt());
	// uhfXeBl[V
	material->setBlendDestination((Material::BlendState)reader_->readInt());
	// tHOIvV
	material->setFogOption((Material::FogOption)reader_->readInt());
	// Cg}XN
	material->setLightMask(reader_->readUInt());
	// Z
	material->setZWrite(reader_->readBool());
	// ZeXg
	material->setZTest(reader_->readBool());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// {}eA̓ǂݍ
void BinarySceneLoader::readBasicMaterial(){
	BasicMaterial* material =
		materialManager_->createBasicMaterial(readString());
	// }eAǂݍ
	readMaterial(material);
	// x[XUVCfbNX
	material->setBaseUVIndex(reader_->readInt());
	// UVCfbNX
	material->setGlossUVIndex(reader_->readInt());
	// CgUVCfbNX
	material->setLightUVIndex(reader_->readInt());
	// UVCfbNX
	material->setStainUVIndex(reader_->readInt());
	// fBt[YJ[
	material->setDiffuseColor(readColor3f());
	// XyLJ[
	material->setSpecularColor(readColor3f());
	// XyLp[
	material->setSpecularPower(reader_->readFloat());
	// ArGgJ[
	material->setAmbientColor(readColor3f());
	// G~bVuJ[
	material->setEmissiveColor(readColor3f());
}
//------------------------------------------------------------------------------
// eNX`
//------------------------------------------------------------------------------
// eNX`Xg̓ǂݍ
void BinarySceneLoader::readTextureList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		String blockName = readID();
		int blockSize = reader_->readInt();
		// ubNJnɃACgƂ
		align();
		int blockAddress = reader_->getPosition();
		if(blockName == "Surface"){
			readSurfaceTexture();
		}else{
			// mȂubNǂݔ΂
			DebugOut("BinarySceneLoader::readTextureList() "
				"Skip unknown texture %s (%d)\n",
				blockName.getBytes(), reader_->getPosition());
			reader_->skip(blockSize);
		}
		// SẴubN̓ACgƂ
		align();
		Assert((reader_->getPosition() - blockAddress) == blockSize);
	}
}
//------------------------------------------------------------------------------
// T[tF[XeNX`̓ǂݍ
void BinarySceneLoader::readSurfaceTexture(){
	SurfaceTexture* texture =
		textureManager_->createSurfaceTexture(readString());
	// AhX[hU
	texture->setAddressModeU((Texture::AddressMode)reader_->readInt());
	// AhX[hV
	texture->setAddressModeV((Texture::AddressMode)reader_->readInt());
	// s[gUV
	texture->setRepeatUV(readTexCoord2());
	// ItZbgUV
	texture->setOffsetUV(readTexCoord2());
}
//------------------------------------------------------------------------------
// sN`
//------------------------------------------------------------------------------
// sN`Xg̓ǂݍ
void BinarySceneLoader::readPictureList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		String name = readString();
		String path = readString();
		// gq擾
		FilePath filePath(path);
		String extension = filePath.getExtension();
		// gqɂ킹ăsN`ǂݕ
		Picture* picture;
		if(extension == "tga"){
			TargaLoader loader(basePath_ + path);
			loader.loadHeader();
			// At@̗L
			if(loader.hasAlpha()){
				picture = readPictureRGBA8(name, loader);
			}else{
				picture = readPictureRGB8(name, loader);
			}
		}else{
			ErrorOut("BinarySceneLoader::readPictureList() "
				"Unsupported picture format %s (%d)",
				path.getBytes(), reader_->getPosition());
			return;
		}
		// pXݒ肷
		picture->setPath(path);
	}
}
//------------------------------------------------------------------------------
// RGB8rbgsN`̓ǂݍ
Picture* BinarySceneLoader::readPictureRGB8(
	const String& name, TargaLoader& loader){
	PictureRGB8* picture = pictureManager_->createPictureRGB8(name);
	picture->setSize(loader.getSize());
	loader.loadImage(picture->getImageBuffer());
	return picture;
}
//------------------------------------------------------------------------------
// RGBA8rbgsN`̓ǂݍ
Picture* BinarySceneLoader::readPictureRGBA8(
	const String& name, TargaLoader& loader){
	PictureRGBA8* picture = pictureManager_->createPictureRGBA8(name);
	picture->setSize(loader.getSize());
	loader.loadImage(picture->getImageBuffer());
	return picture;
}
//------------------------------------------------------------------------------
// V[m[hN
//------------------------------------------------------------------------------
// V[m[hNXg̓ǂݍ
void BinarySceneLoader::readSceneNodeLinkList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		readSceneNodeLink();
	}
}
//------------------------------------------------------------------------------
// V[m[hN̓ǂݍ
void BinarySceneLoader::readSceneNodeLink(){
	String name = readString();
	SceneNode* sceneNode;
	if(name.equals("RootNode")){
		sceneNode = rootNode_;
	}else{
		sceneNode = sceneNodeManager_->search(name);
		if(sceneNode == NULL){
			ErrorOut("BinarySceneLoader::readSceneNodeLink() "
				"not found source sceneNode %s (%d)",
				name.getBytes(), reader_->getPosition());
		}
	}
	// V[m[hN
	int sceneNodeCount = reader_->readInt();
	for(int i = 0; i < sceneNodeCount; i++){
		String targetName = readString();
		SceneNode* target = sceneNodeManager_->search(targetName);
		if(target == NULL){
			ErrorOut("BinarySceneLoader::readSceneNodeLink() "
				"not found target sceneNode %s (%d)",
				targetName.getBytes(), reader_->getPosition());
		}
		sceneNode->addSceneNode(target);
	}
	// V[[tN
	int sceneLeafCount = reader_->readInt();
	for(int i = 0; i < sceneLeafCount; i++){
		int type = reader_->readInt();
		String targetName = readString();
		SceneLeaf* sceneLeaf = NULL;
		if(type == 0){
			sceneLeaf = modelManager_->search(targetName);
		}else if(type == 1){
			sceneLeaf = lightManager_->search(targetName);
		}else{
			ErrorOut("BinarySceneLoader::readSceneNodeLink() "
				"unsupported SceneLeafType %s %d (%d)",
				targetName.getBytes(), type, reader_->getPosition());
		}
		if(sceneLeaf == NULL){
			ErrorOut("BinarySceneLoader::readSceneNodeLink() "
				"not found target SceneLeaf %s (%d)",
				targetName.getBytes(), reader_->getPosition());
		}
		sceneNode->addSceneLeaf(sceneLeaf);
	}
}
//------------------------------------------------------------------------------
// fN
//------------------------------------------------------------------------------
// fNXg̓ǂݍ
void BinarySceneLoader::readModelLinkList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		readModelLink();
	}
}
//------------------------------------------------------------------------------
// fN̓ǂݍ
void BinarySceneLoader::readModelLink(){
	String name = readString();
	Model* model = modelManager_->search(name);
	if(model == NULL){
		ErrorOut("BinarySceneLoader::readModelLink() "
			"not found source model %s (%d)",
			name.getBytes(), reader_->getPosition());
	}
	// bVN
	int meshCount = reader_->readInt();
	for(int i = 0; i < meshCount; i++){
		String targetName = readString();
		Mesh* target = meshManager_->search(targetName);
		if(target == NULL){
			ErrorOut("BinarySceneLoader::readModelLink() "
				"not found target mesh %s (%d)",
				targetName.getBytes(), reader_->getPosition());
		}
		model->addMesh(target);
	}
}
//------------------------------------------------------------------------------
// bVN
//------------------------------------------------------------------------------
// bVNXg̓ǂݍ
void BinarySceneLoader::readMeshLinkList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		readMeshLink();
	}
}
//------------------------------------------------------------------------------
// bVN̓ǂݍ
void BinarySceneLoader::readMeshLink(){
	String name = readString();
	Mesh* mesh = meshManager_->search(name);
	if(mesh == NULL){
		ErrorOut("BinarySceneLoader::readMeshLink() "
			"not found source mesh %s (%d)",
			name.getBytes(), reader_->getPosition());
	}
	// bVf[^
	bool hasMeshData = reader_->readBool();
	// }eA
	bool hasMaterial = reader_->readBool();
	Assert(hasMeshData || hasMaterial);
	// ACg
	align();
	// bVf[^N
	if(hasMeshData){
		String targetName = readString();
		MeshData* target = meshDataManager_->search(targetName);
		if(target == NULL){
			ErrorOut("BinarySceneLoader::readMeshLink() "
				"not found target meshData %s (%d)",
				targetName.getBytes(), reader_->getPosition());
		}
		mesh->setMeshData(target);
	}
	// }eAN
	if(hasMaterial){
		String targetName = readString();
		Material* target = materialManager_->search(targetName);
		if(target == NULL){
			ErrorOut("BinarySceneLoader::readMeshLink() "
				"not found target material %s (%d)",
				targetName.getBytes(), reader_->getPosition());
		}
		mesh->setMaterial(target);
	}
}
//------------------------------------------------------------------------------
// }eAN
//------------------------------------------------------------------------------
// }eANXg̓ǂݍ
void BinarySceneLoader::readMaterialLinkList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		String blockName = readID();
		int blockSize = reader_->readInt();
		// ubNJnɃACgƂ
		align();
		int blockAddress = reader_->getPosition();
		if(blockName == "Basic"){
			readBasicMaterialLink();
		}else{
			// mȂubNǂݔ΂
			DebugOut("BinarySceneLoader::readMaterialList() "
				"Skip unknown material %s (%d)\n",
				blockName.getBytes(), reader_->getPosition());
			reader_->skip(blockSize);
		}
		// SẴubN̓ACgƂ
		align();
		Assert((reader_->getPosition() - blockAddress) == blockSize);
	}

}
//------------------------------------------------------------------------------
// {}eAN̓ǂݍ
void BinarySceneLoader::readBasicMaterialLink(){
	String name = readString();
	Material* material = materialManager_->search(name);
	if(material == NULL){
		ErrorOut("BinarySceneLoader::readBasicMaterialLink() "
			"Not found material %s (%d)",
			name.getBytes(), reader_->getPosition());
	}
	BasicMaterial* basicMaterial = material->castBasicMaterial();
	if(basicMaterial == NULL){
		ErrorOut("BinarySceneLoader::readBasicMaterialLink() "
			"Invalid type material %s (%d)",
			name.getBytes(), reader_->getPosition());
	}
	bool hasBaseTexture = reader_->readBool();
	bool hasGlossTexture = reader_->readBool();
	bool hasLightTexture = reader_->readBool();
	bool hasStainTexture = reader_->readBool();
	align();
	if(hasBaseTexture){
		String textureName = readString();
		Texture* texture = textureManager_->search(textureName);
		if(texture == NULL){
			ErrorOut("BinarySceneLoader::readBasicMaterialLink() "
				"Not found texture %s (%d)",
				textureName.getBytes(), reader_->getPosition());
		}
		basicMaterial->setBaseTexture(texture);
	}
	if(hasGlossTexture){
		String textureName = readString();
		Texture* texture = textureManager_->search(textureName);
		if(texture == NULL){
			ErrorOut("BinarySceneLoader::readBasicMaterialLink() "
				"Not found texture %s (%d)",
				textureName.getBytes(), reader_->getPosition());
		}
		basicMaterial->setGlossTexture(texture);
	}
	if(hasLightTexture){
		String textureName = readString();
		Texture* texture = textureManager_->search(textureName);
		if(texture == NULL){
			ErrorOut("BinarySceneLoader::readBasicMaterialLink() "
				"Not found texture %s (%d)",
				textureName.getBytes(), reader_->getPosition());
		}
		basicMaterial->setLightTexture(texture);
	}
	if(hasStainTexture){
		String textureName = readString();
		Texture* texture = textureManager_->search(textureName);
		if(texture == NULL){
			ErrorOut("BinarySceneLoader::readBasicMaterialLink() "
				"Not found texture %s (%d)",
				textureName.getBytes(), reader_->getPosition());
		}
		basicMaterial->setStainTexture(texture);
	}
}
//------------------------------------------------------------------------------
// eNX`N
//------------------------------------------------------------------------------
// eNX`NXg̓ǂݍ
void BinarySceneLoader::readTextureLinkList(int listSize){
	int endAddress = reader_->getPosition() + listSize;
	while(true){
		if(endAddress == reader_->getPosition()){ break; }
		Assert(!reader_->isEnd());
		readTextureLink();
	}
}
//------------------------------------------------------------------------------
// eNX`N̓ǂݍ
void BinarySceneLoader::readTextureLink(){
	String name = readString();
	Texture* texture = textureManager_->search(name);
	if(texture == NULL){
		ErrorOut("BinarySceneLoader::readTextureLink() "
			"not found source texture %s (%d)",
			name.getBytes(), reader_->getPosition());
	}
	// sN`N
	int pictureCount = reader_->readInt();
	for(int i = 0; i < pictureCount; i++){
		String targetName = readString();
		Picture* target = pictureManager_->search(targetName);
		if(target == NULL){
			ErrorOut("BinarySceneLoader::readTextureLink() "
				"not found target picture %s (%d)",
				targetName.getBytes(), reader_->getPosition());
		}
		texture->addPicture(target);
	}
}
//------------------------------------------------------------------------------
// l̓ǂݍ
//------------------------------------------------------------------------------
// ̓ǂݍ
String BinarySceneLoader::readString(){
	String result;
	int size = reader_->readInt();
	if(size > stringBufferSize_){
		char* allocateBuffer = new char[size];
		reader_->readBytes(allocateBuffer, size);
		result = allocateBuffer;
		delete allocateBuffer;
	}else{
		reader_->readBytes(stringBuffer_, size);
		result = stringBuffer_;
	}
	align();
	return result;
}
//------------------------------------------------------------------------------
// OxNg̓ǂݍ
Vector3 BinarySceneLoader::readVector3(){
	Vector3 result;
	reader_->readBytes(result.array, sizeof(Vector3));
	return result;
}
//------------------------------------------------------------------------------
// Matrix34l̓ǂݍ
Matrix34 BinarySceneLoader::readMatrix34(){
	Matrix34 result;
	reader_->readBytes(result.array, sizeof(Matrix34));
	return result;
}
//------------------------------------------------------------------------------
// OvfJ[l̓ǂݍ
Color3c BinarySceneLoader::readColor3c(){
	Color3c result;
	reader_->readBytes(result.array, sizeof(Color3c));
	return result;
}
//------------------------------------------------------------------------------
// lvfJ[l̓ǂݍ
Color4c BinarySceneLoader::readColor4c(){
	Color4c result;
	reader_->readBytes(result.array, sizeof(Color4c));
	return result;
}
//------------------------------------------------------------------------------
// OvfJ[l̓ǂݍ
Color3f BinarySceneLoader::readColor3f(){
	Color3f result;
	reader_->readBytes(result.array, sizeof(Color3f));
	return result;
}
//------------------------------------------------------------------------------
// lvfJ[l̓ǂݍ
Color4f BinarySceneLoader::readColor4f(){
	Color4f result;
	reader_->readBytes(result.array, sizeof(Color4f));
	return result;
}
//------------------------------------------------------------------------------
// 񎟌eNX`Wl̓ǂݍ
TexCoord2 BinarySceneLoader::readTexCoord2(){
	TexCoord2 result;
	reader_->readBytes(result.array, sizeof(TexCoord2));
	return result;
}
//------------------------------------------------------------------------------
// {bNX̓ǂݍ
AxisAlignedBox BinarySceneLoader::readAxisAlignedBox(){
	AxisAlignedBox result;
	reader_->readBytes(&result, sizeof(AxisAlignedBox));
	return result;
}
//------------------------------------------------------------------------------
// ̓ǂݍ
Sphere BinarySceneLoader::readSphere(){
	Sphere result;
	reader_->readBytes(&result, sizeof(Sphere));
	return result;
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// ACg
void BinarySceneLoader::align(){
	reader_->align(16);
}
//------------------------------------------------------------------------------
// ID̓ǂݍ
String BinarySceneLoader::readID(){
	char buffer[13];
	buffer[12] = 0;
	reader_->readBytes(buffer, 12);
	return String(buffer);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
