//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * ό`bVWIg
 * @author Junpee
 */

#include "LampBasic.h"
#include "Geometry/Mesh/DeformedMeshGeometry.h"
#include "Geometry/System/IntersectionResult.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
DeformedMeshGeometry::DeformedMeshGeometry() : MeshGeometry(),
	boundingBox_(AxisAlignedBox::zero), boundingSphere_(Sphere::zero),
	triangles_(NULL), triangleCount_(0){
}
//------------------------------------------------------------------------------
// Rs[RXgN^
DeformedMeshGeometry::DeformedMeshGeometry(const DeformedMeshGeometry& copy) :
	MeshGeometry(copy), boundingBox_(AxisAlignedBox::zero),
	boundingSphere_(Sphere::zero), triangles_(NULL), triangleCount_(0){
	copyDeformedMeshGeometryData(copy);
}
//------------------------------------------------------------------------------
// Rs[
const DeformedMeshGeometry& DeformedMeshGeometry::operator =(
	const DeformedMeshGeometry& copy){
	// gȂ烊^[
	if(this == &copy){ return *this; }
	MeshGeometry::operator =(copy);
	copyDeformedMeshGeometryData(copy);
	return *this;
}
//------------------------------------------------------------------------------
// ό`bVWIgf[^̃Rs[
void DeformedMeshGeometry::copyDeformedMeshGeometryData(
	const DeformedMeshGeometry& copy){
	int triangleCount = copy.getTriangleCount();
	setTriangleCount(triangleCount);
	for(int i = 0; i < triangleCount; i++){
		setTriangle(i, copy.getTriangle(i));
	}
	setBoundingBox(copy.getBoundingBox());
	setBoundingSphere(copy.getBoundingSphere());
}
//------------------------------------------------------------------------------
// fXgN^
DeformedMeshGeometry::~DeformedMeshGeometry(){
	SafeArrayDelete(triangles_);
}
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
bool DeformedMeshGeometry::intersect(const Sphere& sphere) const{
	// oEfBO
	if(!intersectBounding(sphere)){ return false; }
	// bV
	return intersectMesh(sphere);
}
//------------------------------------------------------------------------------
// 
void DeformedMeshGeometry::intersect(
	IntersectionResult* result, const Sphere& sphere) const{
	// oEfBO
	if(!intersectBounding(sphere)){ return; }
	// bV
	intersectMesh(result, sphere);
}
//------------------------------------------------------------------------------
// oEfBO
bool DeformedMeshGeometry::intersectBounding(const Sphere& sphere) const{
	// oEfBOXtBA
	if(!boundingSphere_.intersect(sphere)){ return false; }
	// oEfBO{bNX
	if(!boundingBox_.intersect(sphere)){ return false; }
	return true;
}
//------------------------------------------------------------------------------
// bV
bool DeformedMeshGeometry::intersectMesh(const Sphere& sphere) const{
	for(int i = 0; i < triangleCount_; i++){
		if(triangles_[i].intersect(sphere)){ return true; }
	}
	return false;
}
//------------------------------------------------------------------------------
// bV
void DeformedMeshGeometry::intersectMesh(
	IntersectionResult* result, const Sphere& sphere) const{
	Intersection intersection;
	for(int i = 0; i < triangleCount_; i++){
		if(triangles_[i].intersect(&intersection, sphere)){
			result->add(intersection);
		}
	}
}
//------------------------------------------------------------------------------
// gCAO
//------------------------------------------------------------------------------
// gCAO̐ݒ
void DeformedMeshGeometry::setTriangleCount(int triangleCount){
	Assert(triangleCount >= 0);
	SafeArrayDelete(triangles_);
	triangleCount_ = triangleCount;
	if(triangleCount_ == 0){ return; }
	triangles_ = new Triangle[triangleCount_];
}
//------------------------------------------------------------------------------
// oEfBO
//------------------------------------------------------------------------------
// oEfBO̎Zo
void DeformedMeshGeometry::calculateBounding(){
	if(triangleCount_ == 0){ return; }

	// oEfBO{bNX
	const Vector3& initializePosition = triangles_[0].getVertex(0);
	boundingBox_.set(initializePosition, initializePosition);
	for(int i = 0; i < triangleCount_; i++){
		const Triangle& triangle = triangles_[i];
		for(int j = 0; j < 3; j++){
			boundingBox_.merge(triangle.getVertex(j));
		}
	}

	// oEfBOXtBA
	boundingSphere_.set(boundingBox_.getCenter(), 0.f);
	for(int i = 0; i < triangleCount_; i++){
		const Triangle& triangle = triangles_[i];
		for(int j = 0; j < 3; j++){
			boundingSphere_.append(triangle.getVertex(j));
		}
	}
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
