//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * 
 * @author Junpee
 */

#include "LampBasic.h"
#include "Core/Primitive/String.h"
#include "Core/System/StringMethod.h"

namespace Lamp{

//------------------------------------------------------------------------------
// [eBeB֐
//------------------------------------------------------------------------------
/**
 * V̍쐬
 * @param length 
 * @return 
 */
inline static char* newString(int length){
	char* returnString = new char[sizeof(int) + length + 1];
	// t@XJE^1ݒ
	*(int*)returnString = 1;
	returnString += sizeof(int);
	return returnString;
}
//------------------------------------------------------------------------------
/**
 * ̉
 * @param string 镶
 * @return t@XJE^
 */
inline static int releaseString(char** string){
	if((*string) != NULL){
		// t@XJE^fNg
		int* refCount = (int*)((*string) - sizeof(int));
		(*refCount) -= 1;
		int returnCount = *refCount;
		if(returnCount == 0){ delete[] refCount; }
		(*string) = NULL;
		return returnCount;
	}
	return 0;
}
//------------------------------------------------------------------------------
/**
 * t@XJE^̒ǉ
 * @param string t@Xǉ镶
 * @return t@XJE^
 */
inline static int addReferenceCounter(char* string){
	// t@XJE^CNg
	int* counterAddress = (int*)(string - sizeof(int));
	*counterAddress += 1;
	return *counterAddress;
}
//------------------------------------------------------------------------------
/**
 * t@XJE^̎擾
 * @param string t@XJE^擾镶
 * @return t@XJE^
 */
inline static int getReferenceCounter(char* string){
	return *(int*)(string - sizeof(int));
}
//------------------------------------------------------------------------------
// ̍\zAj
//------------------------------------------------------------------------------
// RXgN^
String::String() : string_(NULL), size_(0){
}
//------------------------------------------------------------------------------
// RXgN^
String::String(const char* initString){
	if(initString == NULL){
		string_ = NULL;
		size_ = 0;
		return;
	}
	size_ = (int)StdStrlen(initString);
	Assert(size_ >= 0);
	if(size_ == 0){
		string_ = NULL;
		return;
	}
	string_ = newString(size_);
	StdStrcpy(string_, initString);
}
//------------------------------------------------------------------------------
// Rs[RXgN^
String::String(const String& copy){
	size_ = copy.getSize();
	Assert(size_ >= 0);
	if(size_ == 0){
		string_ = NULL;
		return;
	}
	string_ = copy.string_;
	addReferenceCounter(string_);
}
//------------------------------------------------------------------------------
// gpRXgN^
String::String(char* initString, int initLength){
	Assert(initString != NULL);
	Assert(initLength >= 0);
	string_ = initString;
	size_ = initLength;
}
//------------------------------------------------------------------------------
// Zq
String& String::operator =(const String& copy){
	// gȂ烊^[
	if(this == &copy){ return *this; }
	releaseString(&string_);
	size_ = copy.getSize();
	Assert(size_ >= 0);
	if(size_ == 0){
		string_ = NULL;
		return *this;
	}
	string_ = copy.string_;
	addReferenceCounter(string_);
	return *this;
}
//------------------------------------------------------------------------------
// Zq
String& String::operator =(const char* copy){
	releaseString(&string_);
	if(copy == NULL){
		string_ = NULL;
		size_ = 0;
		return *this;
	}
	size_ = (int)StdStrlen(copy);
	Assert(size_ >= 0);
	if(size_ == 0){
		string_ = NULL;
		return *this;
	}
	string_ = newString(size_);
	StdStrcpy(string_, copy);
	return *this;
}
//------------------------------------------------------------------------------
// fXgN^
String::~String(){
	releaseString(&string_);
}
//------------------------------------------------------------------------------
// ̏擾
//------------------------------------------------------------------------------
// Ԃ
int String::getCharacterCount() const{
	return (int)StdMbslen(getBytes());
}
//------------------------------------------------------------------------------
// ̎擾
String String::getSubstring(int beginIndex) const{
	if(beginIndex == 0){ return *this; }
	Assert(beginIndex >= 0);
	Assert((beginIndex <= size_));
	// 󕶎ɂȂ
	if(beginIndex == size_){ return String(); }
	int newLength = size_ - beginIndex;
	char* newArray = newString(newLength);
	StdStrcpy(newArray, string_ + beginIndex);
	return String(newArray, newLength);
}
//------------------------------------------------------------------------------
// ̎擾
String String::getSubstring(int beginIndex, int endIndex) const{
	Assert(beginIndex >= 0);
	Assert(endIndex >= 0);
	Assert((endIndex <= size_));
	Assert((beginIndex <= endIndex));
	// 󕶎ɂȂ
	if(beginIndex == endIndex){ return String(); }
	int newLength = endIndex - beginIndex;
	char* newArray = newString(newLength);
	for(int i = 0; i < newLength; i++){
		newArray[i] = string_[i + beginIndex];
	}
	newArray[newLength] = '\0';
	return String(newArray, newLength);
}
//------------------------------------------------------------------------------
// 啶̎擾
String String::getUpperCase() const{
	// Rs[쐬Kv
	String returnString(this->getBytes());
	if(returnString.getSize() == 0){ return returnString; }
	StdStrupr(returnString.string_);
	return returnString;
}
//------------------------------------------------------------------------------
// ̎擾
String String::getLowerCase() const{
	// Rs[쐬Kv
	String returnString(this->getBytes());
	if(returnString.getSize() == 0){ return returnString; }
	StdStrlwr(returnString.string_);
	return returnString;
}
//------------------------------------------------------------------------------
// nbVR[h̎擾
u_int String::getHashCode() const{
	u_int hashCode = 0;
	if(size_ == 0){ return hashCode; }
	const char* position = getBytes();
	for(int i = 0;i < size_;i++){
		hashCode = hashCode * 31 + position[i];
	}
	return hashCode;
}
//------------------------------------------------------------------------------
// p[X
//------------------------------------------------------------------------------
// charւ̃p[X
bool String::parseChar(char* value) const{
	Assert(value != NULL);
	int scanValue;
	int scanCount = StdSscanf(string_, "%d", &scanValue);
	if((scanCount == 1) &&
		(scanValue <= Limit::charMax) && (scanValue >= Limit::charMin)){
		*value = (char)scanValue;
		return true;
	}
	return false;
}
//------------------------------------------------------------------------------
// u_charւ̃p[X
bool String::parseUChar(u_char* value) const{
	Assert(value != NULL);
	int scanValue;
	int scanCount = StdSscanf(string_, "%d", &scanValue);
	if((scanCount == 1) &&
		(scanValue <= Limit::uCharMax) && (scanValue >= Limit::uCharMin)){
		*value = (u_char)scanValue;
		return true;
	}
	return false;
}
//------------------------------------------------------------------------------
// shortւ̃p[X
bool String::parseShort(short* value) const{
	Assert(value != NULL);
	int scanValue;
	int scanCount = StdSscanf(string_, "%d", &scanValue);
	if((scanCount == 1) &&
		(scanValue <= Limit::shortMax) && (scanValue >= Limit::shortMin)){
		*value = (short)scanValue;
		return true;
	}
	return false;
}
//------------------------------------------------------------------------------
// u_shortւ̃p[X
bool String::parseUShort(u_short* value) const{
	Assert(value != NULL);
	int scanValue;
	int scanCount = StdSscanf(string_, "%d", &scanValue);
	if((scanCount == 1) &&
		(scanValue <= Limit::uShortMax) && (scanValue >= Limit::uShortMin)){
		*value = (u_short)scanValue;
		return true;
	}
	return false;
}
//------------------------------------------------------------------------------
// intւ̃p[X
bool String::parseInt(int* value) const{
	Assert(value != NULL);
	int scanCount = StdSscanf(string_, "%d", value);
	if(scanCount == 1){ return true; }
	return false;
}
//------------------------------------------------------------------------------
// u_intւ̃p[X
bool String::parseUInt(u_int* value) const{
	Assert(value != NULL);
	int scanCount = StdSscanf(string_, "%u", value);
	if(scanCount == 1){ return true; }
	return false;
}
//------------------------------------------------------------------------------
// floatւ̃p[X
bool String::parseFloat(float* value) const{
	Assert(value != NULL);
	int scanCount = StdSscanf(string_, "%f", value);
	if(scanCount == 1){ return true; }
	return false;
}
//------------------------------------------------------------------------------
// doubleւ̃p[X
bool String::parseDouble(double* value) const{
	Assert(value != NULL);
	int scanCount = StdSscanf(string_, "%lf", value);
	if(scanCount == 1){ return true; }
	return false;
}
//------------------------------------------------------------------------------
// ύX
//------------------------------------------------------------------------------
// ̒ǉ
String& String::append(const String& appendString){
	int appendLength = appendString.getSize();
	if(appendLength == 0){ return *this; }
	int newLength = size_ + appendLength;
	char* newArray = newString(newLength);
	if(size_ != 0){ std::memcpy(newArray, string_, size_); }
	std::memcpy(newArray + size_, appendString.getBytes(), appendLength);
	newArray[newLength] = '\0';
	releaseString(&string_);
	string_ = newArray;
	size_ = newLength;
	return *this;
}
//------------------------------------------------------------------------------
// ̒ǉ
String& String::append(const char* appendString){
	Assert(appendString != NULL);
	int appendLength = (int)StdStrlen(appendString);
	if(appendLength == 0){ return *this; }
	int newLength = size_ + appendLength;
	char* newArray = newString(newLength);
	if(size_ != 0){ std::memcpy(newArray, string_, size_); }
	std::memcpy(newArray + size_, appendString, appendLength);
	newArray[newLength] = '\0';
	releaseString(&string_);
	string_ = newArray;
	size_ = newLength;
	return *this;
}
//------------------------------------------------------------------------------
// tH[}bg
String& String::format(const char* formatString, ...){
	Assert(formatString != NULL);
	releaseString(&string_);
	// tH[}bg񂪋󕶎
	if(formatString[0] == '\0'){
		string_ = NULL;
		size_ = 0;
		return *this;
	}
	int allocateSize = formatDefaultLength;
	va_list args;
	va_start(args, formatString);
	while(true){
		int maxLength = allocateSize - sizeof(int) - 1;
		string_ = newString(maxLength);
		int result = StdVsnprintf(string_, maxLength, formatString, args);
		if(result != -1){
			size_ = result;
			if(size_ == 0){
				releaseString(&string_);
			}else{
				// _vsntprintf̎w蒷xNULLI[Ȃ̂
				string_[size_] = '\0';
			}
			break;
		}
		allocateSize *= 2;
		releaseString(&string_);
	}
	va_end(args);
	return *this;
}
//------------------------------------------------------------------------------
// r
//------------------------------------------------------------------------------
// ̎r
int String::compareTo(const String& compareString) const{
	return StdStrcmp(getBytes(), compareString.getBytes());
}
//------------------------------------------------------------------------------
// ̎r
int String::compareTo(const char* compareString) const{
	Assert(compareString != NULL);
	return StdStrcmp(getBytes(), compareString);
}
//------------------------------------------------------------------------------
// 啶A𖳎̎r
int String::compareToIgnoreCase(const String& compareString) const{
	String source = this->getLowerCase();
	String destination = compareString.getLowerCase();
	return StdStrcmp(source.getBytes(), destination.getBytes());
}
//------------------------------------------------------------------------------
// ̔r
bool String::equals(const String& compareString) const{
	return (StdStrcmp(getBytes(), compareString.getBytes()) == 0);
}
//------------------------------------------------------------------------------
// ̔r
bool String::equals(const char* compareString) const{
	Assert(compareString != NULL);
	return (StdStrcmp(getBytes(), compareString) == 0);
}
//------------------------------------------------------------------------------
// 啶A𖳎̔r
bool String::equalsIsIgnoreCase(const String& compareString) const{
	String source = this->getLowerCase();
	String destination = compareString.getLowerCase();
	return (StdStrcmp(source.getBytes(), destination.getBytes()) == 0);
}
//------------------------------------------------------------------------------
// w肵Ŏn܂邩ǂ
bool String::startsWith(const String& prefix) const{
	return (StdStrncmp(getBytes(), prefix.getBytes(), prefix.getSize()) == 0);
}
//------------------------------------------------------------------------------
// w肵Ŏn܂邩ǂ
bool String::startsWith(const char* prefix) const{
	Assert(prefix != NULL);
	return (StdStrncmp(getBytes(), prefix, StdStrlen(prefix)) == 0);
}
//------------------------------------------------------------------------------
// w肵ŏI邩ǂ
bool String::endsWith(const String& suffix) const{
	int index = size_ - suffix.getSize();
	if(index < 0){ return false; }
	return (StdStrncmp(getBytes() + index,
		suffix.getBytes(), suffix.getSize()) == 0);
}
//------------------------------------------------------------------------------
// w肵ŏI邩ǂ
bool String::endsWith(const char* suffix) const{
	Assert(suffix != NULL);
	int suffixLength = (int)StdStrlen(suffix);
	int index = size_ - suffixLength;
	if(index < 0){ return false; }
	return (StdStrncmp(getBytes() + index, suffix, suffixLength) == 0);
}
//------------------------------------------------------------------------------
// w肳ꂽŏɏoʒũCfbNX擾
int String::getIndexOf(const char searchChar) const{
	char* indexPointer = StdStrchr(getBytes(), searchChar);
	if(indexPointer == NULL){ return -1; }
	return (int)(indexPointer - getBytes());
}
//------------------------------------------------------------------------------
// w肳ꂽ񂪍ŏɏoʒũCfbNX擾
int String::getIndexOf(const char* searchString) const{
	Assert(searchString != NULL);
	char* indexPointer = StdStrstr(getBytes(), searchString);
	if(indexPointer == NULL){ return -1; }
	return (int)(indexPointer - getBytes());
}
//------------------------------------------------------------------------------
// w肳ꂽ񂪍ŏɏoʒũCfbNX擾
int String::getIndexOf(const String& searchString) const{
	char* indexPointer = StdStrstr(getBytes(), searchString.getBytes());
	if(indexPointer == NULL){ return -1; }
	return (int)(indexPointer - getBytes());
}
//------------------------------------------------------------------------------
// w肳ꂽŌɏoʒũCfbNX擾
int String::getLastIndexOf(const char searchChar) const{
	char* indexPointer = StdStrrchr(getBytes(), searchChar);
	if(indexPointer == NULL){ return -1; }
	return (int)(indexPointer - getBytes());
}
//------------------------------------------------------------------------------
// w肳ꂽ񂪍ŌɏoʒũCfbNX擾
int String::getLastIndexOf(const char* searchString) const{
	Assert(searchString != NULL);
	if(StdStrlen(searchString) == 0){ return size_; }
	const char* indexPointer = NULL;
	const char* target = getBytes();
	while(true){
		target = StdStrstr(target, searchString);
		if(target == NULL){ break; }
		indexPointer = target;
		target++;
	}
	if(indexPointer == NULL){ return -1; }
	return (int)(indexPointer - getBytes());
}
//------------------------------------------------------------------------------
// w肳ꂽ񂪍ŌɏoʒũCfbNX擾
int String::getLastIndexOf(const String& searchString) const{
	if(searchString.getSize() == 0){ return size_; }
	const char* search = searchString.getBytes();
	const char* indexPointer = NULL;
	const char* target = getBytes();
	while(true){
		target = StdStrstr(target, search);
		if(target == NULL){ break; }
		indexPointer = target;
		target++;
	}
	if(indexPointer == NULL){ return -1; }
	return (int)(indexPointer - getBytes());
}
//------------------------------------------------------------------------------
#ifdef _DEBUG
// fobOpo
void String::debugPrint(){
	DebugOut("[%2d] %s\n", getReferenceCounter(string_), string_);
}
#endif// End of _DEBUG
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
// ̘A
const Lamp::String operator+(
	const Lamp::String& leftString, const Lamp::String& rightString){
	Lamp::String returnString(leftString);
	returnString.append(rightString);
	return returnString;
}
//------------------------------------------------------------------------------
