//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * RWV[
 * @author Junpee
 */

#include "LampBasic.h"
#include "Collision/System/CollisionScene.h"
#include "Core/Renamer/CountRenamer.h"
#include "Collision/System/CollisionNode.h"
#include "Collision/Leaf/StaticSphereCollision.h"
#include "Collision/Leaf/StaticDeformedMeshCollision.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
CollisionScene::CollisionScene() : nodeDatabase_(256, 0.75f), nodeArray_(256),
	leafDatabase_(256, 0.75f), leafArray_(256), tick_(0){
	// l[}̏
	nodeRenamer_ = new CountRenamer();
	nodeRenamerCallback_.scene_ = this;
	leafRenamer_ = new CountRenamer();
	leafRenamerCallback_.scene_ = this;
	// [gm[h̏
	rootNode_ = createCollisionNode("RootNode");
}
//------------------------------------------------------------------------------
// fXgN^
CollisionScene::~CollisionScene(){
	Assert(nodeDatabase_.getCount() == 1);
	Assert(nodeArray_.getCount() == 1);
	Assert(leafDatabase_.getCount() == 0);
	Assert(leafArray_.getCount() == 0);
	// [gm[ȟn
	if(nodeArray_.removeByValue(rootNode_) == -1){
		ErrorOut("CollisionScene::~CollisionScene() "
			"Not found RootNode in array");
	}
	if(nodeDatabase_.remove(rootNode_->getName()) == NULL){
		ErrorOut("CollisionScene::~CollisionScene() "
			"Not found RootNode in hashmap");
	}
	SafeDelete(rootNode_);
	// l[}̌n
	SafeDelete(leafRenamer_);
	SafeDelete(nodeRenamer_);
}
//------------------------------------------------------------------------------
// V[֘A
//------------------------------------------------------------------------------
// 
void CollisionScene::traverse(){
	// `bÑCNg
	tick_++;
	getRootNode()->traverse();
}
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
void CollisionScene::intersection(
	IntersectionResult* result, const Sphere& sphere, u_int collisionMask){
	int leafCount = getLeafCount();
	for(int i = 0; i < leafCount; i++){
		getLeaf(i)->intersection(result, sphere, collisionMask);
	}
}
//------------------------------------------------------------------------------
// RW
void CollisionScene::intersection(IntersectionResult* result,
	StaticSphereCollision* sphere, u_int collisionMask){
	Assert(sphere->isGlobalEnabled());
	int leafCount = getLeafCount();
	for(int i = 0; i < leafCount; i++){
		CollisionLeaf* leaf = getLeaf(i);
		leaf->intersection(result, sphere, collisionMask);
	}
}
//------------------------------------------------------------------------------
// RWm[h
//------------------------------------------------------------------------------
// RWm[h̍쐬
CollisionNode* CollisionScene::createCollisionNode(const String& name){
	// O̒`FbN
	if(name.getSize() == 0){
		ErrorOut("CollisionScene::createCollisionNode() O󕶎ł");
		return NULL;
	}
	// Ȍd`FbN
	if(existNodeName(name)){
		ErrorOut("CollisionScene::createCollisionNode() "
			"OdĂ܂ %s", name.getBytes());
		return NULL;
	}
	CollisionNode* collisionNode = new CollisionNode(name, this);
	nodeDatabase_.put(name, collisionNode);
	nodeArray_.add(collisionNode);
	return collisionNode;
}
//------------------------------------------------------------------------------
// ÓIRW[t
//------------------------------------------------------------------------------
// ÓIRW̍쐬
StaticSphereCollision* CollisionScene::createStaticSphereCollision(
	const String& name){
	if(!checkLeafName(name)){ return NULL; }
	StaticSphereCollision* sphere = new StaticSphereCollision(name, this);
	leafDatabase_.put(name, sphere);
	leafArray_.add(sphere);
	return sphere;
}
//------------------------------------------------------------------------------
// ÓIό`bVRW̍쐬
StaticDeformedMeshCollision* CollisionScene::createStaticDeformedMeshCollision(
	const String& name){
	if(!checkLeafName(name)){ return NULL; }
	StaticDeformedMeshCollision* mesh =
		new StaticDeformedMeshCollision(name, this);
	leafDatabase_.put(name, mesh);
	leafArray_.add(mesh);
	return mesh;
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// [t̖O`FbN
bool CollisionScene::checkLeafName(const String& name){
	// O̒`FbN
	if(name.getSize() == 0){
		ErrorOut("CollisionScene::checkLeafName() O󕶎ł");
		return false;
	}
	// Ȍd`FbN
	if(existLeafName(name)){
		ErrorOut("CollisionScene::checkLeafName() OdĂ܂ %s",
			name.getBytes());
		return false;
	}
	return true;
}
//------------------------------------------------------------------------------
// RWIuWFNgj
//------------------------------------------------------------------------------
// m[hIuWFNg̔j
void CollisionScene::destroyNode(CollisionNode* node){
	Assert(node != rootNode_);
	if(nodeArray_.removeByValue(node) == -1){
		ErrorOut("CollisionScene::~destroyNode() Not found node in array");
	}
	if(nodeDatabase_.remove(node->getName()) == NULL){
		ErrorOut("CollisionScene::~destroyNode() Not found node in hashmap");
	}
	delete node;
}
//------------------------------------------------------------------------------
// [tIuWFNg̔j
void CollisionScene::destroyLeaf(CollisionLeaf* leaf){
	if(leafArray_.removeByValue(leaf) == -1){
		ErrorOut("CollisionScene::~destroyLeaf() Not found leaf in array");
	}
	if(leafDatabase_.remove(leaf->getName()) == NULL){
		ErrorOut("CollisionScene::~destroyLeaf() Not found leaf in hashmap");
	}
	delete leaf;
}
//------------------------------------------------------------------------------
// NA
int CollisionScene::clear(){
	// [gm[h̎qNA
	int childCount = rootNode_->getChildCount();
	for(int i = childCount - 1; i >= 0; i--){
		rootNode_->removeChild(rootNode_->getChild(i));
	}
	// [t̍폜
	int result = getLeafCount();
	for(int i = getLeafCount() - 1; i >= 0; i--){ delete getLeaf(i); }
	// [gm[hȊOj
	result += getNodeCount();
	for(int i = getNodeCount() - 1; i >= 1; i--){ delete getNode(i); }
	leafArray_.clear();
	leafDatabase_.clear();
	nodeArray_.clear();
	nodeDatabase_.clear();
	nodeArray_.add(rootNode_);
	nodeDatabase_.put(rootNode_->getName(), rootNode_);
	return (result - 1);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
