//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * eLXgRW[_
 * @author Junpee
 */

#include "LampBasic.h"
#include "Collision/InputOutput/TextCollisionLoader.h"
#include "Core/InputOutput/TextFileReader.h"
#include "Core/InputOutput/StreamTokenizer.h"
#include "Core/InputOutput/FilePath.h"
#include "Collision/System/CollisionScene.h"
#include "Collision/System/CollisionNode.h"
#include "Collision/Leaf/StaticSphereCollision.h"
#include "Collision/Leaf/StaticDeformedMeshCollision.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
TextCollisionLoader::TextCollisionLoader(){
}
//------------------------------------------------------------------------------
// fXgN^
TextCollisionLoader::~TextCollisionLoader(){
}
//------------------------------------------------------------------------------
// [h
void TextCollisionLoader::load(const String& filePath, CollisionScene* scene){
	load(filePath, scene, scene->getRootNode());
}
//------------------------------------------------------------------------------
// [h
void TextCollisionLoader::load(const String& filePath, CollisionScene* scene,
	CollisionNode* node){
	FilePath path(filePath);
	if(!path.existFile()){
		ErrorOut("TextCollisionLoader::load() t@C܂ł %s",
			filePath.getBytes());
		return;
	}
	TextFileReader* textFileReader = new TextFileReader(filePath);
	load(textFileReader, scene, node);
	delete textFileReader;
}
//------------------------------------------------------------------------------
// [h
void TextCollisionLoader::load(TextReader* textReader, CollisionScene* scene){
	load(textReader, scene, scene->getRootNode());
}
//------------------------------------------------------------------------------
// [h
void TextCollisionLoader::load(TextReader* textReader, CollisionScene* scene,
	CollisionNode* node){
	// 
	tokenizer_ = new StreamTokenizer(textReader);
	scene_ = scene;
	rootNode_ = node;

	// [gm[h𖳌ɂĂ
	bool rootNodeEnabled = rootNode_->isEnabled();
	rootNode_->setEnabled(false);

	// wb_̓ǂݍ
	readHeader();

	while(true){
		// g[NǂݏI
		if(!tokenizer_->nextToken()){ break; }
		String chunkName = tokenizer_->getToken();
		if(chunkName == "CollisionNode"){
			readCollisionNodeList();
		}else if(chunkName == "CollisionLeaf"){
			readCollisionLeafList();
		}else if(chunkName == "CollisionNodeLink"){
			readCollisionLinkList();
		}else{
			ErrorOut("TextCollisionLoader::load() "
				"invalid chunk %s (line %d)",
				chunkName.getBytes(), tokenizer_->getLineNumber());
		}
	}
	delete tokenizer_;

	rootNode_->setEnabled(rootNodeEnabled);
}
///------------------------------------------------------------------------------
// wb_̓ǂݍ
void TextCollisionLoader::readHeader(){
	skipWord("Header");
	openChunk();
	skipWord("type");
	skipWord("LampTextCollisionFormat");
	skipWord("version");
	skipWord("0_1_0");
	closeChunk();
}
//------------------------------------------------------------------------------
// RWm[h
//------------------------------------------------------------------------------
// RWm[hXg̓ǂݍ
void TextCollisionLoader::readCollisionNodeList(){
	openChunk();
	while(true){
		String token = readToken("readCollisionNodeList()");
		if(token == "}"){ break; }
		readCollisionNode(token);
	}
}
//------------------------------------------------------------------------------
// RWm[h̓ǂݍ
void TextCollisionLoader::readCollisionNode(const String& name){
	CollisionNode* node = scene_->createCollisionNode(name);
	openChunk();
	// XP[
	skipWord("scale");
	node->setScale(readVector3());
	// ]
	skipWord("rotation");
	node->setRotationXYZ(readVector3());
	// ړ
	skipWord("translation");
	node->setTranslation(readVector3());
	// LAtO
	skipWord("enabled");
	node->setEnabled(readBool());
	closeChunk();
}
//------------------------------------------------------------------------------
// RW[t
//------------------------------------------------------------------------------
// RW[tXg̓ǂݍ
void TextCollisionLoader::readCollisionLeafList(){
	openChunk();
	while(true){
		String token = readToken("readCollisionLeafList()");
		if(token == "}"){ break; }
		openChunk();
		skipWord("type");
		String type = readToken("readCollisionLeafList()");
		if(type == "StaticSphere"){
			readStaticSphere(token);
		}else if(type == "StaticDeformedMesh"){
			readStaticDeformedMesh(token);
		}else{
			ErrorOut("TextCollisionLoader::readCollisionLeafList() "
				"sȃ^Cvł %s (line %d)",
				type.getBytes(), tokenizer_->getLineNumber());
		}
		closeChunk();
	}
}
//------------------------------------------------------------------------------
// ÓIRW̓ǂݍ
void TextCollisionLoader::readStaticSphere(const String& name){
	StaticSphereCollision* sphere = scene_->createStaticSphereCollision(name);
	// RW}XN
	skipWord("collisionMask");
	sphere->setCollisionMask(readUInt());
	// LAtO
	skipWord("enabled");
	sphere->setEnabled(readBool());
	// 
	skipWord("sphere");
	sphere->setSphere(readSphere());
}
//------------------------------------------------------------------------------
// ÓIό`bVRW̓ǂݍ
void TextCollisionLoader::readStaticDeformedMesh(const String& name){
	StaticDeformedMeshCollision* mesh =
		scene_->createStaticDeformedMeshCollision(name);
	// RW}XN
	skipWord("collisionMask");
	mesh->setCollisionMask(readUInt());
	// LAtO
	skipWord("enabled");
	mesh->setEnabled(readBool());
	// oEfBO{bNX
	skipWord("boundingBox");
	mesh->setBoundingBox(readAxisAlignedBox());
	// oEfBOXtBA
	skipWord("boundingSphere");
	mesh->setBoundingSphere(readSphere());
	// gCAO
	skipWord("triangleCount");
	int triangleCount = readInt();
	mesh->setTriangleCount(triangleCount);
	openChunk();
	for(int i = 0; i < triangleCount; i++){
		mesh->setTriangle(i, readTriangle());
	}
	closeChunk();
}
//------------------------------------------------------------------------------
// RWN
//------------------------------------------------------------------------------
// RWNXg̓ǂݍ
void TextCollisionLoader::readCollisionLinkList(){
	openChunk();
	while(true){
		String token = readToken("readCollisionLinkList()");
		if(token == "}"){ break; }
		readCollisionLink(token);
	}
}
//------------------------------------------------------------------------------
// RWN̓ǂݍ
void TextCollisionLoader::readCollisionLink(const String& name){
	CollisionNode* node;
	if(name.equals("RootNode")){
		node = rootNode_;
	}else{
		node = scene_->searchNode(name);
		if(node == NULL){
			ErrorOut("TextCollisionLoader::readCollisionLink() "
				"em[h܂ %s (line %d)",
				name.getBytes(), tokenizer_->getLineNumber());
		}
	}

	openChunk();
	while(true){
		String token = readToken("readCollisionLink()");
		if(token == "}"){ break; }
		if(token == "node"){
			String childName = readToken("readCollisionLink()");
			CollisionNode* child = scene_->searchNode(childName);
			if(child == NULL){
				ErrorOut("TextCollisionLoader::readCollisionLink() "
					"qm[h܂ %s (line %d)",
					childName.getBytes(), tokenizer_->getLineNumber());
			}
			node->addChild(child);
		}else if(token == "leaf"){
			String childName = readToken("readCollisionLink()");
			CollisionLeaf* child = scene_->searchLeaf(childName);
			if(child == NULL){
				ErrorOut("TextCollisionLoader::readCollisionLink() "
					"q[t܂ %s (line %d)",
					childName.getBytes(), tokenizer_->getLineNumber());
			}
			node->addChild(child);
		}else{
			ErrorOut("TextCollisionLoader::readCollisionLink() "
				"sȃg[Nł %s (line %d)",
				token.getBytes(), tokenizer_->getLineNumber());
		}
	}
}
//------------------------------------------------------------------------------
// l̓ǂݍ
//------------------------------------------------------------------------------
// booll̓ǂݍ
bool TextCollisionLoader::readBool(){
	String token = readToken("readBool()");
	if(token == "true"){ return true; }
	else if(token == "false"){ return false; }
	ErrorOut("TextCollisionLoader::readBool() sȃg[Nł %s (line %d)",
		token.getBytes(), tokenizer_->getLineNumber());
	return false;
}
//------------------------------------------------------------------------------
// intl̓ǂݍ
int TextCollisionLoader::readInt(){
	String token = readToken("readInt()");
	int value = 0;
	bool result = token.parseInt(&value);
	if(!result){
		ErrorOut("TextCollisionLoader::readInt() "
			"sȃg[Nł %s (line %d)",
			token.getBytes(), tokenizer_->getLineNumber());
	}
	return value;
}
//------------------------------------------------------------------------------
// u_intl̓ǂݍ
u_int TextCollisionLoader::readUInt(){
	String token = readToken("readInt()");
	u_int value = 0;
	bool result = token.parseUInt(&value);
	if(!result){
		ErrorOut("TextCollisionLoader::readUInt() "
			"sȃg[Nł %s (line %d)",
			token.getBytes(), tokenizer_->getLineNumber());
	}
	return value;
}
//------------------------------------------------------------------------------
// floatl̓ǂݍ
float TextCollisionLoader::readFloat(){
	String token = readToken("readFloat()");
	float value = 0.f;
	bool result = token.parseFloat(&value);
	if(!result){
		ErrorOut("TextCollisionLoader::readFloat() "
			"sȃg[Nł %s (line %d)",
			token.getBytes(), tokenizer_->getLineNumber());
	}
	return value;
}
//------------------------------------------------------------------------------
// Vector3l̓ǂݍ
Vector3 TextCollisionLoader::readVector3(){
	Vector3 result;
	openChunk();
	result.x = readFloat();
	result.y = readFloat();
	result.z = readFloat();
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// {bNX̓ǂݍ
AxisAlignedBox TextCollisionLoader::readAxisAlignedBox(){
	AxisAlignedBox result;
	Vector3 minimum, maximum;
	openChunk();
	minimum.x = readFloat();
	minimum.y = readFloat();
	minimum.z = readFloat();
	closeChunk();
	openChunk();
	maximum.x = readFloat();
	maximum.y = readFloat();
	maximum.z = readFloat();
	closeChunk();
	result.set(minimum, maximum);
	return result;
}
//------------------------------------------------------------------------------
// ̓ǂݍ
Sphere TextCollisionLoader::readSphere(){
	Sphere result;
	Vector3 vector;
	openChunk();
	vector.x = readFloat();
	vector.y = readFloat();
	vector.z = readFloat();
	result.setCenter(vector);
	result.setRadius(readFloat());
	closeChunk();
	return result;
}
//------------------------------------------------------------------------------
// Op̓ǂݍ
Triangle TextCollisionLoader::readTriangle(){
	Triangle result;
	Vector3 vector;
	for(int i = 0; i < 3; i++){
		openChunk();
		vector.x = readFloat();
		vector.y = readFloat();
		vector.z = readFloat();
		result.setVertex(i, vector);
		closeChunk();
	}
	return result;
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// g[N̓ǂݍ
String TextCollisionLoader::readToken(const String& caller){
	bool hasNext = tokenizer_->nextToken();
	if(!hasNext){
		ErrorOut("TextCollisionLoader::%s g[N܂ (line %d)",
			caller.getBytes(), tokenizer_->getLineNumber());
	}
	return tokenizer_->getToken();
}
//------------------------------------------------------------------------------
// w胏[h̓ǂݔ΂
void TextCollisionLoader::skipWord(const String& word){
	String token = readToken("skipWord()");
	if(token != word){
		ErrorOut("TextCollisionLoader::skipWord(%s) "
			"sȃg[Nł %s (line %d)",
			word.getBytes(), token.getBytes(), tokenizer_->getLineNumber());
	}
}
//------------------------------------------------------------------------------
// `N̓ǂݔ΂
void TextCollisionLoader::skipChunk(){
	int chunkLine = tokenizer_->getLineNumber();
	int chunkCounter = 0;
	while(true){
		bool hasNext = tokenizer_->nextToken();
		if(!hasNext){
			ErrorOut("TextCollisionLoader::skipChunk(%d) "
				"g[N܂ (line %d)",
				chunkLine, tokenizer_->getLineNumber());
		}
		String token = tokenizer_->getToken();
		if(token == "{"){
			chunkCounter++;
		}else if(token == "}"){
			chunkCounter--;
			// fNgɃ[v𔲂`FbNs
			if(chunkCounter == 0){ break; }
			if(chunkCounter < 0){
				ErrorOut("TextCollisionLoader::skipChunk(%d) "
					"sȃ`Nł (line %d)",
					chunkLine, tokenizer_->getLineNumber());
			}
		}
	}
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
