//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * xNgԈk
 * @author Junpee
 */

#include "LampBasic.h"
#include "Animation/VectorInterpolator/VectorInterpolationCompressor.h"
#include "Animation/VectorInterpolator/VectorArrayInterpolator.h"
#include "Animation/VectorInterpolator/VectorConstantInterpolator.h"
#include "Animation/VectorInterpolator/VectorLinearInterpolator.h"
#include "Core/Container/Deque.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
VectorInterpolationCompressor::VectorInterpolationCompressor() :
	tolerance_(0.f), length_(0.f), sourceKeyCount_(0), compressedKeyCount_(0),
	compressedKeySize_(0){
}
//------------------------------------------------------------------------------
// fXgN^
VectorInterpolationCompressor::~VectorInterpolationCompressor(){
}
//------------------------------------------------------------------------------
// k
//------------------------------------------------------------------------------
// k
VectorInterpolator* VectorInterpolationCompressor::compress(
	VectorArrayInterpolator* source, float tolerance){
	// k
	compressSetup(source, tolerance);
	VectorInterpolator* result = NULL;

	// 萔k
	result = compressConstant(source);
	if(result != NULL){ return result; }

	// `k
	result = compressLinear(source);
	if(result != NULL){ return result; }

	// XvCn̈kƐ`ǩʂdݕtŔrďo͉\
	// ̏ꍇsetCompressedDataYȂ悤ɒ

	// SĂ̈kɎsȂ猳f[^̃Rs[Ԃ
	result = source->duplicate();
	setCompressedData(getSourceKeyCount(), sourceKeySize_);
	
	return result;
}
//------------------------------------------------------------------------------
// k
void VectorInterpolationCompressor::compressSetup(
	VectorArrayInterpolator* source, float tolerance){
	Assert((source != NULL) && (tolerance >= 0.f));
	tolerance_ = tolerance;
	length_ = source->getLength();
	sourceKeyCount_ = source->getSize();
	Assert(sourceKeyCount_ > 1);
}
//------------------------------------------------------------------------------
// 萔k
VectorInterpolator* VectorInterpolationCompressor::compressConstant(
	VectorArrayInterpolator* source){
	// ^[QbgƂȂl̎ZoAoEfBO{bNX̒S
	AxisAlignedBox boundingBox(source->getValue(0), source->getValue(0));
	for(int i = 1; i < sourceKeyCount_; i++){
		boundingBox.merge(source->getValue(i));
	}
	Vector3 targetValue = boundingBox.getCenter();

	// 덷`FbN
	float squaredTolerance = getTolerance() * getTolerance();
	for(int i = 0; i < sourceKeyCount_; i++){
		Vector3 distance = source->getValue(i) - targetValue;
		// 萔kɎs
		if(distance.getSquaredLength() > squaredTolerance){ return NULL; }
	}

	// 萔kɐ
	VectorConstantInterpolator* result = new VectorConstantInterpolator();
	result->setLength(getLength());
	result->setValue(targetValue);
	setCompressedData(1, sizeof(Vector3) + sizeof(float));
	return result;
}
//------------------------------------------------------------------------------
// `k
VectorInterpolator* VectorInterpolationCompressor::compressLinear(
	VectorArrayInterpolator* source){
	// L[̃Xg쐬A덷L[ԈĂ

	// L[Xg̍쐬
	int sourceKeyCount = getSourceKeyCount();
	Deque<LinearKey> keys_(sourceKeyCount);
	for(int i = 0; i < sourceKeyCount; i++){
		LinearKey key;
		key.value_ = source->getValue(i);
		key.time_ = (float)i;
		keys_.pushBack(key);
	}

	// S덷̎Zo
	int keyCount = keys_.getCount();
	int maxKeyCount = keyCount - 1;
	// ŏƍŌ̒l͑삵Ȃ
	keys_[0].squaredError_ = 0.f;
	keys_[maxKeyCount].squaredError_ = 0.f;
	Assert(keys_[0].time_ == 0.f)
	Assert(keys_[maxKeyCount].time_ == getLength());
	for(int i = 1; i < maxKeyCount; i++){
		LinearKey& key = keys_[i];
		LinearKey& preKey = keys_[i - 1];
		LinearKey& postKey = keys_[i + 1];
		// 덷̏͑ÕL[ƌ̃L[̕ςԒlɂȂ
		Vector3 value = (preKey.value_ + postKey.value_) * 0.5f;
		key.squaredError_ = (value - key.value_).getSquaredLength();
	}

	// k[vAL[2ɂȂ甲
	float squaredTolerance = getTolerance() * getTolerance();
	while(keys_.getCount() > 2){
		// 덷ŏ̃L[𒲂ׂ
		int targetIndex = 1;
		float targetSquaredError = keys_[targetIndex].squaredError_;
		maxKeyCount = keys_.getCount() - 1;
		for(int i = 2; i < maxKeyCount; i++){
			if(keys_[i].squaredError_ < targetSquaredError){
				targetSquaredError = keys_[i].squaredError_;
				targetIndex = i;
			}
		}

		// ŏ덷e͈͊OȂ爳k[v𔲂
		if(targetSquaredError > squaredTolerance){ break; }

		// ŏ덷L[̍폜ƌ덷̃Abvf[g
		keys_.remove(targetIndex);
		// ÕL[擪L[łȂΌ덷ČvZ
		if((targetIndex - 1) > 0){
			recalcLinearError(source, keys_[targetIndex - 2],
				keys_[targetIndex - 1], keys_[targetIndex]);
		}
		// ̃L[ŌL[łȂΌ덷ČvZ
		if(targetIndex < (keys_.getCount() - 1)){
			recalcLinearError(source, keys_[targetIndex - 1],
				keys_[targetIndex], keys_[targetIndex + 1]);
		}

	}

	// TCYȂĂȂΎsAŝ
	keyCount = keys_.getCount();
	int keySize = sizeof(Vector3) + sizeof(float);
	if((keyCount * keySize) >= getSourceSize()){ return NULL; }

	// `kɐ
	VectorLinearInterpolator* result = new VectorLinearInterpolator();
	result->setKeyCount(keyCount);
	for(int i = 0; i < keyCount; i++){
		LinearKey& key = keys_[i];
		result->setKey(i, key.time_, key.value_);
	}
	setCompressedData(keyCount, keySize);
	return result;
}
//------------------------------------------------------------------------------
// `k덷̍ČvZ
void VectorInterpolationCompressor::recalcLinearError(
	VectorArrayInterpolator* source,
	LinearKey& preKey, LinearKey& key, LinearKey& postKey){
	int preTime = (int)preKey.time_;
	Vector3 preValue = preKey.value_;
	int postTime = (int)postKey.time_;
	Vector3 postValue = postKey.value_;
	int timeRange = postTime - preTime;
	Vector3 distance = postValue - preValue;

	// ő̓덷Zo
	float maxSquaredError = 0.f;
	for(int i = 1; i < timeRange; i++){
		int time = preTime + i;
		Vector3 sourceValue = source->getValue(time);
		float rate = (float)i / (float)timeRange;
		Vector3 targetValue = distance * rate + preValue;
		float squaredError = (targetValue - sourceValue).getSquaredLength();
		if(squaredError > maxSquaredError){ maxSquaredError = squaredError; }
	}
	key.squaredError_ = maxSquaredError;
}
//------------------------------------------------------------------------------
// k
//------------------------------------------------------------------------------
// ʕ̎擾
String VectorInterpolationCompressor::getResultString() const{
	String result;
	result.format(
		"%.2f%%  Key %d/%d  Size %.2f/%.2fKb  Length %.0f",
		getCompressionRate() * 100.f,
		getCompressedKeyCount(), getSourceKeyCount(),
		(float)getCompressedSize() / 1024.f, (float)getSourceSize() / 1024.f,
		getLength());
	return result;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
