//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * Aj[V}l[W
 * @author Junpee
 */

#include "LampBasic.h"
#include "Animation/System/AnimationManager.h"
#include "Core/Renamer/CountRenamer.h"
#include "Animation/System/AnimationSet.h"
#include "Animation/Camera/CameraAnimation.h"
#include "Animation/SceneNode/SceneNodeAnimation.h"
#include "Animation/Model/CharacterModelAnimation.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
AnimationManager::AnimationManager() :
	animationHash_(256, 0.75f), animationArray_(256),
	animationDataHash_(256, 0.75f), animationDataArray_(256){
	animationRenamer_ = new CountRenamer();
	animationDataRenamer_ = new CountRenamer();
}
//------------------------------------------------------------------------------
// fXgN^
AnimationManager::~AnimationManager(){
	Assert(animationHash_.getCount() == 0);
	Assert(animationArray_.getCount() == 0);
	Assert(animationDataHash_.getCount() == 0);
	Assert(animationDataArray_.getCount() == 0);
	if(getCount() != 0){ clear(); }
	if(getDataCount() != 0){ clear(); }
	SafeDelete(animationDataRenamer_);
	SafeDelete(animationRenamer_);
}
//------------------------------------------------------------------------------
// Aj[VZbg
//------------------------------------------------------------------------------
// Aj[VZbg̍쐬
AnimationSet* AnimationManager::createAnimationSet(const String& name){
	AnimationSet* animation = new AnimationSet(rename(name), this);
	animationHash_.put(animation->getName(), animation);
	animationArray_.add(animation);
	return animation;
}
//------------------------------------------------------------------------------
// JAj[V
//------------------------------------------------------------------------------
// JAj[V̍쐬
CameraAnimation* AnimationManager::createCamera(
	const String& name){
	CameraAnimation* animation = new CameraAnimation(rename(name), this);
	animationHash_.put(animation->getName(), animation);
	animationArray_.add(animation);
	return animation;
}
//------------------------------------------------------------------------------
// JAj[Vf[^̍쐬
CameraAnimationData* AnimationManager::createCameraData(
	const String& name){
	CameraAnimationData* animationData =
		new CameraAnimationData(renameData(name), this);
	animationDataHash_.put(animationData->getName(), animationData);
	animationDataArray_.add(animationData);
	return animationData;
}
//------------------------------------------------------------------------------
// V[m[hAj[V
//------------------------------------------------------------------------------
// V[m[hAj[V̍쐬
SceneNodeAnimation* AnimationManager::createSceneNode(
	const String& name){
	SceneNodeAnimation* animation = new SceneNodeAnimation(rename(name), this);
	animationHash_.put(animation->getName(), animation);
	animationArray_.add(animation);
	return animation;
}
//------------------------------------------------------------------------------
// V[m[hAj[Vf[^̍쐬
SceneNodeAnimationData* AnimationManager::createSceneNodeData(
	const String& name){
	SceneNodeAnimationData* animationData =
		new SceneNodeAnimationData(renameData(name), this);
	animationDataHash_.put(animationData->getName(), animationData);
	animationDataArray_.add(animationData);
	return animationData;
}
//------------------------------------------------------------------------------
// fAj[V
//------------------------------------------------------------------------------
// LN^fAj[V̍쐬
CharacterModelAnimation* AnimationManager::createCharacterModel(
	const String& name){
	CharacterModelAnimation* animation =
		new CharacterModelAnimation(rename(name), this);
	animationHash_.put(animation->getName(), animation);
	animationArray_.add(animation);
	return animation;
}
//------------------------------------------------------------------------------
// LN^fAj[Vf[^̍쐬
CharacterModelAnimationData* AnimationManager::createCharacterModelData(
	const String& name){
	CharacterModelAnimationData* animationData =
		new CharacterModelAnimationData(renameData(name), this);
	animationDataHash_.put(animationData->getName(), animationData);
	animationDataArray_.add(animationData);
	return animationData;
}
//------------------------------------------------------------------------------
// jANA
//------------------------------------------------------------------------------
// Aj[V̔j
void AnimationManager::destroy(Animation* animation){
	// Aj[VZbg̏ꍇ͉ʂ̃Aj[V폜
/*
	if(animation->isAnimationSet()){
		AnimationSet* animationSet = animation->castAnimationSet();
		int animationCount = animationSet->getAnimationCount();
		for(int i = animationCount - 1; i >= 0; i--){
			Animation* child = animationSet->getAnimation(i);
			animationSet->removeAnimation(child);
			destroy(child);
		}
	}
*/
	// f[^x[X猟č폜
	if(animationArray_.removeByValue(animation) == -1){
		ErrorOut("AnimationManager::destroy() "
			"Not found object in array");
	}
	if(animationHash_.remove(animation->getName()) == NULL){
		ErrorOut("AnimationManager::destroy() "
			"Not found object in hashmap");
	}
	delete animation;
}
//------------------------------------------------------------------------------
// Aj[Vf[^̔j
int AnimationManager::destroyData(AnimationData* animationData){
	// QƂcĂ΍폜Ȃ
	int referenceCount = animationData->getReferenceCount();
	Assert(referenceCount >= 0);
	if(referenceCount != 0){ return referenceCount; }
	// f[^x[X猟č폜
	if(animationDataArray_.removeByValue(animationData) == -1){
		ErrorOut("AnimationManager::destroyData() "
			"Not found object in array");
	}
	if(animationDataHash_.remove(animationData->getName()) == NULL){
		ErrorOut("AnimationManager::destroyData() "
			"Not found object in hashmap");
	}
	delete animationData;
	return 0;
}
//------------------------------------------------------------------------------
// NA
int AnimationManager::clear(){
	int result = 0;
	// Aj[V̍폜
	int count = getCount();
	for(int i = 0; i < count; i++){ delete get(i); }
	result += count;
	// Aj[Vf[^̍폜
	int dataCount = getDataCount();
	for(int i = 0; i < dataCount; i++){ delete getData(i); }
	result += dataCount;
	// Rei̍폜
	animationArray_.clear();
	animationHash_.clear();
	animationDataArray_.clear();
	animationDataHash_.clear();
	return result;
}
//------------------------------------------------------------------------------
// l[}
//------------------------------------------------------------------------------
// Aj[Vl[}̐ݒ
void AnimationManager::setRenamer(Renamer* renamer){
	Assert(renamer != NULL);
	SafeDelete(animationRenamer_);
	animationRenamer_ = renamer;
}
//------------------------------------------------------------------------------
// Aj[Vf[^l[}̐ݒ
void AnimationManager::setDataRenamer(Renamer* renamer){
	Assert(renamer != NULL);
	SafeDelete(animationDataRenamer_);
	animationDataRenamer_ = renamer;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
