//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * ]Ԉk
 * @author Junpee
 */

#include "LampBasic.h"
#include "Animation/RotationInterpolator/RotationInterpolationCompressor.h"
#include "Animation/RotationInterpolator/QuaternionArrayInterpolator.h"
#include "Animation/RotationInterpolator/EulerArrayInterpolator.h"
#include "Animation/RotationInterpolator/RotationConstantInterpolator.h"
#include "Animation/RotationInterpolator/QuaternionLinearInterpolator.h"
#include "Core/Container/Deque.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
RotationInterpolationCompressor::RotationInterpolationCompressor() :
	tolerance_(0.f), length_(0.f), sourceKeyCount_(0), compressedKeyCount_(0),
	compressedKeySize_(0){
}
//------------------------------------------------------------------------------
// fXgN^
RotationInterpolationCompressor::~RotationInterpolationCompressor(){
}
//------------------------------------------------------------------------------
// k
//------------------------------------------------------------------------------
// k
RotationInterpolator* RotationInterpolationCompressor::compress(
	EulerArrayInterpolator* source, float tolerance){
	// l]zԂɕϊĈk
	QuaternionArrayInterpolator* interpolator =
		source->convertQuaternionArrayInterpolator();
	RotationInterpolator* result = compress(interpolator, tolerance);
	delete interpolator;
	return result;
}
//------------------------------------------------------------------------------
// k
RotationInterpolator* RotationInterpolationCompressor::compress(
	QuaternionArrayInterpolator* source, float tolerance){
	// k
	compressSetup(source, tolerance);
	RotationInterpolator* result = NULL;

	// 萔k
	result = compressConstant(source);
	if(result != NULL){ return result; }

	// `k
	result = compressLinear(source);
	if(result != NULL){ return result; }

	// XvCn̈kƐ`ǩʂdݕtŔrďo͉\
	// ̏ꍇsetCompressedDataYȂ悤ɒ

	// SĂ̈kɎsȂ猳f[^̃Rs[Ԃ
	result = source->duplicate();
	setCompressedData(getSourceKeyCount(), sourceKeySize_);
	
	return result;
}
//------------------------------------------------------------------------------
// k
void RotationInterpolationCompressor::compressSetup(
	QuaternionArrayInterpolator* source, float tolerance){
	Assert(source != NULL); 
	Assert((tolerance >= 0.f) && (tolerance < Math::halfPI));
	tolerance_ = tolerance;
	length_ = source->getLength();
	sourceKeyCount_ = source->getSize();
	Assert(sourceKeyCount_ > 1);
}
//------------------------------------------------------------------------------
// 萔k
RotationInterpolator* RotationInterpolationCompressor::compressConstant(
	QuaternionArrayInterpolator* source){
	// ^[QbgƂȂl̎ZoAŏ̒lgp
	const Quaternion& targetValue = source->getValue(0);
	Assert(targetValue.isUnit());

	// 덷`FbN
	float toleranceCos = Math::cos(getTolerance());
	for(int i = 0; i < sourceKeyCount_; i++){
		const Quaternion& sourceValue = source->getValue(i);
		Assert(sourceValue.isUnit());
		// 萔kɎs
		if(targetValue.dotProduct(sourceValue) < toleranceCos){ return NULL; }
	}

	// 萔kɐ
	RotationConstantInterpolator* result = new RotationConstantInterpolator();
	result->setLength(getLength());
	result->setQuaternion(targetValue);
	setCompressedData(1, sizeof(Quaternion) + sizeof(float));
	return result;
}
//------------------------------------------------------------------------------
// `k
RotationInterpolator* RotationInterpolationCompressor::compressLinear(
	QuaternionArrayInterpolator* source){
	// L[̃Xg쐬A덷L[ԈĂ

	// L[Xg̍쐬
	int sourceKeyCount = getSourceKeyCount();
	Deque<LinearKey> keys_(sourceKeyCount);
	for(int i = 0; i < sourceKeyCount; i++){
		LinearKey key;
		key.value_ = source->getValue(i);
		key.time_ = (float)i;
		keys_.pushBack(key);
	}

	// S덷̎Zo
	int keyCount = keys_.getCount();
	int maxKeyCount = keyCount - 1;
	// ŏƍŌ̒l͑삵Ȃ
	keys_[0].errorCos_ = 1.f;
	keys_[maxKeyCount].errorCos_ = 1.f;
	Assert(keys_[0].time_ == 0.f)
	Assert(keys_[maxKeyCount].time_ == getLength());
	Assert(keys_[0].value_.dotProduct(keys_[1].value_) >= 0.f);
	for(int i = 1; i < maxKeyCount; i++){
		LinearKey& key = keys_[i];
		LinearKey& preKey = keys_[i - 1];
		LinearKey& postKey = keys_[i + 1];
		Assert(key.value_.dotProduct(postKey.value_) >= 0.f);
		// ÕL[ƌ̃L[̒Ԃ̒lƂ̌덷
		Quaternion value =
			Quaternion::slerp(preKey.value_, postKey.value_, 0.5f);
		key.errorCos_ = key.value_.dotProduct(value);
	}

	// k[vAL[2ɂȂ甲
	float toleranceCos = Math::cos(getTolerance());
	while(keys_.getCount() > 2){
		// 덷ŏ̃L[𒲂ׂ
		int targetIndex = 0;
		float targetErrorCos = -1.f;
		maxKeyCount = keys_.getCount() - 1;
		for(int i = 1; i < maxKeyCount; i++){
			// Cos1ɋ߂ƌ덷
			// Oƌ̃L[̓ς0ȏ
//			if(keys_[i].errorCos_ > targetErrorCos){
			if((keys_[i].errorCos_ > targetErrorCos) &&
				(keys_[i - 1].value_.dotProduct(keys_[i + 1].value_) >= 0.f)){
				targetErrorCos = keys_[i].errorCos_;
				targetIndex = i;
			}
		}

		// ^[QbgȂ
		if(targetIndex == 0){ break; }
		// ŏ덷e͈͊OȂ爳k[v𔲂
		if(targetErrorCos < toleranceCos){ break; }

		// ŏ덷L[̍폜ƌ덷̃Abvf[g
		keys_.remove(targetIndex);
		// ÕL[擪L[łȂΌ덷ČvZ
		if((targetIndex - 1) > 0){
			recalcLinearError(source, keys_[targetIndex - 2],
				keys_[targetIndex - 1], keys_[targetIndex]);
		}
		// ̃L[ŌL[łȂΌ덷ČvZ
		if(targetIndex < (keys_.getCount() - 1)){
			recalcLinearError(source, keys_[targetIndex - 1],
				keys_[targetIndex], keys_[targetIndex + 1]);
		}
	}

	// TCYȂĂȂΎsAŝ
	keyCount = keys_.getCount();
	int keySize = sizeof(Quaternion) + sizeof(float);
	if((keyCount * keySize) >= getSourceSize()){ return NULL; }

	// `kɐ
	QuaternionLinearInterpolator* result = new QuaternionLinearInterpolator();
	result->setKeyCount(keyCount);
	for(int i = 0; i < keyCount; i++){
		LinearKey& key = keys_[i];
		result->setKey(i, key.time_, key.value_);
	}
	setCompressedData(keyCount, keySize);
	return result;
}
//------------------------------------------------------------------------------
// `k덷̍ČvZ
void RotationInterpolationCompressor::recalcLinearError(
	QuaternionArrayInterpolator* source,
	LinearKey& preKey, LinearKey& key, LinearKey& postKey){
	int preTime = (int)preKey.time_;
	Quaternion preValue = preKey.value_;
	int postTime = (int)postKey.time_;
	Quaternion postValue = postKey.value_;
	int timeRange = postTime - preTime;

	// ő덷CosZo
	float maxErrorCos = 1.f;
	for(int i = 1; i < timeRange; i++){
		int time = preTime + i;
		Quaternion sourceValue = source->getValue(time);
		float rate = (float)i / (float)timeRange;
		Quaternion targetValue = Quaternion::slerp(preValue, postValue, rate);
		float errorCos = sourceValue.dotProduct(targetValue);
		if(errorCos < maxErrorCos){ maxErrorCos = errorCos; }
	}
	key.errorCos_ = maxErrorCos;
}
//------------------------------------------------------------------------------
// k
//------------------------------------------------------------------------------
// ʕ̎擾
String RotationInterpolationCompressor::getResultString() const{
	String result;
	result.format(
		"%.2f%%  Key %d/%d  Size %.2f/%.2fKb  Length %.0f",
		getCompressionRate() * 100.f,
		getCompressedKeyCount(), getSourceKeyCount(),
		(float)getCompressedSize() / 1024.f, (float)getSourceSize() / 1024.f,
		getLength());
	return result;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
