//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * LN^fAj[V
 * @author Junpee
 */

#include "LampBasic.h"
#include "Animation/Model/CharacterModelAnimation.h"
#include "Animation/System/AnimationManager.h"
#include "Graphics/Scene/Scene.h"
#include "Graphics/Model/ModelManager.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
CharacterModelAnimation::CharacterModelAnimation(
	String name, AnimationManager* manager) : ObjectAnimation(name, manager),
	animationData_(NULL), target_(NULL),
	boneCount_(0), boneNames_(NULL), targetBones_(NULL){
}
//------------------------------------------------------------------------------
// fXgN^
CharacterModelAnimation::~CharacterModelAnimation(){
	SafeArrayDelete(targetBones_);
	SafeArrayDelete(boneNames_);
}
//------------------------------------------------------------------------------
// oCh
bool CharacterModelAnimation::bind(Scene* scene){
	// ^[Qbgw肳ĂȂΎs
	if(getTargetName().getSize() == 0){ return false; }
	// LN^f̌
	Model* model = scene->getModelManager()->search(getTargetName());
	if(model == NULL){ return false; }
	CharacterModel* characterModel = model->castCharacterModel();
	if(characterModel == NULL){ return false; }
	target_ = characterModel;
	// {[̃oCh
	for(int i = 0; i < boneCount_; i++){
		Assert(boneNames_[i].getSize() > 0);
		Bone* bone = target_->searchBone(boneNames_[i]);
		Assert(bone != NULL);
		if(bone == NULL){ return false; }
		targetBones_[i] = bone;
	}
	return true;
}
//------------------------------------------------------------------------------
// oCh
bool CharacterModelAnimation::bind(CharacterModel* model){
	if(model == NULL){ return false; }
	setTargetName(model->getName());
	target_ = model;
	// {[̃oCh
	for(int i = 0; i < boneCount_; i++){
		Assert(boneNames_[i].getSize() > 0);
		Bone* bone = target_->searchBone(boneNames_[i]);
		Assert(bone != NULL);
		if(bone == NULL){ return false; }
		targetBones_[i] = bone;
	}
	return true;
}
//------------------------------------------------------------------------------
// Aj[V
bool CharacterModelAnimation::animate(float deltaTime, AnimationMask mask){
	// `FbN
	if((mask & maskCharacterModel) == 0){ return true; }
	if(!isEnabled()){ return true; } 
	if(animationData_ == NULL){ return true; }
	if(target_ == NULL){ return true; }
	// ΏۂLłȂ΃Aj[VȂA^C}CNg
	if(!target_->isGlobalEnabled()){
		setTime(getTime() + deltaTime);
		return true;
	}
	// Aj[V
	int sequence = getSequence();
	float correctTime = increasesTime(deltaTime);
	CharacterModelAnimationData* data = getCharacterModelAnimationData();
	Assert(boneCount_ == data->getBoneCount());
	for(int i = 0; i < boneCount_; i++){
		Bone* bone = targetBones_[i];
		// XP[
		VectorInterpolator* scale = data->getScale(sequence, i);
		if(scale != NULL){ bone->setScale(scale->interpolate(correctTime)); }
		// ]
		RotationInterpolator* rotation = data->getRotation(sequence, i);
		if(rotation != NULL){
			if(rotation->isQuaternionInterpolator()){
				bone->setRotationQuaternion(
					rotation->quaternionInterpolate(correctTime));
			}else if(rotation->isEulerInterpolator()){
				bone->setRotationXYZ(rotation->eulerInterpolate(correctTime));
			}else{
				ErrorOut("CharacterModelAnimation::animate() "
					"Unsupported Rotation type");
			}
		}
		// ړ
		VectorInterpolator* translation = data->getTranslation(sequence, i);
		if(translation != NULL){
			bone->setTranslation(translation->interpolate(correctTime));
		}
	}
	return isFinished();
}
//------------------------------------------------------------------------------
// LN^fAj[ṼRs[
CharacterModelAnimation* CharacterModelAnimation::copyCharacterModelAnimation(
	DataCopyMask dataCopyMask) const{
	CharacterModelAnimation* animation =
		getManager()->createCharacterModel(getName());
	copyObjectAnimationValue(animation);
	// {[̃Rs[
	animation->setBoneCount(boneCount_);
	for(int i = 0; i < boneCount_; i++){
		animation->setBoneName(i, getBoneName(i));
	}
	// Aj[Vf[^̃Rs[
	if((dataCopyMask & copyCharacterModel) == copyCharacterModel){
		animation->animationData_ =
			animationData_->copyCharacterModelAnimationData();
	}else{
		animation->animationData_ = animationData_;
	}
	animation->animationData_->addReference();
	animation->bind(target_);
	return animation;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
