//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * eLXgAj[VZ[o
 * @author Junpee
 */

#include "LampBasic.h"
#include "Animation/InputOutput/TextAnimationSaver.h"
#include "Core/InputOutput/TextFileWriter.h"

#include "Animation/VectorInterpolator/VectorConstantInterpolator.h"
#include "Animation/VectorInterpolator/VectorArrayInterpolator.h"
#include "Animation/VectorInterpolator/VectorLinearInterpolator.h"
#include "Animation/RotationInterpolator/RotationConstantInterpolator.h"
#include "Animation/RotationInterpolator/EulerArrayInterpolator.h"
#include "Animation/RotationInterpolator/QuaternionArrayInterpolator.h"
#include "Animation/RotationInterpolator/QuaternionLinearInterpolator.h"

#include "Animation/System/AnimationManager.h"
#include "Animation/System/AnimationSet.h"
#include "Animation/Camera/CameraAnimation.h"
#include "Animation/SceneNode/SceneNodeAnimation.h"
#include "Animation/Model/CharacterModelAnimation.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
TextAnimationSaver::TextAnimationSaver(){
}
//------------------------------------------------------------------------------
// fXgN^
TextAnimationSaver::~TextAnimationSaver(){
}
//------------------------------------------------------------------------------
// Z[u
void TextAnimationSaver::save(
	const String& filePath, AnimationManager* manager){
	TextFileWriter* textFileWriter = new TextFileWriter(filePath);
	save(textFileWriter, manager);
	delete textFileWriter;
}
//------------------------------------------------------------------------------
// Z[u
void TextAnimationSaver::save(
	TextWriter* textWriter, AnimationManager* manager){
	writer_ = textWriter;
	manager_ = manager;
	// wb_̏o
	writeHeader();
	// IuWFNg̃JEg
	if(countObjects() == 0){ return; }
	// Aj[VXg̏o
	writeAnimationList();
	// N̏o
	writeLink();
	writeLineComment();
}
//------------------------------------------------------------------------------
// wb_̏o
void TextAnimationSaver::writeHeader(){
	writeBlockComment("Header");
	writer_->writeText("Header {\n");
	writer_->writeText("\ttype LampTextAnimationFormat\n");
	writer_->writeText("\tversion 0_9_0\n");
	writer_->writeText("}\n\n");
}
//------------------------------------------------------------------------------
// IuWFNg̃JEg
int TextAnimationSaver::countObjects(){
	// Aj[VJEgNA
	animationCount_ = manager_->getCount();
	animationSetCount_ = 0;
	cameraCount_ = 0;
	sceneNodeCount_ = 0;
	characterModelCount_ = 0;
	// eAj[ṼJEg
	for(int i = 0; i < animationCount_; i++){
		Animation* animation = manager_->get(i);
		if(animation->isAnimationSet()){
			animationSetCount_++;
		}else if(animation->isCameraAnimation()){
			cameraCount_++;
		}else if(animation->isSceneNodeAnimation()){
			sceneNodeCount_++;
		}else if(animation->isCharacterModelAnimation()){
			characterModelCount_++;
		}else{
			ErrorOut("TextAnimationSaver::countObjects() "
				"Unsupported animation type");
		}
	}
	// Aj[Vf[^JEgNA
	animationDataCount_ = manager_->getDataCount();
	cameraDataCount_ = 0;
	sceneNodeDataCount_ = 0;
	characterModelDataCount_ = 0;
	// eAj[Vf[^̃JEg
	for(int i = 0; i < animationDataCount_; i++){
		AnimationData* data = manager_->getData(i);
		if(data->isCameraAnimationData()){
			cameraDataCount_++;
		}else if(data->isSceneNodeAnimationData()){
			sceneNodeDataCount_++;
		}else if(data->isCharacterModelAnimationData()){
			characterModelDataCount_++;
		}else{
			ErrorOut("TextAnimationSaver::countObjects() "
				"Unsupported animation data type");
		}
	}
	return (animationCount_ + animationDataCount_);
}
//------------------------------------------------------------------------------
// Aj[VXg̏o
void TextAnimationSaver::writeAnimationList(){
	// Aj[VZbg̏o
	if(animationSetCount_ != 0){
		writeBlockComment("AnimationSet");
		writer_->writeText("AnimationSet {\n");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isAnimationSet()){
				writeAnimationSet(animation->castAnimationSet());
			}
		}
		writer_->writeText("}\n\n");
	}
	// JAj[V̏o
	if(cameraCount_ != 0){
		writeBlockComment("CameraAnimation");
		writer_->writeText("CameraAnimation {\n");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isCameraAnimation()){
				writeCameraAnimation(animation->castCameraAnimation());
			}
		}
		writer_->writeText("}\n\n");
	}
	// JAj[Vf[^̏o
	if(cameraDataCount_ != 0){
		writeBlockComment("CameraAnimationData");
		writer_->writeText("CameraAnimationData {\n");
		for(int i = 0; i < animationDataCount_; i++){
			AnimationData* data = manager_->getData(i);
			if(data->isCameraAnimationData()){
				writeCameraAnimationData(
					data->castCameraAnimationData());
			}
		}
		writer_->writeText("}\n\n");
	}
	// V[m[hAj[V̏o
	if(sceneNodeCount_ != 0){
		writeBlockComment("SceneNodeAnimation");
		writer_->writeText("SceneNodeAnimation {\n");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isSceneNodeAnimation()){
				writeSceneNodeAnimation(animation->castSceneNodeAnimation());
			}
		}
		writer_->writeText("}\n\n");
	}
	// V[m[hAj[Vf[^̏o
	if(sceneNodeDataCount_ != 0){
		writeBlockComment("SceneNodeAnimationData");
		writer_->writeText("SceneNodeAnimationData {\n");
		for(int i = 0; i < animationDataCount_; i++){
			AnimationData* data = manager_->getData(i);
			if(data->isSceneNodeAnimationData()){
				writeSceneNodeAnimationData(
					data->castSceneNodeAnimationData());
			}
		}
		writer_->writeText("}\n\n");
	}
	// LN^fAj[V̏o
	if(characterModelCount_ != 0){
		writeBlockComment("CharacterModelAnimation");
		writer_->writeText("CharacterModelAnimation {\n");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isCharacterModelAnimation()){
				writeCharacterModelAnimation(
					animation->castCharacterModelAnimation());
			}
		}
		writer_->writeText("}\n\n");
	}
	// LN^fAj[Vf[^̏o
	if(characterModelDataCount_ != 0){
		writeBlockComment("CharacterModelAnimationData");
		writer_->writeText("CharacterModelAnimationData {\n");
		for(int i = 0; i < animationDataCount_; i++){
			AnimationData* data = manager_->getData(i);
			if(data->isCharacterModelAnimationData()){
				writeCharacterModelAnimationData(
					data->castCharacterModelAnimationData());
			}
		}
		writer_->writeText("}\n\n");
	}
}
//------------------------------------------------------------------------------
// N̏o
void TextAnimationSaver::writeLink(){
	writeBlockComment("Link");
	// Aj[VZbgN
	if(animationSetCount_ != 0){
		writer_->writeText("AnimationSetLink {\n");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isAnimationSet()){
				writeAnimationSetLink(animation->castAnimationSet());
			}
		}
		writer_->writeText("}\n\n");
	}
	// JN
	if(cameraCount_ != 0){
		writer_->writeText("CameraLink {\n");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isCameraAnimation()){
				writeCameraAnimationLink(
					animation->castCameraAnimation());
			}
		}
		writer_->writeText("}\n\n");
	}
	// V[m[hN
	if(sceneNodeCount_ != 0){
		writer_->writeText("SceneNodeLink {\n");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isSceneNodeAnimation()){
				writeSceneNodeAnimationLink(
					animation->castSceneNodeAnimation());
			}
		}
		writer_->writeText("}\n\n");
	}
	// LN^fN
	if(characterModelCount_ != 0){
		writer_->writeText("CharacterModelLink {\n");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isCharacterModelAnimation()){
				writeCharacterModelAnimationLink(
					animation->castCharacterModelAnimation());
			}
		}
		writer_->writeText("}\n\n");
	}
}
//------------------------------------------------------------------------------
// Aj[VZbg
//------------------------------------------------------------------------------
// Aj[VZbg̏o
void TextAnimationSaver::writeAnimationSet(AnimationSet* animation){
	writer_->writeText("\t");
	writer_->writeText(animation->getName());
	writer_->writeText(" {\n");
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(animation->isEnabled());
	writer_->writeText("\n");
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// J
//------------------------------------------------------------------------------
// JAj[V̏o
void TextAnimationSaver::writeCameraAnimation(CameraAnimation* animation){
	writer_->writeText("\t");
	writer_->writeText(animation->getName());
	writer_->writeText(" {\n");
	// ^[Qbg
	writer_->writeText("\t\ttargetName ");
	writer_->writeText(animation->getTargetName());
	writer_->writeText("\n");
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(animation->isEnabled());
	writer_->writeText("\n");
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// JAj[Vf[^̏o
void TextAnimationSaver::writeCameraAnimationData(
	CameraAnimationData* data){
	writer_->writeText("\t");
	writer_->writeText(data->getName());
	writer_->writeText(" {\n");
	// V[PX
	int sequenceCount = data->getSequenceCount();
	writer_->writeFormat("\t\tsequenceCount %d\n", sequenceCount);
	// Aj[Vf[^̏o
	String tabDepth = "\t\t\t";
	for(int i = 0; i < sequenceCount; i++){
		writer_->writeFormat("\t\tsequence %d {\n", i);
		// ]
		writeRotationInterpolator(
			data->getRotation(i), "rotation", tabDepth);
		// ړ
		writeVectorInterpolator(
			data->getTranslation(i), "translation", tabDepth);
		// [vtO
		writer_->writeText("\t\t\tlooped ");
		writeBool(data->isLooped(i));
		writer_->writeText("\n\t\t}\n");
	}
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// V[m[h
//------------------------------------------------------------------------------
// V[m[hAj[V̏o
void TextAnimationSaver::writeSceneNodeAnimation(SceneNodeAnimation* animation){
	writer_->writeText("\t");
	writer_->writeText(animation->getName());
	writer_->writeText(" {\n");
	// ^[Qbg
	writer_->writeText("\t\ttargetName ");
	writer_->writeText(animation->getTargetName());
	writer_->writeText("\n");
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(animation->isEnabled());
	writer_->writeText("\n");
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// V[m[hAj[Vf[^̏o
void TextAnimationSaver::writeSceneNodeAnimationData(
	SceneNodeAnimationData* data){
	writer_->writeText("\t");
	writer_->writeText(data->getName());
	writer_->writeText(" {\n");
	// V[PX
	int sequenceCount = data->getSequenceCount();
	writer_->writeFormat("\t\tsequenceCount %d\n", sequenceCount);
	// Aj[Vf[^̏o
	String tabDepth = "\t\t\t";
	for(int i = 0; i < sequenceCount; i++){
		writer_->writeFormat("\t\tsequence %d {\n", i);
		// XP[
		writeVectorInterpolator(
			data->getScale(i), "scale", tabDepth);
		// ]
		writeRotationInterpolator(
			data->getRotation(i), "rotation", tabDepth);
		// ړ
		writeVectorInterpolator(
			data->getTranslation(i), "translation", tabDepth);
		// [vtO
		writer_->writeText("\t\t\tlooped ");
		writeBool(data->isLooped(i));
		writer_->writeText("\n\t\t}\n");
	}
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// f
//------------------------------------------------------------------------------
// LN^fAj[V̏o
void TextAnimationSaver::writeCharacterModelAnimation(
	CharacterModelAnimation* animation){
	writer_->writeText("\t");
	writer_->writeText(animation->getName());
	writer_->writeText(" {\n");
	// ^[Qbg
	writer_->writeText("\t\ttargetName ");
	writer_->writeText(animation->getTargetName());
	writer_->writeText("\n");
	// {[
	int boneCount = animation->getBoneCount();
	writer_->writeFormat("\t\tboneCount %d {\n", boneCount);
	for(int i = 0; i < boneCount; i++){
		writer_->writeText("\t\t\t");
		writer_->writeText(animation->getBoneName(i));
		writer_->writeText("\n");
	}
	writer_->writeText("\t\t}\n");
	// LA
	writer_->writeText("\t\tenabled ");
	writeBool(animation->isEnabled());
	writer_->writeText("\n");
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// LN^fAj[Vf[^̏o
void TextAnimationSaver::writeCharacterModelAnimationData(
	CharacterModelAnimationData* data){
	writer_->writeText("\t");
	writer_->writeText(data->getName());
	writer_->writeText(" {\n");
	// {[
	int boneCount = data->getBoneCount();
	writer_->writeFormat("\t\tboneCount %d\n", boneCount);
	// V[PX
	int sequenceCount = data->getSequenceCount();
	writer_->writeFormat("\t\tsequenceCount %d\n", sequenceCount);
	// Aj[Vf[^̏o
	String tabDepth = "\t\t\t\t";
	for(int i = 0; i < sequenceCount; i++){
		writer_->writeFormat("\t\tsequence %d {\n", i);
		for(int j = 0; j < boneCount; j++){
			writer_->writeFormat("\t\t\tbone %d {\n", j);
			// XP[
			writeVectorInterpolator(
				data->getScale(i, j), "scale", tabDepth);
			// ]
			writeRotationInterpolator(
				data->getRotation(i, j), "rotation", tabDepth);
			// ړ
			writeVectorInterpolator(
				data->getTranslation(i, j), "translation", tabDepth);
			writer_->writeText("\t\t\t}\n");
		}
		// [vtO
		writer_->writeText("\t\t\tlooped ");
		writeBool(data->isLooped(i));
		writer_->writeText("\n\t\t}\n");
	}
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// N
//------------------------------------------------------------------------------
// Aj[VZbgN̏o
void TextAnimationSaver::writeAnimationSetLink(AnimationSet* animation){
	int childCount = animation->getAnimationCount();
	if(childCount == 0){ return; }
	writer_->writeText("\t");
	writer_->writeText(animation->getName());
	writer_->writeText(" {\n");
	for(int i = 0; i < childCount; i++){
		Animation* child = animation->getAnimation(i);
		writer_->writeText("\t\t");
		writer_->writeText(child->getName());
		writer_->writeText("\n");
	}
	writer_->writeText("\t}\n\n");
}
//------------------------------------------------------------------------------
// JAj[VN̏o
void TextAnimationSaver::writeCameraAnimationLink(
	CameraAnimation* animation){
	writer_->writeText("\t");
	writer_->writeText(animation->getName());
	writer_->writeText(" ");
	writer_->writeText(animation->getAnimationData()->getName());
	writer_->writeText("\n");
}
//------------------------------------------------------------------------------
// V[m[hAj[VN̏o
void TextAnimationSaver::writeSceneNodeAnimationLink(
	SceneNodeAnimation* animation){
	writer_->writeText("\t");
	writer_->writeText(animation->getName());
	writer_->writeText(" ");
	writer_->writeText(animation->getAnimationData()->getName());
	writer_->writeText("\n");
}
//------------------------------------------------------------------------------
// LN^fAj[VN̏o
void TextAnimationSaver::writeCharacterModelAnimationLink(
	CharacterModelAnimation* animation){
	writer_->writeText("\t");
	writer_->writeText(animation->getName());
	writer_->writeText(" ");
	writer_->writeText(animation->getAnimationData()->getName());
	writer_->writeText("\n");
}
//------------------------------------------------------------------------------
// xNg
//------------------------------------------------------------------------------
// xNgԂ̏o
void TextAnimationSaver::writeVectorInterpolator(
	VectorInterpolator* interpolator,
	const String& name, const String& tabDepth){
	// NULLΉ
	if(interpolator == NULL){
		writer_->writeText(tabDepth);
		writer_->writeText(name);
		writer_->writeText(" NULL\n");
		return;
	}
	// eԂ̏o
	if(interpolator->isVectorConstantInterpolator()){
		writeVectorConstantInterpolator(
			interpolator->castVectorConstantInterpolator(), name, tabDepth);
	}else if(interpolator->isVectorArrayInterpolator()){
		writeVectorArrayInterpolator(
			interpolator->castVectorArrayInterpolator(), name, tabDepth);
	}else if(interpolator->isVectorLinearInterpolator()){
		writeVectorLinearInterpolator(
			interpolator->castVectorLinearInterpolator(), name, tabDepth);
	}else{
		ErrorOut("TextAnimationSaver::writeVectorInterpolator() "
			"Unsupported interpolater");
	}
}
//------------------------------------------------------------------------------
// xNg萔Ԃ̏o
void TextAnimationSaver::writeVectorConstantInterpolator(
	VectorConstantInterpolator* interpolator,
	const String& name, const String& tabDepth){
	// O
	writer_->writeText(tabDepth);
	writer_->writeText(name);
	writer_->writeText(" {\n");
	// ^Cv
	writer_->writeText(tabDepth);
	writer_->writeText("\ttype Constant\n");
	// 
	writer_->writeText(tabDepth);
	writer_->writeFormat("\tlength %.8f\n", interpolator->getLength());
	// l
	writer_->writeText(tabDepth);
	Vector3 value = interpolator->getValue();
	writer_->writeFormat("\tvalue { %.8f %.8f %.8f }\n",
		value.x, value.y, value.z);
	writer_->writeText(tabDepth);
	writer_->writeText("}\n");
}
//------------------------------------------------------------------------------
// xNgzԂ̏o
void TextAnimationSaver::writeVectorArrayInterpolator(
	VectorArrayInterpolator* interpolator,
	const String& name, const String& tabDepth){
	// O
	writer_->writeText(tabDepth);
	writer_->writeText(name);
	writer_->writeText(" {\n");
	// ^Cv
	writer_->writeText(tabDepth);
	writer_->writeText("\ttype Array\n");
	// l
	writer_->writeText(tabDepth);
	int size = interpolator->getSize();
	writer_->writeFormat("\tvalue %d {\n", size);
	String valueTab = tabDepth + "\t\t";
	for(int i = 0; i < size; i++){
		writer_->writeText(valueTab);
		const Vector3& value = interpolator->getValue(i);
		writer_->writeFormat("{ %.8f %.8f %.8f }\n",
			value.x, value.y, value.z);
	}
	writer_->writeText(tabDepth);
	writer_->writeText("\t}\n");
	writer_->writeText(tabDepth);
	writer_->writeText("}\n");
}
//------------------------------------------------------------------------------
// xNg`Ԃ̏o
void TextAnimationSaver::writeVectorLinearInterpolator(
	VectorLinearInterpolator* interpolator,
	const String& name, const String& tabDepth){
	// O
	writer_->writeText(tabDepth);
	writer_->writeText(name);
	writer_->writeText(" {\n");
	// ^Cv
	writer_->writeText(tabDepth);
	writer_->writeText("\ttype Linear\n");
	// l
	writer_->writeText(tabDepth);
	int keyCount = interpolator->getKeyCount();
	writer_->writeFormat("\tvalue %d {\n", keyCount);
	String valueTab = tabDepth + "\t\t";
	for(int i = 0; i < keyCount; i++){
		writer_->writeText(valueTab);
		float time = interpolator->getTime(i);
		const Vector3& value = interpolator->getValue(i);
		writer_->writeFormat("%.8f { %.8f %.8f %.8f }\n",
			time, value.x, value.y, value.z);
	}
	writer_->writeText(tabDepth);
	writer_->writeText("\t}\n");
	writer_->writeText(tabDepth);
	writer_->writeText("}\n");
}
//------------------------------------------------------------------------------
// ]
//------------------------------------------------------------------------------
// ]Ԃ̏o
void TextAnimationSaver::writeRotationInterpolator(
	RotationInterpolator* interpolator,
	const String& name, const String& tabDepth){
	// NULLΉ
	if(interpolator == NULL){
		writer_->writeText(tabDepth);
		writer_->writeText(name);
		writer_->writeText(" NULL\n");
		return;
	}
	// eԂ̏o
	if(interpolator->isRotationConstantInterpolator()){
		writeRotationConstantInterpolator(
			interpolator->castRotationConstantInterpolator(), name, tabDepth);
	}else if(interpolator->isEulerArrayInterpolator()){
		writeEulerArrayInterpolator(
			interpolator->castEulerArrayInterpolator(), name, tabDepth);
	}else if(interpolator->isQuaternionArrayInterpolator()){
		writeQuaternionArrayInterpolator(
			interpolator->castQuaternionArrayInterpolator(), name, tabDepth);
	}else if(interpolator->isQuaternionLinearInterpolator()){
		writeQuaternionLinearInterpolator(
			interpolator->castQuaternionLinearInterpolator(), name, tabDepth);
	}else{
		ErrorOut("TextAnimationSaver::writeRotationInterpolator() "
			"Unsupported interpolater");
	}
}
//------------------------------------------------------------------------------
// ]萔Ԃ̏o
void TextAnimationSaver::writeRotationConstantInterpolator(
	RotationConstantInterpolator* interpolator,
	const String& name, const String& tabDepth){
	// O
	writer_->writeText(tabDepth);
	writer_->writeText(name);
	writer_->writeText(" {\n");
	// ^Cv
	writer_->writeText(tabDepth);
	writer_->writeText("\ttype Constant\n");
	// 
	writer_->writeText(tabDepth);
	writer_->writeFormat("\tlength %.8f\n", interpolator->getLength());
	// l
	writer_->writeText(tabDepth);
	Quaternion value = interpolator->getQuaternion();
	writer_->writeFormat("\tquaternionValue { %.8f %.8f %.8f %.8f }\n",
		value.x, value.y, value.z, value.w);
	writer_->writeText(tabDepth);
	writer_->writeText("}\n");
}
//------------------------------------------------------------------------------
// IC[]zԂ̏o
void TextAnimationSaver::writeEulerArrayInterpolator(
	EulerArrayInterpolator* interpolator,
	const String& name, const String& tabDepth){
	// O
	writer_->writeText(tabDepth);
	writer_->writeText(name);
	writer_->writeText(" {\n");
	// ^Cv
	writer_->writeText(tabDepth);
	writer_->writeText("\ttype EulerArray\n");
	// l
	writer_->writeText(tabDepth);
	int size = interpolator->getSize();
	writer_->writeFormat("\tvalue %d {\n", size);
	String valueTab = tabDepth + "\t\t";
	for(int i = 0; i < size; i++){
		writer_->writeText(valueTab);
		const Vector3& value = interpolator->getValue(i);
		writer_->writeFormat("{ %.8f %.8f %.8f }\n",
			value.x, value.y, value.z);
	}
	writer_->writeText(tabDepth);
	writer_->writeText("\t}\n");
	writer_->writeText(tabDepth);
	writer_->writeText("}\n");
}
//------------------------------------------------------------------------------
// l]zԂ̏o
void TextAnimationSaver::writeQuaternionArrayInterpolator(
	QuaternionArrayInterpolator* interpolator,
	const String& name, const String& tabDepth){
	// O
	writer_->writeText(tabDepth);
	writer_->writeText(name);
	writer_->writeText(" {\n");
	// ^Cv
	writer_->writeText(tabDepth);
	writer_->writeText("\ttype QuaternionArray\n");
	// l
	writer_->writeText(tabDepth);
	int size = interpolator->getSize();
	writer_->writeFormat("\tvalue %d {\n", size);
	String valueTab = tabDepth + "\t\t";
	for(int i = 0; i < size; i++){
		writer_->writeText(valueTab);
		const Quaternion& value = interpolator->getValue(i);
		writer_->writeFormat("{ %.8f %.8f %.8f %.8f }\n",
			value.x, value.y, value.z, value.w);
	}
	writer_->writeText(tabDepth);
	writer_->writeText("\t}\n");
	writer_->writeText(tabDepth);
	writer_->writeText("}\n");
}
//------------------------------------------------------------------------------
// l]`Ԃ̏o
void TextAnimationSaver::writeQuaternionLinearInterpolator(
	QuaternionLinearInterpolator* interpolator,
	const String& name, const String& tabDepth){
	// O
	writer_->writeText(tabDepth);
	writer_->writeText(name);
	writer_->writeText(" {\n");
	// ^Cv
	writer_->writeText(tabDepth);
	writer_->writeText("\ttype QuaternionLinear\n");
	// l
	writer_->writeText(tabDepth);
	int keyCount = interpolator->getKeyCount();
	writer_->writeFormat("\tvalue %d {\n", keyCount);
	String valueTab = tabDepth + "\t\t";
	for(int i = 0; i < keyCount; i++){
		writer_->writeText(valueTab);
		float time = interpolator->getTime(i);
		const Quaternion& value = interpolator->getValue(i);
		writer_->writeFormat("%.8f { %.8f %.8f %.8f %.8f }\n",
			time, value.x, value.y, value.z, value.w);
	}
	writer_->writeText(tabDepth);
	writer_->writeText("\t}\n");
	writer_->writeText(tabDepth);
	writer_->writeText("}\n");
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// bool̏o
void TextAnimationSaver::writeBool(bool flag){
	if(flag){
		writer_->writeText("true");
	}else{
		writer_->writeText("false");
	}
}
//------------------------------------------------------------------------------
// Rg̏o
void TextAnimationSaver::writeLineComment(){
	writer_->writeText("//----------------------------"
		"--------------------------------------------------\n");
}
//------------------------------------------------------------------------------
// ubNRg̏o
void TextAnimationSaver::writeBlockComment(const String& blockName){
	writeLineComment();
	writer_->writeText("// ");
	writer_->writeText(blockName);
	writer_->writeText(" Block\n");
	writeLineComment();
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
