//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * oCiAj[VZ[o
 * @author Junpee
 */

#include "LampBasic.h"
#include "Animation/InputOutput/BinaryAnimationSaver.h"
#include "Core/InputOutput/BinaryFileWriter.h"
#include "Animation/VectorInterpolator/VectorConstantInterpolator.h"
#include "Animation/VectorInterpolator/VectorArrayInterpolator.h"
#include "Animation/VectorInterpolator/VectorLinearInterpolator.h"
#include "Animation/RotationInterpolator/RotationConstantInterpolator.h"
#include "Animation/RotationInterpolator/EulerArrayInterpolator.h"
#include "Animation/RotationInterpolator/QuaternionArrayInterpolator.h"
#include "Animation/RotationInterpolator/QuaternionLinearInterpolator.h"

#include "Animation/System/AnimationManager.h"
#include "Animation/System/AnimationSet.h"
#include "Animation/Camera/CameraAnimation.h"
#include "Animation/SceneNode/SceneNodeAnimation.h"
#include "Animation/Model/CharacterModelAnimation.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
BinaryAnimationSaver::BinaryAnimationSaver(){
}
//------------------------------------------------------------------------------
// fXgN^
BinaryAnimationSaver::~BinaryAnimationSaver(){
}
//------------------------------------------------------------------------------
// Z[u
//------------------------------------------------------------------------------
// Z[u
void BinaryAnimationSaver::save(
	const String& filePath, AnimationManager* manager){
	BinaryFileWriter* binaryFileWriter = new BinaryFileWriter(filePath);
	save(binaryFileWriter, manager);
	delete binaryFileWriter;
}
//------------------------------------------------------------------------------
// Z[u
void BinaryAnimationSaver::save(
	BinaryWriter* binaryWriter, AnimationManager* manager){
	Assert((binaryWriter != NULL) && (manager != NULL))
	writer_ = binaryWriter;
	manager_ = manager;

	// wb_̏o
	writeHeader();
	// IuWFNg̃JEg
	if(countObjects() == 0){ return; }

	// Aj[VXg̏o
	writeAnimationList();
	// N̏o
	writeLink();
}
//------------------------------------------------------------------------------
// wb_̏o
void BinaryAnimationSaver::writeHeader(){
	writeString("LampBinaryAnimation");
	u_int version = 0x00000900;
	writer_->writeUInt(version);
	// ACg
	align();
}
//------------------------------------------------------------------------------
// IuWFNg̃JEg
int BinaryAnimationSaver::countObjects(){
	// Aj[VJEgNA
	clearObjectCount();
	// eAj[ṼJEg
	animationCount_ = manager_->getCount();
	for(int i = 0; i < animationCount_; i++){
		Animation* animation = manager_->get(i);
		if(animation->isAnimationSet()){
			animationSetCount_++;
		}else if(animation->isCameraAnimation()){
			cameraCount_++;
		}else if(animation->isSceneNodeAnimation()){
			sceneNodeCount_++;
		}else if(animation->isCharacterModelAnimation()){
			characterModelCount_++;
		}else{
			ErrorOut("BinaryAnimationSaver::countObjects() "
				"Unsupported animation type");
		}
	}
	// eAj[Vf[^̃JEg
	animationDataCount_ = manager_->getDataCount();
	for(int i = 0; i < animationDataCount_; i++){
		AnimationData* data = manager_->getData(i);
		if(data->isCameraAnimationData()){
			cameraDataCount_++;
		}else if(data->isSceneNodeAnimationData()){
			sceneNodeDataCount_++;
		}else if(data->isCharacterModelAnimationData()){
			characterModelDataCount_++;
		}else{
			ErrorOut("BinaryAnimationSaver::countObjects() "
				"Unsupported animation data type");
		}
	}
	return (animationCount_ + animationDataCount_);
}
//------------------------------------------------------------------------------
// Aj[VXg
//------------------------------------------------------------------------------
// Aj[VXg̏o
void BinaryAnimationSaver::writeAnimationList(){
	// Aj[VZbg̏o
	if(animationSetCount_ != 0){
		int startAddress = startBlock("AnimationSet");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isAnimationSet()){
				writeAnimationSet(animation->castAnimationSet());
			}
		}
		endBlock(startAddress, animationSetCount_);
	}
	// JAj[V̏o
	if(cameraCount_ != 0){
		int startAddress = startBlock("Camera");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isCameraAnimation()){
				writeCameraAnimation(animation->castCameraAnimation());
			}
		}
		endBlock(startAddress, cameraCount_);
	}
	// JAj[Vf[^̏o
	if(cameraDataCount_ != 0){
		int startAddress = startBlock("CameraData");
		for(int i = 0; i < animationDataCount_; i++){
			AnimationData* data = manager_->getData(i);
			if(data->isCameraAnimationData()){
				writeCameraAnimationData(
					data->castCameraAnimationData());
			}
		}
		endBlock(startAddress, cameraDataCount_);
	}
	// V[m[hAj[V̏o
	if(sceneNodeCount_ != 0){
		int startAddress = startBlock("SceneNode");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isSceneNodeAnimation()){
				writeSceneNodeAnimation(animation->castSceneNodeAnimation());
			}
		}
		endBlock(startAddress, sceneNodeCount_);
	}
	// V[m[hAj[Vf[^̏o
	if(sceneNodeDataCount_ != 0){
		int startAddress = startBlock("SceneNodeData");
		for(int i = 0; i < animationDataCount_; i++){
			AnimationData* data = manager_->getData(i);
			if(data->isSceneNodeAnimationData()){
				writeSceneNodeAnimationData(
					data->castSceneNodeAnimationData());
			}
		}
		endBlock(startAddress, sceneNodeDataCount_);
	}
	// LN^fAj[V̏o
	if(characterModelCount_ != 0){
		int startAddress = startBlock("CharacterModel");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isCharacterModelAnimation()){
				writeCharacterModelAnimation(
					animation->castCharacterModelAnimation());
			}
		}
		endBlock(startAddress, characterModelCount_);
	}
	// LN^fAj[Vf[^̏o
	if(characterModelDataCount_ != 0){
		int startAddress = startBlock("CharacterModelData");
		for(int i = 0; i < animationDataCount_; i++){
			AnimationData* data = manager_->getData(i);
			if(data->isCharacterModelAnimationData()){
				writeCharacterModelAnimationData(
					data->castCharacterModelAnimationData());
			}
		}
		endBlock(startAddress, characterModelDataCount_);
	}
}
//------------------------------------------------------------------------------
// Aj[VZbg
//------------------------------------------------------------------------------
// Aj[VZbg̏o
void BinaryAnimationSaver::writeAnimationSet(AnimationSet* animation){
	// O
	writeString(animation->getName());
	// LA
	writer_->writeBool(animation->isEnabled());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// J
//------------------------------------------------------------------------------
// JAj[V̏o
void BinaryAnimationSaver::writeCameraAnimation(CameraAnimation* animation){
	// O
	writeString(animation->getName());
	// LA
	writer_->writeBool(animation->isEnabled());
	// ^[Qbg
	writeString(animation->getTargetName());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// JAj[Vf[^̏o
void BinaryAnimationSaver::writeCameraAnimationData(
	CameraAnimationData* data){
	// O
	writeString(data->getName());
	// V[PX
	int sequenceCount = data->getSequenceCount();
	writer_->writeInt(sequenceCount);
	// Aj[Vf[^
	for(int i = 0; i < sequenceCount; i++){
		// ]
		writeRotationInterpolator(data->getRotation(i));
		// ړ
		writeVectorInterpolator(data->getTranslation(i));
	}
	// [vtO
	for(int i = 0; i < sequenceCount; i++){
		writer_->writeBool(data->isLooped(i));
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// V[m[h
//------------------------------------------------------------------------------
// V[m[hAj[V̏o
void BinaryAnimationSaver::writeSceneNodeAnimation(
	SceneNodeAnimation* animation){
	// O
	writeString(animation->getName());
	// LA
	writer_->writeBool(animation->isEnabled());
	// ^[Qbg
	writeString(animation->getTargetName());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// V[m[hAj[Vf[^̏o
void BinaryAnimationSaver::writeSceneNodeAnimationData(
	SceneNodeAnimationData* data){
	// O
	writeString(data->getName());
	// V[PX
	int sequenceCount = data->getSequenceCount();
	writer_->writeInt(sequenceCount);
	// Aj[Vf[^
	for(int i = 0; i < sequenceCount; i++){
		// XP[
		writeVectorInterpolator(data->getScale(i));
		// ]
		writeRotationInterpolator(data->getRotation(i));
		// ړ
		writeVectorInterpolator(data->getTranslation(i));
	}
	// [vtO
	for(int i = 0; i < sequenceCount; i++){
		writer_->writeBool(data->isLooped(i));
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// LN^f
//------------------------------------------------------------------------------
// LN^fAj[V̏o
void BinaryAnimationSaver::writeCharacterModelAnimation(
	CharacterModelAnimation* animation){
	// O
	writeString(animation->getName());
	// LA
	writer_->writeBool(animation->isEnabled());
	// ^[Qbg
	writeString(animation->getTargetName());
	// {[
	int boneCount = animation->getBoneCount();
	writer_->writeInt(boneCount);
	// {[
	for(int i = 0; i < boneCount; i++){
		writeString(animation->getBoneName(i));
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// LN^fAj[Vf[^̏o
void BinaryAnimationSaver::writeCharacterModelAnimationData(
	CharacterModelAnimationData* data){
	// O
	writeString(data->getName());
	// V[PX
	int sequenceCount = data->getSequenceCount();
	writer_->writeInt(sequenceCount);
	// {[
	int boneCount = data->getBoneCount();
	writer_->writeInt(boneCount);
	// Aj[Vf[^
	for(int i = 0; i < sequenceCount; i++){
		for(int j = 0; j < boneCount; j++){
			// XP[
			writeVectorInterpolator(data->getScale(i, j));
			// ]
			writeRotationInterpolator(data->getRotation(i, j));
			// ړ
			writeVectorInterpolator(data->getTranslation(i, j));
		}
	}
	// [vtO
	for(int i = 0; i < sequenceCount; i++){
		writer_->writeBool(data->isLooped(i));
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// N
//------------------------------------------------------------------------------
// N̏o
void BinaryAnimationSaver::writeLink(){
	// Aj[VZbgN
	if(animationSetCount_ != 0){
		int startAddress = startBlock("AnimationSetLink");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isAnimationSet()){
				writeAnimationSetLink(animation->castAnimationSet());
			}
		}
		endBlock(startAddress, animationSetCount_);
	}
	// JN
	if(cameraCount_ != 0){
		int startAddress = startBlock("CameraLink");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isCameraAnimation()){
				writeCameraAnimationLink(
					animation->castCameraAnimation());
			}
		}
		endBlock(startAddress, cameraCount_);
	}
	// V[m[hN
	if(sceneNodeCount_ != 0){
		int startAddress = startBlock("SceneNodeLink");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isSceneNodeAnimation()){
				writeSceneNodeAnimationLink(
					animation->castSceneNodeAnimation());
			}
		}
		endBlock(startAddress, sceneNodeCount_);
	}
	// LN^fN
	if(characterModelCount_ != 0){
		int startAddress = startBlock("CharacterModelLink");
		for(int i = 0; i < animationCount_; i++){
			Animation* animation = manager_->get(i);
			if(animation->isCharacterModelAnimation()){
				writeCharacterModelAnimationLink(
					animation->castCharacterModelAnimation());
			}
		}
		endBlock(startAddress, characterModelCount_);
	}
}
//------------------------------------------------------------------------------
// Aj[VZbgN̏o
void BinaryAnimationSaver::writeAnimationSetLink(AnimationSet* animation){
	// O
	writeString(animation->getName());
	// q
	int childCount = animation->getAnimationCount();
	writer_->writeInt(childCount);
	for(int i = 0; i < childCount; i++){
		writeString(animation->getAnimation(i)->getName());
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// JAj[VN̏o
void BinaryAnimationSaver::writeCameraAnimationLink(
	CameraAnimation* animation){
	// O
	writeString(animation->getName());
	// f[^̖O
	writeString(animation->getAnimationData()->getName());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// V[m[hAj[VN̏o
void BinaryAnimationSaver::writeSceneNodeAnimationLink(
	SceneNodeAnimation* animation){
	// O
	writeString(animation->getName());
	// f[^̖O
	writeString(animation->getAnimationData()->getName());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// LN^fAj[VN̏o
void BinaryAnimationSaver::writeCharacterModelAnimationLink(
	CharacterModelAnimation* animation){
	// O
	writeString(animation->getName());
	// f[^̖O
	writeString(animation->getAnimationData()->getName());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// l̏o
//------------------------------------------------------------------------------
// ̏o
void BinaryAnimationSaver::writeString(const String& string){
	int writeSize = string.getSize() + 1;
	writer_->writeInt(writeSize);
	writer_->writeBytes(string.getBytes(), writeSize);
	align();
}
//------------------------------------------------------------------------------
// OxNg̏o
void BinaryAnimationSaver::writeVector3(const Vector3& vector){
	writer_->writeBytes(vector.array, sizeof(Vector3));
}
//------------------------------------------------------------------------------
// l̏o
void BinaryAnimationSaver::writeQuaternion(const Quaternion& quaternion){
	writer_->writeBytes(quaternion.array, sizeof(Quaternion));
}
//------------------------------------------------------------------------------
// xNg
//------------------------------------------------------------------------------
// xNgԂ̏o
void BinaryAnimationSaver::writeVectorInterpolator(
	VectorInterpolator* interpolator){
	// NULLΉ
	if(interpolator == NULL){
		// NULLΉ
		writer_->writeInt(0);
	}else if(interpolator->isVectorConstantInterpolator()){
		writer_->writeInt(1);
		writeVectorConstantInterpolator(
			interpolator->castVectorConstantInterpolator());
	}else if(interpolator->isVectorArrayInterpolator()){
		writer_->writeInt(2);
		writeVectorArrayInterpolator(
			interpolator->castVectorArrayInterpolator());
	}else if(interpolator->isVectorLinearInterpolator()){
		writer_->writeInt(3);
		writeVectorLinearInterpolator(
			interpolator->castVectorLinearInterpolator());
	}else{
		ErrorOut("BinaryAnimationSaver::writeVectorInterpolator() "
			"Unsupported interpolater");
	}
}
//------------------------------------------------------------------------------
// xNg萔Ԃ̏o
void BinaryAnimationSaver::writeVectorConstantInterpolator(
	VectorConstantInterpolator* interpolator){
	// 
	writer_->writeFloat(interpolator->getLength());
	// l
	writeVector3(interpolator->getValue());
}
//------------------------------------------------------------------------------
// xNgzԂ̏o
void BinaryAnimationSaver::writeVectorArrayInterpolator(
	VectorArrayInterpolator* interpolator){
	// l
	int size = interpolator->getSize();
	writer_->writeInt(size);
	for(int i = 0; i < size; i++){ writeVector3(interpolator->getValue(i)); }
}
//------------------------------------------------------------------------------
// xNg`Ԃ̏o
void BinaryAnimationSaver::writeVectorLinearInterpolator(
	VectorLinearInterpolator* interpolator){
	// l
	int keyCount = interpolator->getKeyCount();
	writer_->writeInt(keyCount);
	for(int i = 0; i < keyCount; i++){
		writer_->writeFloat(interpolator->getTime(i));
		writeVector3(interpolator->getValue(i));
	}
}
//------------------------------------------------------------------------------
// ]
//------------------------------------------------------------------------------
// ]Ԃ̏o
void BinaryAnimationSaver::writeRotationInterpolator(
	RotationInterpolator* interpolator){
	if(interpolator == NULL){
		// NULLΉ
		writer_->writeInt(0);
	}else if(interpolator->isRotationConstantInterpolator()){
		writer_->writeInt(1);
		writeRotationConstantInterpolator(
			interpolator->castRotationConstantInterpolator());
	}else if(interpolator->isEulerArrayInterpolator()){
		writer_->writeInt(2);
		writeEulerArrayInterpolator(
			interpolator->castEulerArrayInterpolator());
	}else if(interpolator->isQuaternionArrayInterpolator()){
		writer_->writeInt(3);
		writeQuaternionArrayInterpolator(
			interpolator->castQuaternionArrayInterpolator());
	}else if(interpolator->isQuaternionLinearInterpolator()){
		writer_->writeInt(4);
		writeQuaternionLinearInterpolator(
			interpolator->castQuaternionLinearInterpolator());
	}else{
		ErrorOut("BinaryAnimationSaver::writeRotationInterpolator() "
			"Unsupported interpolater");
	}
}
//------------------------------------------------------------------------------
// ]萔Ԃ̏o
void BinaryAnimationSaver::writeRotationConstantInterpolator(
	RotationConstantInterpolator* interpolator){
	// 
	writer_->writeFloat(interpolator->getLength());
	// l
	writeQuaternion(interpolator->getQuaternion());
}
//------------------------------------------------------------------------------
// IC[]zԂ̏o
void BinaryAnimationSaver::writeEulerArrayInterpolator(
	EulerArrayInterpolator* interpolator){
	// l
	int size = interpolator->getSize();
	writer_->writeInt(size);
	for(int i = 0; i < size; i++){ writeVector3(interpolator->getValue(i)); }
}
//------------------------------------------------------------------------------
// l]zԂ̏o
void BinaryAnimationSaver::writeQuaternionArrayInterpolator(
	QuaternionArrayInterpolator* interpolator){
	// l
	int size = interpolator->getSize();
	writer_->writeInt(size);
	for(int i = 0; i < size; i++){ writeQuaternion(interpolator->getValue(i)); }
}
//------------------------------------------------------------------------------
// l]`Ԃ̏o
void BinaryAnimationSaver::writeQuaternionLinearInterpolator(
	QuaternionLinearInterpolator* interpolator){
	// l
	int keyCount = interpolator->getKeyCount();
	writer_->writeInt(keyCount);
	for(int i = 0; i < keyCount; i++){
		writer_->writeFloat(interpolator->getTime(i));
		writeQuaternion(interpolator->getValue(i));
	}
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// ACg
void BinaryAnimationSaver::align(){
	// ύXꍇendBlock()ɏCKv
	writer_->align(16);
}
//------------------------------------------------------------------------------
// ubN̊Jn
int BinaryAnimationSaver::startBlock(const String& blockName){
	writeString(blockName);
	// ubNTCY̗\
	int blockAddress = writer_->getPosition();
	writer_->writeInt(0);
	// IuWFNg̗\
	writer_->writeInt(0);
	align();
	return blockAddress;
}
//------------------------------------------------------------------------------
// ubN̏I
void BinaryAnimationSaver::endBlock(int blockStartAddress, int objectCount){
	align();
	int blockEndPosition = writer_->getPosition();
	// ubNwb_eubNTCYAACgˑ
	int blockSize = blockEndPosition - blockStartAddress - (sizeof(int) * 4);
	writer_->setPosition(blockStartAddress);
	// ubNTCY
	writer_->writeInt(blockSize);
	// IuWFNg
	writer_->writeInt(objectCount);
	// J[\߂
	writer_->setPosition(blockEndPosition);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
