//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * oCiAj[V[_
 * @author Junpee
 */

#include "LampBasic.h"
#include "Animation/InputOutput/BinaryAnimationLoader.h"
#include "Core/InputOutput/BinaryFileReader.h"
#include "Core/InputOutput/FilePath.h"
#include "Animation/VectorInterpolator/VectorConstantInterpolator.h"
#include "Animation/VectorInterpolator/VectorArrayInterpolator.h"
#include "Animation/VectorInterpolator/VectorLinearInterpolator.h"
#include "Animation/RotationInterpolator/RotationConstantInterpolator.h"
#include "Animation/RotationInterpolator/EulerArrayInterpolator.h"
#include "Animation/RotationInterpolator/QuaternionArrayInterpolator.h"
#include "Animation/RotationInterpolator/QuaternionLinearInterpolator.h"

#include "Animation/System/AnimationManager.h"
#include "Animation/System/AnimationSet.h"
#include "Animation/Camera/CameraAnimation.h"
#include "Animation/SceneNode/SceneNodeAnimation.h"
#include "Animation/Model/CharacterModelAnimation.h"

namespace Lamp{

//------------------------------------------------------------------------------
// RXgN^
BinaryAnimationLoader::BinaryAnimationLoader(){
}
//------------------------------------------------------------------------------
// fXgN^
BinaryAnimationLoader::~BinaryAnimationLoader(){
}
//------------------------------------------------------------------------------
// [h
//------------------------------------------------------------------------------
// [h
void BinaryAnimationLoader::load(
	const String& filePath, AnimationManager* manager){
	FilePath path(filePath);
	if(!path.existFile()){
		ErrorOut("BinaryAnimationLoader::load() File not Found");
	}
	BinaryFileReader* binaryFileReader = new BinaryFileReader(filePath);
	load(binaryFileReader, manager);
	delete binaryFileReader;
}
//------------------------------------------------------------------------------
// [h
void BinaryAnimationLoader::load(
	BinaryReader* binaryReader, AnimationManager* manager){
	reader_ = binaryReader;
	manager_ = manager;

	// wb_̓ǂݍ
	readHeader();

	// eubN̓ǂݍ
	while(true){
		if(reader_->isEnd()){ break; }
		// ubNwb_ǂݍ
		String blockName = readString();
		int blockSize = reader_->readInt();
		int objectCount = reader_->readInt();
		align();
		// ubNe̓ǂݍ
		int blockAddress = reader_->getPosition();
		if(blockName == "AnimationSet"){
			for(int i = 0; i < objectCount; i++){
				readAnimationSet();
			}
		}else if(blockName == "Camera"){
			for(int i = 0; i < objectCount; i++){
				readCameraAnimation();
			}
		}else if(blockName == "CameraData"){
			for(int i = 0; i < objectCount; i++){
				readCameraAnimationData();
			}
		}else if(blockName == "SceneNode"){
			for(int i = 0; i < objectCount; i++){
				readSceneNodeAnimation();
			}
		}else if(blockName == "SceneNodeData"){
			for(int i = 0; i < objectCount; i++){
				readSceneNodeAnimationData();
			}
		}else if(blockName == "CharacterModel"){
			for(int i = 0; i < objectCount; i++){
				readCharacterModelAnimation();
			}
		}else if(blockName == "CharacterModelData"){
			for(int i = 0; i < objectCount; i++){
				readCharacterModelAnimationData();
			}
		}else if(blockName == "AnimationSetLink"){
			for(int i = 0; i < objectCount; i++){
				readAnimationSetLink();
			}
		}else if(blockName == "CameraLink"){
			for(int i = 0; i < objectCount; i++){
				readCameraAnimationLink();
			}
		}else if(blockName == "SceneNodeLink"){
			for(int i = 0; i < objectCount; i++){
				readSceneNodeAnimationLink();
			}
		}else if(blockName == "CharacterModelLink"){
			for(int i = 0; i < objectCount; i++){
				readCharacterModelAnimationLink();
			}
		}else{
			// mȂubNǂݔ΂
			DebugOut("BinaryAnimationLoader::load() "
				"Skip unknown block %s (%d)\n",
				blockName.getBytes(), reader_->getPosition());
			reader_->skip(blockSize);
		}
		// SẴubN̓ACgƂ
		align();
		Assert((reader_->getPosition() - blockAddress) == blockSize);
	}
}
//------------------------------------------------------------------------------
// wb_̓ǂݍ
void BinaryAnimationLoader::readHeader(){
	String header = readString();
	if(header != "LampBinaryAnimation"){
		ErrorOut("BinaryAnimationLoader::readHeader() Invalid header");
	}
	u_int version = reader_->readUInt();
	if(version != 0x00000900){
		ErrorOut("BinaryAnimationLoader::readHeader() Invalid version");
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// Aj[VZbg
//------------------------------------------------------------------------------
// Aj[VZbg̓ǂݍ
void BinaryAnimationLoader::readAnimationSet(){
	// O
	String name = readString();
	AnimationSet* animation = manager_->createAnimationSet(name);
	// LA
	animation->setEnabled(reader_->readBool());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// J
//------------------------------------------------------------------------------
// JAj[V̓ǂݍ
void BinaryAnimationLoader::readCameraAnimation(){
	// O
	String name = readString();
	CameraAnimation* animation = manager_->createCamera(name);
	// LA
	animation->setEnabled(reader_->readBool());
	// ^[Qbg
	animation->setTargetName(readString());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// JAj[Vf[^̓ǂݍ
void BinaryAnimationLoader::readCameraAnimationData(){
	// O
	String name = readString();
	CameraAnimationData* data = manager_->createCameraData(name);
	// V[PX
	int sequenceCount = reader_->readInt();
	data->setSequenceCount(sequenceCount);
	// Aj[Vf[^
	for(int i = 0; i < sequenceCount; i++){
		// ]
		data->setRotation(i, readRotationInterpolator());
		// ړ
		data->setTranslation(i, readVectorInterpolator());
	}
	// [vtO
	for(int i = 0; i < sequenceCount; i++){
		data->setLooped(i, reader_->readBool());
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// V[m[h
//------------------------------------------------------------------------------
// V[m[hAj[V̓ǂݍ
void BinaryAnimationLoader::readSceneNodeAnimation(){
	// O
	String name = readString();
	SceneNodeAnimation* animation = manager_->createSceneNode(name);
	// LA
	animation->setEnabled(reader_->readBool());
	// ^[Qbg
	animation->setTargetName(readString());
	// ACg
	align();
}
//------------------------------------------------------------------------------
// V[m[hAj[Vf[^̓ǂݍ
void BinaryAnimationLoader::readSceneNodeAnimationData(){
	// O
	String name = readString();
	SceneNodeAnimationData* data = manager_->createSceneNodeData(name);
	// V[PX
	int sequenceCount = reader_->readInt();
	data->setSequenceCount(sequenceCount);
	// Aj[Vf[^
	for(int i = 0; i < sequenceCount; i++){
		// XP[
		data->setScale(i, readVectorInterpolator());
		// ]
		data->setRotation(i, readRotationInterpolator());
		// ړ
		data->setTranslation(i, readVectorInterpolator());
	}
	// [vtO
	for(int i = 0; i < sequenceCount; i++){
		data->setLooped(i, reader_->readBool());
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// LN^f
//------------------------------------------------------------------------------
// LN^fAj[V̓ǂݍ
void BinaryAnimationLoader::readCharacterModelAnimation(){
	// O
	String name = readString();
	CharacterModelAnimation* animation = manager_->createCharacterModel(name);
	// LA
	animation->setEnabled(reader_->readBool());
	// ^[Qbg
	animation->setTargetName(readString());
	// {[
	int boneCount = reader_->readInt();
	animation->setBoneCount(boneCount);
	// {[
	for(int i = 0; i < boneCount; i++){
		animation->setBoneName(i, readString());
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// LN^fAj[Vf[^̓ǂݍ
void BinaryAnimationLoader::readCharacterModelAnimationData(){
	// O
	String name = readString();
	CharacterModelAnimationData* data =
		manager_->createCharacterModelData(name);
	// V[PX
	int sequenceCount = reader_->readInt();
	data->setSequenceCount(sequenceCount);
	// {[
	int boneCount = reader_->readInt();
	data->setBoneCount(boneCount);
	// Aj[Vf[^
	for(int i = 0; i < sequenceCount; i++){
		for(int j = 0; j < boneCount; j++){
			// XP[
			data->setScale(i, j, readVectorInterpolator());
			// ]
			data->setRotation(i, j, readRotationInterpolator());
			// ړ
			data->setTranslation(i, j, readVectorInterpolator());
		}
	}
	// [vtO
	for(int i = 0; i < sequenceCount; i++){
		data->setLooped(i, reader_->readBool());
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// N
//------------------------------------------------------------------------------
// Aj[VZbgN̓ǂݍ
void BinaryAnimationLoader::readAnimationSetLink(){
	// Aj[V
	String animationName = readString();
	AnimationSet* animation = NULL;
	Animation* searchResult = manager_->search(animationName);
	if(searchResult != NULL){ animation = searchResult->castAnimationSet(); }
	if(animation == NULL){
		ErrorOut("BinaryAnimationLoader::readAnimationSetLink() "
			"AnimationSet not found %s (%d)",
			animationName.getBytes(), reader_->getPosition());
	}
	// q
	int childCount = reader_->readInt();
	for(int i = 0; i < childCount; i++){
		String childName = readString();
		Animation* child = manager_->search(childName);
		if(child == NULL){
			ErrorOut("BinaryAnimationLoader::readAnimationSetLink() "
				"Child animation not found %s (%d)",
				childName.getBytes(), reader_->getPosition());
		}
		animation->addAnimation(child);
	}
	// ACg
	align();
}
//------------------------------------------------------------------------------
// JAj[VN̓ǂݍ
void BinaryAnimationLoader::readCameraAnimationLink(){
	// Aj[V
	String animationName = readString();
	CameraAnimation* animation = NULL;
	Animation* animationSearchResult = manager_->search(animationName);
	if(animationSearchResult != NULL){
		animation = animationSearchResult->castCameraAnimation();
	}
	if(animation == NULL){
		ErrorOut("BinaryAnimationLoader::readCameraAnimationLink() "
			"CameraAnimation not found %s (%d)",
			animationName.getBytes(), reader_->getPosition());
	}
	// f[^
	CameraAnimationData* data = NULL;
	String dataName = readString();
	AnimationData* dataSearchResult = manager_->searchData(dataName);
	if(dataSearchResult != NULL){
		data = dataSearchResult->castCameraAnimationData();
	}
	if(data == NULL){
		ErrorOut("BinaryAnimationLoader::readCameraAnimationLink() "
			"CameraAnimationData not found %s (%d)",
			dataName.getBytes(), reader_->getPosition());
	}
	// N
	animation->setCameraAnimationData(data);
	// ACg
	align();
}
//------------------------------------------------------------------------------
// V[m[hAj[VN̓ǂݍ
void BinaryAnimationLoader::readSceneNodeAnimationLink(){
	// Aj[V
	String animationName = readString();
	SceneNodeAnimation* animation = NULL;
	Animation* animationSearchResult = manager_->search(animationName);
	if(animationSearchResult != NULL){
		animation = animationSearchResult->castSceneNodeAnimation();
	}
	if(animation == NULL){
		ErrorOut("BinaryAnimationLoader::readSceneNodeAnimationLink() "
			"SceneNodeAnimation not found %s (%d)",
			animationName.getBytes(), reader_->getPosition());
	}
	// f[^
	SceneNodeAnimationData* data = NULL;
	String dataName = readString();
	AnimationData* dataSearchResult = manager_->searchData(dataName);
	if(dataSearchResult != NULL){
		data = dataSearchResult->castSceneNodeAnimationData();
	}
	if(data == NULL){
		ErrorOut("BinaryAnimationLoader::readSceneNodeAnimationLink() "
			"SceneNodeAnimationData not found %s (%d)",
			dataName.getBytes(), reader_->getPosition());
	}
	// N
	animation->setSceneNodeAnimationData(data);
	// ACg
	align();
}
//------------------------------------------------------------------------------
// LN^fAj[VN̓ǂݍ
void BinaryAnimationLoader::readCharacterModelAnimationLink(){
	// Aj[V
	String animationName = readString();
	CharacterModelAnimation* animation = NULL;
	Animation* animationSearchResult = manager_->search(animationName);
	if(animationSearchResult != NULL){
		animation = animationSearchResult->castCharacterModelAnimation();
	}
	if(animation == NULL){
		ErrorOut("BinaryAnimationLoader::readCharacterModelAnimationLink() "
			"CharacterModelAnimation not found %s (%d)",
			animationName.getBytes(), reader_->getPosition());
	}
	// f[^
	CharacterModelAnimationData* data = NULL;
	String dataName = readString();
	AnimationData* dataSearchResult = manager_->searchData(dataName);
	if(dataSearchResult != NULL){
		data = dataSearchResult->castCharacterModelAnimationData();
	}
	if(data == NULL){
		ErrorOut("BinaryAnimationLoader::readCharacterModelAnimationLink() "
			"CharacterModelAnimationData not found %s (%d)",
			dataName.getBytes(), reader_->getPosition());
	}
	// N
	animation->setCharacterModelAnimationData(data);
	// ACg
	align();
}
//------------------------------------------------------------------------------
// l̓ǂݍ
//------------------------------------------------------------------------------
// ̓ǂݍ
String BinaryAnimationLoader::readString(){
	String result;
	const int stringBufferSize = 256;
	char stringBuffer[stringBufferSize];
	// TCY̓ǂݍ
	int size = reader_->readInt();
	if(size > stringBufferSize){
		// TCY傫̂ŃAP[gĕǂݍ
		char* allocateBuffer = new char[size];
		reader_->readBytes(allocateBuffer, size);
		result = allocateBuffer;
		delete allocateBuffer;
	}else{
		// ǂݍ
		reader_->readBytes(stringBuffer, size);
		result = stringBuffer;
	}
	align();
	return result;
}
//------------------------------------------------------------------------------
// OxNg̓ǂݍ
Vector3 BinaryAnimationLoader::readVector3(){
	Vector3 result;
	reader_->readBytes(result.array, sizeof(Vector3));
	return result;
}
//------------------------------------------------------------------------------
// l̓ǂݍ
Quaternion BinaryAnimationLoader::readQuaternion(){
	Quaternion result;
	reader_->readBytes(result.array, sizeof(Quaternion));
	return result;
}
//------------------------------------------------------------------------------
// xNgԂ̓ǂݍ
//------------------------------------------------------------------------------
// xNgԂ̓ǂݍ
VectorInterpolator* BinaryAnimationLoader::readVectorInterpolator(){
	VectorInterpolator* result = NULL;
	int type = reader_->readInt();
	if(type == 0){
		result = NULL;
	}else if(type == 1){
		result = readVectorConstantInterpolator();
	}else if(type == 2){
		result = readVectorArrayInterpolator();
	}else if(type == 3){
		result = readVectorLinearInterpolator();
	}else{
		ErrorOut("BinaryAnimationLoader::readVectorInterpolator() "
			"Unsupported interpolater (%d)", reader_->getPosition());
	}
	return result;
}
//------------------------------------------------------------------------------
// xNg萔Ԃ̓ǂݍ
VectorInterpolator* BinaryAnimationLoader::readVectorConstantInterpolator(){
	VectorConstantInterpolator* interpolator = new VectorConstantInterpolator();
	// 
	interpolator->setLength(reader_->readFloat());
	// l
	interpolator->setValue(readVector3());
	return interpolator;
}
//------------------------------------------------------------------------------
// xNgzԂ̓ǂݍ
VectorInterpolator* BinaryAnimationLoader::readVectorArrayInterpolator(){
	VectorArrayInterpolator* interpolator = new VectorArrayInterpolator();
	// l
	int size = reader_->readInt();
	interpolator->setSize(size);
	for(int i = 0; i < size; i++){ interpolator->setValue(i, readVector3()); }
	return interpolator;
}
//------------------------------------------------------------------------------
// xNg`Ԃ̓ǂݍ
VectorInterpolator* BinaryAnimationLoader::readVectorLinearInterpolator(){
	VectorLinearInterpolator* interpolator = new VectorLinearInterpolator();
	// l
	int keyCount = reader_->readInt();
	interpolator->setKeyCount(keyCount);
	for(int i = 0; i < keyCount; i++){
		float time = reader_->readFloat();
		interpolator->setKey(i, time, readVector3());
	}
	return interpolator;
}
//------------------------------------------------------------------------------
// ]Ԃ̓ǂݍ
//------------------------------------------------------------------------------
// ]Ԃ̓ǂݍ
RotationInterpolator* BinaryAnimationLoader::readRotationInterpolator(){
	RotationInterpolator* result = NULL;
	int type = reader_->readInt();
	if(type == 0){
		result = NULL;
	}else if(type == 1){
		result = readRotationConstantInterpolator();
	}else if(type == 2){
		result = readEulerArrayInterpolator();
	}else if(type == 3){
		result = readQuaternionArrayInterpolator();
	}else if(type == 4){
		result = readQuaternionLinearInterpolator();
	}else{
		ErrorOut("BinaryAnimationLoader::readRotationInterpolator() "
			"Unsupported interpolater (%d)", reader_->getPosition());
	}
	return result;
}
//------------------------------------------------------------------------------
// ]萔Ԃ̓ǂݍ
RotationInterpolator* BinaryAnimationLoader::readRotationConstantInterpolator(){
	RotationConstantInterpolator* interpolator =
		new RotationConstantInterpolator();
	// 
	interpolator->setLength(reader_->readFloat());
	// l
	interpolator->setQuaternion(readQuaternion());
	return interpolator;
}
//------------------------------------------------------------------------------
// IC[]zԂ̓ǂݍ
RotationInterpolator* BinaryAnimationLoader::readEulerArrayInterpolator(){
	EulerArrayInterpolator* interpolator = new EulerArrayInterpolator();
	// l
	int size = reader_->readInt();
	interpolator->setSize(size);
	for(int i = 0; i < size; i++){ interpolator->setValue(i, readVector3()); }
	return interpolator;
}
//------------------------------------------------------------------------------
// l]zԂ̓ǂݍ
RotationInterpolator* BinaryAnimationLoader::readQuaternionArrayInterpolator(){
	QuaternionArrayInterpolator* interpolator =
		new QuaternionArrayInterpolator();
	// l
	int size = reader_->readInt();
	interpolator->setSize(size);
	for(int i = 0; i < size; i++){
		interpolator->setValue(i, readQuaternion());
	}
	return interpolator;
}
//------------------------------------------------------------------------------
// l]`Ԃ̓ǂݍ
RotationInterpolator* BinaryAnimationLoader::readQuaternionLinearInterpolator(){
	QuaternionLinearInterpolator* interpolator =
		new QuaternionLinearInterpolator();
	// l
	int keyCount = reader_->readInt();
	interpolator->setKeyCount(keyCount);
	for(int i = 0; i < keyCount; i++){
		float time = reader_->readFloat();
		interpolator->setKey(i, time, readQuaternion());
	}
	return interpolator;
}
//------------------------------------------------------------------------------
// [eBeB
//------------------------------------------------------------------------------
// ACg
void BinaryAnimationLoader::align(){
	reader_->align(16);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
