//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * Cwb_
 * @author Junpee
 */

#ifndef RAY_H_
#define RAY_H_

#include <Core/Primitive/Vector3.h>
#include <Core/Primitive/Matrix33.h>
#include <Core/Primitive/Matrix34.h>
#include <Core/Primitive/Matrix44.h>

namespace Lamp{

class AxisAlignedBox;
class Capsule;
class Cone;
class Line;
class OrientedBox;
class Plane;
class Segment;
class Sphere;
class Triangle;

//------------------------------------------------------------------------------
/**
 * C
 *
 * ̃NX͌pȂŉB
 */
class Ray{
public:
	//--------------------------------------------------------------------------
	// 萔
	//--------------------------------------------------------------------------
	/// [C
	static const Ray zero;

	//--------------------------------------------------------------------------
	// RXgN^
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 *
	 * ̃RXgN^͏l̐ݒsȂߒl͕słB
	 */
	Ray(){}

	/**
	 * RXgN^
	 * @param origin _̏l
	 * @param direction ̏l
	 */
	inline Ray(const Vector3& origin, const Vector3& direction) :
		origin_(origin), direction_(direction){
	}

	/**
	 * RXgN^
	 * @param originX _X̏l
	 * @param originY _Y̏l
	 * @param originZ _Z̏l
	 * @param directionX X̏l
	 * @param directionY Y̏l
	 * @param directionZ Z̏l
	 */
	inline Ray(float originX, float originY, float originZ,
		float directionX, float directionY, float directionZ) :
		origin_(originX, originY, originZ),
		direction_(directionX, directionY, directionZ){
	}

	/**
	 * RXgN^
	 * @param source lz
	 */
	inline explicit Ray(const float* const source) :
		origin_(source[0], source[1], source[2]),
		direction_(source[3], source[4], source[5]){
	}

	//--------------------------------------------------------------------------
	// l̐ݒ
	//--------------------------------------------------------------------------
	/**
	 * l̐ݒ
	 * @param origin ݒ肷錴_
	 * @param direction ݒ肷
	 */
	inline void set(const Vector3& origin, const Vector3& direction){
		origin_ = origin;
		direction_ = direction;
	}

	/**
	 * l̐ݒ
	 * @param originX ݒ肷錴_X
	 * @param originY ݒ肷錴_Y
	 * @param originZ ݒ肷錴_Z
	 * @param directionX ݒ肷X
	 * @param directionY ݒ肷Y
	 * @param directionZ ݒ肷Z
	 */
	inline void set(float originX, float originY, float originZ,
		float directionX, float directionY, float directionZ){
		origin_.set(originX, originY, originZ);
		direction_.set(directionX, directionY, directionZ);
	}

	/**
	 * l̐ݒ
	 * @param source ݒlz
	 */
	inline void set(const float* const source){
		origin_.set(source[0], source[1], source[2]);
		direction_.set(source[3], source[4], source[5]);
	}

	//--------------------------------------------------------------------------
	/**
	 * _̐ݒ
	 * @param origin ݒ肷錴_
	 */
	inline void setOrigin(const Vector3& origin){ origin_ = origin; }

	/**
	 * ̐ݒ
	 * @param direction ݒ肷
	 */
	inline void setDirection(const Vector3& direction){
		direction_ = direction;
	}

	//--------------------------------------------------------------------------
	// l̎擾
	//--------------------------------------------------------------------------
	/**
	 * _̎擾
	 * @return _
	 */
	inline const Vector3& getOrigin() const{ return origin_; }

	/**
	 * ̎擾
	 * @return 
	 */
	inline const Vector3& getDirection() const{ return direction_; }

	//--------------------------------------------------------------------------
	// CZ
	//--------------------------------------------------------------------------
	/**
	 * [Cǂ
	 * @return [CȂtrue
	 */
	inline bool isZero() const{
		return (direction_.epsilonEquals(Vector3::zero, Math::epsilon));
	}

	//--------------------------------------------------------------------------
	// gXtH[
	//--------------------------------------------------------------------------
	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̃C
	 */
	inline Ray transform(const Matrix33& matrix) const{
		return Ray(matrix * origin_, matrix * direction_);
	}

	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̃C
	 */
	inline Ray transform(const Matrix34& matrix) const{
		return Ray(matrix * origin_, matrix.multiply33(direction_));
	}

	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̃C
	 */
	inline Ray transform(const Matrix44& matrix) const{
		return Ray(matrix * origin_, matrix.multiply33(direction_));
	}

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @return 
	 */
	float getDistance(const Vector3& point) const{
		return Math::sqrt(getSquaredDistance(point));
	}

	/**
	 * _̓
	 * @param point 肷_
	 * @return ̓
	 */
	float getSquaredDistance(const Vector3& point) const;

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return 
	 */
	float getDistance(const AxisAlignedBox& axisAlignedBox) const{
		return Math::sqrt(getSquaredDistance(axisAlignedBox));
	}

	/**
	 * {bNX̓
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const AxisAlignedBox& axisAlignedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return 
	 */
	float getDistance(const Capsule& capsule) const{
		return Math::sqrt(getSquaredDistance(capsule));
	}

	/**
	 * JvZ̓
	 * @param capsule 肷JvZ
	 * @return ̓
	 */
	float getSquaredDistance(const Capsule& capsule) const;

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return 
	 */
	float getDistance(const Cone& cone) const{
		return Math::sqrt(getSquaredDistance(cone));
	}

	/**
	 * R[̓
	 * @param cone 肷R[
	 * @return ̓
	 */
	float getSquaredDistance(const Cone& cone) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @return 
	 */
	float getDistance(const Line& line) const{
		return Math::sqrt(getSquaredDistance(line));
	}

	/**
	 * C̓
	 * @param line 肷郉C
	 * @return ̓
	 */
	float getSquaredDistance(const Line& line) const;

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return 
	 */
	float getDistance(const OrientedBox& orientedBox) const{
		return Math::sqrt(getSquaredDistance(orientedBox));
	}

	/**
	 * w{bNX̓
	 * @param orientedBox 肷w{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const OrientedBox& orientedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * ʋ
	 * @param plane 肷镽
	 * @return 
	 */
	float getDistance(const Plane& plane) const;

	/**
	 * ʋ̓
	 * @param plane 肷镽
	 * @return ̓
	 */
	float getSquaredDistance(const Plane& plane) const{
		float distance = getDistance(plane);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @return 
	 */
	float getDistance(const Ray& ray) const{
		return Math::sqrt(getSquaredDistance(ray));
	}

	/**
	 * C̓
	 * @param ray 肷郌C
	 * @return ̓
	 */
	float getSquaredDistance(const Ray& ray) const;

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @return 
	 */
	float getDistance(const Segment& segment) const{
		return Math::sqrt(getSquaredDistance(segment));
	}

	/**
	 * ZOg̓
	 * @param segment 肷ZOg
	 * @return ̓
	 */
	float getSquaredDistance(const Segment& segment) const;

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return 
	 */
	float getDistance(const Sphere& sphere) const{
		return Math::sqrt(getSquaredDistance(sphere));
	}

	/**
	 * ̓
	 * @param sphere 肷鋅
	 * @return ̓
	 */
	float getSquaredDistance(const Sphere& sphere) const;

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return 
	 */
	float getDistance(const Triangle& triangle) const{
		return Math::sqrt(getSquaredDistance(triangle));
	}

	/**
	 * Op̓
	 * @param triangle 肷Op
	 * @return ̓
	 */
	float getSquaredDistance(const Triangle& triangle) const;

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @param range ͈
	 * @return Ătrue
	 */
	bool intersect(const Vector3& point, float range = Math::epsilon) const;

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return Ătrue
	 */
	bool intersect(const AxisAlignedBox& axisAlignedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return Ătrue
	 */
	bool intersect(const Capsule& capsule) const;

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return Ătrue
	 */
	bool intersect(const Cone& cone) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @param range ͈
	 * @return Ătrue
	 */
	bool intersect(const Line& line, float range = Math::epsilon) const;

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return Ătrue
	 */
	bool intersect(const OrientedBox& orientedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * ʌ
	 * @param plane 肷镽
	 * @return Ătrue
	 */
	bool intersect(const Plane& plane) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @param range ͈
	 * @return Ătrue
	 */
	bool intersect(const Ray& ray, float range = Math::epsilon) const;

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @param range ͈
	 * @return Ătrue
	 */
	bool intersect(const Segment& segment, float range = Math::epsilon) const;

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return Ătrue
	 */
	bool intersect(const Sphere& sphere) const;

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return Ătrue
	 */
	bool intersect(const Triangle& triangle) const;

	//--------------------------------------------------------------------------
	// _Z
	//--------------------------------------------------------------------------
	/**
	 * Cǂ
	 * @param target r郌C
	 * @return lłtrueԂ
	 */
	inline bool operator ==(const Ray& target) const{
		return ((origin_ == target.origin_) &&
			(direction_ == target.direction_));
	}

	/**
	 * Cǂ
	 * @param target r郌C
	 * @param epsilon 덷
	 * @return 덷͈͓̔œlłtrueԂ
	 */
	inline bool epsilonEquals(
		const Ray& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (origin_.epsilonEquals(target.origin_, epsilon) &&
			direction_.epsilonEquals(target.direction_, epsilon));
	}

	/**
	 * CłȂǂ
	 * @param target r郌C
	 * @return łȂlłtrueԂ
	 */
	inline bool operator !=(const Ray& target) const{
		return ((origin_ != target.origin_) ||
			(direction_ != target.direction_));
	}

	/**
	 * CłȂǂ
	 * @param target r郌C
	 * @param epsilon 덷
	 * @return 덷͈͓̔œłȂlłtrueԂ
	 */
	inline bool notEpsilonEquals(
		const Ray& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (origin_.notEpsilonEquals(target.origin_, epsilon) ||
			direction_.notEpsilonEquals(target.direction_, epsilon));
	}

	//--------------------------------------------------------------------------
	// ̑
	//--------------------------------------------------------------------------
	/**
	 * 
	 * @return C̕\L
	 */
	inline String toString() const{
		String returnString;
		returnString.format("{ ( %.8f, %.8f, %.8f ) ( %.8f, %.8f, %.8f ) }",
			origin_.x, origin_.y, origin_.z,
			direction_.x, direction_.y, direction_.z);
		return returnString;
	}

private:
	//--------------------------------------------------------------------------
	// oϐ
	//--------------------------------------------------------------------------
	// _
	Vector3 origin_;
	// 
	Vector3 direction_;

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of RAY_H_
//------------------------------------------------------------------------------
