//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * ʃwb_
 * @author Junpee
 */

#ifndef PLANE_H_
#define PLANE_H_

#include <Core/Primitive/Vector3.h>
#include <Core/Primitive/Matrix33.h>
#include <Core/Primitive/Matrix34.h>
#include <Core/Primitive/Matrix44.h>

namespace Lamp{

class AxisAlignedBox;
class Capsule;
class Cone;
class Line;
class OrientedBox;
class Ray;
class Segment;
class Sphere;
class Triangle;

//------------------------------------------------------------------------------
/**
 * 
 *
 * ̃NX͌pȂŉB
 */
class Plane{
public:
	//--------------------------------------------------------------------------
	// 萔
	//--------------------------------------------------------------------------
	/// [
	static const Plane zero;

	/// XPʕ
	static const Plane unitX;

	/// YPʕ
	static const Plane unitY;

	/// ZPʕ
	static const Plane unitZ;

	//--------------------------------------------------------------------------
	// RXgN^
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 *
	 * ̃RXgN^͏l̐ݒsȂߒl͕słB
	 */
	inline Plane(){}

	/**
	 * RXgN^
	 * @param normal @̏l
	 * @param constant 萔̏l
	 */

	inline Plane(const Vector3& normal, float constant) :
		normal_(normal), constant_(constant){
	}

	/**
	 * RXgN^
	 * @param normalX @X̏l
	 * @param normalY @Y̏l
	 * @param normalZ @Z̏l
	 * @param constant 萔̏l
	 */

	inline Plane(float normalX, float normalY, float normalZ, float constant) :
		normal_(normalX, normalY, normalZ), constant_(constant){
	}

	/**
	 * RXgN^
	 * @param source lz
	 */
	inline explicit Plane(const float* const source) :
		normal_(source[0], source[1], source[2]), constant_(source[3]){
	}

	/**
	 * RXgN^
	 * @param vertex0 ʏ̒_0
	 * @param vertex1 ʏ̒_1
	 * @param vertex2 ʏ̒_2
	 */
	inline Plane(
		const Vector3& vertex0, const Vector3& vertex1, const Vector3& vertex2){
		normal_ = Math3D::calculateNormal(vertex0, vertex1, vertex2);
		constant_ = -(normal_.dotProduct(vertex0));
	}

	/**
	 * RXgN^
	 * @param normal @̏l
	 * @param vertex ʏ̒_
	 */
	inline Plane(const Vector3& normal, const Vector3& vertex) :
		normal_(normal){
		constant_ = -(normal_.dotProduct(vertex));
	}

	//--------------------------------------------------------------------------
	// l̐ݒ
	//--------------------------------------------------------------------------
	/**
	 * l̐ݒ
	 * @param normal ݒ肷@
	 * @param constant ݒ肷萔
	 */
	inline void set(const Vector3& normal, float constant){
		normal_ = normal;
		constant_ = constant;
	}

	/**
	 * l̐ݒ
	 * @param normalX ݒ肷@X
	 * @param normalY ݒ肷@Y
	 * @param normalZ ݒ肷@Z
	 * @param constant ݒ肷萔
	 */
	inline void set(
		float normalX, float normalY, float normalZ, float constant){
		normal_.set(normalX, normalY, normalZ);
		constant_ = constant;
	}

	/**
	 * l̐ݒ
	 * @param source ݒlz
	 */
	inline void set(const float* const source){
		normal_.set(source[0], source[1], source[2]);
		constant_ = source[3];
	}

	/**
	 * l̐ݒ
	 * @param vertex0 ʏ̒_0
	 * @param vertex1 ʏ̒_1
	 * @param vertex2 ʏ̒_2
	 */
	inline void set(
		const Vector3& vertex0, const Vector3& vertex1, const Vector3& vertex2){
		normal_ = Math3D::calculateNormal(vertex0, vertex1, vertex2);
		constant_ = -(normal_.dotProduct(vertex0));
	}

	/**
	 * l̐ݒ
	 * @param normal ݒ肷@
	 * @param vertex ʏ̒_
	 */
	inline void set(const Vector3& normal, const Vector3& vertex){
		normal_ = normal;
		constant_ = -(normal_.dotProduct(vertex));
	}

	/**
	 * @̐ݒ
	 * @param normalX ݒ肷@X
	 * @param normalY ݒ肷@Y
	 * @param normalZ ݒ肷@Z
	 */
	inline void setNormal(float normalX, float normalY, float normalZ){
		normal_.set(normalX, normalY, normalZ);
	}

	/**
	 * @̐ݒ
	 * @param normal ݒ肷@
	 */
	inline void setNormal(const Vector3& normal){ normal_ = normal; }

	/**
	 * 萔̐ݒ
	 * @param constant ݒ肷萔
	 */
	inline void setConstant(float constant){ constant_ = constant; }

	/**
	 * ̐ݒ
	 * @param length ݒ肷钷
	 */
	inline const Plane& setLength(float length){
		float nowLength = normal_.getLength();
		// 0̏ꍇɂΏ̓AvP[VɂĈႤ̂Assert
		Assert(nowLength >= Math::epsilon);
		float inverseLength = (length / nowLength);
		normal_ *= inverseLength;
		constant_ *= inverseLength;
		return *this;
	}

	//--------------------------------------------------------------------------
	// l̎擾
	//--------------------------------------------------------------------------
	/**
	 * @̎擾
	 * @return @
	 */
	inline const Vector3& getNormal() const{ return normal_; }

	/**
	 * 萔̎擾
	 * @return 萔
	 */
	inline float getConstant() const{ return constant_; }

	/**
	 * ̎擾
	 * @return 
	 */
	inline float getLength() const{
		return Math::sqrt(normal_.x * normal_.x +
			normal_.y * normal_.y + normal_.z * normal_.z);
	}

	/**
	 * ̓擾
	 * @return ̓
	 */
	inline float getSquaredLength() const{
		return (normal_.x * normal_.x +
			normal_.y * normal_.y + normal_.z * normal_.z);
	}

	//--------------------------------------------------------------------------
	// ʉZ
	//--------------------------------------------------------------------------
	/**
	 * K
	 * @return Kꂽ
	 */
	inline const Plane& normalize(){ return setLength(1.f); }

	/**
	 * ςs
	 * @return ςl
	 */
	inline float dotProduct(const Vector3& vector) const{
		return normal_.dotProduct(vector) + constant_;
	}

	//--------------------------------------------------------------------------
	/**
	 * [ʂǂ
	 * @return [ʂȂtrueԂ
	 */
	inline bool isZero() const{
		return (getLength() <= Math::epsilon);
	}

	/**
	 * Pʕʂǂ
	 * @return PʕʂȂtrueԂ
	 */
	inline bool isUnit() const{
		return (Math::abs(getLength() - 1.f) <= Math::epsilon);
	}

	//--------------------------------------------------------------------------
	// gXtH[
	//--------------------------------------------------------------------------
	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̋
	 */
	inline Plane transform(const Matrix33& matrix) const{
		Plane result;
		result.normal_ = matrix * normal_;
		result.constant_ = constant_;
		return result;
	}

	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̋
	 */
	inline Plane transform(const Matrix34& matrix) const{
		Plane result;
		result.normal_ = matrix.multiply33(normal_);
		Vector3 trans = matrix.getTranslation();
		trans.set(
			matrix.m00 * trans.x + matrix.m10 * trans.y + matrix.m20 * trans.z,
			matrix.m01 * trans.x + matrix.m11 * trans.y + matrix.m21 * trans.z,
			matrix.m02 * trans.x + matrix.m12 * trans.y + matrix.m22 * trans.z);
		result.constant_ = constant_ - trans.dotProduct(normal_);
		return result;
	}

	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̋
	 */
	inline Plane transform(const Matrix44& matrix) const{
		Plane result;
		result.normal_ = matrix.multiply33(normal_);
		Vector3 trans = matrix.getTranslation();
		trans.set(
			matrix.m00 * trans.x + matrix.m10 * trans.y + matrix.m20 * trans.z,
			matrix.m01 * trans.x + matrix.m11 * trans.y + matrix.m21 * trans.z,
			matrix.m02 * trans.x + matrix.m12 * trans.y + matrix.m22 * trans.z);
		result.constant_ = constant_ - trans.dotProduct(normal_);
		return result;
	}

	//--------------------------------------------------------------------------
	/**
	 * XP[LgXtH[
	 * @param matrix Zs
	 * @return ϊ̋
	 */
	inline Plane scaledTransform(const Matrix33& matrix) const{
		Plane result;
		Matrix33 normalMatrix;
		matrix.invert(&normalMatrix);
		normalMatrix.transpose();
		result.normal_ = normalMatrix * normal_;
		result.constant_ = constant_;
		return result;
	}

	/**
	 * XP[LgXtH[
	 * @param matrix Zs
	 * @return ϊ̋
	 */
	inline Plane scaledTransform(const Matrix34& matrix) const{
		Plane result;
		Matrix33 normalMatrix;
		normalMatrix.set(matrix);
		normalMatrix.invert();
		result.constant_ = constant_ -
			(normalMatrix * matrix.getTranslation()).dotProduct(normal_);
		normalMatrix.transpose();
		result.normal_ = normalMatrix * normal_;
		return result;
	}

	/**
	 * XP[LgXtH[
	 * @param matrix Zs
	 * @return ϊ̋
	 */
	inline Plane scaledTransform(const Matrix44& matrix) const{
		Plane result;
		Matrix33 normalMatrix;
		normalMatrix.set(matrix);
		normalMatrix.invert();
		result.constant_ = constant_ -
			(normalMatrix * matrix.getTranslation()).dotProduct(normal_);
		normalMatrix.transpose();
		result.normal_ = normalMatrix * normal_;
		return result;
	}

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @return 
	 */
	float getDistance(const Vector3& point) const;

	/**
	 * _̓
	 * @param point 肷_
	 * @return ̓
	 */
	float getSquaredDistance(const Vector3& point) const{
		float distance = getDistance(point);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return 
	 */
	float getDistance(const AxisAlignedBox& axisAlignedBox) const;

	/**
	 * {bNX̓
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const AxisAlignedBox& axisAlignedBox) const{
		float distance = getDistance(axisAlignedBox);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return 
	 */
	float getDistance(const Capsule& capsule) const;

	/**
	 * JvZ̓
	 * @param capsule 肷JvZ
	 * @return ̓
	 */
	float getSquaredDistance(const Capsule& capsule) const{
		float distance = getDistance(capsule);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return 
	 */
	float getDistance(const Cone& cone) const;

	/**
	 * R[̓
	 * @param cone 肷R[
	 * @return ̓
	 */
	float getSquaredDistance(const Cone& cone) const{
		float distance = getDistance(cone);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @return 
	 */
	float getDistance(const Line& line) const;

	/**
	 * C̓
	 * @param line 肷郉C
	 * @return ̓
	 */
	float getSquaredDistance(const Line& line) const{
		float distance = getDistance(line);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return 
	 */
	float getDistance(const OrientedBox& orientedBox) const;

	/**
	 * w{bNX̓
	 * @param orientedBox 肷w{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const OrientedBox& orientedBox) const{
		float distance = getDistance(orientedBox);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * ʋ
	 * @param plane 肷镽
	 * @return 
	 */
	float getDistance(const Plane& plane) const;

	/**
	 * ʋ̓
	 * @param plane 肷镽
	 * @return ̓
	 */
	float getSquaredDistance(const Plane& plane) const{
		float distance = getDistance(plane);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @return 
	 */
	float getDistance(const Ray& ray) const;

	/**
	 * C̓
	 * @param ray 肷郌C
	 * @return ̓
	 */
	float getSquaredDistance(const Ray& ray) const{
		float distance = getDistance(ray);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @return 
	 */
	float getDistance(const Segment& segment) const;

	/**
	 * ZOg̓
	 * @param segment 肷ZOg
	 * @return ̓
	 */
	float getSquaredDistance(const Segment& segment) const{
		float distance = getDistance(segment);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return 
	 */
	float getDistance(const Sphere& sphere) const;

	/**
	 * ̓
	 * @param sphere 肷鋅
	 * @return ̓
	 */
	float getSquaredDistance(const Sphere& sphere) const{
		float distance = getDistance(sphere);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return 
	 */
	float getDistance(const Triangle& triangle) const;

	/**
	 * Op̓
	 * @param triangle 肷Op
	 * @return ̓
	 */
	float getSquaredDistance(const Triangle& triangle) const{
		float distance = getDistance(triangle);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @param range ͈
	 * @return Ătrue
	 */
	bool intersect(const Vector3& point, float range = Math::epsilon) const;

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return Ătrue
	 */
	bool intersect(const AxisAlignedBox& axisAlignedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return Ătrue
	 */
	bool intersect(const Capsule& capsule) const;

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return Ătrue
	 */
	bool intersect(const Cone& cone) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @return Ătrue
	 */
	bool intersect(const Line& line) const;

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return Ătrue
	 */
	bool intersect(const OrientedBox& orientedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * ʌ
	 * @param plane 肷镽
	 * @return Ătrue
	 */
	bool intersect(const Plane& plane) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @return Ătrue
	 */
	bool intersect(const Ray& ray) const;

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @return Ătrue
	 */
	bool intersect(const Segment& segment) const;

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return Ătrue
	 */
	bool intersect(const Sphere& sphere) const;

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return Ătrue
	 */
	bool intersect(const Triangle& triangle) const;

	//--------------------------------------------------------------------------
	// _Z
	//--------------------------------------------------------------------------
	/**
	 * ʂǂ
	 * @param target r镽
	 * @return lłtrueԂ
	 */
	inline bool operator ==(const Plane& target) const{
		return ((normal_ == target.normal_) && (constant_ == target.constant_));
	}

	/**
	 * ʂǂ
	 * @param target r镽
	 * @param epsilon 덷
	 * @return 덷͈͓̔œlłtrueԂ
	 */
	inline bool epsilonEquals(
		const Plane& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (normal_.epsilonEquals(target.normal_, epsilon) &&
			(Math::abs(constant_ - target.constant_) <= epsilon));
	}

	/**
	 * ʂłȂǂ
	 * @param target r镽
	 * @return łȂlłtrueԂ
	 */
	inline bool operator !=(const Plane& target) const{
		return ((normal_ != target.normal_) || (constant_ != target.constant_));
	}

	/**
	 * ʂłȂǂ
	 * @param target r镽
	 * @param epsilon 덷
	 * @return 덷͈͓̔œłȂlłtrueԂ
	 */
	inline bool notEpsilonEquals(
		const Plane& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (normal_.notEpsilonEquals(target.normal_, epsilon) ||
			(Math::abs(constant_ - target.constant_) > epsilon));
	}

	//--------------------------------------------------------------------------
	// ̑
	//--------------------------------------------------------------------------
	/**
	 * 
	 * @return ʂ̕\L
	 */
	inline String toString() const{
		String returnString;
		returnString.format("{ ( %.8f, %.8f, %.8f ) %.8f }",
			normal_.x, normal_.y, normal_.z, constant_);
		return returnString;
	}

private:
	//--------------------------------------------------------------------------
	// oϐ
	//--------------------------------------------------------------------------
	// @
	Vector3 normal_;
	// 萔
	float constant_;

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of PLANE_H_
//------------------------------------------------------------------------------
