//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * w{bNXwb_
 * @author Junpee
 */

#ifndef ORIENTED_BOX_H_
#define ORIENTED_BOX_H_

#include <Core/Primitive/Vector3.h>
#include <Core/Primitive/Matrix33.h>
#include <Core/Primitive/Matrix34.h>
#include <Core/Primitive/Matrix44.h>

namespace Lamp{

class AxisAlignedBox;
class Capsule;
class Cone;
class Line;
class Plane;
class Ray;
class Segment;
class Sphere;
class Triangle;

//------------------------------------------------------------------------------
/**
 * w{bNX
 *
 * ̃NX͌pȂŉB
 */
class OrientedBox{
public:
	//--------------------------------------------------------------------------
	// 萔
	//--------------------------------------------------------------------------
	/// [{bNX
	static const OrientedBox zero;

	/// Pʃ{bNX
	static const OrientedBox unit;

	//--------------------------------------------------------------------------
	// RXgN^
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 *
	 * ̃RXgN^͏l̐ݒsȂߒl͕słB
	 */
	inline OrientedBox(){}

	/**
	 * RXgN^
	 * @param rotationMatrix ]s̏l
	 * @param center S̏l
	 * @param extent 傫̏l
	 */
	inline OrientedBox(const Matrix33& rotationMatrix,
		const Vector3& center, const Vector3& extent) :
		rotationMatrix_(rotationMatrix), center_(center), extent_(extent){
		Assert((extent_.x >= 0.f) && (extent_.y >= 0.f) && (extent_.z >= 0.f));
	}

	/**
	 * RXgN^
	 * @param rotation00 ]sm00l
	 * @param rotation01 ]sm01l
	 * @param rotation02 ]sm02l
	 * @param rotation10 ]sm10l
	 * @param rotation11 ]sm11l
	 * @param rotation12 ]sm12l
	 * @param rotation20 ]sm20l
	 * @param rotation21 ]sm21l
	 * @param rotation22 ]sm22l
	 * @param centerX SX̏l
	 * @param centerY SY̏l
	 * @param centerZ SZ̏l
	 * @param extentX 傫X̏l
	 * @param extentY 傫Y̏l
	 * @param extentZ 傫Z̏l
	 */
	inline OrientedBox(
		float rotation00, float rotation01, float rotation02,
		float rotation10, float rotation11, float rotation12,
		float rotation20, float rotation21, float rotation22,
		float centerX, float centerY, float centerZ,
		float extentX, float extentY, float extentZ) :
		rotationMatrix_(rotation00, rotation01, rotation02,
			rotation10, rotation11, rotation12,
			rotation20, rotation21, rotation22),
		center_(centerX, centerY, centerZ),
		extent_(extentX, extentY, extentZ){
		Assert((extent_.x >= 0.f) && (extent_.y >= 0.f) && (extent_.z >= 0.f));
	}

	/**
	 * RXgN^
	 * @param source lz
	 */
	inline explicit OrientedBox(const float* const source) :
		rotationMatrix_(source[0], source[1], source[2],
			source[3], source[4], source[5],
			source[6], source[7], source[8]),
		center_(source[9], source[10], source[11]),
		extent_(source[12], source[13], source[14]){
		Assert((extent_.x >= 0.f) && (extent_.y >= 0.f) && (extent_.z >= 0.f));
	}

	//--------------------------------------------------------------------------
	// l̐ݒ
	//--------------------------------------------------------------------------
	/**
	 * l̐ݒ
	 * @param rotationMatrix ݒ肷]s
	 * @param center ݒ肷钆S
	 * @param extent ݒ肷傫
	 */
	inline void set(const Matrix33& rotationMatrix,
		const Vector3& center, const Vector3& extent){
		rotationMatrix_ = rotationMatrix;
		center_ = center;
		extent_ = extent;
		Assert((extent_.x >= 0.f) && (extent_.y >= 0.f) && (extent_.z >= 0.f));
	}

	/**
	 * l̐ݒ
	 * @param rotation00 ݒ肷]sm00
	 * @param rotation01 ݒ肷]sm01
	 * @param rotation02 ݒ肷]sm02
	 * @param rotation10 ݒ肷]sm10
	 * @param rotation11 ݒ肷]sm11
	 * @param rotation12 ݒ肷]sm12
	 * @param rotation20 ݒ肷]sm20
	 * @param rotation21 ݒ肷]sm21
	 * @param rotation22 ݒ肷]sm22
	 * @param centerX ݒ肷钆SX
	 * @param centerY ݒ肷钆SY
	 * @param centerZ ݒ肷钆SZ
	 * @param extentX ݒ肷傫X
	 * @param extentY ݒ肷傫Y
	 * @param extentZ ݒ肷傫Z
	 */
	inline void set(
		float rotation00, float rotation01, float rotation02,
		float rotation10, float rotation11, float rotation12,
		float rotation20, float rotation21, float rotation22,
		float centerX, float centerY, float centerZ,
		float extentX, float extentY, float extentZ){
		rotationMatrix_.set(rotation00, rotation01, rotation02,
			rotation10, rotation11, rotation12,
			rotation20, rotation21, rotation22);
		center_.set(centerX, centerY, centerZ);
		extent_.set(extentX, extentY, extentZ);
		Assert((extent_.x >= 0.f) && (extent_.y >= 0.f) && (extent_.z >= 0.f));
	}

	/**
	 * l̐ݒ
	 * @param source ݒlz
	 */
	inline void set(const float* const source){
		rotationMatrix_.set(source[0], source[1], source[2],
			source[3], source[4], source[5],
			source[6], source[7], source[8]);
		center_.set(source[9], source[10], source[11]);
		extent_.set(source[12], source[13], source[14]);
		Assert((extent_.x >= 0.f) && (extent_.y >= 0.f) && (extent_.z >= 0.f));
	}

	//--------------------------------------------------------------------------
	/**
	 * ]s̐ݒ
	 * @param rotationMatrix ]s
	 */
	inline void setRotationMatrix(const Matrix33& rotationMatrix){
		rotationMatrix_ = rotationMatrix;
	}

	/**
	 * XYZ]̐ݒ
	 * @param rotationXYZ XYZ]
	 */
	inline void setRotationXYZ(const Vector3 rotationXYZ){
		rotationMatrix_.setRotationXYZ(rotationXYZ);
	}

	/**
	 * l]̐ݒ
	 * @param rotationQuaternion l]
	 */
	inline void setRotationQuaternion(const Quaternion& rotationQuaternion){
		rotationMatrix_.setRotationQuaternion(rotationQuaternion);
	}

	//--------------------------------------------------------------------------
	/**
	 * S̐ݒ
	 * @param center S
	 */
	inline void setCenter(const Vector3& center){ center_ = center; }

	//--------------------------------------------------------------------------
	/**
	 * 傫̐ݒ
	 * @param extent 傫
	 */
	inline void setExtent(const Vector3& extent){
		extent_ = extent;
		Assert((extent_.x >= 0.f) && (extent_.y >= 0.f) && (extent_.z >= 0.f));
	}

	//--------------------------------------------------------------------------
	// l̎擾
	//--------------------------------------------------------------------------
	/**
	 * ]s̎擾
	 * @return ]s
	 */
	inline const Matrix33& getRotationMatrix() const{ return rotationMatrix_; }

	/**
	 * S̎擾
	 * @return S
	 */
	inline const Vector3& getCenter() const{ return center_; }

	/**
	 * 傫̎擾
	 * @return 傫
	 */
	inline const Vector3& getExtent() const{ return extent_; }

	/**
	 * TCY̎擾
	 * @return TCY
	 */
	inline Vector3 getSize() const{ return (extent_ * 2.f); }

	//--------------------------------------------------------------------------
	/**
	 * ̎擾
	 * @param index CfbNX
	 * @return xNg
	 */
	inline Vector3 getAxis(int index) const{
		Assert((index >= 0) && (index < 3));
		return Vector3(rotationMatrix_.m[index][0],
			rotationMatrix_.m[index][1], rotationMatrix_.m[index][2]);
	}

	/**
	 * X̎擾
	 * @return XxNg
	 */
	inline Vector3 getAxisX() const{
		return Vector3(
			rotationMatrix_.m00, rotationMatrix_.m01, rotationMatrix_.m02);
	}

	/**
	 * Y̎擾
	 * @return YxNg
	 */
	inline Vector3 getAxisY() const{
		return Vector3(
			rotationMatrix_.m10, rotationMatrix_.m11, rotationMatrix_.m12);
	}

	/**
	 * Z̎擾
	 * @return XxNg
	 */
	inline Vector3 getAxisZ() const{
		return Vector3(
			rotationMatrix_.m20, rotationMatrix_.m21, rotationMatrix_.m22);
	}

	//--------------------------------------------------------------------------
	/**
	 * 傫Kp̎擾
	 * @param index CfbNX
	 * @return 傫KpxNg
	 */
	inline Vector3 getExtendedAxis(int index) const{
		Assert((index >= 0) && (index < 3));
		float extent = extent_.array[index];
		return Vector3(rotationMatrix_.m[index][0] * extent,
			rotationMatrix_.m[index][1] * extent,
			rotationMatrix_.m[index][2] * extent);
	}

	/**
	 * 傫KpX̎擾
	 * @return 傫KpXxNg
	 */
	inline Vector3 getExtendedAxisX() const{
		return Vector3(rotationMatrix_.m00 * extent_.x,
			rotationMatrix_.m01 * extent_.x, rotationMatrix_.m02 * extent_.x);
	}

	/**
	 * 傫KpY̎擾
	 * @return 傫KpYxNg
	 */
	inline Vector3 getExtendedAxisY() const{
		return Vector3(rotationMatrix_.m10 * extent_.y,
			rotationMatrix_.m11 * extent_.y, rotationMatrix_.m12 * extent_.y);
	}

	/**
	 * 傫KpZ̎擾
	 * @return 傫KpXxNg
	 */
	inline Vector3 getExtendedAxisZ() const{
		return Vector3(rotationMatrix_.m20 * extent_.z,
			rotationMatrix_.m21 * extent_.z, rotationMatrix_.m22 * extent_.z);
	}

	//--------------------------------------------------------------------------
	/**
	 * a̎擾
	 * @param direction a𑪂
	 * @return a
	 */
	inline float getEffectiveDiameter(const Vector3& direction) const{
		return (Math::abs(direction.dotProduct(getExtendedAxisX())) +
			Math::abs(direction.dotProduct(getExtendedAxisY())) +
			Math::abs(direction.dotProduct(getExtendedAxisZ())));
	}

	/**
	 * a̎擾
	 * @param direction a𑪂
	 * @return a
	 */
	inline float getEffectiveRadius(const Vector3& direction) const{
		return getEffectiveDiameter(direction) * 0.5f;
	}

	//--------------------------------------------------------------------------
	/**
	 * R[i[̎擾
	 *
	 * ȉ̐}̃CfbNXɉăR[i[擾܂B
	 * 0S4傫łB
	 * <pre>
	 *     y+
	 *     |
	 *     1----2
	 *    /|   /|
	 *   5-+--4 |
	 *   | 0--+-3-- x+
	 *   |/   |/
	 *   6----7
	 *  /
	 * z+
	 * </pre>
	 * @param index CfbNX
	 * @return R[i[
	 */
	inline Vector3 getCorner(int index) const{
		Assert((index >= 0) && (index < 8));
		Vector3 axisX = getExtendedAxisX();
		Vector3 axisY = getExtendedAxisY();
		Vector3 axisZ = getExtendedAxisZ();
		if(index == 0){
			return center_ - axisX - axisY - axisZ;
		}else if(index == 1){
			return center_ - axisX + axisY - axisZ;
		}else if(index == 2){
			return center_ + axisX + axisY - axisZ;
		}else if(index == 3){
			return center_ + axisX - axisY - axisZ;
		}else if(index == 4){
			return center_ + axisX + axisY + axisZ;
		}else if(index == 5){
			return center_ - axisX + axisY + axisZ;
		}else if(index == 6){
			return center_ - axisX - axisY + axisZ;
		}else if(index == 7){
			return center_ + axisX - axisY + axisZ;
		}
		ErrorOut("OrientedBox::getCorner() Out of index");
		return center_;
	}

	/**
	 * R[i[z̎擾
	 *
	 * ȉ̐}̃CfbNXɉăR[i[擾܂B
	 * 0S4傫łB
	 * <pre>
	 *     y+
	 *     |
	 *     1----2
	 *    /|   /|
	 *   5-+--4 |
	 *   | 0--+-3-- x+
	 *   |/   |/
	 *   6----7
	 *  /
	 * z+
	 * </pre>
	 * @param corner [out] R[i[z
	 */
	inline void getCornerArray(Vector3 corner[8]) const{
		const Matrix33& matrix = rotationMatrix_;
		Vector3 axisX = getExtendedAxisX();
		Vector3 axisY = getExtendedAxisY();
		Vector3 axisZ = getExtendedAxisZ();
		corner[0] = center_ - axisX - axisY - axisZ;
		corner[1] = center_ - axisX + axisY - axisZ;
		corner[2] = center_ + axisX + axisY - axisZ;
		corner[3] = center_ + axisX - axisY - axisZ;
		corner[4] = center_ + axisX + axisY + axisZ;
		corner[5] = center_ - axisX + axisY + axisZ;
		corner[6] = center_ - axisX - axisY + axisZ;
		corner[7] = center_ + axisX - axisY + axisZ;
	}

	//--------------------------------------------------------------------------
	// {bNXZ
	//--------------------------------------------------------------------------
	/**
	 * [{bNXǂ
	 * @return [{bNXȂtrueԂ
	 */
	inline bool isZero() const{
		return extent_.epsilonEquals(Vector3::zero, Math::epsilon);
	}

	/**
	 * Pʃ{bNXǂ
	 * @return Pʃ{bNXȂtrueԂ
	 */
	inline bool isUnit() const{
		return extent_.epsilonEquals(Vector3(0.5f, 0.5f, 0.5f), Math::epsilon);
	}

	//--------------------------------------------------------------------------
	// gXtH[
	//--------------------------------------------------------------------------
	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̎w{bNX
	 */
	inline OrientedBox transform(const Matrix33& matrix) const{
		OrientedBox result;
		result.rotationMatrix_ = matrix * rotationMatrix_;
		result.center_ = matrix * center_;
		result.extent_ = extent_;
		return result;
	}

	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̎w{bNX
	 */
	inline OrientedBox transform(const Matrix34& matrix) const{
		OrientedBox result;
		Matrix33 matrix33;
		matrix33.set(matrix);
		result.rotationMatrix_ = matrix33 * rotationMatrix_;
		result.center_ = matrix * center_;
		result.extent_ = extent_;
		return result;
	}

	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̎w{bNX
	 */
	inline OrientedBox transform(const Matrix44& matrix) const{
		OrientedBox result;
		Matrix33 matrix33;
		matrix33.set(matrix);
		result.rotationMatrix_ = matrix33 * rotationMatrix_;
		result.center_ = matrix * center_;
		result.extent_ = extent_;
		return result;
	}

	//--------------------------------------------------------------------------
	/**
	 * XP[LgXtH[
	 * @param matrix Zs
	 * @return ϊ̎{bNX
	 */
	AxisAlignedBox scaledTransform(const Matrix33& matrix) const;

	/**
	 * XP[LgXtH[
	 * @param matrix Zs
	 * @return ϊ̎{bNX
	 */
	AxisAlignedBox scaledTransform(const Matrix34& matrix) const;

	/**
	 * XP[LgXtH[
	 * @param matrix Zs
	 * @return ϊ̎{bNX
	 */
	AxisAlignedBox scaledTransform(const Matrix44& matrix) const;

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @return 
	 */
	float getDistance(const Vector3& point) const{
		return Math::sqrt(getSquaredDistance(point));
	}

	/**
	 * _̓
	 * @param point 肷_
	 * @return ̓
	 */
	float getSquaredDistance(const Vector3& point) const;

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return 
	 */
	float getDistance(const AxisAlignedBox& axisAlignedBox) const{
		return Math::sqrt(getSquaredDistance(axisAlignedBox));
	}

	/**
	 * {bNX̓
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const AxisAlignedBox& axisAlignedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return 
	 */
	float getDistance(const Capsule& capsule) const{
		return Math::sqrt(getSquaredDistance(capsule));
	}

	/**
	 * JvZ̓
	 * @param capsule 肷JvZ
	 * @return ̓
	 */
	float getSquaredDistance(const Capsule& capsule) const;

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return 
	 */
	float getDistance(const Cone& cone) const{
		return Math::sqrt(getSquaredDistance(cone));
	}

	/**
	 * R[̓
	 * @param cone 肷R[
	 * @return ̓
	 */
	float getSquaredDistance(const Cone& cone) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @return 
	 */
	float getDistance(const Line& line) const{
		return Math::sqrt(getSquaredDistance(line));
	}

	/**
	 * C̓
	 * @param line 肷郉C
	 * @return ̓
	 */
	float getSquaredDistance(const Line& line) const;

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return 
	 */
	float getDistance(const OrientedBox& orientedBox) const{
		return Math::sqrt(getSquaredDistance(orientedBox));
	}

	/**
	 * w{bNX̓
	 * @param orientedBox 肷w{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const OrientedBox& orientedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * ʋ
	 * @param plane 肷镽
	 * @return 
	 */
	float getDistance(const Plane& plane) const;

	/**
	 * ʋ̓
	 * @param plane 肷镽
	 * @return ̓
	 */
	float getSquaredDistance(const Plane& plane) const{
		float distance = getDistance(plane);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @return 
	 */
	float getDistance(const Ray& ray) const{
		return Math::sqrt(getSquaredDistance(ray));
	}

	/**
	 * C̓
	 * @param ray 肷郌C
	 * @return ̓
	 */
	float getSquaredDistance(const Ray& ray) const;

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @return 
	 */
	float getDistance(const Segment& segment) const{
		return Math::sqrt(getSquaredDistance(segment));
	}

	/**
	 * ZOg̓
	 * @param segment 肷ZOg
	 * @return ̓
	 */
	float getSquaredDistance(const Segment& segment) const;

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return 
	 */
	float getDistance(const Sphere& sphere) const{
		return Math::sqrt(getSquaredDistance(sphere));
	}

	/**
	 * ̓
	 * @param sphere 肷鋅
	 * @return ̓
	 */
	float getSquaredDistance(const Sphere& sphere) const;

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return 
	 */
	float getDistance(const Triangle& triangle) const{
		return Math::sqrt(getSquaredDistance(triangle));
	}

	/**
	 * Op̓
	 * @param triangle 肷Op
	 * @return ̓
	 */
	float getSquaredDistance(const Triangle& triangle) const;

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @return Ătrue
	 */
	bool intersect(const Vector3& point) const;

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return Ătrue
	 */
	bool intersect(const AxisAlignedBox& axisAlignedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return Ătrue
	 */
	bool intersect(const Capsule& capsule) const;

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return Ătrue
	 */
	bool intersect(const Cone& cone) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @return Ătrue
	 */
	bool intersect(const Line& line) const;

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return Ătrue
	 */
	bool intersect(const OrientedBox& orientedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * ʌ
	 * @param plane 肷镽
	 * @return Ătrue
	 */
	bool intersect(const Plane& plane) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @return Ătrue
	 */
	bool intersect(const Ray& ray) const;

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @return Ătrue
	 */
	bool intersect(const Segment& segment) const;

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return Ătrue
	 */
	bool intersect(const Sphere& sphere) const;

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return Ătrue
	 */
	bool intersect(const Triangle& triangle) const;

	//--------------------------------------------------------------------------
	// _Z
	//--------------------------------------------------------------------------
	/**
	 * w{bNXǂ
	 * @param target rw{bNX
	 * @return lłtrueԂ
	 */
	inline bool operator ==(const OrientedBox& target) const{
		return ((rotationMatrix_ == target.rotationMatrix_) &&
			(center_ == target.center_) && (extent_ == target.extent_));
	}

	/**
	 * w{bNXǂ
	 * @param target rw{bNX
	 * @param epsilon 덷
	 * @return 덷͈͓̔œlłtrueԂ
	 */
	inline bool epsilonEquals(
		const OrientedBox& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			rotationMatrix_.epsilonEquals(target.rotationMatrix_, epsilon) &&
			center_.epsilonEquals(target.center_, epsilon) &&
			extent_.epsilonEquals(target.extent_, epsilon));
	}

	/**
	 * w{bNXłȂǂ
	 * @param target rw{bNX
	 * @return łȂlłtrueԂ
	 */
	inline bool operator !=(const OrientedBox& target) const{
		return ((rotationMatrix_ != target.rotationMatrix_) ||
			(center_ != target.center_) || (extent_ != target.extent_));
	}

	/**
	 * w{bNXłȂǂ
	 * @param target rw{bNX
	 * @param epsilon 덷
	 * @return 덷͈͓̔œłȂlłtrueԂ
	 */
	inline bool notEpsilonEquals(
		const OrientedBox& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			rotationMatrix_.notEpsilonEquals(target.rotationMatrix_, epsilon) ||
			center_.notEpsilonEquals(target.center_, epsilon) ||
			extent_.notEpsilonEquals(target.extent_, epsilon));
	}

	//--------------------------------------------------------------------------
	// ̑
	//--------------------------------------------------------------------------
	/**
	 * 
	 * @return w{bNX̕\L
	 */
	inline String toString() const{
		String returnString;
		returnString.format("{ { ( %.8f, %.8f, %.8f ) "
			"( %.8f, %.8f, %.8f ) ( %.8f, %.8f, %.8f ) } "
			"( %.8f, %.8f, %.8f ) ( %.8f, %.8f, %.8f ) }",
			rotationMatrix_.m00, rotationMatrix_.m01, rotationMatrix_.m02,
			rotationMatrix_.m10, rotationMatrix_.m11, rotationMatrix_.m12,
			rotationMatrix_.m20, rotationMatrix_.m21, rotationMatrix_.m22,
			center_.x, center_.y, center_.z, extent_.x, extent_.y, extent_.z);
		return returnString;
	}

private:
	//--------------------------------------------------------------------------
	// oϐ
	//--------------------------------------------------------------------------
	// ]s
	Matrix33 rotationMatrix_;
	// S
	Vector3 center_;
	// 傫
	Vector3 extent_;

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of ORIENTED_BOX_H_
//------------------------------------------------------------------------------
