//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * R[wb_
 * @author Junpee
 */

#ifndef CONE_H_
#define CONE_H_

#include <Core/Primitive/Vector3.h>
#include <Core/Primitive/Matrix33.h>
#include <Core/Primitive/Matrix34.h>
#include <Core/Primitive/Matrix44.h>

namespace Lamp{

class AxisAlignedBox;
class Capsule;
class Line;
class OrientedBox;
class Plane;
class Ray;
class Segment;
class Sphere;
class Triangle;

//------------------------------------------------------------------------------
/**
 * R[
 *
 * ̃NX͌pȂŉB
 */
class Cone{
public:
	//--------------------------------------------------------------------------
	// 萔
	//--------------------------------------------------------------------------
	/// [R[
	static const Cone zero;

	//--------------------------------------------------------------------------
	// RXgN^
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 *
	 * ̃RXgN^͏l̐ݒsȂߒl͕słB
	 */
	Cone(){}

	/**
	 * RXgN^
	 * @param origin _̏l
	 * @param direction ̏l
	 * @param theta V[^̏l
	 */
	inline Cone(
		const Vector3& origin, const Vector3& direction, float theta) :
		origin_(origin), direction_(direction), theta_(theta){
		Assert((theta_ >= 0.f) && (theta_ <= Math::halfPI));
	}

	/**
	 * RXgN^
	 * @param originX _X̏l
	 * @param originY _Y̏l
	 * @param originZ _Z̏l
	 * @param directionX X̏l
	 * @param directionY Y̏l
	 * @param directionZ Z̏l
	 * @param theta V[^̏l
	 */
	inline Cone(float originX, float originY, float originZ,
		float directionX, float directionY, float directionZ, float theta) :
		origin_(originX, originY, originZ),
		direction_(directionX, directionY, directionZ), theta_(theta){
		Assert((theta_ >= 0.f) && (theta_ <= Math::halfPI));
	}

	/**
	 * RXgN^
	 * @param source lz
	 */
	inline explicit Cone(const float* const source) :
		origin_(source[0], source[1], source[2]),
		direction_(source[3], source[4], source[5]), theta_(source[6]){
		Assert((theta_ >= 0.f) && (theta_ <= Math::halfPI));
	}

	//--------------------------------------------------------------------------
	// l̐ݒ
	//--------------------------------------------------------------------------
	/**
	 * l̐ݒ
	 * @param origin ݒ肷錴_
	 * @param direction ݒ肷
	 * @param theta ݒ肷V[^
	 */
	inline void set(
		const Vector3& origin, const Vector3& direction, float theta){
		origin_ = origin;
		direction_ = direction;
		theta_ = theta;
		Assert((theta_ >= 0.f) && (theta_ <= Math::halfPI));
	}

	/**
	 * l̐ݒ
	 * @param originX ݒ肷錴_X
	 * @param originY ݒ肷錴_Y
	 * @param originZ ݒ肷錴_Z
	 * @param directionX ݒ肷X
	 * @param directionY ݒ肷Y
	 * @param directionZ ݒ肷Z
	 * @param theta ݒ肷V[^
	 */
	inline void set(float originX, float originY, float originZ,
		float directionX, float directionY, float directionZ, float theta){
		origin_.set(originX, originY, originZ);
		direction_.set(directionX, directionY, directionZ);
		theta_ = theta;
		Assert((theta_ >= 0.f) && (theta_ <= Math::halfPI));
	}

	/**
	 * l̐ݒ
	 * @param source ݒlz
	 */
	inline void set(const float* const source){
		origin_.set(source[0], source[1], source[2]);
		direction_.set(source[3], source[4], source[5]);
		theta_ = source[6];
		Assert((theta_ >= 0.f) && (theta_ <= Math::halfPI));
	}

	//--------------------------------------------------------------------------
	/**
	 * _̐ݒ
	 * @param origin ݒ肷錴_
	 */
	inline void setOrigin(const Vector3& origin){ origin_ = origin; }

	/**
	 * ̐ݒ
	 * @param direction ݒ肷
	 */
	inline void setDirection(const Vector3& direction){
		direction_ = direction;
	}

	/**
	 * V[^̐ݒ
	 * @param theta ݒ肷V[^
	 */
	inline void setTheta(float theta){
		theta_ = theta;
		Assert((theta_ >= 0.f) && (theta_ <= Math::halfPI));
	}

	//--------------------------------------------------------------------------
	/**
	 * ʒu̐ݒ
	 * @param source \[Xʒu
	 * @param target ^[Qbgʒu
	 */
	inline void setPositions(const Vector3& source, const Vector3& target){
		origin_ = source;
		direction_ = (target - source);
	}

	/**
	 * px̐ݒ
	 * @param angle px
	 */
	inline void setAngle(float angle){
		theta_ = angle * 0.5f;
		Assert((theta_ >= 0.f) && (theta_ <= Math::halfPI));
	}

	//--------------------------------------------------------------------------
	// l̎擾
	//--------------------------------------------------------------------------
	/**
	 * _̎擾
	 * @return _
	 */
	inline const Vector3& getOrigin() const{ return origin_; }

	/**
	 * ̎擾
	 * @return 
	 */
	inline const Vector3& getDirection() const{ return direction_; }

	/**
	 * V[^̎擾
	 * @return V[^
	 */
	inline float getTheta() const{ return theta_; }

	//--------------------------------------------------------------------------
	/**
	 * \[Xʒu̎擾
	 * @return \[Xʒu
	 */
	inline const Vector3& getSourcePosition() const{ return origin_; }

	/**
	 * ^[Qbgʒu̎擾
	 * @return ^[Qbgʒu
	 */
	inline Vector3 getTargetPosition() const{ return (origin_ + direction_); }

	/**
	 * px̎擾
	 * @return px
	 */
	inline float getAngle() const{ return (theta_ * 2.f); }

	/**
	 * TC擾
	 * @return TC
	 */
	inline float getSin() const{ return Math::sin(theta_); }

	/**
	 * RTC擾
	 * @return RTC
	 */
	inline float getCos() const{ return Math::cos(theta_); }

	//--------------------------------------------------------------------------
	// R[Z
	//--------------------------------------------------------------------------
	/**
	 * [R[ǂ
	 * @return [R[Ȃtrue
	 */
	inline bool isZero() const{
		return direction_.epsilonEquals(Vector3::zero, Math::epsilon);
	}

	//--------------------------------------------------------------------------
	// gXtH[
	//--------------------------------------------------------------------------
	/**
	 * gXtH[
	 *
	 * XP[ɂ͑ΉĂ܂
	 * @param matrix Zs
	 * @return ϊ̃R[
	 */
	inline Cone transform(const Matrix33& matrix) const{
		return Cone(matrix * origin_, matrix * direction_, theta_);
	}

	/**
	 * gXtH[
	 *
	 * XP[ɂ͑ΉĂ܂
	 * @param matrix Zs
	 * @return ϊ̃R[
	 */
	inline Cone transform(const Matrix34& matrix) const{
		return Cone(matrix * origin_, matrix.multiply33(direction_), theta_);
	}

	/**
	 * gXtH[
	 *
	 * XP[ɂ͑ΉĂ܂
	 * @param matrix Zs
	 * @return ϊ̃R[
	 */
	inline Cone transform(const Matrix44& matrix) const{
		return Cone(matrix * origin_, matrix.multiply33(direction_), theta_);
	}

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @return 
	 */
	float getDistance(const Vector3& point) const{
		return Math::sqrt(getSquaredDistance(point));
	}

	/**
	 * _̓
	 * @param point 肷_
	 * @return ̓
	 */
	float getSquaredDistance(const Vector3& point) const;

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return 
	 */
	float getDistance(const AxisAlignedBox& axisAlignedBox) const{
		return Math::sqrt(getSquaredDistance(axisAlignedBox));
	}

	/**
	 * {bNX̓
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const AxisAlignedBox& axisAlignedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return 
	 */
	float getDistance(const Capsule& capsule) const{
		return Math::sqrt(getSquaredDistance(capsule));
	}

	/**
	 * JvZ̓
	 * @param capsule 肷JvZ
	 * @return ̓
	 */
	float getSquaredDistance(const Capsule& capsule) const;

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return 
	 */
	float getDistance(const Cone& cone) const{
		return Math::sqrt(getSquaredDistance(cone));
	}

	/**
	 * R[̓
	 * @param cone 肷R[
	 * @return ̓
	 */
	float getSquaredDistance(const Cone& cone) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @return 
	 */
	float getDistance(const Line& line) const{
		return Math::sqrt(getSquaredDistance(line));
	}

	/**
	 * C̓
	 * @param line 肷郉C
	 * @return ̓
	 */
	float getSquaredDistance(const Line& line) const;

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return 
	 */
	float getDistance(const OrientedBox& orientedBox) const{
		return Math::sqrt(getSquaredDistance(orientedBox));
	}

	/**
	 * w{bNX̓
	 * @param orientedBox 肷w{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const OrientedBox& orientedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * ʋ
	 * @param plane 肷镽
	 * @return 
	 */
	float getDistance(const Plane& plane) const;

	/**
	 * ʋ̓
	 * @param plane 肷镽
	 * @return ̓
	 */
	float getSquaredDistance(const Plane& plane) const{
		float distance = getDistance(plane);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @return 
	 */
	float getDistance(const Ray& ray) const{
		return Math::sqrt(getSquaredDistance(ray));
	}

	/**
	 * C̓
	 * @param ray 肷郌C
	 * @return ̓
	 */
	float getSquaredDistance(const Ray& ray) const;

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @return 
	 */
	float getDistance(const Segment& segment) const{
		return Math::sqrt(getSquaredDistance(segment));
	}

	/**
	 * ZOg̓
	 * @param segment 肷ZOg
	 * @return ̓
	 */
	float getSquaredDistance(const Segment& segment) const;

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return 
	 */
	float getDistance(const Sphere& sphere) const{
		return Math::sqrt(getSquaredDistance(sphere));
	}

	/**
	 * ̓
	 * @param sphere 肷鋅
	 * @return ̓
	 */
	float getSquaredDistance(const Sphere& sphere) const;

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return 
	 */
	float getDistance(const Triangle& triangle) const{
		return Math::sqrt(getSquaredDistance(triangle));
	}

	/**
	 * Op̓
	 * @param triangle 肷Op
	 * @return ̓
	 */
	float getSquaredDistance(const Triangle& triangle) const;

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @return Ătrue
	 */
	bool intersect(const Vector3& point) const;

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return Ătrue
	 */
	bool intersect(const AxisAlignedBox& axisAlignedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return Ătrue
	 */
	bool intersect(const Capsule& capsule) const;

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return Ătrue
	 */
	bool intersect(const Cone& cone) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @return Ătrue
	 */
	bool intersect(const Line& line) const;

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return Ătrue
	 */
	bool intersect(const OrientedBox& orientedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * ʌ
	 * @param plane 肷镽
	 * @return Ătrue
	 */
	bool intersect(const Plane& plane) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @return Ătrue
	 */
	bool intersect(const Ray& ray) const;

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @return Ătrue
	 */
	bool intersect(const Segment& segment) const;

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return Ătrue
	 */
	bool intersect(const Sphere& sphere) const;

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return Ătrue
	 */
	bool intersect(const Triangle& triangle) const;

	//--------------------------------------------------------------------------
	// _Z
	//--------------------------------------------------------------------------
	/**
	 * R[ǂ
	 * @param target rR[
	 * @return lłtrueԂ
	 */
	inline bool operator ==(const Cone& target) const{
		return ((origin_ == target.origin_) &&
			(direction_ == target.direction_) &&
			(theta_ == target.theta_));
	}

	/**
	 * R[ǂ
	 * @param target rR[
	 * @param epsilon 덷
	 * @return 덷͈͓̔œlłtrueԂ
	 */
	inline bool epsilonEquals(
		const Cone& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (origin_.epsilonEquals(target.origin_, epsilon) &&
			direction_.epsilonEquals(target.direction_, epsilon) &&
			(Math::abs(theta_ - target.theta_) <= epsilon));
	}

	/**
	 * R[łȂǂ
	 * @param target rR[
	 * @return łȂlłtrueԂ
	 */
	inline bool operator !=(const Cone& target) const{
		return ((origin_ != target.origin_) ||
			(direction_ != target.direction_) || (theta_ != target.theta_));
	}

	/**
	 * R[łȂǂ
	 * @param target rR[
	 * @param epsilon 덷
	 * @return 덷͈͓̔œłȂlłtrueԂ
	 */
	inline bool notEpsilonEquals(
		const Cone& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (origin_.notEpsilonEquals(target.origin_, epsilon) ||
			direction_.notEpsilonEquals(target.direction_, epsilon) ||
			(Math::abs(theta_ - target.theta_) > epsilon));
	}

	//--------------------------------------------------------------------------
	// ̑
	//--------------------------------------------------------------------------
	/**
	 * 
	 * @return R[̕\L
	 */
	inline String toString() const{
		String returnString;
		returnString.format(
			"{ ( %.8f, %.8f, %.8f ) ( %.8f, %.8f, %.8f ) %.8f }",
			origin_.x, origin_.y, origin_.z,
			direction_.x, direction_.y, direction_.z, theta_);
		return returnString;
	}

private:
	//--------------------------------------------------------------------------
	// oϐ
	//--------------------------------------------------------------------------
	// _
	Vector3 origin_;
	// 
	Vector3 direction_;
	// V[^
	float theta_;

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of CONE_H_
//------------------------------------------------------------------------------
