//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * {bNXwb_
 * @author Junpee
 */

#ifndef AXIS_ALIGNED_BOX_H_
#define AXIS_ALIGNED_BOX_H_

#include <Core/Primitive/Vector3.h>
#include <Core/Primitive/Matrix33.h>
#include <Core/Primitive/Matrix34.h>
#include <Core/Primitive/Matrix44.h>

namespace Lamp{

class Capsule;
class Cone;
class Line;
class OrientedBox;
class Plane;
class Ray;
class Segment;
class Sphere;
class Triangle;

//------------------------------------------------------------------------------
/**
 * {bNX
 *
 * ̃NX͌pȂŉB
 */
class AxisAlignedBox{
public:
	//--------------------------------------------------------------------------
	// 萔
	//--------------------------------------------------------------------------
	/// [{bNX
	static const AxisAlignedBox zero;

	/// Pʃ{bNX
	static const AxisAlignedBox unit;

	//--------------------------------------------------------------------------
	// RXgN^
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 *
	 * ̃RXgN^͏l̐ݒsȂߒl͕słB
	 */
	inline AxisAlignedBox(){}

	/**
	 * RXgN^
	 * @param minimum ŏl̏l
	 * @param maximum ől̏l
	 */
	inline AxisAlignedBox(const Vector3& minimum, const Vector3& maximum) :
		minimum_(minimum), maximum_(maximum){
		Assert(minimum_.x <= maximum_.x);
		Assert(minimum_.y <= maximum_.y);
		Assert(minimum_.z <= maximum_.z);
	}

	/**
	 * RXgN^
	 * @param minimumX ŏlX̏l
	 * @param minimumY ŏlY̏l
	 * @param minimumZ ŏlZ̏l
	 * @param maximumX őlX̏l
	 * @param maximumY őlY̏l
	 * @param maximumZ őlZ̏l
	 */
	inline AxisAlignedBox(
		float minimumX, float minimumY, float minimumZ,
		float maximumX, float maximumY, float maximumZ) :
		minimum_(minimumX, minimumY, minimumZ),
		maximum_(maximumX, maximumY, maximumZ){
		Assert(minimum_.x <= maximum_.x);
		Assert(minimum_.y <= maximum_.y);
		Assert(minimum_.z <= maximum_.z);
	}

	/**
	 * RXgN^
	 * @param source lz
	 */
	inline explicit AxisAlignedBox(const float* const source) :
		minimum_(source[0], source[1], source[2]),
		maximum_(source[3], source[4], source[5]){
		Assert(minimum_.x <= maximum_.x);
		Assert(minimum_.y <= maximum_.y);
		Assert(minimum_.z <= maximum_.z);
	}

	//--------------------------------------------------------------------------
	// l̐ݒ
	//--------------------------------------------------------------------------
	/**
	 * l̐ݒ
	 * @param minimum ݒ肷ŏl
	 * @param maximum ݒ肷ől
	 */
	inline void set(const Vector3& minimum, const Vector3& maximum){
		minimum_ = minimum;
		maximum_ = maximum;
		Assert(minimum_.x <= maximum_.x);
		Assert(minimum_.y <= maximum_.y);
		Assert(minimum_.z <= maximum_.z);
	}

	/**
	 * l̐ݒ
	 * @param minimumX ݒ肷ŏlX
	 * @param minimumY ݒ肷ŏlY
	 * @param minimumZ ݒ肷ŏlZ
	 * @param maximumX ݒ肷őlX
	 * @param maximumY ݒ肷őlY
	 * @param maximumZ ݒ肷őlZ
	 */
	inline void set(
		float minimumX, float minimumY, float minimumZ,
		float maximumX, float maximumY, float maximumZ){
		minimum_.set(minimumX, minimumY, minimumZ);
		maximum_.set(maximumX, maximumY, maximumZ);
		Assert(minimum_.x <= maximum_.x);
		Assert(minimum_.y <= maximum_.y);
		Assert(minimum_.z <= maximum_.z);
	}

	/**
	 * l̐ݒ
	 * @param source ݒlz
	 */
	inline void set(const float* const source){
		minimum_.set(source[0], source[1], source[2]);
		maximum_.set(source[3], source[4], source[5]);
		Assert(minimum_.x <= maximum_.x);
		Assert(minimum_.y <= maximum_.y);
		Assert(minimum_.z <= maximum_.z);
	}

	//--------------------------------------------------------------------------
	// l̎擾
	//--------------------------------------------------------------------------
	/**
	 * ŏl̎擾
	 * @return ŏl
	 */
	inline const Vector3& getMinimum() const{ return minimum_; }

	/**
	 * ől̎擾
	 * @return ől
	 */
	inline const Vector3& getMaximum() const{ return maximum_; }

	/**
	 * TCY̎擾
	 * @return TCY
	 */
	inline Vector3 getSize() const{ return (maximum_ - minimum_); }

	/**
	 * S̎擾
	 * @return S
	 */
	inline Vector3 getCenter() const{ return (maximum_ + minimum_) * 0.5f; }

	/**
	 * R[i[̎擾
	 *
	 * ȉ̐}̃CfbNXɉăR[i[擾܂B
	 * 0ŏl4őlłB
	 * <pre>
	 *     y+
	 *     |
	 *     1----2
	 *    /|   /|
	 *   5-+--4 |
	 *   | 0--+-3-- x+
	 *   |/   |/
	 *   6----7
	 *  /
	 * z+
	 * </pre>
	 * @param index CfbNX
	 * @return R[i[
	 */
	inline Vector3 getCorner(int index) const{
		if(index == 0){
			return minimum_;
		}else if(index == 1){
			return Vector3(minimum_.x, maximum_.y, minimum_.z);
		}else if(index == 2){
			return Vector3(maximum_.x, maximum_.y, minimum_.z);
		}else if(index == 3){
			return Vector3(maximum_.x, minimum_.y, minimum_.z);
		}else if(index == 4){
			return maximum_;
		}else if(index == 5){
			return Vector3(minimum_.x, maximum_.y, maximum_.z);
		}else if(index == 6){
			return Vector3(minimum_.x, minimum_.y, maximum_.z);
		}else if(index == 7){
			return Vector3(maximum_.x, minimum_.y, maximum_.z);
		}
		ErrorOut("AxisAlignedBox::getCorner() Out of index");
		return Vector3(0.f, 0.f, 0.f);
	}

	/**
	 * R[i[z̎擾
	 *
	 * ȉ̐}̃CfbNXɉăR[i[擾܂B
	 * 0ŏl4őlłB
	 * <pre>
	 *     y+
	 *     |
	 *     1----2
	 *    /|   /|
	 *   5-+--4 |
	 *   | 0--+-3-- x+
	 *   |/   |/
	 *   6----7
	 *  /
	 * z+
	 * </pre>
	 * @param corner [out] R[i[z
	 */
	inline void getCornerArray(Vector3 corner[8]) const{
		corner[0] = minimum_;
		corner[1].set(minimum_.x, maximum_.y, minimum_.z);
		corner[2].set(maximum_.x, maximum_.y, minimum_.z);
		corner[3].set(maximum_.x, minimum_.y, minimum_.z);
		corner[4] = maximum_;
		corner[5].set(minimum_.x, maximum_.y, maximum_.z);
		corner[6].set(minimum_.x, minimum_.y, maximum_.z);
		corner[7].set(maximum_.x, minimum_.y, maximum_.z);
	}

	//--------------------------------------------------------------------------
	// {bNXZ
	//--------------------------------------------------------------------------
	/**
	 * }[W
	 * @param box }[W{bNX
	 * @return }[Wꂽ{bNX
	 */
	inline const AxisAlignedBox& merge(const AxisAlignedBox& box){
		if(minimum_.x > box.minimum_.x){ minimum_.x = box.minimum_.x; }
		if(maximum_.x < box.maximum_.x){ maximum_.x = box.maximum_.x; }
		if(minimum_.y > box.minimum_.y){ minimum_.y = box.minimum_.y; }
		if(maximum_.y < box.maximum_.y){ maximum_.y = box.maximum_.y; }
		if(minimum_.z > box.minimum_.z){ minimum_.z = box.minimum_.z; }
		if(maximum_.z < box.maximum_.z){ maximum_.z = box.maximum_.z; }
		return *this;
	}

	/**
	 * }[W
	 * @param vector }[WxNg
	 * @return }[Wꂽ{bNX
	 */
	inline const AxisAlignedBox& merge(const Vector3& vector){
		if(minimum_.x > vector.x){ minimum_.x = vector.x; }
		else if(maximum_.x < vector.x){ maximum_.x = vector.x; }
		if(minimum_.y > vector.y){ minimum_.y = vector.y; }
		else if(maximum_.y < vector.y){ maximum_.y = vector.y; }
		if(minimum_.z > vector.z){ minimum_.z = vector.z; }
		else if(maximum_.z < vector.z){ maximum_.z = vector.z; }
		return *this;
	}

	//--------------------------------------------------------------------------
	/**
	 * [{bNXǂ
	 * @return [{bNXȂtrueԂ
	 */
	inline bool isZero() const{
		return epsilonEquals(zero, Math::epsilon);
	}

	/**
	 * Pʃ{bNXǂ
	 * @return Pʃ{bNXȂtrueԂ
	 */
	inline bool isUnit() const{
		return epsilonEquals(unit, Math::epsilon);
	}

	//--------------------------------------------------------------------------
	// gXtH[
	//--------------------------------------------------------------------------
	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̎{bNX
	 */
	inline AxisAlignedBox transform(const Matrix33& matrix) const{
		AxisAlignedBox result;
		Vector3 corner[8];
		getCornerArray(corner);
		result.maximum_ = result.minimum_ = matrix * corner[7];
		for(int i = 0; i < 7; i++){ result.merge(matrix * corner[i]); }
		return result;
	}

	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̎{bNX
	 */
	inline AxisAlignedBox transform(const Matrix34& matrix) const{
		AxisAlignedBox result;
		Vector3 corner[8];
		getCornerArray(corner);
		result.maximum_ = result.minimum_ = matrix * corner[7];
		for(int i = 0; i < 7; i++){ result.merge(matrix * corner[i]); }
		return result;
	}

	/**
	 * gXtH[
	 * @param matrix Zs
	 * @return ϊ̎{bNX
	 */
	inline AxisAlignedBox transform(const Matrix44& matrix) const{
		AxisAlignedBox result;
		Vector3 corner[8];
		getCornerArray(corner);
		result.maximum_ = result.minimum_ = matrix * corner[7];
		for(int i = 0; i < 7; i++){ result.merge(matrix * corner[i]); }
		return result;
	}

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @return 
	 */
	float getDistance(const Vector3& point) const{
		return Math::sqrt(getSquaredDistance(point));
	}

	/**
	 * _̓
	 * @param point 肷_
	 * @return ̓
	 */
	float getSquaredDistance(const Vector3& point) const;

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return 
	 */
	float getDistance(const AxisAlignedBox& axisAlignedBox) const{
		return Math::sqrt(getSquaredDistance(axisAlignedBox));
	}

	/**
	 * {bNX̓
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const AxisAlignedBox& axisAlignedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return 
	 */
	float getDistance(const Capsule& capsule) const{
		return Math::sqrt(getSquaredDistance(capsule));
	}

	/**
	 * JvZ̓
	 * @param capsule 肷JvZ
	 * @return ̓
	 */
	float getSquaredDistance(const Capsule& capsule) const;

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return 
	 */
	float getDistance(const Cone& cone) const{
		return Math::sqrt(getSquaredDistance(cone));
	}

	/**
	 * R[̓
	 * @param cone 肷R[
	 * @return ̓
	 */
	float getSquaredDistance(const Cone& cone) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @return 
	 */
	float getDistance(const Line& line) const{
		return Math::sqrt(getSquaredDistance(line));
	}

	/**
	 * C̓
	 * @param line 肷郉C
	 * @return ̓
	 */
	float getSquaredDistance(const Line& line) const;

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return 
	 */
	float getDistance(const OrientedBox& orientedBox) const{
		return Math::sqrt(getSquaredDistance(orientedBox));
	}

	/**
	 * w{bNX̓
	 * @param orientedBox 肷w{bNX
	 * @return ̓
	 */
	float getSquaredDistance(const OrientedBox& orientedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * ʋ
	 * @param plane 肷镽
	 * @return 
	 */
	float getDistance(const Plane& plane) const;

	/**
	 * ʋ̓
	 * @param plane 肷镽
	 * @return ̓
	 */
	float getSquaredDistance(const Plane& plane) const{
		float distance = getDistance(plane);
		return (distance * distance);
	}

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @return 
	 */
	float getDistance(const Ray& ray) const{
		return Math::sqrt(getSquaredDistance(ray));
	}

	/**
	 * C̓
	 * @param ray 肷郌C
	 * @return ̓
	 */
	float getSquaredDistance(const Ray& ray) const;

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @return 
	 */
	float getDistance(const Segment& segment) const{
		return Math::sqrt(getSquaredDistance(segment));
	}

	/**
	 * ZOg̓
	 * @param segment 肷ZOg
	 * @return ̓
	 */
	float getSquaredDistance(const Segment& segment) const;

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return 
	 */
	float getDistance(const Sphere& sphere) const{
		return Math::sqrt(getSquaredDistance(sphere));
	}

	/**
	 * ̓
	 * @param sphere 肷鋅
	 * @return ̓
	 */
	float getSquaredDistance(const Sphere& sphere) const;

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return 
	 */
	float getDistance(const Triangle& triangle) const{
		return Math::sqrt(getSquaredDistance(triangle));
	}

	/**
	 * Op̓
	 * @param triangle 肷Op
	 * @return ̓
	 */
	float getSquaredDistance(const Triangle& triangle) const;

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * _
	 * @param point 肷_
	 * @return Ătrue
	 */
	bool intersect(const Vector3& point) const;

	//--------------------------------------------------------------------------
	/**
	 * {bNX
	 * @param axisAlignedBox 肷鎲{bNX
	 * @return Ătrue
	 */
	bool intersect(const AxisAlignedBox& axisAlignedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * JvZ
	 * @param capsule 肷JvZ
	 * @return Ătrue
	 */
	bool intersect(const Capsule& capsule) const;

	//--------------------------------------------------------------------------
	/**
	 * R[
	 * @param cone 肷R[
	 * @return Ătrue
	 */
	bool intersect(const Cone& cone) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param line 肷郉C
	 * @return Ătrue
	 */
	bool intersect(const Line& line) const;

	//--------------------------------------------------------------------------
	/**
	 * w{bNX
	 * @param orientedBox 肷w{bNX
	 * @return Ătrue
	 */
	bool intersect(const OrientedBox& orientedBox) const;

	//--------------------------------------------------------------------------
	/**
	 * ʌ
	 * @param plane 肷镽
	 * @return Ătrue
	 */
	bool intersect(const Plane& plane) const;

	//--------------------------------------------------------------------------
	/**
	 * C
	 * @param ray 肷郌C
	 * @return Ătrue
	 */
	bool intersect(const Ray& ray) const;

	//--------------------------------------------------------------------------
	/**
	 * ZOg
	 * @param segment 肷ZOg
	 * @return Ătrue
	 */
	bool intersect(const Segment& segment) const;

	//--------------------------------------------------------------------------
	/**
	 * 
	 * @param sphere 肷鋅
	 * @return Ătrue
	 */
	bool intersect(const Sphere& sphere) const;

	//--------------------------------------------------------------------------
	/**
	 * Op
	 * @param triangle 肷Op
	 * @return Ătrue
	 */
	bool intersect(const Triangle& triangle) const;

	//--------------------------------------------------------------------------
	// _Z
	//--------------------------------------------------------------------------
	/**
	 * {bNXǂ
	 * @param target r鎲{bNX
	 * @return lłtrueԂ
	 */
	inline bool operator ==(const AxisAlignedBox& target) const{
		return ((minimum_ == target.minimum_) && (maximum_ == target.maximum_));
	}

	/**
	 * {bNXǂ
	 * @param target r鎲{bNX
	 * @param epsilon 덷
	 * @return 덷͈͓̔œlłtrueԂ
	 */
	inline bool epsilonEquals(
		const AxisAlignedBox& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (minimum_.epsilonEquals(target.minimum_, epsilon) &&
			maximum_.epsilonEquals(target.maximum_, epsilon));
	}

	/**
	 * {bNXłȂǂ
	 * @param target r鎲{bNX
	 * @return łȂlłtrueԂ
	 */
	inline bool operator !=(const AxisAlignedBox& target) const{
		return ((minimum_ != target.minimum_) || (maximum_ != target.maximum_));
	}

	/**
	 * {bNXłȂǂ
	 * @param target r鎲{bNX
	 * @param epsilon 덷
	 * @return 덷͈͓̔œłȂlłtrueԂ
	 */
	inline bool notEpsilonEquals(
		const AxisAlignedBox& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (minimum_.notEpsilonEquals(target.minimum_, epsilon) ||
			maximum_.notEpsilonEquals(target.maximum_, epsilon));
	}

	//--------------------------------------------------------------------------
	// ̑
	//--------------------------------------------------------------------------
	/**
	 * 
	 * @return {bNX̕\L
	 */
	inline String toString() const{
		String returnString;
		returnString.format("{ ( %.8f, %.8f, %.8f ) ( %.8f, %.8f, %.8f ) }",
			minimum_.x, minimum_.y, minimum_.z,
			maximum_.x, maximum_.y, maximum_.z);
		return returnString;
	}

private:
	//--------------------------------------------------------------------------
	// oϐ
	//--------------------------------------------------------------------------
	// ŏl
	Vector3 minimum_;
	// ől
	Vector3 maximum_;

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of AXIS_ALIGNED_BOX_H_
//------------------------------------------------------------------------------
