//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * lwb_
 * @author Junpee
 */

#ifndef QUATERNION_H_
#define QUATERNION_H_

#include <Core/Primitive/Vector3.h>

namespace Lamp{

//------------------------------------------------------------------------------
/**
 * l
 *
 * ̃NX͌pȂŉB
 */
class Quaternion{
public:
	//--------------------------------------------------------------------------
	// oϐ
	//--------------------------------------------------------------------------
	/// oϐ
	union{
		/// evf
		struct{
			/// Xl
			float x;
			/// Yl
			float y;
			/// Zl
			float z;
			/// Wl
			float w;
		};

		/// z
		float array[4];
	};

	//--------------------------------------------------------------------------
	// 萔
	//--------------------------------------------------------------------------
	/// [l
	static const Quaternion zero;

	/// Pl
	static const Quaternion identity;

	//--------------------------------------------------------------------------
	// RXgN^
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 *
	 * ̃RXgN^͏l̐ݒsȂߒl͕słB
	 */
	inline Quaternion(){}

	/**
	 * RXgN^
	 * @param sourceX X̏l
	 * @param sourceY Y̏l
	 * @param sourceZ Z̏l
	 * @param sourceW W̏l
	 */
	inline Quaternion(
		float sourceX, float sourceY, float sourceZ, float sourceW) :
		x(sourceX), y(sourceY), z(sourceZ), w(sourceW){
	}

	/**
	 * RXgN^
	 * @param source lz
	 */
	inline explicit Quaternion(const float* const source) :
		x(source[0]), y(source[1]), z(source[2]), w(source[3]){
	}

	//--------------------------------------------------------------------------
	// l̐ݒ
	//--------------------------------------------------------------------------
	/**
	 * l̐ݒ
	 * @param sourceX X̐ݒl
	 * @param sourceY Y̐ݒl
	 * @param sourceZ Z̐ݒl
	 * @param sourceW W̐ݒl
	 */
	inline void set(float sourceX, float sourceY, float sourceZ, float sourceW){
		x = sourceX;
		y = sourceY;
		z = sourceZ;
		w = sourceW;
	}

	/**
	 * l̐ݒ
	 * @param source ݒlz
	 */
	inline void set(const float* const source){
		x = source[0];
		y = source[1];
		z = source[2];
		w = source[3];
	}

	/**
	 * [l̐ݒ
	 */
	inline void setZero(){
		set(0.f, 0.f, 0.f, 0.f);
	}

	/**
	 * Pl̐ݒ
	 */
	inline void setIdentity(){
		set(0.f, 0.f, 0.f, 1.f);
	}

	//--------------------------------------------------------------------------
	// ]
	//--------------------------------------------------------------------------
	/**
	 * w]̐ݒ
	 * @param axis ]
	 * @param radian WAPʂł̉]px
	 */
	inline void setRotationAxis(const Vector3& axis, float radian){
		Assert(axis.isUnit());
		float halfRadian = radian * 0.5f;
		float sin = Math::sin(halfRadian);
		x = axis.x * sin;
		y = axis.y * sin;
		z = axis.z * sin;
		w = Math::cos(halfRadian);
	}

	/**
	 * w]̒ǉ
	 * @param axis ]
	 * @param radian WAPʂł̉]px
	 */
	inline void addRotationAxis(const Vector3& axis, float radian){
		Quaternion quaternion;
		quaternion.setRotationAxis(axis, radian);
		(*this) = quaternion * (*this);
	}

	/**
	 * w]̎擾
	 * @param axis [out]]i[xNgւ̃|C^B
	 *	]ĂȂƂ̓[xNg
	 * @param radian [out]WAPʂł̉]pxi[floatւ̃|C^
	 */
	inline void getRotationAxis(Vector3* axis, float* radian) const{
		Assert(isUnit());
		axis->set(array);
		float halfRadian = Math::acos(w);
		*radian = 2.f * halfRadian;
		(*axis) *= 1.f / Math::sin(halfRadian);
	}

	//--------------------------------------------------------------------------
	// OIC[]
	//--------------------------------------------------------------------------
	/**
	 * XYZ]̐ݒ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void setRotationXYZ(const Vector3& radian){
		float xRadian = radian.x * 0.5f;
		float yRadian = radian.y * 0.5f;
		float zRadian = radian.z * 0.5f;
		float sinX = Math::sin(xRadian);
		float cosX = Math::cos(xRadian);
		float sinY = Math::sin(yRadian);
		float cosY = Math::cos(yRadian);
		float sinZ = Math::sin(zRadian);
		float cosZ = Math::cos(zRadian);
		x =  sinX * cosY * cosZ - cosX * sinY * sinZ;
		y =  cosX * sinY * cosZ + sinX * cosY * sinZ;
		z =  cosX * cosY * sinZ - sinX * sinY * cosZ;
		w =  cosX * cosY * cosZ + sinX * sinY * sinZ;
	}

	/**
	 * XYZ]̒ǉ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void addRotationXYZ(const Vector3& radian){
		Quaternion quaternion;
		quaternion.setRotationXYZ(radian);
		(*this) = quaternion * (*this);
	}

	/**
	 * XYZ]̎擾
	 * @param radian [out] eɂ郉WAPʂł̉]px
	 * @return Płtrue
	 */
	inline bool getRotationXYZ(Vector3* radian) const{
		// Y]߂
		float x2 = x + x;
		float y2 = y + y;
		float z2 = z + z;
		float xz2 = x * z2;
		float wy2 = w * y2;
		float temp = -(xz2 - wy2);
		// 덷΍
		if(temp >= 1.f){ temp = 1.f; }
		else if(temp <= -1.f){ temp = -1.f; }
		float yRadian = Math::asin(temp);
		radian->y = yRadian;
		// ̉]߂
		float xx2 = x * x2;
		float xy2 = x * y2;
		float zz2 = z * z2;
		float wz2 = w * z2;
		if(yRadian < Math::halfPI){
			if(yRadian > -Math::halfPI){
				float yz2 = y * z2;
				float wx2 = w * x2;
				float yy2 = y * y2;
				radian->x = Math::atan2((yz2 + wx2), (1.f - (xx2 + yy2)));
				radian->z = Math::atan2((xy2 + wz2), (1.f - (yy2 + zz2)));
				return true;
			}else{
				radian->x = -Math::atan2((xy2 - wz2), (1.f - (xx2 + zz2)));
				radian->z = 0.f;
				return false;
			}
		}else{
			radian->x = Math::atan2((xy2 - wz2), (1.f - (xx2 + zz2)));
			radian->z = 0.f;
			return false;
		}
	}

	//--------------------------------------------------------------------------
	/**
	 * ZYX]̐ݒ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void setRotationZYX(const Vector3& radian){
		float xRadian = radian.x * 0.5f;
		float yRadian = radian.y * 0.5f;
		float zRadian = radian.z * 0.5f;
		float sinX = Math::sin(xRadian);
		float cosX = Math::cos(xRadian);
		float sinY = Math::sin(yRadian);
		float cosY = Math::cos(yRadian);
		float sinZ = Math::sin(zRadian);
		float cosZ = Math::cos(zRadian);
		x =  sinX * cosY * cosZ + cosX * sinY * sinZ;
		y =  cosX * sinY * cosZ - sinX * cosY * sinZ;
		z =  cosX * cosY * sinZ + sinX * sinY * cosZ;
		w =  cosX * cosY * cosZ - sinX * sinY * sinZ;
	}

	/**
	 * ZYX]̒ǉ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void addRotationZYX(const Vector3& radian){
		Quaternion quaternion;
		quaternion.setRotationZYX(radian);
		(*this) = quaternion * (*this);
	}

	/**
	 * ZYX]̎擾
	 * @param radian [out] eɂ郉WAPʂł̉]px
	 * @return Płtrue
	 */
	inline bool getRotationZYX(Vector3* radian) const{
		// Y]߂
		float x2 = x + x;
		float y2 = y + y;
		float z2 = z + z;
		float xz2 = x * z2;
		float wy2 = w * y2;
		float temp = (xz2 + wy2);
		// 덷΍
		if(temp >= 1.f){ temp = 1.f; }
		else if(temp <= -1.f){ temp = -1.f; }
		float yRadian = Math::asin(temp);
		radian->y = yRadian;
		// ̉]߂
		float xy2 = x * y2;
		float wz2 = w * z2;
		if(yRadian < Math::halfPI){
			if(yRadian > -Math::halfPI){
				float yy2 = y * y2;
				float zz2 = z * z2;
				float yz2 = y * z2;
				float wx2 = w * x2;
				float xx2 = x * x2;
				radian->z = Math::atan2(-(xy2 - wz2), (1.f - (yy2 + zz2)));
				radian->x = Math::atan2(-(yz2 - wx2), (1.f - (xx2 + yy2)));
				return true;
			}else{
				radian->z = -Math::atan2(-(xy2 + wz2), (xz2 - wy2));
				radian->x = 0.f;
				return false;
			}
		}else{
			radian->z = Math::atan2((xy2 + wz2), -(xz2 - wy2));
			radian->x = 0.f;
			return false;
		}
	}

	//--------------------------------------------------------------------------
	// Z
	//--------------------------------------------------------------------------
	/**
	 * Z
	 * @param addQuaternion Zl
	 * @return Zꂽl
	 */
	inline Quaternion operator +(const Quaternion& addQuaternion) const{
		return Quaternion(
			x + addQuaternion.x,
			y + addQuaternion.y,
			z + addQuaternion.z,
			w + addQuaternion.w);
	}

	/**
	 * Z
	 * @param subQuaternion Zl
	 * @return Zꂽl
	 */
	inline Quaternion operator -(const Quaternion& subQuaternion) const{
		return Quaternion(
			x - subQuaternion.x,
			y - subQuaternion.y,
			z - subQuaternion.z,
			w - subQuaternion.w);
	}

	/**
	 * Z
	 * @param mulQuat Zl
	 * @return Zꂽl
	 */
	inline Quaternion operator *(const Quaternion& mulQuat) const{
		// xyz = (q0.xyz * q1.w) + (q1.xyz * q0.w) + cross(q0.xyz, q1.xyz)
		// w = q0.w * q1.w - dot(q0.xyz, q1.xyz)
		return Quaternion(
			y * mulQuat.z - z * mulQuat.y + x * mulQuat.w + w * mulQuat.x,
			z * mulQuat.x - x * mulQuat.z + y * mulQuat.w + w * mulQuat.y,
			x * mulQuat.y - y * mulQuat.x + z * mulQuat.w + w * mulQuat.z,
			w * mulQuat.w - x * mulQuat.x - y * mulQuat.y - z * mulQuat.z);
	}

	/**
	 * Z
	 * @param mulValue Zl
	 * @return Zꂽl
	 */
	inline Quaternion operator *(float mulValue) const{
		return Quaternion(
			x * mulValue, y * mulValue, z * mulValue, w * mulValue);
	}

	/**
	 * Z
	 * @param mulValue Zl
	 * @param mulQuaternion Zl
	 * @return Zꂽl
	 */
	inline friend Quaternion operator *(
		float mulValue, const Quaternion& mulQuaternion){
		return Quaternion(
			mulQuaternion.x * mulValue,
			mulQuaternion.y * mulValue,
			mulQuaternion.z * mulValue,
			mulQuaternion.w * mulValue);
	}

	/**
	 * xNgZ
	 * @param mulVector ZxNg
	 * @return ZꂽxNg
	 */
	inline Vector3 operator *(const Vector3& mulVector) const{
		Vector3 quaternion(x, y, z);
		Vector3 cross1 = quaternion.crossProduct(mulVector);
		Vector3 cross2 = quaternion.crossProduct(cross1);
		cross1 *= (w * 2.f);
		cross1 += mulVector;
		cross2 *= 2.f;
		cross1 += cross2;
		return cross1;
	}

	/**
	 * +Zq
	 * @return l̃Rs[
	 */
	inline Quaternion operator +() const{ return *this; }

	/**
	 * -Zq
	 * @return l̕]l
	 */
	inline Quaternion operator -() const{ return Quaternion(-x, -y, -z, -w); }

	//--------------------------------------------------------------------------
	// Z
	//--------------------------------------------------------------------------
	/**
	 * Z
	 * @param addQuaternion Zl
	 * @return Zꂽl
	 */
	inline Quaternion& operator +=(const Quaternion& addQuaternion){
		x += addQuaternion.x;
		y += addQuaternion.y;
		z += addQuaternion.z;
		w += addQuaternion.w;
		return *this;
	}

	/**
	 * Z
	 * @param subQuaternion Zl
	 * @return Zꂽl
	 */
	inline Quaternion& operator -=(const Quaternion& subQuaternion){
		x -= subQuaternion.x;
		y -= subQuaternion.y;
		z -= subQuaternion.z;
		w -= subQuaternion.w;
		return *this;
	}

	/**
	 * Z
	 * @param mulQuat Zl
	 * @return Zꂽl
	 */
	inline Quaternion& operator *=(const Quaternion& mulQuat){
		// xyz = (q0.xyz * q1.w) + (q1.xyz * q0.w) + cross(q0.xyz, q1.xyz)
		// w = q0.w * q1.w - dot(q0.xyz, q1.xyz)
		set(y * mulQuat.z - z * mulQuat.y + x * mulQuat.w + w * mulQuat.x,
			z * mulQuat.x - x * mulQuat.z + y * mulQuat.w + w * mulQuat.y,
			x * mulQuat.y - y * mulQuat.x + z * mulQuat.w + w * mulQuat.z,
			w * mulQuat.w - x * mulQuat.x - y * mulQuat.y - z * mulQuat.z);
		return *this;
	}

	/**
	 * Z
	 * @param mulValue Zl
	 * @return Zꂽl
	 */
	inline Quaternion& operator *=(float mulValue){
		x *= mulValue;
		y *= mulValue;
		z *= mulValue;
		w *= mulValue;
		return *this;
	}

	//--------------------------------------------------------------------------
	// ֘A
	//--------------------------------------------------------------------------
	/**
	 * ̎擾
	 * @return 
	 */
	inline float getLength() const{
		return Math::sqrt(x * x + y * y + z * z + w * w);
	}

	/**
	 * [lǂ
	 * @return [lȂtrueԂ
	 */
	inline bool isZero() const{
		return (getLength() <= Math::epsilon);
	}

	/**
	 * Pʎlǂ
	 * @return PʎlȂtrueԂ
	 */
	inline bool isUnit() const{
		return (Math::abs(getLength() - 1.f) <= Math::epsilon);
	}

	/**
	 * K
	 * @return Kꂽl
	 */
	inline Quaternion& normalize(){
		float legnth = getLength();
		Assert(legnth >= Math::epsilon);
		float inverseLength = 1.f / legnth;
		(*this) *= inverseLength;
		return *this;
	}

	//--------------------------------------------------------------------------
	// lZ
	//--------------------------------------------------------------------------
	/**
	 * m̎擾
	 * @return m
	 */
	inline float getNorm() const{
		return x * x + y * y + z * z + w * w;
	}

	/**
	 * 
	 * @return ꂽl
	 */
	inline Quaternion& conjugate(){
		x = -x;
		y = -y;
		z = -z;
		return *this;
	}

	/**
	 * 
	 * @param dotQuaternion ςƂl
	 * @return ϒl
	 */
	inline float dotProduct(const Quaternion& dotQuaternion) const{
		return x * dotQuaternion.x + y * dotQuaternion.y +
			z * dotQuaternion.z + w * dotQuaternion.w;
	}

	/**
	 * tl
	 * @return tl
	 */
	inline Quaternion& invert(){
		Assert(!isZero());
		float inverseNorm = 1.f / getNorm();
		x = -x * inverseNorm;
		y = -y * inverseNorm;
		z = -z * inverseNorm;
		w =  w * inverseNorm;
		return *this;
	}

	/**
	 * Pʋtl
	 * @return tl
	 */
	inline Quaternion& unitInvert(){
		Assert(isUnit());
		return conjugate();
	}

	//--------------------------------------------------------------------------
	// 
	//--------------------------------------------------------------------------
	/**
	 * ʐ`
	 * @param source ԊJnl
	 * @param target ԏIl
	 * @param alpha ԌW
	 * @return ԍςݎl
	 */
	static Quaternion slerp(
		const Quaternion& source, const Quaternion& target, float alpha){
		Assert(source.isUnit());
		Assert(target.isUnit());
		float dot = source.dotProduct(target);
		// 덷΍
		if(dot >= 1.f){
			dot = 1.f;
		}else if(dot <= -1.f){
			dot = -1.f;
		}
		float radian = Math::acos(dot);
		if(Math::abs(radian) < Math::epsilon){ return target; }
		float inverseSin = 1.f / Math::sin(radian);
		float leftScale = Math::sin((1.f - alpha) * radian) * inverseSin;
		float rightScale = Math::sin(alpha * radian) * inverseSin;
		return source * leftScale + target * rightScale;
	}

	/**
	 * ␳tʐ`
	 * @param source ԊJnl
	 * @param target ԏIl
	 * @param alpha ԌW
	 * @return ԍςݎl
	 */
	static Quaternion correctSlerp(
		const Quaternion& source, const Quaternion& target, float alpha){
		Assert(source.isUnit());
		Assert(target.isUnit());
		float dot = source.dotProduct(target);
		// ]
		Quaternion correctTarget;
		if(dot < 0.f){
			correctTarget = -target;
			dot = -dot;
		}else{
			correctTarget = target;
		}
		// 덷΍
		if(dot >= 1.f){ dot = 1.f; }
		float radian = Math::acos(dot);
		if(Math::abs(radian) < Math::epsilon){ return correctTarget; }
		float inverseSin = 1.f / Math::sin(radian);
		float leftScale = Math::sin((1.f - alpha) * radian) * inverseSin;
		float rightScale = Math::sin(alpha * radian) * inverseSin;
		return source * leftScale + correctTarget * rightScale;
	}

	/**
	 * ʓ񎟕
	 * @param source ԊJnl
	 * @param sourceHandle ԊJnlnh
	 * @param targetHandle ԏIlnh
	 * @param target ԏIl
	 * @param alpha ԌW
	 * @return ԍςݎl
	 */
	static Quaternion squad(
		const Quaternion& source, const Quaternion& sourceHandle,
		const Quaternion& targetHandle, const Quaternion& target, float alpha){
		Quaternion slerpOriginal = slerp(source, target, alpha);
		Quaternion slerpHandle = slerp(sourceHandle, targetHandle, alpha);
		return slerp(slerpOriginal, slerpHandle, 2.f * alpha * (1.f - alpha));
	}

	/**
	 * ␳tʓ񎟕
	 * @param source ԊJnl
	 * @param sourceHandle ԊJnlnh
	 * @param targetHandle ԏIlnh
	 * @param target ԏIl
	 * @param alpha ԌW
	 * @return ԍςݎl
	 */
	static Quaternion correctSquad(
		const Quaternion& source, const Quaternion& sourceHandle,
		const Quaternion& targetHandle, const Quaternion& target, float alpha){
		Quaternion slerpOriginal =
			correctSlerp(source, target, alpha);
		Quaternion slerpHandle =
			correctSlerp(sourceHandle, targetHandle, alpha);
		return correctSlerp(
			slerpOriginal, slerpHandle, 2.f * alpha * (1.f - alpha));
	}

	//--------------------------------------------------------------------------
	// _Z
	//--------------------------------------------------------------------------
	/**
	 * lǂ
	 * @param target rl
	 * @return lłtrueԂ
	 */
	inline bool operator ==(const Quaternion& target) const{
		return ((x == target.x) && (y == target.y) &&
			(z == target.z) && (w == target.w));
	}

	/**
	 * lǂ
	 * @param target rl
	 * @param epsilon 덷
	 * @return 덷͈͓̔œlłtrueԂ
	 */
	inline bool epsilonEquals(const Quaternion& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			(Math::abs(x - target.x) <= epsilon) &&
			(Math::abs(y - target.y) <= epsilon) &&
			(Math::abs(z - target.z) <= epsilon) &&
			(Math::abs(w - target.w) <= epsilon));
	}

	/**
	 * lłȂǂ
	 * @param target rl
	 * @return łȂlłtrueԂ
	 */
	inline bool operator !=(const Quaternion& target) const{
		return ((x != target.x) || (y != target.y) ||
			(z != target.z) || (w != target.w));
	}

	/**
	 * lłȂǂ
	 * @param target rl
	 * @param epsilon 덷
	 * @return 덷͈͓̔œłȂlłtrueԂ
	 */
	inline bool notEpsilonEquals(const Quaternion& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			(Math::abs(x - target.x) > epsilon) ||
			(Math::abs(y - target.y) > epsilon) ||
			(Math::abs(z - target.z) > epsilon) ||
			(Math::abs(w - target.w) > epsilon));
	}

	//--------------------------------------------------------------------------
	/**
	 * l]킷ǂ
	 * @param target rl
	 * @return ]킹trueԂ
	 */
	inline bool rotationEquals(const Quaternion& target) const{
		if((*this) == target){ return true; }
		if((*this) == -target){ return true; }
		return false;
	}

	/**
	 * l]킷ǂ
	 * @param target rl
	 * @param epsilon 덷
	 * @return 덷͈͓̔œ]킹trueԂ
	 */
	inline bool epsilonRotationEquals(
		const Quaternion& target, float epsilon) const{
		if(epsilonEquals(target, epsilon)){ return true; }
		if(epsilonEquals(-target, epsilon)){ return true; }
		return false;
	}

	//--------------------------------------------------------------------------
	// ̑
	//--------------------------------------------------------------------------
	/**
	 * 
	 * @return xNg̕\L
	 */
	inline String toString() const{
		String returnString;
		returnString.format("( < %.8f, %.8f, %.8f > %.8f )", x, y, z, w);
		return returnString;
	}

	//--------------------------------------------------------------------------
private:

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of QUATERNION_H_
//------------------------------------------------------------------------------
