//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * 4~4swb_
 * @author Junpee
 */

#ifndef MATRIX44_H_
#define MATRIX44_H_

#include <Core/Primitive/Vector3.h>
#include <Core/Primitive/Quaternion.h>

namespace Lamp{

class Matrix33;
class Matrix34;

//------------------------------------------------------------------------------
/**
 * 4~4s
 *
 * EWngp܂B
 * ̃NX͌pȂŉB
 */
class Matrix44{
public:
	//--------------------------------------------------------------------------
	// oϐ
	//--------------------------------------------------------------------------
	/// oϐ
	union{
		/// evf
		struct{
			/// 00vf
			float m00;
			/// 01vf
			float m01;
			/// 02vf
			float m02;
			/// 03vf
			float m03;
			/// 10vf
			float m10;
			/// 11vf
			float m11;
			/// 12vf
			float m12;
			/// 13vf
			float m13;
			/// 20vf
			float m20;
			/// 21vf
			float m21;
			/// 22vf
			float m22;
			/// 23vf
			float m23;
			/// 30vf
			float m30;
			/// 31vf
			float m31;
			/// 32vf
			float m32;
			/// 33vf
			float m33;
		};

		/// sl
		float m[4][4];

		/// z
		float array[16];
	};

	//--------------------------------------------------------------------------
	// 萔
	//--------------------------------------------------------------------------
	/// [s
	static const Matrix44 zero;

	/// Pʍs
	static const Matrix44 unit;

	//--------------------------------------------------------------------------
	// RXgN^
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 *
	 * ̃RXgN^͏l̐ݒsȂߒl͕słB
	 */
	Matrix44(){}

	/**
	 * RXgN^
	 * @param i00 m00̏l
	 * @param i01 m01̏l
	 * @param i02 m02̏l
	 * @param i03 m03̏l
	 * @param i10 m10̏l
	 * @param i11 m11̏l
	 * @param i12 m12̏l
	 * @param i13 m13̏l
	 * @param i20 m20̏l
	 * @param i21 m21̏l
	 * @param i22 m22̏l
	 * @param i23 m23̏l
	 * @param i30 m30̏l
	 * @param i31 m31̏l
	 * @param i32 m32̏l
	 * @param i33 m33̏l
	 */
	inline Matrix44(
		float i00, float i01, float i02, float i03,
		float i10, float i11, float i12, float i13,
		float i20, float i21, float i22, float i23,
		float i30, float i31, float i32, float i33) :
		m00(i00), m01(i01), m02(i02), m03(i03),
		m10(i10), m11(i11), m12(i12), m13(i13),
		m20(i20), m21(i21), m22(i22), m23(i23),
		m30(i30), m31(i31), m32(i32), m33(i33){
	}

	/**
	 * RXgN^
	 * @param source lz
	 */
	inline explicit Matrix44(const float* const source) :
		m00(source[ 0]), m01(source[ 1]), m02(source[ 2]), m03(source[ 3]),
		m10(source[ 4]), m11(source[ 5]), m12(source[ 6]), m13(source[ 7]),
		m20(source[ 8]), m21(source[ 9]), m22(source[10]), m23(source[11]),
		m30(source[12]), m31(source[13]), m32(source[14]), m33(source[15]){
	}

	//--------------------------------------------------------------------------
	// l̐ݒ
	//--------------------------------------------------------------------------
	/**
	 * l̐ݒ
	 * @param s00 m00̐ݒl
	 * @param s01 m01̐ݒl
	 * @param s02 m02̐ݒl
	 * @param s03 m03̐ݒl
	 * @param s10 m10̐ݒl
	 * @param s11 m11̐ݒl
	 * @param s12 m12̐ݒl
	 * @param s13 m13̐ݒl
	 * @param s20 m20̐ݒl
	 * @param s21 m21̐ݒl
	 * @param s22 m22̐ݒl
	 * @param s23 m23̐ݒl
	 * @param s30 m30̐ݒl
	 * @param s31 m31̐ݒl
	 * @param s32 m32̐ݒl
	 * @param s33 m33̐ݒl
	 */
	inline void set(
		float s00, float s01, float s02, float s03,
		float s10, float s11, float s12, float s13,
		float s20, float s21, float s22, float s23,
		float s30, float s31, float s32, float s33){
		m00 = s00; m01 = s01; m02 = s02; m03 = s03;
		m10 = s10; m11 = s11; m12 = s12; m13 = s13;
		m20 = s20; m21 = s21; m22 = s22; m23 = s23;
		m30 = s30; m31 = s31; m32 = s32; m33 = s33;
	}

	/**
	 * l̐ݒ
	 * @param source ݒlz
	 */
	inline void set(const float* const source){
		m00 = source[ 0]; m01 = source[ 1]; m02 = source[ 2]; m03 = source[ 3];
		m10 = source[ 4]; m11 = source[ 5]; m12 = source[ 6]; m13 = source[ 7];
		m20 = source[ 8]; m21 = source[ 9]; m22 = source[10]; m23 = source[11];
		m30 = source[12]; m31 = source[13]; m32 = source[14]; m33 = source[15];
	}

	/**
	 * [sݒ
	 */
	inline void setZero(){
		set(0.f, 0.f, 0.f, 0.f,
			0.f, 0.f, 0.f, 0.f,
			0.f, 0.f, 0.f, 0.f,
			0.f, 0.f, 0.f, 0.f);
	}

	/**
	 * Pʍsݒ
	 */
	inline void setUnit(){
		set(1.f, 0.f, 0.f, 0.f,
			0.f, 1.f, 0.f, 0.f,
			0.f, 0.f, 1.f, 0.f,
			0.f, 0.f, 0.f, 1.f);
	}

	/**
	 * 3~3s̐ݒ
	 * @param source ݒ茳s
	 */
	void set(const Matrix33& source);

	/**
	 * 3~4s̐ݒ
	 * @param source ݒ茳s
	 */
	void set(const Matrix34& source);

	//--------------------------------------------------------------------------
	// XP[
	//--------------------------------------------------------------------------
	/**
	 * XP[̐ݒ
	 * @param scaleX XXP[l
	 * @param scaleY YXP[l
	 * @param scaleZ ZXP[l
	 */
	inline void setScale(float scaleX, float scaleY, float scaleZ){
		set(scaleX,    0.f,    0.f,    0.f,
			   0.f, scaleY,    0.f,    0.f,
			   0.f,    0.f, scaleZ,    0.f,
			   0.f,    0.f,    0.f,    1.f);
	}

	/**
	 * XP[̐ݒ
	 * @param scale XP[l
	 */
	inline void setScale(const Vector3& scale){
		setScale(scale.x, scale.y, scale.z);
	}

	/**
	 * XP[̒ǉ
	 * @param scaleX XXP[l
	 * @param scaleY YXP[l
	 * @param scaleZ ZXP[l
	 */
	inline void addScale(float scaleX, float scaleY, float scaleZ){
		Matrix44 matrix;
		matrix.setScale(scaleX, scaleY, scaleZ);
		(*this) = matrix * (*this);
	}

	/**
	 * XP[̒ǉ
	 * @param scale XP[l
	 */
	inline void addScale(const Vector3& scale){
		Matrix44 matrix;
		matrix.setScale(scale.x, scale.y, scale.z);
		(*this) = matrix * (*this);
	}

	//--------------------------------------------------------------------------
	// ]
	//--------------------------------------------------------------------------
	/**
	 * X]̐ݒ
	 * @param radian WAPʂł̉]px
	 */
	inline void setRotationX(float radian){
		float sin = Math::sin(radian);
		float cos = Math::cos(radian);
		set( 1.f,  0.f,  0.f,  0.f,
			 0.f,  cos, -sin,  0.f,
			 0.f,  sin,  cos,  0.f,
			 0.f,  0.f,  0.f,  1.f);
	}

	/**
	 * X]̒ǉ
	 * @param radian WAPʂł̉]px
	 */
	inline void addRotationX(float radian){
		Matrix44 matrix;
		matrix.setRotationX(radian);
		(*this) = matrix * (*this);
	}

	//--------------------------------------------------------------------------
	/**
	 * Y]̐ݒ
	 * @param radian WAPʂł̉]px
	 */
	inline void setRotationY(float radian){
		float sin = Math::sin(radian);
		float cos = Math::cos(radian);
		set( cos,  0.f,  sin,  0.f,
			 0.f,  1.f,  0.f,  0.f,
			-sin,  0.f,  cos,  0.f,
			 0.f,  0.f,  0.f,  1.f);
	}

	/**
	 * Y]̒ǉ
	 * @param radian WAPʂł̉]px
	 */
	inline void addRotationY(float radian){
		Matrix44 matrix;
		matrix.setRotationY(radian);
		(*this) = matrix * (*this);
	}

	//--------------------------------------------------------------------------
	/**
	 * Z]̐ݒ
	 * @param radian WAPʂł̉]px
	 */
	inline void setRotationZ(float radian){
		float sin = Math::sin(radian);
		float cos = Math::cos(radian);
		set( cos, -sin,  0.f,  0.f,
			 sin,  cos,  0.f,  0.f,
			 0.f,  0.f,  1.f,  0.f,
			 0.f,  0.f,  0.f,  1.f);
	}

	/**
	 * Z]̒ǉ
	 * @param radian WAPʂł̉]px
	 */
	inline void addRotationZ(float radian){
		Matrix44 matrix;
		matrix.setRotationZ(radian);
		(*this) = matrix * (*this);
	}

	//--------------------------------------------------------------------------
	/**
	 * w]̐ݒ
	 * @param axis ]
	 * @param radian WAPʂł̉]px
	 */
	inline void setRotationAxis(const Vector3& axis, float radian){
		Assert(axis.isUnit());
		float sin = Math::sin(radian);
		float cos = Math::cos(radian);
		float invCos = 1.f - cos;
		float xyInv = axis.x * axis.y * invCos;
		float xzInv = axis.x * axis.z * invCos;
		float yzInv = axis.y * axis.z * invCos;
		float xSin = axis.x * sin;
		float ySin = axis.y * sin;
		float zSin = axis.z * sin;
		set((axis.x * axis.x * invCos + cos), (xyInv - zSin), (xzInv + ySin),
			0.f,
			(xyInv + zSin), (axis.y * axis.y * invCos + cos), (yzInv - xSin),
			0.f,
			(xzInv - ySin), (yzInv + xSin), (axis.z * axis.z * invCos + cos),
			0.f,
			0.f, 0.f, 0.f, 1.f);
	}

	/**
	 * w]̒ǉ
	 * @param axis ]
	 * @param radian WAPʂł̉]px
	 */
	inline void addRotationAxis(const Vector3& axis, float radian){
		Matrix44 matrix;
		matrix.setRotationAxis(axis, radian);
		(*this) = matrix * (*this);
	}

	/**
	 * w]̎擾
	 * @param axis [out]]i[xNgւ̃|C^
	 * @param radian [out]WAPʂł̉]pxi[floatւ̃|C^
	 */
	inline void getRotationAxis(Vector3* axis, float* radian) const{
		float radianResult = Math::acos(0.5f * ((m00 + m11 + m22) - 1.f));
		*radian = radianResult;
		if(radianResult > 0.f){
			if(radianResult < Math::PI){
				axis->set(m21 - m12, m02 - m20, m10 - m01);
				axis->normalize();
			}else{
				if(m00 >= m11){
					if(m00 >= m22){
						axis->x = 0.5f * Math::sqrt(m00 - m11 - m22 + 1.f);
						float halfInverse = 0.5f / axis->x;
						axis->y = halfInverse * m01;
						axis->z = halfInverse * m02;
					}else{
						axis->z = 0.5f * Math::sqrt(m22 - m00 - m11 + 1.f);
						float halfInverse = 0.5f / axis->z;
						axis->x = halfInverse * m02;
						axis->y = halfInverse * m12;
					}
				}else{
					if(m11 >= m22){
						axis->y = 0.5f * Math::sqrt(m11 - m00 - m22 + 1.f);
						float halfInverse = 0.5f / axis->y;
						axis->x = halfInverse * m01;
						axis->z = halfInverse * m12;
					}else{
						axis->z = 0.5f * Math::sqrt(m22 - m00 - m11 + 1.f);
						float halfInverse = 0.5f / axis->z;
						axis->x = halfInverse * m02;
						axis->y = halfInverse * m12;
					}
				}
			}
		}else{
			axis->set(1.f, 0.f, 0.f);
		}
	}

	//--------------------------------------------------------------------------
	/**
	 * l]̐ݒ
	 * @param quaternion l
	 */
	inline void setRotationQuaternion(const Quaternion& quaternion){
		Assert(quaternion.isUnit());
		float x2 = quaternion.x + quaternion.x;
		float y2 = quaternion.y + quaternion.y;
		float z2 = quaternion.z + quaternion.z;
		float xx2 = quaternion.x * x2;
		float xy2 = quaternion.x * y2;
		float xz2 = quaternion.x * z2;
		float yy2 = quaternion.y * y2;
		float yz2 = quaternion.y * z2;
		float zz2 = quaternion.z * z2;
		float wx2 = quaternion.w * x2;
		float wy2 = quaternion.w * y2;
		float wz2 = quaternion.w * z2;
		m00 = 1.f - (yy2 + zz2);
		m01 = xy2 - wz2;
		m02 = xz2 + wy2;
		m03 = 0.f;

		m10 = xy2 + wz2;
		m11 = 1.f - (xx2 + zz2);
		m12 = yz2 - wx2;
		m13 = 0.f;

		m20 = xz2 - wy2;
		m21 = yz2 + wx2;
		m22 = 1.f - (xx2 + yy2);
		m23 = 0.f;

		m30 = 0.f;
		m31 = 0.f;
		m32 = 0.f;
		m33 = 1.f;
	}

	/**
	 * l]̒ǉ
	 * @param quaternion l
	 */
	inline void addRotationQuaternion(const Quaternion& quaternion){
		Matrix44 matrix;
		matrix.setRotationQuaternion(quaternion);
		(*this) = matrix * (*this);
	}

	/**
	 * l]̎擾
	 * @return ]\l
	 */
	inline Quaternion getRotationQuaternion() const{
		Quaternion result;
		float trace = m00 + m11 + m22;
		if(trace > 0.f){
			float scale = Math::sqrt(trace + 1.f);
			result.w = scale * 0.5f;
			scale = 0.5f / scale;
			result.x = (m21 - m12) * scale;
			result.y = (m02 - m20) * scale;
			result.z = (m10 - m01) * scale;
		}else{
			int i = 0;
			if(m11 > m00){ i = 1; }
			if(m22 > m[i][i]){ i = 2; }
			int nextTable[] = { 1, 2, 0 };
			int j = nextTable[i];
			int k = nextTable[j];
			float scale = Math::sqrt(m[i][i] - m[j][j] - m[k][k] + 1.f);
			result.array[i] = 0.5f * scale;
			scale = 0.5f / scale;
			result.w = (m[k][j] - m[j][k]) * scale;
			result.array[j] = (m[j][i] + m[i][j]) * scale;
			result.array[k] = (m[k][i] + m[i][k]) * scale;
		}
		return result;
	}

	//--------------------------------------------------------------------------
	// Ow]
	//--------------------------------------------------------------------------
	/**
	 * XYZ]̐ݒ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void setRotationXYZ(const Vector3& radian){
		float sinX = Math::sin(radian.x);
		float cosX = Math::cos(radian.x);
		float sinY = Math::sin(radian.y);
		float cosY = Math::cos(radian.y);
		float sinZ = Math::sin(radian.z);
		float cosZ = Math::cos(radian.z);
		m00 = cosY * cosZ;
		m01 = sinX * sinY * cosZ - cosX * sinZ;
		m02 = cosX * sinY * cosZ + sinX * sinZ;
		m03 = 0.f;

		m10 = cosY * sinZ;
		m11 = sinX * sinY * sinZ + cosX * cosZ;
		m12 = cosX * sinY * sinZ - sinX * cosZ;
		m13 = 0.f;

		m20 = -sinY;
		m21 = sinX * cosY;
		m22 = cosX * cosY;
		m23 = 0.f;

		m30 = 0.f;
		m31 = 0.f;
		m32 = 0.f;
		m33 = 1.f;
	}

	/**
	 * XYZ]̒ǉ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void addRotationXYZ(const Vector3& radian){
		Matrix44 matrix;
		matrix.setRotationXYZ(radian);
		(*this) = matrix * (*this); 
	}

	/**
	 * XYZ]̎擾
	 * @param radian [out] eɂ郉WAPʂł̉]px
	 * @return Płtrue
	 */
	inline bool getRotationXYZ(Vector3* radian) const{
		float yRadian = Math::asin(-m20);
		radian->y = yRadian;
		if(yRadian < Math::halfPI){
			if(yRadian > -Math::halfPI){
				radian->x = Math::atan2(m21, m22);
				radian->z = Math::atan2(m10, m00);
				return true;
			}else{
				radian->x = -Math::atan2(m01, m11);
				radian->z = 0.f;
				return false;
			}
		}else{
			radian->x = Math::atan2(m01, m11);
			radian->z = 0.f;
			return false;
		}
	}

	//--------------------------------------------------------------------------
	/**
	 * XZY]̐ݒ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	// WJ΍ł
	inline void setRotationXZY(const Vector3& radian){
		setRotationX(radian.x);
		addRotationZ(radian.z);
		addRotationY(radian.y);
	}

	/**
	 * XZY]̒ǉ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void addRotationXZY(const Vector3& radian){
		Matrix44 matrix;
		matrix.setRotationXZY(radian);
		(*this) = matrix * (*this); 
	}

	/**
	 * XZY]̎擾
	 * @param radian [out] eɂ郉WAPʂł̉]px
	 * @return Płtrue
	 */
	inline bool getRotationXZY(Vector3* radian) const{
		float zRadian = Math::asin(m10);
		radian->z = zRadian;
		if(zRadian < Math::halfPI){
			if(zRadian > -Math::halfPI){
				radian->x = Math::atan2(-m12, m11);
				radian->y = Math::atan2(-m20, m00);
				return true;
			}else{
				radian->x = -Math::atan2(m02, m22);
				radian->y = 0.f;
				return false;
			}
		}else{
			radian->x = Math::atan2(m02, m22);
			radian->y = 0.f;
			return false;
		}
	}

	//--------------------------------------------------------------------------
	/**
	 * YXZ]̐ݒ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	// WJ΍ł
	inline void setRotationYXZ(const Vector3& radian){
		setRotationY(radian.y);
		addRotationX(radian.x);
		addRotationZ(radian.z);
	}

	/**
	 * YXZ]̒ǉ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void addRotationYXZ(const Vector3& radian){
		Matrix44 matrix;
		matrix.setRotationYXZ(radian);
		(*this) = matrix * (*this); 
	}

	/**
	 * YXZ]̎擾
	 * @param radian [out] eɂ郉WAPʂł̉]px
	 * @return Płtrue
	 */
	inline bool getRotationYXZ(Vector3* radian) const{
		float xRadian = Math::asin(m21);
		radian->x = xRadian;
		if(xRadian < Math::halfPI){
			if(xRadian > -Math::halfPI){
				radian->y = Math::atan2(-m20, m22);
				radian->z = Math::atan2(-m01, m11);
				return true;
			}else{
				radian->y = -Math::atan2(-m10, m00);
				radian->z = 0.f;
				return false;
			}
		}else{
			radian->y = Math::atan2(-m10, m00);
			radian->z = 0.f;
			return false;
		}
	}

	//--------------------------------------------------------------------------
	/**
	 * YZX]̐ݒ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	// WJ΍ł
	inline void setRotationYZX(const Vector3& radian){
		setRotationY(radian.y);
		addRotationZ(radian.z);
		addRotationX(radian.x);
	}

	/**
	 * YZX]̒ǉ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void addRotationYZX(const Vector3& radian){
		Matrix44 matrix;
		matrix.setRotationYZX(radian);
		(*this) = matrix * (*this); 
	}

	/**
	 * YZX]̎擾
	 * @param radian [out] eɂ郉WAPʂł̉]px
	 * @return Płtrue
	 */
	inline bool getRotationYZX(Vector3* radian) const{
		float zRadian = Math::asin(-m01);
		radian->z = zRadian;
		if(zRadian < Math::halfPI){
			if(zRadian > -Math::halfPI){
				radian->y = Math::atan2(m02, m00);
				radian->x = Math::atan2(m21, m11);
				return true;
			}else{
				radian->y = -Math::atan2(m12, m22);
				radian->x = 0.f;
				return false;
			}
		}else{
			radian->y = Math::atan2(m12, m22);
			radian->x = 0.f;
			return false;
		}
	}

	//--------------------------------------------------------------------------
	/**
	 * ZXY]̐ݒ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	// WJ΍ł
	inline void setRotationZXY(const Vector3& radian){
		setRotationZ(radian.z);
		addRotationX(radian.x);
		addRotationY(radian.y);
	}

	/**
	 * ZXY]̒ǉ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void addRotationZXY(const Vector3& radian){
		Matrix44 matrix;
		matrix.setRotationZXY(radian);
		(*this) = matrix * (*this); 
	}

	/**
	 * ZXY]̎擾
	 * @param radian [out] eɂ郉WAPʂł̉]px
	 * @return Płtrue
	 */
	inline bool getRotationZXY(Vector3* radian) const{
		float xRadian = Math::asin(-m12);
		radian->x = xRadian;
		if(xRadian < Math::halfPI){
			if(xRadian > -Math::halfPI){
				radian->z = Math::atan2(m10, m11);
				radian->y = Math::atan2(m02, m22);
				return true;
			}else{
				radian->z = -Math::atan2(m20, m00);
				radian->y = 0.f;
				return false;
			}
		}else{
			radian->z = Math::atan2(m20, m00);
			radian->y = 0.f;
			return false;
		}
	}

	//--------------------------------------------------------------------------
	/**
	 * ZYX]̐ݒ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	// WJ΍ł
	inline void setRotationZYX(const Vector3& radian){
		setRotationZ(radian.z);
		addRotationY(radian.y);
		addRotationX(radian.x);
	}

	/**
	 * ZYX]̒ǉ
	 * @param radian eɂ郉WAPʂł̉]px
	 */
	inline void addRotationZYX(const Vector3& radian){
		Matrix44 matrix;
		matrix.setRotationZYX(radian);
		(*this) = matrix * (*this); 
	}

	/**
	 * ZYX]̎擾
	 * @param radian [out] eɂ郉WAPʂł̉]px
	 * @return Płtrue
	 */
	inline bool getRotationZYX(Vector3* radian) const{
		float yRadian = Math::asin(m02);
		radian->y = yRadian;
		if(yRadian < Math::halfPI){
			if(yRadian > -Math::halfPI){
				radian->z = Math::atan2(-m01, m00);
				radian->x = Math::atan2(-m12, m22);
				return true;
			}else{
				radian->z = -Math::atan2(-m10, m20);
				radian->x = 0.f;
				return false;
			}
		}else{
			radian->z = Math::atan2(m10, -m20);
			radian->x = 0.f;
			return false;
		}
	}

	//--------------------------------------------------------------------------
	// ړ
	//--------------------------------------------------------------------------
	/**
	 * ړ̐ݒ
	 * @param translationX Xړl
	 * @param translationY Yړl
	 * @param translationZ Zړl
	 */
	inline void setTranslation(
		float translationX, float translationY, float translationZ){
		set(1.f, 0.f, 0.f, translationX,
			0.f, 1.f, 0.f, translationY,
			0.f, 0.f, 1.f, translationZ,
			0.f, 0.f, 0.f, 1.f);
	}

	/**
	 * ړ̐ݒ
	 * @param translation ړl
	 */
	inline void setTranslation(const Vector3& translation){
		setTranslation(translation.x, translation.y, translation.z);
	}

	/**
	 * ړ̒ǉ
	 * @param translationX Xړl
	 * @param translationY Yړl
	 * @param translationZ Zړl
	 */
	inline void addTranslation(
		float translationX, float translationY, float translationZ){
		Matrix44 matrix;
		matrix.setTranslation(translationX, translationY, translationZ);
		(*this) = matrix * (*this);
	}

	/**
	 * ړ̒ǉ
	 * @param translation ړl
	 */
	inline void addTranslation(const Vector3& translation){
		Matrix44 matrix;
		matrix.setTranslation(translation.x, translation.y, translation.z);
		(*this) = matrix * (*this);
	}

	/**
	 * ړ̎擾
	 * @return ړl
	 */
	inline Vector3 getTranslation() const{ return Vector3(m03, m13, m23); }

	//--------------------------------------------------------------------------
	// ϊ
	//--------------------------------------------------------------------------
	/**
	 * XYZ]̕ϊݒ
	 *
	 * XYZ]Aړ̏ɕϊs쐬܂
	 * @param radian eɂ郉WAPʂł̉]px
	 * @param translation ړl
	 */
	inline void setTransformationXYZ(
		const Vector3& radian, const Vector3& translation){
		float sinX = Math::sin(radian.x);
		float cosX = Math::cos(radian.x);
		float sinY = Math::sin(radian.y);
		float cosY = Math::cos(radian.y);
		float sinZ = Math::sin(radian.z);
		float cosZ = Math::cos(radian.z);
		m00 = cosY * cosZ;
		m01 = sinX * sinY * cosZ - cosX * sinZ;
		m02 = cosX * sinY * cosZ + sinX * sinZ;
		m03 = translation.x;

		m10 = cosY * sinZ;
		m11 = sinX * sinY * sinZ + cosX * cosZ;
		m12 = cosX * sinY * sinZ - sinX * cosZ;
		m13 = translation.y;

		m20 = -sinY;
		m21 = sinX * cosY;
		m22 = cosX * cosY;
		m23 = translation.z;

		m30 = 0.f;
		m31 = 0.f;
		m32 = 0.f;
		m33 = 1.f;
	}

	/**
	 * XYZ]̕ϊǉ
	 *
	 * XYZ]Aړ̏ɕϊǉ܂
	 * @param radian eɂ郉WAPʂł̉]px
	 * @param translation ړl
	 */
	inline void addTransformationXYZ(
		const Vector3& radian, const Vector3& translation){
		Matrix44 matrix;
		matrix.setTransformationXYZ(radian, translation);
		(*this) = matrix * (*this); 
	}

	//--------------------------------------------------------------------------
	/**
	 * XYZ]̕ϊݒ
	 *
	 * XP[AXYZ]Aړ̏ɕϊs쐬܂
	 * @param scale XP[l
	 * @param radian eɂ郉WAPʂł̉]px
	 * @param translation ړl
	 */
	inline void setTransformationXYZ(const Vector3& scale,
		const Vector3& radian, const Vector3& translation){
		float sinX = Math::sin(radian.x);
		float cosX = Math::cos(radian.x);
		float sinY = Math::sin(radian.y);
		float cosY = Math::cos(radian.y);
		float sinZ = Math::sin(radian.z);
		float cosZ = Math::cos(radian.z);
		m00 = scale.x * (cosY * cosZ);
		m01 = scale.y * (sinX * sinY * cosZ - cosX * sinZ);
		m02 = scale.z * (cosX * sinY * cosZ + sinX * sinZ);
		m03 = translation.x;

		m10 = scale.x * (cosY * sinZ);
		m11 = scale.y * (sinX * sinY * sinZ + cosX * cosZ);
		m12 = scale.z * (cosX * sinY * sinZ - sinX * cosZ);
		m13 = translation.y;

		m20 = scale.x * (-sinY);
		m21 = scale.y * (sinX * cosY);
		m22 = scale.z * (cosX * cosY);
		m23 = translation.z;

		m30 = 0.f;
		m31 = 0.f;
		m32 = 0.f;
		m33 = 1.f;
	}

	/**
	 * XYZ]̕ϊǉ
	 *
	 * XP[AXYZ]Aړ̏ɕϊǉ܂
	 * @param scale XP[l
	 * @param radian eɂ郉WAPʂł̉]px
	 * @param translation ړl
	 */
	inline void addTransformationXYZ(const Vector3& scale,
		const Vector3& radian, const Vector3& translation){
		Matrix44 matrix;
		matrix.setTransformationXYZ(scale, radian, translation);
		(*this) = matrix * (*this); 
	}

	//--------------------------------------------------------------------------
	/**
	 * l]̕ϊݒ
	 *
	 * l]Aړ̏ɕϊs쐬܂
	 * @param translation ړl
	 * @param quaternion l
	 */
	inline void setTransformationQuaternion(
		const Quaternion& quaternion, const Vector3& translation){
		Assert(quaternion.isUnit());
		float x2 = quaternion.x + quaternion.x;
		float y2 = quaternion.y + quaternion.y;
		float z2 = quaternion.z + quaternion.z;
		float xx2 = quaternion.x * x2;
		float xy2 = quaternion.x * y2;
		float xz2 = quaternion.x * z2;
		float yy2 = quaternion.y * y2;
		float yz2 = quaternion.y * z2;
		float zz2 = quaternion.z * z2;
		float wx2 = quaternion.w * x2;
		float wy2 = quaternion.w * y2;
		float wz2 = quaternion.w * z2;
		m00 = 1.f - (yy2 + zz2);
		m01 = xy2 - wz2;
		m02 = xz2 + wy2;
		m03 = translation.x;

		m10 = xy2 + wz2;
		m11 = 1.f - (xx2 + zz2);
		m12 = yz2 - wx2;
		m13 = translation.y;

		m20 = xz2 - wy2;
		m21 = yz2 + wx2;
		m22 = 1.f - (xx2 + yy2);
		m23 = translation.z;

		m30 = 0.f;
		m31 = 0.f;
		m32 = 0.f;
		m33 = 1.f;
	}

	/**
	 * l]̕ϊǉ
	 *
	 * l]Aړ̏ɕϊǉ܂
	 * @param translation ړl
	 * @param quaternion l
	 */
	inline void addTransformationQuaternion(
		const Quaternion& quaternion, const Vector3& translation){
		Matrix44 matrix;
		matrix.setTransformationQuaternion(quaternion, translation);
		(*this) = matrix * (*this); 
	}

	//--------------------------------------------------------------------------
	/**
	 * l]̕ϊݒ
	 *
	 * XP[Al]Aړ̏ɕϊs쐬܂
	 * @param scale XP[l
	 * @param translation ړl
	 * @param quaternion l
	 */
	inline void setTransformationQuaternion(const Vector3& scale,
		const Quaternion& quaternion, const Vector3& translation){
		Assert(quaternion.isUnit());
		float x2 = quaternion.x + quaternion.x;
		float y2 = quaternion.y + quaternion.y;
		float z2 = quaternion.z + quaternion.z;
		float xx2 = quaternion.x * x2;
		float xy2 = quaternion.x * y2;
		float xz2 = quaternion.x * z2;
		float yy2 = quaternion.y * y2;
		float yz2 = quaternion.y * z2;
		float zz2 = quaternion.z * z2;
		float wx2 = quaternion.w * x2;
		float wy2 = quaternion.w * y2;
		float wz2 = quaternion.w * z2;
		m00 = scale.x * (1.f - (yy2 + zz2));
		m01 = scale.y * (xy2 - wz2);
		m02 = scale.z * (xz2 + wy2);
		m03 = translation.x;

		m10 = scale.x * (xy2 + wz2);
		m11 = scale.y * (1.f - (xx2 + zz2));
		m12 = scale.z * (yz2 - wx2);
		m13 = translation.y;

		m20 = scale.x * (xz2 - wy2);
		m21 = scale.y * (yz2 + wx2);
		m22 = scale.z * (1.f - (xx2 + yy2));
		m23 = translation.z;

		m30 = 0.f;
		m31 = 0.f;
		m32 = 0.f;
		m33 = 1.f;
	}

	/**
	 * l]̕ϊǉ
	 *
	 * XP[Al]Aړ̏ɕϊǉ܂
	 * @param scale XP[l
	 * @param translation ړl
	 * @param quaternion l
	 */
	inline void addTransformationQuaternion(const Vector3& scale,
		const Quaternion& quaternion, const Vector3& translation){
		Matrix44 matrix;
		matrix.setTransformationQuaternion(scale, quaternion, translation);
		(*this) = matrix * (*this); 
	}

	//--------------------------------------------------------------------------
	// Z
	//--------------------------------------------------------------------------
	/**
	 * sZ
	 *
	 * ̍s񂩂珇ɌvZKp܂B
	 * @param mtx Zs
	 * @return Zꂽs
	 */
	inline Matrix44 operator *(const Matrix44& mtx) const{
		return Matrix44(
			(m00 * mtx.m00) + (m01 * mtx.m10) + (m02 * mtx.m20) + (m03 * mtx.m30),
			(m00 * mtx.m01) + (m01 * mtx.m11) + (m02 * mtx.m21) + (m03 * mtx.m31),
			(m00 * mtx.m02) + (m01 * mtx.m12) + (m02 * mtx.m22) + (m03 * mtx.m32),
			(m00 * mtx.m03) + (m01 * mtx.m13) + (m02 * mtx.m23) + (m03 * mtx.m33),
			(m10 * mtx.m00) + (m11 * mtx.m10) + (m12 * mtx.m20) + (m13 * mtx.m30),
			(m10 * mtx.m01) + (m11 * mtx.m11) + (m12 * mtx.m21) + (m13 * mtx.m31),
			(m10 * mtx.m02) + (m11 * mtx.m12) + (m12 * mtx.m22) + (m13 * mtx.m32),
			(m10 * mtx.m03) + (m11 * mtx.m13) + (m12 * mtx.m23) + (m13 * mtx.m33),
			(m20 * mtx.m00) + (m21 * mtx.m10) + (m22 * mtx.m20) + (m23 * mtx.m30),
			(m20 * mtx.m01) + (m21 * mtx.m11) + (m22 * mtx.m21) + (m23 * mtx.m31),
			(m20 * mtx.m02) + (m21 * mtx.m12) + (m22 * mtx.m22) + (m23 * mtx.m32),
			(m20 * mtx.m03) + (m21 * mtx.m13) + (m22 * mtx.m23) + (m23 * mtx.m33),
			(m30 * mtx.m00) + (m31 * mtx.m10) + (m32 * mtx.m20) + (m33 * mtx.m30),
			(m30 * mtx.m01) + (m31 * mtx.m11) + (m32 * mtx.m21) + (m33 * mtx.m31),
			(m30 * mtx.m02) + (m31 * mtx.m12) + (m32 * mtx.m22) + (m33 * mtx.m32),
			(m30 * mtx.m03) + (m31 * mtx.m13) + (m32 * mtx.m23) + (m33 * mtx.m33));
	}

	/**
	 * sZ
	 *
	 * ̍s񂩂珇ɌvZKp܂B
	 * @param mtx Zs
	 * @return Zꂽs
	 */
	inline Matrix44& operator *=(Matrix44 mtx){
		float old00 = m00;
		float old01 = m01;
		float old02 = m02;
		m00 = (old00 * mtx.m00) + (old01 * mtx.m10) +
			(old02 * mtx.m20) + (m03 * mtx.m30);
		m01 = (old00 * mtx.m01) + (old01 * mtx.m11) +
			(old02 * mtx.m21) + (m03 * mtx.m31);
		m02 = (old00 * mtx.m02) + (old01 * mtx.m12) +
			(old02 * mtx.m22) + (m03 * mtx.m32);
		m03 = (old00 * mtx.m03) + (old01 * mtx.m13) +
			(old02 * mtx.m23) + (m03 * mtx.m33);
		float old10 = m10;
		float old11 = m11;
		float old12 = m12;
		m10 = (old10 * mtx.m00) + (old11 * mtx.m10) +
			(old12 * mtx.m20) + (m13 * mtx.m30);
		m11 = (old10 * mtx.m01) + (old11 * mtx.m11) +
			(old12 * mtx.m21) + (m13 * mtx.m31);
		m12 = (old10 * mtx.m02) + (old11 * mtx.m12) +
			(old12 * mtx.m22) + (m13 * mtx.m32);
		m13 = (old10 * mtx.m03) + (old11 * mtx.m13) +
			(old12 * mtx.m23) + (m13 * mtx.m33);
		float old20 = m20;
		float old21 = m21;
		float old22 = m22;
		m20 = (old20 * mtx.m00) + (old21 * mtx.m10) +
			(old22 * mtx.m20) + (m23 * mtx.m30);
		m21 = (old20 * mtx.m01) + (old21 * mtx.m11) +
			(old22 * mtx.m21) + (m23 * mtx.m31);
		m22 = (old20 * mtx.m02) + (old21 * mtx.m12) +
			(old22 * mtx.m22) + (m23 * mtx.m32);
		m23 = (old20 * mtx.m03) + (old21 * mtx.m13) +
			(old22 * mtx.m23) + (m23 * mtx.m33);
		float old30 = m30;
		float old31 = m31;
		float old32 = m32;
		m30 = (old30 * mtx.m00) + (old31 * mtx.m10) +
			(old32 * mtx.m20) + (m33 * mtx.m30);
		m31 = (old30 * mtx.m01) + (old31 * mtx.m11) +
			(old32 * mtx.m21) + (m33 * mtx.m31);
		m32 = (old30 * mtx.m02) + (old31 * mtx.m12) +
			(old32 * mtx.m22) + (m33 * mtx.m32);
		m33 = (old30 * mtx.m03) + (old31 * mtx.m13) +
			(old32 * mtx.m23) + (m33 * mtx.m33);
		return *this;
	}

	/**
	 * xNgZ
	 * @param vector ZxNg
	 * @return ZꂽxNg
	 */
	// WvflĂȂBVector4Ƃ̏ZAW = 1ŎZo邩B
	inline Vector3 operator *(const Vector3& vector) const{
		return Vector3(
			vector.x * m00 + vector.y * m01 + vector.z * m02 + m03,
			vector.x * m10 + vector.y * m11 + vector.z * m12 + m13,
			vector.x * m20 + vector.y * m21 + vector.z * m22 + m23);
	}

	/**
	 * 3x3̃xNgZ
	 * @param vector 3x3̏ZxNg
	 * @return 3x3̏ZꂽxNg
	 */
	inline Vector3 multiply33(const Vector3& vector) const{
		return Vector3(
			vector.x * m00 + vector.y * m01 + vector.z * m02,
			vector.x * m10 + vector.y * m11 + vector.z * m12,
			vector.x * m20 + vector.y * m21 + vector.z * m22);
	}

	/**
	 * XJ[Z
	 * @param value ZXJ[
	 * @return Zꂽs
	 */
	inline Matrix44 operator *(float value) const{
		return Matrix44(
			m00 * value, m01 * value, m02 * value, m03 * value,
			m10 * value, m11 * value, m12 * value, m13 * value,
			m20 * value, m21 * value, m22 * value, m23 * value,
			m30 * value, m31 * value, m32 * value, m33 * value);
	}

	/**
	 * XJ[Z
	 * @param value ZXJ[
	 * @return Zꂽs
	 */
	inline Matrix44& operator *=(float value){
		m00 *= value;
		m01 *= value;
		m02 *= value;
		m03 *= value;
		m10 *= value;
		m11 *= value;
		m12 *= value;
		m13 *= value;
		m20 *= value;
		m21 *= value;
		m22 *= value;
		m23 *= value;
		m30 *= value;
		m31 *= value;
		m32 *= value;
		m33 *= value;
		return *this;
	}

	//--------------------------------------------------------------------------
	// s񉉎Z
	//--------------------------------------------------------------------------
	/**
	 * ]u
	 */
	inline void transpose(){
		float swap;
		swap = m01; m01 = m10; m10 = swap;
		swap = m02; m02 = m20; m20 = swap;
		swap = m03; m03 = m30; m30 = swap;
		swap = m12; m12 = m21; m21 = swap;
		swap = m13; m13 = m31; m31 = swap;
		swap = m23; m23 = m32; m32 = swap;
	}

	/**
	 * s
	 * @return s
	 */
	inline float determinant() const{
		return
			m00 * (
				m11 * (m22 * m33 - m32 * m23) -
				m12 * (m21 * m33 - m31 * m23) +
				m13 * (m21 * m32 - m31 * m22)
			) -
			m01 * (
				m10 * (m22 * m33 - m32 * m23) -
				m12 * (m20 * m33 - m30 * m23) +
				m13 * (m20 * m32 - m30 * m22)
			) +
			m02 * (
				m10 * (m21 * m33 - m31 * m23) -
				m11 * (m20 * m33 - m30 * m23) +
				m13 * (m20 * m31 - m30 * m21)
			) -
			m03 * (
				m10 * (m21 * m32 - m31 * m22) -
				m11 * (m20 * m32 - m30 * m22) +
				m12 * (m20 * m31 - m30 * m21)
			);
	}

	/**
	 * ts
	 * @return s
	 */
	inline float invert(){
		Matrix44 invertMatrix;
		// s񎮂o
		invertMatrix.m00 = (
			m11 * (m22 * m33 - m32 * m23) -
			m12 * (m21 * m33 - m31 * m23) +
			m13 * (m21 * m32 - m31 * m22));
		invertMatrix.m10 = -(
			m10 * (m22 * m33 - m32 * m23) -
			m12 * (m20 * m33 - m30 * m23) +
			m13 * (m20 * m32 - m30 * m22));
		invertMatrix.m20 = (
			m10 * (m21 * m33 - m31 * m23) -
			m11 * (m20 * m33 - m30 * m23) +
			m13 * (m20 * m31 - m30 * m21));
		invertMatrix.m30 = -(
			m10 * (m21 * m32 - m31 * m22) -
			m11 * (m20 * m32 - m30 * m22) +
			m12 * (m20 * m31 - m30 * m21));
		float determ =
			m00 * invertMatrix.m00 +
			m01 * invertMatrix.m10 +
			m02 * invertMatrix.m20 +
			m03 * invertMatrix.m30;
		Assert(Math::abs(determ) > Math::epsilon);
		// evf̎Zo
		invertMatrix.m01 = -(
			m01 * (m22 * m33 - m32 * m23) -
			m02 * (m21 * m33 - m31 * m23) +
			m03 * (m21 * m32 - m31 * m22));
		invertMatrix.m02 = (
			m01 * (m12 * m33 - m32 * m13) -
			m02 * (m11 * m33 - m31 * m13) +
			m03 * (m11 * m32 - m31 * m12));
		invertMatrix.m03 = -(
			m01 * (m12 * m23 - m22 * m13) -
			m02 * (m11 * m23 - m21 * m13) +
			m03 * (m11 * m22 - m21 * m12));
		invertMatrix.m11 = (
			m00 * (m22 * m33 - m32 * m23) -
			m02 * (m20 * m33 - m30 * m23) +
			m03 * (m20 * m32 - m30 * m22));
		invertMatrix.m12 = -(
			m00 * (m12 * m33 - m32 * m13) -
			m02 * (m10 * m33 - m30 * m13) +
			m03 * (m10 * m32 - m30 * m12));
		invertMatrix.m13 = (
			m00 * (m12 * m23 - m22 * m13) -
			m02 * (m10 * m23 - m20 * m13) +
			m03 * (m10 * m22 - m20 * m12));
		invertMatrix.m21 = -(
			m00 * (m21 * m33 - m31 * m23) -
			m01 * (m20 * m33 - m30 * m23) +
			m03 * (m20 * m31 - m30 * m21));
		invertMatrix.m22 = (
			m00 * (m11 * m33 - m31 * m13) -
			m01 * (m10 * m33 - m30 * m13) +
			m03 * (m10 * m31 - m30 * m11));
		invertMatrix.m23 = -(
			m00 * (m11 * m23 - m21 * m13) -
			m01 * (m10 * m23 - m20 * m13) +
			m03 * (m10 * m21 - m20 * m11));
		invertMatrix.m31 = (
			m00 * (m21 * m32 - m31 * m22) -
			m01 * (m20 * m32 - m30 * m22) +
			m02 * (m20 * m31 - m30 * m21));
		invertMatrix.m32 = -(
			m00 * (m11 * m32 - m31 * m12) -
			m01 * (m10 * m32 - m30 * m12) +
			m02 * (m10 * m31 - m30 * m11));
		invertMatrix.m33 = (
			m00 * (m11 * m22 - m21 * m12) -
			m01 * (m10 * m22 - m20 * m12) +
			m02 * (m10 * m21 - m20 * m11));
		// s񎮂̋t|
		float invDeterm = 1.f / determ;
		invertMatrix *= invDeterm;
		(*this) = invertMatrix;
		return determ;
	}

	/**
	 * ts
	 * @param invertMatrix [out] tsi[sւ̃|C^
	 * @return s
	 */
	inline float invert(Matrix44* invertMatrix) const{
		Assert(invertMatrix != NULL);
		Assert(invertMatrix != this);
		// s񎮂o
		invertMatrix->m00 = (
			m11 * (m22 * m33 - m32 * m23) -
			m12 * (m21 * m33 - m31 * m23) +
			m13 * (m21 * m32 - m31 * m22));
		invertMatrix->m10 = -(
			m10 * (m22 * m33 - m32 * m23) -
			m12 * (m20 * m33 - m30 * m23) +
			m13 * (m20 * m32 - m30 * m22));
		invertMatrix->m20 = (
			m10 * (m21 * m33 - m31 * m23) -
			m11 * (m20 * m33 - m30 * m23) +
			m13 * (m20 * m31 - m30 * m21));
		invertMatrix->m30 = -(
			m10 * (m21 * m32 - m31 * m22) -
			m11 * (m20 * m32 - m30 * m22) +
			m12 * (m20 * m31 - m30 * m21));
		float determ =
			m00 * invertMatrix->m00 +
			m01 * invertMatrix->m10 +
			m02 * invertMatrix->m20 +
			m03 * invertMatrix->m30;
		Assert(determ > Math::epsilon);
		// evf̎Zo
		invertMatrix->m01 = -(
			m01 * (m22 * m33 - m32 * m23) -
			m02 * (m21 * m33 - m31 * m23) +
			m03 * (m21 * m32 - m31 * m22));
		invertMatrix->m02 = (
			m01 * (m12 * m33 - m32 * m13) -
			m02 * (m11 * m33 - m31 * m13) +
			m03 * (m11 * m32 - m31 * m12));
		invertMatrix->m03 = -(
			m01 * (m12 * m23 - m22 * m13) -
			m02 * (m11 * m23 - m21 * m13) +
			m03 * (m11 * m22 - m21 * m12));
		invertMatrix->m11 = (
			m00 * (m22 * m33 - m32 * m23) -
			m02 * (m20 * m33 - m30 * m23) +
			m03 * (m20 * m32 - m30 * m22));
		invertMatrix->m12 = -(
			m00 * (m12 * m33 - m32 * m13) -
			m02 * (m10 * m33 - m30 * m13) +
			m03 * (m10 * m32 - m30 * m12));
		invertMatrix->m13 = (
			m00 * (m12 * m23 - m22 * m13) -
			m02 * (m10 * m23 - m20 * m13) +
			m03 * (m10 * m22 - m20 * m12));
		invertMatrix->m21 = -(
			m00 * (m21 * m33 - m31 * m23) -
			m01 * (m20 * m33 - m30 * m23) +
			m03 * (m20 * m31 - m30 * m21));
		invertMatrix->m22 = (
			m00 * (m11 * m33 - m31 * m13) -
			m01 * (m10 * m33 - m30 * m13) +
			m03 * (m10 * m31 - m30 * m11));
		invertMatrix->m23 = -(
			m00 * (m11 * m23 - m21 * m13) -
			m01 * (m10 * m23 - m20 * m13) +
			m03 * (m10 * m21 - m20 * m11));
		invertMatrix->m31 = (
			m00 * (m21 * m32 - m31 * m22) -
			m01 * (m20 * m32 - m30 * m22) +
			m02 * (m20 * m31 - m30 * m21));
		invertMatrix->m32 = -(
			m00 * (m11 * m32 - m31 * m12) -
			m01 * (m10 * m32 - m30 * m12) +
			m02 * (m10 * m31 - m30 * m11));
		invertMatrix->m33 = (
			m00 * (m11 * m22 - m21 * m12) -
			m01 * (m10 * m22 - m20 * m12) +
			m02 * (m10 * m21 - m20 * m11));
		// s񎮂̋t|
		float invDeterm = 1.f / determ;
		(*invertMatrix) *= invDeterm;
		return determ;
	}

	//--------------------------------------------------------------------------
	/**
	 * ϊs̋ts
	 *
	 * ϊsɂ͉]ƈړ܂܂ĂȂƉ肷
	 */
	inline void invertTransformation(){
		Assert((m30 == 0.f) && (m31 == 0.f) && (m32 == 0.f) && (m33 == 1.f));
		// ]s]u
		float swap;
		swap = m01; m01 = m10; m10 = swap;
		swap = m02; m02 = m20; m20 = swap;
		swap = m12; m12 = m21; m21 = swap;
		// ړltϊ
		Vector3 trans(-m03, -m13, -m23);
		m03 = m00 * trans.x + m01 * trans.y + m02 * trans.z;
		m13 = m10 * trans.x + m11 * trans.y + m12 * trans.z;
		m23 = m20 * trans.x + m21 * trans.y + m22 * trans.z;
	}

	/**
	 * ϊs̋ts
	 *
	 * ϊsɂ͉]ƈړ܂܂ĂȂƉ肷
	 * @param invertMatrix [out] tsi[sւ̃|C^
	 */
	inline void invertTransformation(Matrix44* invertMatrix) const{
		Assert((m30 == 0.f) && (m31 == 0.f) && (m32 == 0.f) && (m33 == 1.f));
		// ]s]u
		invertMatrix->m00 = m00;
		invertMatrix->m01 = m10;
		invertMatrix->m02 = m20;
		invertMatrix->m10 = m01;
		invertMatrix->m11 = m11;
		invertMatrix->m12 = m21;
		invertMatrix->m20 = m02;
		invertMatrix->m21 = m12;
		invertMatrix->m22 = m22;
		// ړltϊ
		Vector3 trans(-m03, -m13, -m23);
		invertMatrix->m03 = m00 * trans.x + m10 * trans.y + m20 * trans.z;
		invertMatrix->m13 = m01 * trans.x + m11 * trans.y + m21 * trans.z;
		invertMatrix->m23 = m02 * trans.x + m12 * trans.y + m22 * trans.z;
		// c𖄂߂Ă
		invertMatrix->m30 = 0.f;
		invertMatrix->m31 = 0.f;
		invertMatrix->m32 = 0.f;
		invertMatrix->m33 = 1.f;
	}

	//--------------------------------------------------------------------------
	// _Z
	//--------------------------------------------------------------------------
	/**
	 * s񂪓ǂ
	 * @param target rs
	 * @return lłtrueԂ
	 */
	inline bool operator ==(const Matrix44& target) const{
		return (
			(m00 == target.m00) && (m01 == target.m01) &&
			(m02 == target.m02) && (m03 == target.m03) &&
			(m10 == target.m10) && (m11 == target.m11) &&
			(m12 == target.m12) && (m13 == target.m13) &&
			(m20 == target.m20) && (m21 == target.m21) &&
			(m22 == target.m22) && (m23 == target.m23) &&
			(m30 == target.m30) && (m31 == target.m31) &&
			(m32 == target.m32) && (m33 == target.m33));
	}

	/**
	 * s񂪓ǂ
	 * @param target rs
	 * @param epsilon 덷
	 * @return 덷͈͓̔œlłtrueԂ
	 */
	inline bool epsilonEquals(const Matrix44& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			(Math::abs(m00 - target.m00) <= epsilon) &&
			(Math::abs(m01 - target.m01) <= epsilon) &&
			(Math::abs(m02 - target.m02) <= epsilon) &&
			(Math::abs(m03 - target.m03) <= epsilon) &&
			(Math::abs(m10 - target.m10) <= epsilon) &&
			(Math::abs(m11 - target.m11) <= epsilon) &&
			(Math::abs(m12 - target.m12) <= epsilon) &&
			(Math::abs(m13 - target.m13) <= epsilon) &&
			(Math::abs(m20 - target.m20) <= epsilon) &&
			(Math::abs(m21 - target.m21) <= epsilon) &&
			(Math::abs(m22 - target.m22) <= epsilon) &&
			(Math::abs(m23 - target.m23) <= epsilon) &&
			(Math::abs(m30 - target.m30) <= epsilon) &&
			(Math::abs(m31 - target.m31) <= epsilon) &&
			(Math::abs(m32 - target.m32) <= epsilon) &&
			(Math::abs(m33 - target.m33) <= epsilon));
	}

	/**
	 * s񂪓łȂǂ
	 * @param target rs
	 * @return łȂlłtrueԂ
	 */
	inline bool operator !=(const Matrix44& target) const{
		return (
			(m00 != target.m00) || (m01 != target.m01) ||
			(m02 != target.m02) || (m03 != target.m03) ||
			(m10 != target.m10) || (m11 != target.m11) ||
			(m12 != target.m12) || (m13 != target.m13) ||
			(m20 != target.m20) || (m21 != target.m21) ||
			(m22 != target.m22) || (m23 != target.m23) ||
			(m30 != target.m30) || (m31 != target.m31) ||
			(m32 != target.m32) || (m33 != target.m33));
	}

	/**
	 * s񂪓łȂǂ
	 * @param target rs
	 * @param epsilon 덷
	 * @return 덷͈͓̔œłȂlłtrueԂ
	 */
	inline bool notEpsilonEquals(const Matrix44& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			(Math::abs(m00 - target.m00) > epsilon) ||
			(Math::abs(m01 - target.m01) > epsilon) ||
			(Math::abs(m02 - target.m02) > epsilon) ||
			(Math::abs(m03 - target.m03) > epsilon) ||
			(Math::abs(m10 - target.m10) > epsilon) ||
			(Math::abs(m11 - target.m11) > epsilon) ||
			(Math::abs(m12 - target.m12) > epsilon) ||
			(Math::abs(m13 - target.m13) > epsilon) ||
			(Math::abs(m20 - target.m20) > epsilon) ||
			(Math::abs(m21 - target.m21) > epsilon) ||
			(Math::abs(m22 - target.m22) > epsilon) ||
			(Math::abs(m23 - target.m23) > epsilon) ||
			(Math::abs(m30 - target.m30) > epsilon) ||
			(Math::abs(m31 - target.m31) > epsilon) ||
			(Math::abs(m32 - target.m32) > epsilon) ||
			(Math::abs(m33 - target.m33) > epsilon));
	}

	//--------------------------------------------------------------------------
	// ̑
	//--------------------------------------------------------------------------
	/**
	 * 
	 * @return s̕\L
	 */
	inline String toString() const{
		String returnString;
		returnString.format(
			"{\n  ( %.8f, %.8f, %.8f, %.8f )\n  ( %.8f, %.8f, %.8f, %.8f )\n"
			"  ( %.8f, %.8f, %.8f, %.8f )\n  ( %.8f, %.8f, %.8f, %.8f )\n}",
			m00, m01, m02, m03, m10, m11, m12, m13,
			m20, m21, m22, m23, m30, m31, m32, m33);
		return returnString;
	}

	//--------------------------------------------------------------------------
private:

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of MATRIX44_H_
//------------------------------------------------------------------------------
