//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * lvfJ[wb_
 * @author Junpee
 */

#ifndef COLOR_4F_H_
#define COLOR_4F_H_

#include <Core/System/Math.h>

namespace Lamp{

class Color3c;
class Color4c;
class Color3f;

//------------------------------------------------------------------------------
/**
 * lvfJ[
 *
 * ̃NX͌pȂŉB
 */
class Color4f{
public:
	//--------------------------------------------------------------------------
	// oϐ
	//--------------------------------------------------------------------------
	/// oϐ
	union{
		/// evf
		struct{
			/// 
			float r;
			/// 
			float g;
			/// 
			float b;
			/// At@
			float a;
		};

		/// evf
		struct{
			/// F
			float h;
			/// ʓx
			float s;
			/// x
			float v;
			/// At@
			float a;
		};

		/// z
		float array[4];
	};

	//--------------------------------------------------------------------------
	// 萔
	//--------------------------------------------------------------------------
	/// 
	static const Color4f white;

	/// DF
	static const Color4f gray;

	/// 
	static const Color4f black;

	/// 
	static const Color4f red;

	/// 
	static const Color4f green;

	/// 
	static const Color4f blue;

	/// 
	static const Color4f yellow;

	/// 
	static const Color4f cyan;

	/// Ԏ
	static const Color4f magenta;

	//--------------------------------------------------------------------------
	// RXgN^
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 *
	 * ̃RXgN^͏l̐ݒsȂߒl͕słB
	 */
	Color4f(){}

	/**
	 * RXgN^
	 * @param sourceR Ԃ̏l
	 * @param sourceG ΂̏l
	 * @param sourceB ̏l
	 * @param sourceA At@̏l
	 */
	inline Color4f(float sourceR, float sourceG, float sourceB,
		float sourceA = 1.f) :
		r(sourceR), g(sourceG), b(sourceB), a(sourceA){
	}

	/**
	 * RXgN^
	 * @param source ݒ肷F
	 */
	explicit Color4f(const Color3c& source);

	/**
	 * RXgN^
	 * @param source ݒ肷F
	 */
	explicit Color4f(const Color4c& source);

	/**
	 * RXgN^
	 * @param source ݒ肷F
	 */
	explicit Color4f(const Color3f& source);

	//--------------------------------------------------------------------------
	// l̐ݒ
	//--------------------------------------------------------------------------
	/**
	 * l̐ݒ
	 * @param sourceR Ԃ̐ݒl
	 * @param sourceG ΂̐ݒl
	 * @param sourceB ̐ݒl
	 * @param sourceA At@̐ݒl
	 */
	inline void set(float sourceR, float sourceG, float sourceB,
		float sourceA = 1.f){
		r = sourceR;
		g = sourceG;
		b = sourceB;
		a = sourceA;
	}

	/**
	 * OvfLN^J[̐ݒ
	 * @param source ݒ肷F
	 */
	void set(const Color3c& source);

	/**
	 * lvfLN^J[̐ݒ
	 * @param source ݒ肷F
	 */
	void set(const Color4c& source);

	/**
	 * OvfJ[̐ݒ
	 * @param source ݒ肷F
	 */
	void set(const Color3f& source);

	//--------------------------------------------------------------------------
	// Z
	//--------------------------------------------------------------------------
	/**
	 * Z
	 * @param addColor ZF
	 * @return ZꂽF
	 */
	inline Color4f operator +(const Color4f& addColor) const{
		return Color4f(
			r + addColor.r, g + addColor.g, b + addColor.b, a + addColor.a);
	}

	/**
	 * Z
	 * @param subColor ZF
	 * @return ZꂽF
	 */
	inline Color4f operator -(const Color4f& subColor) const{
		return Color4f(
			r - subColor.r, g - subColor.g, b - subColor.b, a - subColor.a);
	}

	/**
	 * Z
	 * @param mulColor ZF
	 * @return ZꂽF
	 */
	inline Color4f operator *(const Color4f& mulColor) const{
		return Color4f(
			r * mulColor.r, g * mulColor.g, b * mulColor.b, a * mulColor.a);
	}

	/**
	 * Z
	 * @param mulValue Zl
	 * @return ZꂽF
	 */
	inline Color4f operator *(float mulValue) const{
		return Color4f(
			r * mulValue, g * mulValue, b * mulValue, a * mulValue);
	}

	/**
	 * Z
	 * @param mulValue Zl
	 * @param mulColor ZF
	 * @return ZꂽF
	 */
	inline friend Color4f operator *(float mulValue, const Color4f& mulColor){
		return Color4f(
			mulColor.r * mulValue, mulColor.g * mulValue,
			mulColor.b * mulValue, mulColor.a * mulValue);
	}

	/**
	 * +Zq
	 * @return F̃Rs[
	 */
	inline Color4f operator +() const{ return *this; }

	/**
	 * -Zq
	 * @return l̕]F
	 */
	inline Color4f operator -() const{ return Color4f(-r, -g, -b, -a); }

	//--------------------------------------------------------------------------
	// Z
	//--------------------------------------------------------------------------
	/**
	 * Z
	 * @param addColor ZF
	 * @return ZꂽF
	 */
	inline Color4f& operator +=(const Color4f& addColor){
		r += addColor.r;
		g += addColor.g;
		b += addColor.b;
		a += addColor.a;
		return (*this);
	}

	/**
	 * Z
	 * @param subColor ZF
	 * @return ZꂽF
	 */
	inline Color4f& operator -=(const Color4f& subColor){
		r -= subColor.r;
		g -= subColor.g;
		b -= subColor.b;
		a -= subColor.a;
		return (*this);
	}

	/**
	 * Z
	 * @param mulColor ZF
	 * @return ZꂽF
	 */
	inline Color4f& operator *=(const Color4f& mulColor){
		r *= mulColor.r;
		g *= mulColor.g;
		b *= mulColor.b;
		a *= mulColor.a;
		return (*this);
	}

	/**
	 * Z
	 * @param mulValue Zl
	 * @return ZꂽF
	 */
	inline Color4f& operator *=(float mulValue){
		r *= mulValue;
		g *= mulValue;
		b *= mulValue;
		a *= mulValue;
		return (*this);
	}

	//--------------------------------------------------------------------------
	// FZ
	//--------------------------------------------------------------------------
	/**
	 * Nv
	 * @param lower Nvl
	 * @param upper Nvl
	 * @return NvꂽF
	 */
	inline Color4f& clamp(float lower = 0.f, float upper = 1.f){
		Assert(upper > lower);
		if(r > upper){ r = upper; }
		else if(r < lower){ r = lower; }
		if(g > upper){ g = upper; }
		else if(g < lower){ g = lower; }
		if(b > upper){ b = upper; }
		else if(b < lower){ b = lower; }
		if(a > upper){ a = upper; }
		else if(a < lower){ a = lower; }
		return (*this);
	}

	/**
	 * Nv
	 * @param lower Nvl
	 * @return NvꂽF
	 */
	inline Color4f& lowerClamp(float lower = 0.f){
		if(r < lower){ r = lower; }
		if(g < lower){ g = lower; }
		if(b < lower){ b = lower; }
		if(a < lower){ a = lower; }
		return (*this);
	}

	/**
	 * Nv
	 * @param upper Nvl
	 * @return NvꂽF
	 */
	inline Color4f& upperClamp(float upper = 1.f){
		if(r > upper){ r = upper; }
		if(g > upper){ g = upper; }
		if(b > upper){ b = upper; }
		if(a > upper){ a = upper; }
		return (*this);
	}

	/**
	 * ΐF
	 *
	 * At@l͕ω܂B
	 * @return ]ꂽF
	 */
	inline Color4f& negative(){
		set(1.f - r, 1.f - g, 1.f - b, a);
		return (*this);
	}

	//--------------------------------------------------------------------------
	/**
	 * HSV̐ݒ
	 * @param hsv HSVJ[
	 */
	inline void setHSV(const Color4f& hsv){
		float alpha = hsv.a;
		float value = hsv.v;
		if(hsv.s <= Math::epsilon){
			set(value, value, value, alpha);
			return;
		}
		float hue = hsv.h * 6.f;
		float integer = Math::floor(hue);
		float decimal = hue - integer;
		float temp0 = value * (1.f - hsv.s);
		float temp1 = value * (1.f - (hsv.s * decimal));
		float temp2 = value * (1.f - (hsv.s * (1.f - decimal)));
		if(integer < 1.f){
			set(value, temp2, temp0, alpha);
		}else if(integer < 2.f){
			set(temp1, value, temp0, alpha);
		}else if(integer < 3.f){
			set(temp0, value, temp2, alpha);
		}else if(integer < 4.f){
			set(temp0, temp1, value, alpha);
		}else if(integer < 5.f){
			set(temp2, temp0, value, alpha);
		}else{
			set(value, temp0, temp1, alpha);
		}
	}

	/**
	 * HSV̎擾
	 * @return HSVJ[
	 */
	inline Color4f getHSV() const{
		Color4f hsv;
		hsv.a = a;
		// x̎Zo
		float maximum = Math::maximum(r, Math::maximum(g, b));
		hsv.v = maximum;
		// ʓx̎Zo
		float minimum = Math::minimum(r, Math::minimum(g, b));
		float chroma = maximum - minimum;
		if(maximum <= Math::epsilon){
			hsv.s = 0.f;
			hsv.h = 0.f;
			return hsv;
		}else{
			hsv.s = chroma / maximum;
		}
		if(chroma <= Math::epsilon){
			hsv.h = 0.f;
			return hsv;
		}
		float hueScale = (1 / chroma) * 0.1666667f;
		if(r == maximum){
			hsv.h = (g - b) * hueScale;
		}else if(g == maximum){
			hsv.h = (b - r) * hueScale + 0.3333333f;
		}else{
			hsv.h = (r - g) * hueScale + 0.6666667f;
		}
		if(hsv.h < 0.f){ hsv.h += 1.f; }
		return hsv;
	}

	//--------------------------------------------------------------------------
	/**
	 * F̎擾
	 * @return F
	 */
	inline float getHue() const{
		float maximum = Math::maximum(r, Math::maximum(g, b));
		if(maximum <= Math::epsilon){ return 0.f; }
		float minimum = Math::minimum(r, Math::minimum(g, b));
		float chroma = maximum - minimum;
		if(chroma <= Math::epsilon){ return 0.f; }
		float hueScale = (1 / chroma) * 0.16666667f;
		float hue;
		if(r == maximum){
			hue = (g - b) * hueScale;
		}else if(g == maximum){
			hue = (b - r) * hueScale + 0.33333333f;
		}else{
			hue = (r - g) * hueScale + 0.66666667f;
		}
		if(hue < 0.f){ hue += 1.f; }
	}

	/**
	 * ʓx̎擾
	 * @return ʓx
	 */
	inline float getSaturation() const{
		float maximum = Math::maximum(r, Math::maximum(g, b));
		if(maximum <= Math::epsilon){ return 0.f; }
		float minimum = Math::minimum(r, Math::minimum(g, b));
		float chroma = maximum - minimum;
		return chroma / maximum;
	}

	/**
	 * x̎擾
	 * @return x
	 */
	inline float getValue() const{
		return Math::maximum(r, Math::maximum(g, b));
	}

	/**
	 * Px̎擾
	 * @return Px
	 */
	inline float getLuminance() const{
		return (r * 0.298912f + g * 0.586611f + b * 0.114477f);
	}

	//--------------------------------------------------------------------------
	/**
	 * F̐`
	 * @param source JnF
	 * @param target ΏېF
	 * @param alpha uhW
	 * @return `ԂꂽF
	 */
	inline static Color4f lerp(
		const Color4f& source, const Color4f& target, float alpha){
		float beta = 1.f - alpha;
		Color4f result;
		result.r = source.r * beta + target.r * alpha;
		result.g = source.g * beta + target.g * alpha;
		result.b = source.b * beta + target.b * alpha;
		result.a = source.a * beta + target.a * alpha;
		return result;
	}

	//--------------------------------------------------------------------------
	// _Z
	//--------------------------------------------------------------------------
	/**
	 * lǂ
	 * @param target rJ[
	 * @return lłtrueԂ
	 */
	inline bool operator ==(const Color4f& target) const{
		return ((r == target.r) && (g == target.g) &&
			(b == target.b) && (a == target.a));
	}

	/**
	 * lǂ
	 * @param target rJ[
	 * @param epsilon 덷
	 * @return 덷͈͓̔œlłtrueԂ
	 */
	inline bool epsilonEquals(const Color4f& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			(Math::abs(r - target.r) <= epsilon) &&
			(Math::abs(g - target.g) <= epsilon) &&
			(Math::abs(b - target.b) <= epsilon) &&
			(Math::abs(a - target.a) <= epsilon));
	}

	/**
	 * lłȂǂ
	 * @param target rJ[
	 * @return lłȂtrueԂ
	 */
	inline bool operator !=(const Color4f& target) const{
		return ((r != target.r) || (g != target.g) ||
			(b != target.b) || (a != target.a));
	}

	/**
	 * lłȂǂ
	 * @param target rJ[
	 * @param epsilon 덷
	 * @return 덷͈͓̔œłȂlłtrueԂ
	 */
	inline bool notEpsilonEquals(const Color4f& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			(Math::abs(r - target.r) > epsilon) ||
			(Math::abs(g - target.g) > epsilon) ||
			(Math::abs(b - target.b) > epsilon) ||
			(Math::abs(a - target.a) > epsilon));
	}

	//--------------------------------------------------------------------------
	// ̑
	//--------------------------------------------------------------------------
	/**
	 * 
	 * @return J[̕\L
	 */
	inline String toString() const{
		String returnString;
		returnString.format("( %.8f, %.8f, %.8f, %.8f )", r, g, b, a);
		return returnString;
	}

	//--------------------------------------------------------------------------
private:

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of COLOR_4F_H_
//------------------------------------------------------------------------------
