//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * OvfJ[wb_
 * @author Junpee
 */

#ifndef COLOR_3F_H_
#define COLOR_3F_H_

#include <Core/System/Math.h>

namespace Lamp{

class Color3c;
class Color4c;
class Color4f;

//------------------------------------------------------------------------------
/**
 * OvfJ[
 *
 * ̃NX͌pȂŉB
 */
class Color3f{
public:
	//--------------------------------------------------------------------------
	// oϐ
	//--------------------------------------------------------------------------
	/// oϐ
	union{
		/// evf
		struct{
			/// 
			float r;
			/// 
			float g;
			/// 
			float b;
		};

		/// evf
		struct{
			/// F
			float h;
			/// ʓx
			float s;
			/// x
			float v;
		};

		/// z
		float array[3];
	};

	//--------------------------------------------------------------------------
	// 萔
	//--------------------------------------------------------------------------
	/// 
	static const Color3f white;

	/// DF
	static const Color3f gray;

	/// 
	static const Color3f black;

	/// 
	static const Color3f red;

	/// 
	static const Color3f green;

	/// 
	static const Color3f blue;

	/// 
	static const Color3f yellow;

	/// 
	static const Color3f cyan;

	/// Ԏ
	static const Color3f magenta;

	//--------------------------------------------------------------------------
	// RXgN^
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 *
	 * ̃RXgN^͏l̐ݒsȂߒl͕słB
	 */
	Color3f(){}

	/**
	 * RXgN^
	 * @param sourceR Ԃ̏l
	 * @param sourceG ΂̏l
	 * @param sourceB ̏l
	 */
	inline Color3f(float sourceR, float sourceG, float sourceB) :
		r(sourceR), g(sourceG), b(sourceB){
	}

	/**
	 * RXgN^
	 * @param source ݒ肷F
	 */
	explicit Color3f(const Color3c& source);

	/**
	 * RXgN^
	 * @param source ݒ肷F
	 */
	explicit Color3f(const Color4c& source);

	/**
	 * RXgN^
	 * @param source ݒ肷F
	 */
	explicit Color3f(const Color4f& source);

	//--------------------------------------------------------------------------
	// l̐ݒ
	//--------------------------------------------------------------------------
	/**
	 * l̐ݒ
	 * @param sourceR Ԃ̐ݒl
	 * @param sourceG ΂̐ݒl
	 * @param sourceB ̐ݒl
	 */
	inline void set(float sourceR, float sourceG, float sourceB){
		r = sourceR;
		g = sourceG;
		b = sourceB;
	}

	/**
	 * OvfLN^J[̐ݒ
	 * @param source ݒ肷F
	 */
	void set(const Color3c& source);

	/**
	 * lvfLN^J[̐ݒ
	 * @param source ݒ肷F
	 */
	void set(const Color4c& source);

	/**
	 * lvfJ[̐ݒ
	 * @param source ݒ肷F
	 */
	void set(const Color4f& source);

	//--------------------------------------------------------------------------
	// Z
	//--------------------------------------------------------------------------
	/**
	 * Z
	 * @param addColor ZF
	 * @return ZꂽF
	 */
	inline Color3f operator +(const Color3f& addColor) const{
		return Color3f(r + addColor.r, g + addColor.g, b + addColor.b);
	}

	/**
	 * Z
	 * @param subColor ZF
	 * @return ZꂽF
	 */
	inline Color3f operator -(const Color3f& subColor) const{
		return Color3f(r - subColor.r, g - subColor.g, b - subColor.b);
	}

	/**
	 * Z
	 * @param mulColor ZF
	 * @return ZꂽF
	 */
	inline Color3f operator *(const Color3f& mulColor) const{
		return Color3f(r * mulColor.r, g * mulColor.g, b * mulColor.b);
	}

	/**
	 * Z
	 * @param mulValue Zl
	 * @return ZꂽF
	 */
	inline Color3f operator *(float mulValue) const{
		return Color3f(r * mulValue, g * mulValue, b * mulValue);
	}

	/**
	 * Z
	 * @param mulValue Zl
	 * @param mulColor ZF
	 * @return ZꂽF
	 */
	inline friend Color3f operator *(float mulValue, const Color3f& mulColor){
		return Color3f(mulColor.r * mulValue,
			mulColor.g * mulValue, mulColor.b * mulValue);
	}

	/**
	 * +Zq
	 * @return F̃Rs[
	 */
	inline Color3f operator +() const{ return *this; }

	/**
	 * -Zq
	 * @return l̕]F
	 */
	inline Color3f operator -() const{ return Color3f(-r, -g, -b); }

	//--------------------------------------------------------------------------
	// Z
	//--------------------------------------------------------------------------
	/**
	 * Z
	 * @param addColor ZF
	 * @return ZꂽF
	 */
	inline Color3f& operator +=(const Color3f& addColor){
		r += addColor.r;
		g += addColor.g;
		b += addColor.b;
		return (*this);
	}

	/**
	 * Z
	 * @param subColor ZF
	 * @return ZꂽF
	 */
	inline Color3f& operator -=(const Color3f& subColor){
		r -= subColor.r;
		g -= subColor.g;
		b -= subColor.b;
		return (*this);
	}

	/**
	 * Z
	 * @param mulColor ZF
	 * @return ZꂽF
	 */
	inline Color3f& operator *=(const Color3f& mulColor){
		r *= mulColor.r;
		g *= mulColor.g;
		b *= mulColor.b;
		return (*this);
	}

	/**
	 * Z
	 * @param mulValue Zl
	 * @return ZꂽF
	 */
	inline Color3f& operator *=(float mulValue){
		r *= mulValue;
		g *= mulValue;
		b *= mulValue;
		return (*this);
	}

	//--------------------------------------------------------------------------
	// FZ
	//--------------------------------------------------------------------------
	/**
	 * Nv
	 * @param lower Nvl
	 * @param upper Nvl
	 * @return NvꂽF
	 */
	inline Color3f& clamp(float lower = 0.f, float upper = 1.f){
		Assert(upper > lower);
		if(r > upper){ r = upper; }
		else if(r < lower){ r = lower; }
		if(g > upper){ g = upper; }
		else if(g < lower){ g = lower; }
		if(b > upper){ b = upper; }
		else if(b < lower){ b = lower; }
		return (*this);
	}

	/**
	 * Nv
	 * @param lower Nvl
	 * @return NvꂽF
	 */
	inline Color3f& lowerClamp(float lower = 0.f){
		if(r < lower){ r = lower; }
		if(g < lower){ g = lower; }
		if(b < lower){ b = lower; }
		return (*this);
	}

	/**
	 * Nv
	 * @param upper Nvl
	 * @return NvꂽF
	 */
	inline Color3f& upperClamp(float upper = 1.f){
		if(r > upper){ r = upper; }
		if(g > upper){ g = upper; }
		if(b > upper){ b = upper; }
		return (*this);
	}

	/**
	 * ΐF
	 * @return ]ꂽF
	 */
	inline Color3f& negative(){
		set(1.f - r, 1.f - g, 1.f - b);
		return (*this);
	}

	//--------------------------------------------------------------------------
	/**
	 * HSV̐ݒ
	 * @param hsv HSVJ[
	 */
	inline void setHSV(const Color3f& hsv){
		float value = hsv.v;
		if(hsv.s <= Math::epsilon){
			set(value, value, value);
			return;
		}
		float hue = hsv.h * 6.f;
		float integer = Math::floor(hue);
		float decimal = hue - integer;
		float temp0 = value * (1.f - hsv.s);
		float temp1 = value * (1.f - (hsv.s * decimal));
		float temp2 = value * (1.f - (hsv.s * (1.f - decimal)));
		if(integer < 1.f){
			set(value, temp2, temp0);
		}else if(integer < 2.f){
			set(temp1, value, temp0);
		}else if(integer < 3.f){
			set(temp0, value, temp2);
		}else if(integer < 4.f){
			set(temp0, temp1, value);
		}else if(integer < 5.f){
			set(temp2, temp0, value);
		}else{
			set(value, temp0, temp1);
		}
	}

	/**
	 * HSV̎擾
	 * @return HSVJ[
	 */
	inline Color3f getHSV() const{
		Color3f hsv;
		// x̎Zo
		float maximum = Math::maximum(r, Math::maximum(g, b));
		hsv.v = maximum;
		// ʓx̎Zo
		float minimum = Math::minimum(r, Math::minimum(g, b));
		float chroma = maximum - minimum;
		if(maximum <= Math::epsilon){
			hsv.s = 0.f;
			hsv.h = 0.f;
			return hsv;
		}else{
			hsv.s = chroma / maximum;
		}
		if(chroma <= Math::epsilon){
			hsv.h = 0.f;
			return hsv;
		}
		float hueScale = (1 / chroma) * 0.16666667f;
		if(r == maximum){
			hsv.h = (g - b) * hueScale;
		}else if(g == maximum){
			hsv.h = (b - r) * hueScale + 0.33333338f;
		}else{
			hsv.h = (r - g) * hueScale + 0.66666667f;
		}
		if(hsv.h < 0.f){ hsv.h += 1.f; }
		return hsv;
	}

	//--------------------------------------------------------------------------
	/**
	 * F̎擾
	 * @return F
	 */
	inline float getHue() const{
		float maximum = Math::maximum(r, Math::maximum(g, b));
		if(maximum <= Math::epsilon){ return 0.f; }
		float minimum = Math::minimum(r, Math::minimum(g, b));
		float chroma = maximum - minimum;
		if(chroma <= Math::epsilon){ return 0.f; }
		float hueScale = (1 / chroma) * 0.1666667f;
		float hue;
		if(r == maximum){
			hue = (g - b) * hueScale;
		}else if(g == maximum){
			hue = (b - r) * hueScale + 0.3333333f;
		}else{
			hue = (r - g) * hueScale + 0.6666667f;
		}
		if(hue < 0.f){ hue += 1.f; }
	}

	/**
	 * ʓx̎擾
	 * @return ʓx
	 */
	inline float getSaturation() const{
		float maximum = Math::maximum(r, Math::maximum(g, b));
		if(maximum <= Math::epsilon){ return 0.f; }
		float minimum = Math::minimum(r, Math::minimum(g, b));
		float chroma = maximum - minimum;
		return chroma / maximum;
	}

	/**
	 * x̎擾
	 * @return x
	 */
	inline float getValue() const{
		return Math::maximum(r, Math::maximum(g, b));
	}

	/**
	 * Px̎擾
	 * @return Px
	 */
	inline float getLuminance() const{
		return (r * 0.298912f + g * 0.586611f + b * 0.114477f);
	}

	//--------------------------------------------------------------------------
	/**
	 * F̐`
	 * @param source JnF
	 * @param target ΏېF
	 * @param alpha uhW
	 * @return `ԂꂽF
	 */
	inline static Color3f lerp(
		const Color3f& source, const Color3f& target, float alpha){
		float beta = 1.f - alpha;
		Color3f result;
		result.r = source.r * beta + target.r * alpha;
		result.g = source.g * beta + target.g * alpha;
		result.b = source.b * beta + target.b * alpha;
		return result;
	}

	//--------------------------------------------------------------------------
	// _Z
	//--------------------------------------------------------------------------
	/**
	 * lǂ
	 * @param target rJ[
	 * @return lłtrueԂ
	 */
	inline bool operator ==(const Color3f& target) const{
		return ((r == target.r) && (g == target.g) && (b == target.b));
	}

	/**
	 * lǂ
	 * @param target rJ[
	 * @param epsilon 덷
	 * @return 덷͈͓̔œlłtrueԂ
	 */
	inline bool epsilonEquals(const Color3f& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			(Math::abs(r - target.r) <= epsilon) &&
			(Math::abs(g - target.g) <= epsilon) &&
			(Math::abs(b - target.b) <= epsilon));
	}

	/**
	 * lłȂǂ
	 * @param target rJ[
	 * @return lłȂtrueԂ
	 */
	inline bool operator !=(const Color3f& target) const{
		return ((r != target.r) || (g != target.g) || (b != target.b));
	}

	/**
	 * lłȂǂ
	 * @param target rJ[
	 * @param epsilon 덷
	 * @return 덷͈͓̔œłȂlłtrueԂ
	 */
	inline bool notEpsilonEquals(const Color3f& target, float epsilon) const{
		Assert(epsilon >= 0.f);
		return (
			(Math::abs(r - target.r) > epsilon) ||
			(Math::abs(g - target.g) > epsilon) ||
			(Math::abs(b - target.b) > epsilon));
	}

	//--------------------------------------------------------------------------
	// ̑
	//--------------------------------------------------------------------------
	/**
	 * 
	 * @return J[̕\L
	 */
	inline String toString() const{
		String returnString;
		returnString.format("( %.8f, %.8f, %.8f )", r, g, b);
		return returnString;
	}

	//--------------------------------------------------------------------------
private:

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of COLOR_3F_H_
//------------------------------------------------------------------------------
