//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * LN^fAj[Vf[^wb_
 * @author Junpee
 */

#ifndef CHARACTER_MODEL_ANIMATION_DATA_H_
#define CHARACTER_MODEL_ANIMATION_DATA_H_

#include <Animation/System/AnimationData.h>
#include <Animation/VectorInterpolator/VectorInterpolator.h>
#include <Animation/RotationInterpolator/RotationInterpolator.h>

namespace Lamp{

//------------------------------------------------------------------------------
/**
 * LN^fAj[Vf[^
 */
class CharacterModelAnimationData : public AnimationData{
friend class AnimationManager;
protected:
	//--------------------------------------------------------------------------
	/**
	 * LN^fV[PX
	 */
	class CharacterModelSequence : public Sequence{
	public:
		/**
		 * RXgN^
		 */
		CharacterModelSequence() : Sequence(),
			boneCount_(0), scale_(NULL), rotation_(NULL), translation_(NULL){}

		/**
		 * fXgN^
		 */
		virtual ~CharacterModelSequence(){ clear(); }

		/**
		 * Rs[
		 * @param copy Rs[
		 */
		virtual void operator =(const CharacterModelSequence& copy){
			Sequence::operator=(copy);
			int boneCount = copy.boneCount_;
			setBoneCount(boneCount);
			for(int i = 0; i < boneCount; i++){
				if(copy.scale_[i] != NULL){
					scale_[i] = copy.scale_[i]->duplicate();
				}
				if(copy.rotation_[i] != NULL){
					rotation_[i] = copy.rotation_[i]->duplicate();
				}
				if(copy.translation_[i] != NULL){
					translation_[i] = copy.translation_[i]->duplicate();
				}
			}
		}

		/**
		 * ̌vZ
		 */
		virtual void calcLength(){
			// [h̃{glbNɂȂ\L
			// calcLength()𖾎IɌĂԂ悤ɂ邩Azɂf[^ݒ肩
			length_ = 0.f;
			for(int i = 0; i < boneCount_; i++){
				if((scale_[i] != NULL) &&
					(scale_[i]->getLength() > length_)){
					length_ = scale_[i]->getLength();
				}
			}
			for(int i = 0; i < boneCount_; i++){
				if((rotation_[i] != NULL) &&
					(rotation_[i]->getLength() > length_)){
					length_ = rotation_[i]->getLength();
				}
			}
			for(int i = 0; i < boneCount_; i++){
				if((translation_[i] != NULL) &&
					(translation_[i]->getLength() > length_)){
					length_ = translation_[i]->getLength();
				}
			}
		}

		/**
		 * {[̐ݒ
		 * @param boneCount {[
		 */
		virtual void setBoneCount(int boneCount){
			clear();
			boneCount_ = boneCount;
			if(boneCount_ != 0){
				int size = sizeof(VectorInterpolator*) * boneCount_;
				scale_ = new VectorInterpolator*[boneCount_];
				::memset(scale_, 0, size);
				rotation_ = new RotationInterpolator*[boneCount_];
				::memset(rotation_, 0, size);
				translation_ = new VectorInterpolator*[boneCount_];
				::memset(translation_, 0, size);
			}
		}

		/**
		 * NA
		 */
		virtual void clear(){
			for(int i = boneCount_ - 1; i >= 0; i--){
				SafeDelete(translation_[i]);
				SafeDelete(rotation_[i]);
				SafeDelete(scale_[i]);
			}
			SafeArrayDelete(translation_);
			SafeArrayDelete(rotation_);
			SafeArrayDelete(scale_);
			boneCount_ = 0;
		}

		//----------------------------------------------------------------------
		/// {[
		int boneCount_;
		/// XP[
		VectorInterpolator** scale_;
		/// ]
		RotationInterpolator** rotation_;
		/// ړ
		VectorInterpolator** translation_;

	};

public:
	//--------------------------------------------------------------------------
	// Rs[
	//--------------------------------------------------------------------------
	/**
	 * Rs[
	 * @return Rs[ꂽAj[Vf[^
	 */
	virtual AnimationData* copy() const{
		return copyCharacterModelAnimationData();
	}

	/**
	 * LN^fAj[Vf[^̃Rs[
	 */
	virtual CharacterModelAnimationData*
		copyCharacterModelAnimationData() const;

	//--------------------------------------------------------------------------
	// V[PX
	//--------------------------------------------------------------------------
	/**
	 * V[PX̐ݒ
	 * @param sequenceCount V[PX
	 */
	virtual void setSequenceCount(int sequenceCount){
		SafeArrayDelete(sequences_);
		sequenceCount_ = sequenceCount;
		if(sequenceCount_ == 0){ return; }
		sequences_ = new CharacterModelSequence[sequenceCount_];
		if(boneCount_ == 0){ return; }
		for(int i = 0; i < sequenceCount; i++){
			sequences_[i].setBoneCount(boneCount_);
		}
	}

	/**
	 * V[PX̎擾
	 * @return V[PX
	 */
	virtual int getSequenceCount() const{ return sequenceCount_; }

	//--------------------------------------------------------------------------
	// {[
	//--------------------------------------------------------------------------
	/**
	 * {[̎擾
	 * @return {[
	 */
	virtual int getBoneCount() const{ return boneCount_; }

	/**
	 * {[̐ݒ
	 * @param boneCount {[
	 */
	virtual void setBoneCount(int boneCount){
		if(boneCount_ == boneCount){ return; }
		boneCount_ = boneCount;
		for(int i = 0; i < sequenceCount_; i++){
			sequences_[i].setBoneCount(boneCount_);
		}
	}

	//--------------------------------------------------------------------------
	// XP[
	//--------------------------------------------------------------------------
	/**
	 * XP[̐ݒ
	 * @param sequence V[PX
	 * @param index {[̃CfbNX
	 * @param scale ݒ肷XP[
	 */
	virtual void setScale(
		int sequence, int index, VectorInterpolator* scale){
		Assert(sequence >= 0);
		Assert(sequence < sequenceCount_);
		Assert(index >= 0);
		Assert(index < boneCount_);
		CharacterModelSequence& data = sequences_[sequence];
		SafeDelete(data.scale_[index]);
		data.scale_[index] = scale;
		data.calcLength();
	}

	/**
	 * XP[̎擾
	 * @param sequence V[PX
	 * @param index {[̃CfbNX
	 * @return XP[
	 */
	virtual VectorInterpolator* getScale(int sequence, int index) const{
		Assert(sequence >= 0);
		Assert(sequence < sequenceCount_);
		Assert(index >= 0);
		Assert(index < boneCount_);
		return sequences_[sequence].scale_[index];
	}

	//--------------------------------------------------------------------------
	// ]
	//--------------------------------------------------------------------------
	/**
	 * ]̐ݒ
	 * @param sequence V[PX
	 * @param index {[̃CfbNX
	 * @param rotation ݒ肷]
	 */
	virtual void setRotation(
		int sequence, int index, RotationInterpolator* rotation){
		Assert(sequence >= 0);
		Assert(sequence < sequenceCount_);
		Assert(index >= 0);
		Assert(index < boneCount_);
		CharacterModelSequence& data = sequences_[sequence];
		SafeDelete(data.rotation_[index]);
		data.rotation_[index] = rotation;
		data.calcLength();
	}

	/**
	 * ]̎擾
	 * @param sequence V[PX
	 * @param index {[̃CfbNX
	 * @return ]
	 */
	virtual RotationInterpolator* getRotation(int sequence, int index) const{
		Assert(sequence >= 0);
		Assert(sequence < sequenceCount_);
		Assert(index >= 0);
		Assert(index < boneCount_);
		return sequences_[sequence].rotation_[index];
	}

	//--------------------------------------------------------------------------
	// ړ
	//--------------------------------------------------------------------------
	/**
	 * ړ̐ݒ
	 * @param sequence V[PX
	 * @param index {[̃CfbNX
	 * @param translation ݒ肷ړ
	 */
	virtual void setTranslation(
		int sequence, int index, VectorInterpolator* translation){
		Assert(sequence >= 0);
		Assert(sequence < sequenceCount_);
		Assert(index >= 0);
		Assert(index < boneCount_);
		CharacterModelSequence& data = sequences_[sequence];
		SafeDelete(data.translation_[index]);
		data.translation_[index] = translation;
		data.calcLength();
	}

	/**
	 * ړ̎擾
	 * @param sequence V[PX
	 * @param index {[̃CfbNX
	 * @return ړ
	 */
	virtual VectorInterpolator* getTranslation(
		int sequence, int index) const{
		Assert(sequence >= 0);
		Assert(sequence < sequenceCount_);
		Assert(index >= 0);
		Assert(index < boneCount_);
		return sequences_[sequence].translation_[index];
	}

	//--------------------------------------------------------------------------
	// RTTI
	//--------------------------------------------------------------------------
	/**
	 * LN^fAj[Vf[^ǂ
	 * @return LN^fAj[Vf[^Ȃtrue
	 */
	virtual bool isCharacterModelAnimationData() const{ return true; }

	//--------------------------------------------------------------------------
protected:
	/**
	 * RXgN^
	 * @param name O
	 * @param manager Aj[V}l[W
	 */
	CharacterModelAnimationData(
		const String& name, AnimationManager* manager) :
		AnimationData(name, manager),
		sequenceCount_(0), sequences_(NULL), boneCount_(0){}

	/**
	 * fXgN^
	 */
	virtual ~CharacterModelAnimationData(){
		SafeArrayDelete(sequences_);
	}

	//--------------------------------------------------------------------------
	/**
	 * V[PX̎擾
	 * @param sequence V[PX
	 * @return V[PX
	 */
	virtual Sequence* getSequence(int sequence){
		Assert(sequence >= 0);
		Assert(sequence < sequenceCount_);
		return &sequences_[sequence];
	}

	/**
	 * V[PX̎擾
	 * @param sequence V[PX
	 * @return V[PX
	 */
	virtual const Sequence* getSequence(int sequence) const{
		Assert(sequence >= 0);
		Assert(sequence < sequenceCount_);
		return &sequences_[sequence];
	}

	//--------------------------------------------------------------------------
private:
	// V[PX
	int sequenceCount_;
	// V[PX
	CharacterModelSequence* sequences_;
	// {[
	int boneCount_;

};

//------------------------------------------------------------------------------
} // End of namespace Lamp
#endif // End of CHARACTER_MODEL_ANIMATION_DATA_H_
//------------------------------------------------------------------------------
