<?php


/**
 * ソースをブロック要素としてパースする。
 * 
 * @param	string	$source	ソース
 * @param	string	$pagename	$pagenameとして解釈する。ページを指定できない場合は空文字列を渡す。
 * @return	T_Body
 */
function parse_block($source, $pagename)
{
	if(!is_array($source)){
		$source = explode("\n", $source);
	}
	return T_Body::parse($source, new Context(resolvepath($pagename)));
}


/**
 * ソースをインライン要素としてパースする。
 * 
 * @param	string	$source	ソース
 * @param	string	$pagename	$pagenameとして解釈する。ページを指定できない場合は空文字列を渡す。
 * @return	T_Body
 */
function parse_inline($source, $pagename)
{
	return T_Inline::parse($source, new Context(resolvepath($pagename)));
}



/**
 * 文脈としての情報を保持する構造体。
 */
class Context
{
	var $pagename;	//起点となるページ名。
	
	
	function __construct($pagename)
	{
		$this->pagename = $pagename;
	}
}



/**
 * 内部表現の要素を表すクラス。
 */
abstract class T_Element
{
	private $elements = array();	//内包する要素
	
	protected $prev = null;	//前の要素への参照
	protected $next = null;	//次の要素への参照
	
	function getelements(){ return $this->elements; }
	function getelem(){ return $this->elements[0]; }
	function getprev(){ return $this->prev; }
	function getnext(){ return $this->next; }
	
	
	/**
	 * 内包する要素を保存する。
	 * 
	 * @param	T_Element	$elem
	 */
	protected function addelement($elem)
	{
		$this->elements[] = $elem;
		$last = count($this->elements) - 1;
		if($last != 0){
			$this->elements[$last-1]->next = $this->elements[$last];
			$this->elements[$last]->prev = $this->elements[$last-1];
		}
	}
	
	
	/**
	 * パースを行う。
	 * 
	 * @param	array(string)	$source	パースできた部分は削除される。
	 * @param	Context	$context
	 * @return	Element	パースエラーの場合はnullを返す。
	 */
	public static abstract function parse(&$source, $context);
	
	
	/**
	 * Visitorに要素を実行させる。
	 * 
	 * @param	Visitor	$v
	 */
	public function accept($v)
	{
		$call = 'visit' . get_class($this);
		return $v->$call($this);
	}
	
	
	/**
	 * 外部からのコンストラクタ呼び出しの禁止。
	 */
	protected function __construct()
	{
		//do nothing.
	}
}



class T_Body extends T_Element
{
	public static function parse(&$source, $context)
	{
		$ins = new T_Body;
		while($source != array()){
			$ins->addelement(T_Block::parse($source, $context));
		}
		return $ins;
	}
}



abstract class T_Block extends T_Body
{
	static $classlist = array('T_Empty', 'T_Heading', 'T_Horizon', 'T_Pre', 'T_BlockQuote', 'T_List', 'T_DL', 'T_BlockPlugin', 'T_BlockTag', 'T_Comment');
	
	
	public static function parse(&$source, $context)
	{
		foreach(T_Block::$classlist as $class){
			$ret = eval("return $class::parse(\$source, \$context);");
			if($ret != null){
				return $ret;
			}
		}
		return T_P::parse($source, $context);
	}
	
	
	/**
	 * 現在の行がどのクラスに解釈されるかを調べる。
	 * 
	 * @return	string	該当のクラス名。
	 */
	public static function check(&$source, $context)
	{
		foreach(T_Block::$classlist as $class){
			$ret = eval("return $class::isvalid(\$source, \$context);");
			if($ret == true){
				return $class;
			}
		}
		return 'T_P';
	}
	
	
	/**
	 * 現在の行を解釈できるかを返す。
	 * 
	 * @return	bool
	 */
	public static abstract function isvalid(&$source, $context);
}



class T_Empty extends T_Block
{
	public static function parse(&$source, $context)
	{
		if(mb_ereg('^\s*$', $source[0])){
			array_shift($source);
			return new T_Empty;
		}
		return null;
	}
	
	
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^\s*$', $source[0]);
	}
}



class T_Heading extends T_Block
{
	protected $level;
	protected $source;
	
	
	function getlevel(){ return $this->level; }
	function getsource(){ return $this->source; }
	
	
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^[*＊]{1,4}\s*.+?\s*$', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		if(mb_ereg('^([*＊]{1,4})\s*(.+?)\s*$', $source[0], $m)){
			array_shift($source);
			return new T_Heading(mb_strlen($m[1]), $m[2], $context);
		}
		return null;
	}
	
	
	protected function __construct($level, $subject, $context)
	{
		$this->level = $level;
		$this->source = $subject;
		$this->addelement(T_Inline::parse($subject, $context));
	}
}



class T_Horizon extends T_Block
{
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^----\s*$', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		if(mb_ereg('^----\s*$', $source[0])){
			array_shift($source);
			return new T_Horizon;
		}
		return null;
	}
}



class T_Pre extends T_Block
{
	protected $text;	//array
	
	function gettext(){ return $this->text; }
	
	
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^[ \t].*?\s*$', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		$text = array();
		while($source != array() && mb_ereg('^[ \t](.*?)\s*$', $source[0], $m)){
			array_shift($source);
			$text[] = $m[1];
		}
		return $text != array() ? new T_Pre($text) : null;
	}
	
	
	protected function __construct($text)
	{
		$this->text = $text;
	}
}



class T_BlockQuote extends T_Block
{
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^>.*?\s*$', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		$text = array();
		while($source != array() && mb_ereg('^>(.*?)\s*$', $source[0], $m)){
			array_shift($source);
			$text[] = $m[1];
		}
		return $text != array() ? new T_BlockQuote($text, $context) : null;
	}
	
	
	protected function __construct($text, $context)
	{
		$this->addelement(T_Body::parse($text, $context));
	}
}



abstract class T_List extends T_Block
{
	static $list = array('T_UL', 'T_OL');
	
	
	public static function isvalid(&$source, $context)
	{
		foreach(T_List::$list as $class){
			$ret = eval("return $class::isvalid(\$source, \$context);");
			if($ret == true){
				return true;
			}
		}
		return false;
	}
	
	
	public static function parse(&$source, $context)
	{
		foreach(T_List::$list as $class){
			$ret = eval("return $class::parse(\$source, \$context);");
			if($ret != null){
				return $ret;
			}
		}
		return null;
	}
	
	
	protected function __construct($text, $context)
	{
		while($text != array())
		{
			$e = T_List::parse($text, $context);
			if($e == null){
				$e = T_Inline::parse(array_shift($text), $context);
			}
			$this->addelement($e);
		}
	}
}



class T_UL extends T_List
{
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^(?:-|・).*?\s*$', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		$text = array();
		while($source != array() && mb_ereg('^(?:-|・)(.*?)\s*$', $source[0], $m)){
			array_shift($source);
			$text[] = $m[1];
		}
		return $text != array() ? new T_UL($text, $context) : null;
	}
}



class T_OL extends T_List
{
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^\+.*?\s*$', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		$text = array();
		while($source != array() && mb_ereg('^\+(.*?)\s*$', $source[0], $m)){
			array_shift($source);
			$text[] = $m[1];
		}
		return $text != array() ? new T_OL($text, $context) : null;
	}
}



class T_DL extends T_Block
{
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^[:：].+?[:：]', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		$text = array();
		if(mb_ereg('^[:：](.+?)[:：]', $source[0])){
			while($source != array() && !T_Empty::isvalid($source, $context)){
				$text[] = array_shift($source);
			}
		}
		return $text != array() ? new T_DL($text, $context) : null;
	}
	
	
	protected function __construct($text, $context)
	{
		while($text != array()){
			if(mb_ereg('^\:(.+?):\s*(.*)$', $text[0], $m)){
				$this->addelement(new T_DT($m[1], $context));
				if($m[2] != ''){
					$this->addelement(new T_DD($m[2], $context));
				}
			}
			else{
				$this->addelement(new T_DD($text[0], $context));
			}
			array_shift($text);
		}
	}
}



class T_DT extends T_DL
{
	public static function isvalid(&$source, $context)
	{
		die("don't call T_DD::isvalid().");
	}
	
	
	public static function parse(&$source, $context)
	{
		die("don't call T_DT::parse().");
	}
	
	
	protected function __construct($term, $context)
	{
		$this->addelement(T_Inline::parse($term, $context));
	}
}



class T_DD extends T_DL
{
	public static function isvalid(&$source, $context)
	{
		die("don't call T_DD::isvalid().");
	}
	
	
	public static function parse(&$source, $context)
	{
		die("don't call T_DD::parse().");
	}
	
	
	protected function __construct($text, $context)
	{
		$this->addelement(T_Inline::parse($text, $context));
	}
}



class T_BlockPlugin extends T_Block
{
	protected $pluginname;
	protected $param1;
	protected $param2;
	protected $pagename;
	
	function getpluginname(){ return $this->pluginname; }
	function getparam1(){ return $this->param1; }
	function getparam2(){ return $this->param2; }
	function getpagename(){ return $this->pagename; }
	
	
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^#\w+', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		if(mb_ereg('^#\w+', $source[0])){
			$str = join("\n", $source);
			mb_ereg('^#(\w+)(?:\s*\((.*?)\))?(?:\s*{([^}]*)})?', $str, $m);
			$source = explode("\n", substr($str, strlen($m[0])));
			return new T_BlockPlugin($m[1], $m[2], $m[3], $context->pagename);
		}
		return null;
	}
	
	
	protected function __construct($pluginname, $param1, $param2, $pagename)
	{
		$this->pluginname = $pluginname;
		$this->param1 = $param1;
		$this->param2 = $param2;
		$this->pagename = $pagename;
	}
}



class T_BlockTag extends T_BlockPlugin
{
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^<\w+.*?>', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		if(mb_ereg('^<\w+.*?>', $source[0])){
			if(mb_ereg('^<(\w+)(?:\s+(.*?))?\s+/>', $source[0], $m)){
				return new T_BlockTag($m[1], $m[2], '');
			}
			
			$str = join("\n", $source);
			if(mb_ereg('^<(\w+)(?:\s+(.*?))?>(.*?)</\1>', $str, $m)){
				$source = explode("\n", substr($str, strlen($m[0])));
				return new T_BlockTag($m[1], $m[2], $m[3], $context->pagename);
			}
		}
		return null;
	}
}



class T_Comment extends T_Block
{
	public static function isvalid(&$source, $context)
	{
		return (bool)mb_ereg('^//', $source[0]);
	}
	
	
	public static function parse(&$source, $context)
	{
		if(mb_ereg('^//', $source[0])){
			array_shift($source);
			return new T_Comment;
		}
		return null;
	}
}



class T_P extends T_Block
{
	public static function isvalid(&$source, $context)
	{
		return true;
	}
	
	
	public static function parse(&$source, $context)
	{
		$str = mb_ereg_replace('^　', '', rtrim(array_shift($source)));
		return new T_P($str, $context);
	}
	
	
	public function __construct($text, $context)
	{
		$this->addelement(T_Inline::parse($text, $context));
	}
}



class T_Inline extends T_Element
{
	/**
	 * インライン要素のパーサは配列ではなく文字列を受け取る。
	 */
	public static function parse(&$str, $context)
	{
		$ins = new T_Inline;
		while($str != ''){
			$ins->addelement(T_InlineElement::parse($str, $context));
		}
		return $ins;
	}
}



abstract class T_InlineElement extends T_Inline
{
	public static function parse(&$str, $context)
	{
		static $classlist = array('T_URL', 'T_Mail', 'T_AutoLink', 'T_BlacketName', 'T_InlinePlugin', 'T_Footnote', 'T_Strong');
		
		$len = strlen($str);
		foreach($classlist as $class){
			$ret = eval("return $class::getdistance(\$str, \$context);");
			if($ret == 0){
				return eval("return $class::parse(\$str, \$context);");
			}
			if($ret < $len){
				$len = $ret;
			}
		}
		$text = substr($str, 0, $len);
		$str = substr($str, $len);
		return T_Text::parse($text, $context);
	}
	
	
	/**
	 * 行頭から正規表現にマッチするところまでの文字数を返す。
	 * @return	int	マッチしない場合は行末までの文字数を返す。
	 */
	public static abstract function getdistance(&$str, $context);
}



class T_URL extends T_InlineElement
{
	protected $url;
	
	function geturl(){ return $this->url; }
	
	
	public static function getdistance(&$str, $context)
	{
		if(mb_ereg('^(.*?)(?:s?https?|ftp|file):\/\/[-a-zA-Z0-9_:@&?=+,.!\/~*%$]+', $str, $m)){
			return strlen($m[1]);
		}
		else{
			return strlen($str);
		}
	}
	
	
	public static function parse(&$str, $context)
	{
		if(mb_ereg('^(?:s?https?|ftp|file):\/\/[-a-zA-Z0-9_:@&?=+,.!\/~*%$]+', $str, $m)){
			$str = substr($str, strlen($m[0]));
			return new T_URL($m[0]);
		}
		return null;
	}
	
	
	protected function __construct($url)
	{
		$this->url = $url;
	}
}



class T_Mail extends T_InlineElement
{
	protected $address;
	
	function getaddress(){ return $this->address; }
	
	
	public static function getdistance(&$str, $context)
	{
		if(mb_ereg('^(.*?)\w[-\w.]*\@[-\w]+(?:\.[-\w]+)+', $str, $m)){
			return strlen($m[1]);
		}
		else{
			return strlen($str);
		}
	}
	
	
	public static function parse(&$str, $context)
	{
		if(mb_ereg('^\w[-\w.]*\@[-\w]+(?:\.[-\w]+)+', $str, $m)){
			$str = substr($str, strlen($m[0]));
			return new T_Mail($m[0]);
		}
		return null;
	}
	
	
	protected function __construct($address)
	{
		$this->address = $address;
	}
}



class T_AutoLink extends T_InlineElement
{
	protected $pagename;
	protected $alias;
	
	function getpagename(){ return $this->pagename; }
	function getalias(){ return $this->alias; }
	
	
	public static function getdistance(&$str, $context)
	{
		$list[] = $context->pagename;
		$list[] = getdirname($context->pagename);
		$list[] = '';
		
		$ret = strlen($str);
		foreach($list as $dir){
			$exp = AutoLink::getinstance()->getexpression($dir);
			if($exp != '' && mb_ereg("^(.*?)$exp", $str, $m)){
				$ret = min($ret, strlen($m[1]));
			}
		}
		return $ret;
	}
	
	
	public static function parse(&$str, $context)
	{
		$list[] = $context->pagename;
		$list[] = getdirname($context->pagename);
		$list[] = '';
		
		foreach($list as $dir){
			$exp = AutoLink::getinstance()->getexpression($dir);
			if($exp != '' && mb_ereg("^$exp", $str, $m)){
				$str = substr($str, strlen($m[0]));
				$dir = $dir == '' ? '' : "{$dir}/";
				return new T_AutoLink($dir . $m[0], $m[0]);
			}
		}
		return null;
	}
	
	
	protected function __construct($pagename, $alias)
	{
		$this->pagename = $pagename;
		$this->alias = $alias;
	}
}



class T_BlacketName extends T_InlineElement
{
	protected $pagename;
	protected $alias;
	protected $currentpage;
	
	function getpagename(){ return $this->pagename; }
	function getalias(){ return $this->alias; }
	function getcurrentpage(){ return $this->currentpage; }
	
	
	public static function getdistance(&$str, $context)
	{
		if(mb_ereg('^(.*?)\[\[.+?\]\]', $str, $m)){
			return strlen($m[1]);
		}
		else{
			return strlen($str);
		}
	}
	
	
	public static function parse(&$str, $context)
	{
		if(mb_ereg('^\[\[(?:(.+?)>)?(.+?)\]\]', $str, $m)){
			$str = substr($str, strlen($m[0]));
			return new T_BlacketName($m[2], $m[1], $context->pagename);
		}
		return null;
	}
	
	
	protected function __construct($pagename, $alias, $currentpage)
	{
		$this->pagename = $pagename;
		$this->alias = $alias;
		$this->currentpage = $currentpage;
	}
}



class T_InlinePlugin extends T_InlineElement
{
	protected $pluginname;
	protected $param1;
	protected $param2;
	protected $pagename;
	
	function getpluginname(){ return $this->pluginname; }
	function getparam1(){ return $this->param1; }
	function getparam2(){ return $this->param2; }
	function getpagename(){ return $this->pagename; }
	
	
	public static function getdistance(&$str, $context)
	{
		if(mb_ereg('^(.*?)&\w+\s*\(.*?\)(?:\s*{[^}]*})?', $str, $m)){
			return strlen($m[1]);
		}
		else{
			return strlen($str);
		}
	}
	
	
	public static function parse(&$str, $context)
	{
		if(mb_ereg('^&(\w+)\s*\((.*?)\)(?:\s*{([^}]*)})?', $str, $m)){
			$str = substr($str, strlen($m[0]));
			return new T_InlinePlugin($m[1], $m[2], $m[3], $context);
		}
		return null;
	}
	
	
	protected function __construct($pluginname, $param1, $param2, $context)
	{
		$this->pluginname = $pluginname;
		$this->param1 = $param1;
		$this->param2 = $param2;
		$this->pagename = $context->pagename;
	}
}



class T_Footnote extends T_InlineElement
{
	public static function getdistance(&$str, $context)
	{
		if(mb_ereg('^(.*?)\(\(.+?\)\)', $str, $m)){
			return strlen($m[1]);
		}
		else{
			return strlen($str);
		}
	}
	
	
	public static function parse(&$str, $context)
	{
		if(mb_ereg('^\(\((.+?)\)\)', $str, $m)){
			$str = substr($str, strlen($m[0]));
			return new T_Footnote($m[1], $context);
		}
		return null;
	}
	
	
	protected function __construct($text, $context)
	{
		$this->addelement(T_Inline::parse($text, $context));
	}
}



class T_Strong extends T_InlineElement
{
	protected $str;
	protected $level;
	
	function getstr(){ return $this->str; }
	function getlevel(){ return $this->level; }
	
	
	public static function getdistance(&$str, $context)
	{
		if(mb_ereg('^(.*?)(\*\*?)(?!\s).+?(?<!\s)\2', $str, $m)){
			return strlen($m[1]);
		}
		else{
			return strlen($str);
		}
	}
	
	
	public static function parse(&$str, $context)
	{
		if(mb_ereg('^(\*\*?)((?!\s).+?(?<!\s))\1', $str, $m)){
			$str = substr($str, strlen($m[0]));
			return new T_Strong($m[2], strlen($m[1]));
		}
		return null;
	}
	
	
	protected function __construct($str, $level)
	{
		$this->str = $str;
		$this->level = $level;
	}
}



class T_Text extends T_InlineElement
{
	protected $text;
	
	function gettext(){ return $this->text; }
	
	
	public static function getdistance(&$str, $context)
	{
		return 0;
	}
	
	
	public static function parse(&$str, $context)
	{
		$ins = new T_Text;
		$ins->text = $str;
		$str = '';
		return $ins;
	}
}


?>