<?php


/**
 * パースにより生成されたオブジェクトを元にHTML形式の文字列を生成するクラス。
 */
class HTMLConverter
{
	protected $anchormaker;
	
	
	function __construct()
	{
		$this->anchormaker = new SectionAnchorMaker;
	}
	
	
	function visitT_Body($e)
	{
		$elements = $e->getelements();
		$ret = array();
		foreach($elements as $elem){
			$ret[] = $elem->accept($this);
		}
		return join("\n", $ret);
	}
	
	
	function visitT_Empty($e)
	{
		return '';
	}
	
	
	function visitT_Heading($e)
	{
		static $opentag = array('', '<h3 class="subtitle">', '<h4>', '<h5>', '<h6>');
		static $closetag = array('', '</h3>', '</h4>', '</h5>', '</h6>');
		
		$id = $this->anchormaker->makeid($e->getlevel(), $e->getsource());
		$str = "<a id=\"$id\">" . $e->getelem()->accept($this) . '</a>';
		return $opentag[$e->getlevel()] . $str . $closetag[$e->getlevel()];
	}
	
	
	function visitT_Horizon($e)
	{
		return '<hr />';
	}
	
	
	function visitT_Pre($e)
	{
		return '<pre>' . htmlspecialchars(join("\n", $e->gettext())) . '</pre>';
	}
	
	
	function visitT_BlockQuote($e)
	{
		return '<blockquote>' . $e->getelem()->accept($this) . '</blockquote>';
	}
	
	
	function visitT_UL($e)
	{
		$elements = $e->getelements();
		$ret = array();
		
		for($i = 0; $i < count($elements); $i++){
			$str = '';
			if($i > 0 && !is_subclass_of($elements[$i], 'T_List')){
				$str = "</li>\n<li>";
			}
			$ret[] = $str . $elements[$i]->accept($this);
		}
		return "<ul>\n<li>" . join("\n", $ret) . "</li>\n</ul>";
	}
	
	
	function visitT_OL($e)
	{
		$elements = $e->getelements();
		$ret = array();
		
		for($i = 0; $i < count($elements); $i++){
			$str = '';
			if($i > 0 && !is_subclass_of($elements[$i], 'T_List')){
				$str = "</li>\n<li>";
			}
			$ret[] = $str . $elements[$i]->accept($this);
		}
		return "<ol>\n<li>" . join("\n", $ret) . "</li>\n</ol>";
	}
	
	
	function visitT_DL($e)
	{
		$elements = $e->getelements();
		$ret = array();
		foreach($elements as $elem){
			$ret[] = $elem->accept($this);
		}
		return "<dl>\n" . join("\n", $ret) . "\n</dl>";
	}
	
	
	function visitT_DT($e)
	{
		return '<dt>' . $e->getelem()->accept($this) . '</dt>';
	}
	
	
	function visitT_DD($e)
	{
		return '<dd>' . $e->getelem()->accept($this) . '</dd>';
	}
	
	
	function visitT_BlockPlugin($e)
	{
		try{
			$plugin = Plugin::getPlugin($e->getpluginname());
			return $plugin->do_block($e->getpagename(), $e->getparam1(), $e->getparam2());
		}
		catch(PluginException $exc){
			return '<p class="warning">' . htmlspecialchars($exc->message()) . '</p>';
		}
	}
	
	
	function visitT_BlockTag($e)
	{
		try{
			$plugin = Plugin::getPlugin($e->getpluginname());
			return $plugin->do_blocktag($e->getpagename(), $e->getparam1(), $e->getparam2());
		}
		catch(PluginException $exc){
			return '<span class="warning">' . htmlspecialchars($exc->message()) . '</span>';
		}
	}
	
	
	function visitT_Comment($e)
	{
		return '';
	}
	
	
	function visitT_P($e)
	{
		$ret  = get_class($e->getprev()) != 'T_P' ? '<div class="subsection">' : '';
		$ret .= '<p>' . $e->getelem()->accept($this) . '</p>';
		$ret .= get_class($e->getnext()) != 'T_P' ? '</div>' : '';
		return $ret;
	}
	
	
	function visitT_Inline($e)
	{
		$elements = $e->getelements();
		$ret = array();
		foreach($elements as $elem){
			$ret[] = $elem->accept($this);
		}
		return join('', $ret);
	}
	
	
	function visitT_URL($e)
	{
		$url = htmlspecialchars($e->geturl());
		return "<a href=\"$url\">$url</a>";
	}
	
	
	function visitT_Mail($e)
	{
		$address = htmlspecialchars($e->getaddress());
		return "<a href=\"mailto:$address\">$address</a>";
	}
	
	
	function visitT_AutoLink($e)
	{
		return makelink($e->getpagename(), $e->getalias());
	}
	
	
	function visitT_BlacketName($e)
	{
		$pagename = $e->getpagename();
		$alias = $e->getalias() != '' ? $e->getalias() : $e->getpagename();
		if(mb_ereg('^(?:s?https?|ftp|file):\/\/[-a-zA-Z0-9_:@&?=+,.!\/~*%$]+$', $pagename)){
			$alias = htmlspecialchars($alias);
			return "<a href=\"$pagename\">$alias</a>";
		}
		else if(mb_ereg('^\w[-\w.]*\@[-\w]+(?:\.[-\w]+)+$', $pagename)){
			$alias = htmlspecialchars($alias);
			return "<a href=\"mailto:$pagename\">$alias</a>";
		}
		else if(mb_ereg('^(.+?):(.+)$', $pagename, $m) && !Wiki::getinstance()->ispage($pagename)){
			return makeinterwikilink($m[1], $m[2], $alias);
		}
		else{
			$fullname = resolvepath($pagename, $e->getcurrentpage());
			return makelink($fullname, $alias);
		}
	}
	
	
	function visitT_InlinePlugin($e)
	{
		try{
			$plugin = Plugin::getPlugin($e->getpluginname());
			return $plugin->do_inline($e->getpagename(), $e->getparam1(), $e->getparam2());
		}
		catch(PluginException $exc){
			return '<span class="warning">' . htmlspecialchars($exc->message()) . '</span>';
		}
	}
	
	
	function visitT_Footnote($e)
	{
		$footnote = Footnote::getinstance();
		return $footnote->addnote($e->getelem()->accept($this));
	}
	
	
	function visitT_Strong($e)
	{
		$str = htmlspecialchars($e->getstr());
		$level = $e->getlevel();
		return $level == 1 ? "<em>$str</em>" : "<strong>$str</strong>";
	}
	
	
	function visitT_Text($e)
	{
		$exp = '&amp;(#\d{2,4}|#x[0-9a-fA-F]{2,3}|' . CHARACTER_ENTITY_REFERENCES . ');';
		
		$str = htmlspecialchars($e->gettext());
		return mb_ereg_replace($exp, '&\1;', $str);
	}
}



/**
 * 脚注を管理する。シングルトン。
 */
class Footnote
{
	protected $note = array();
	
	
	public static function getinstance()
	{
		static $ins = null;
		if($ins == null){
			$ins = new Footnote;
		}
		return $ins;
	}
	
	
	protected function __construct()
	{
		//do nothing.
	}
	
	
	/**
	 * 脚注を追加する。
	 * 
	 * @param	string	$html	追加するhtml形式文字列。
	 * @return	string	アンカー
	 */
	function addnote($html)
	{
		$this->note[] = $html;
		$num = count($this->note);
		$str  = '<span class="hidden">(</span>';
		$str .= "<a class=\"footnote\" href=\"#footnote_${num}\" id=\"footnote_${num}_r\">*$num</a>";
		$str .= '<span class="hidden">)</span>';
		return $str;
	}
	
	
	/**
	 * 脚注を取得する。
	 * 
	 * @return	string	html形式の文字列。
	 */
	function getnote()
	{
		if($this->note == array()){
			return '';
		}
		$smarty = new MySmarty(TPL_DIR);
		$smarty->assign('note', $this->note);
		return $smarty->fetch('footnote.tpl.htm');
	}
}



/**
 * 見出しアンカーのidを作るクラス。
 */
class SectionAnchorMaker
{
	protected $section = array('');
	
	
	/**
	 * アンカーのidを取得する。
	 * 
	 * @param	int	$level	見出しレベル
	 * @param	string	$str	見出し（wikiのソース）
	 */
	function makeid($level, $str)
	{
		for($i = 1; $i < $level; $i++){
			if(!isset($this->section[$i])){
				$this->section[$i] = '';
			}
		}
		
		$this->section[$i] = $str;
		return md5(join("\0", array_slice($this->section, 1, $level)));
	}
}

?>