unit FullTextSearch;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
	Dialogs, ExtCtrls, StdCtrls, ComCtrls, ToolWin, Katjusha, FavTypes,
  ImgList, ActnList, Menus;

type
	TFullTextSearchThread = class;
	PFullTextSearchInfo = ^TFullTextSearchInfo;
	TFullTextSearchInfo = record
		Title: string;
		Text: string;
		Line: string;
		Number: Integer;
	end;
	TFullTextSearchForm = class(TForm)
    ListView: TListView;
    StatusBar: TStatusBar;
    Memo: TMemo;
    Splitter1: TSplitter;
    ImageList: TImageList;
    ToolBar: TToolBar;
    CloseButton: TToolButton;
    ResearchButton: TToolButton;
    CancelButton: TToolButton;
    ActionList: TActionList;
    ToolBarClose: TAction;
    ToolBarResearch: TAction;
    ToolBarCancel: TAction;
    PopupMenu: TPopupMenu;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ListViewClick(Sender: TObject);
    procedure ToolBarCloseExecute(Sender: TObject);
    procedure ToolBarCancelUpdate(Sender: TObject);
    procedure ToolBarResearchExecute(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ToolBarCancelExecute(Sender: TObject);
  private
		{ Private 錾 }
		FProgressBar: TProgressBar;
		FThread: TFullTextSearchThread;
		FIndexList: T2chIndexList;
		FConfig: TFullTextSearchConfig;
		FInSearching: Boolean;	{  }
		procedure TerminateThread;
		procedure ClearAll;
		procedure CatchResult(AInfo: PFullTextSearchInfo);
		procedure NotifyProgress(Index, Max: Integer; const Text: string);
		procedure SetStatusText(const Value: string);
		procedure SearchComplete(AMatchCount: Integer);
		procedure RefreshColumn;
	public
		{ Public 錾 }
    constructor Create(AOwner: TComponent); override;
		destructor Destroy; override;
		procedure StartSearch(List: T2chIndexList = nil; Config: TFullTextSearchConfig = nil);
		property StatusText: string write SetStatusText;
	end;
{ SXbh }
	TFullTextSearchThread = class(TThread)
	private
		FOwner: TFullTextSearchForm;
		FIndexList: T2chIndexList;
		FConfig: TFullTextSearchConfig;
		FProgressIndex: Integer;
		FProgressMax: Integer;
		FProgressText: string;
		FMatchCount: Integer;
		FInfo: TFullTextSearchInfo;
		procedure SendResult;
		procedure Notify;
		procedure Complete;
		procedure Hit(Text: string; Index, Number: Integer; const Line: string; ThreadIdx: T2chThreadIndex);
	protected
		procedure Execute; override;
	public
		constructor Create(AOwner: TFullTextSearchForm; List: T2chIndexList; Config: TFullTextSearchConfig);
    destructor Destroy; override;
	end;

var
	FullTextSearchForm: TFullTextSearchForm;

implementation

{$R *.dfm}

uses Math, StrUtils, regex, AWKFunc, FavUtils, bmRExp, FullTextSearchConfig,
  FavMain;

procedure TFullTextSearchForm.CatchResult(AInfo: PFullTextSearchInfo);
var
	PInfo: PFullTextSearchInfo;
	Refresh: Boolean;
begin
	PInfo := nil;
	Refresh := False;
	try
		New(PInfo);
		PInfo^ := AInfo^;

		if ListView.VisibleRowCount = ListView.Items.Count then
			Refresh := True;

		with ListView.Items.Add do
		begin
			Caption := PInfo.Title;
			SubItems.Add(IntToStr(PInfo.Number));
			SubItems.Add(PInfo.Text);
			Data := PInfo;
		end;

		if Refresh then RefreshColumn;
	except
		if PInfo <> nil then Dispose(PInfo);
		raise;
	end;
end;

procedure TFullTextSearchForm.ClearAll;
var
	I: Integer;
begin
	for I := 0 to ListView.Items.Count - 1 do
		Dispose(PFullTextSearchInfo(ListView.Items[I].Data));
	ListView.Clear;
	Memo.Lines.Text := '';
end;

constructor TFullTextSearchForm.Create(AOwner: TComponent);
begin
	inherited;
	FThread := nil;
	FInSearching := False;
	FIndexList := T2chIndexList.Create;
	FConfig := TFullTextSearchConfig.Create;
end;

destructor TFullTextSearchForm.Destroy;
begin
	inherited;
	FreeAndNil(FConfig);
	FreeAndNil(FIndexList);
end;

procedure TFullTextSearchForm.FormCreate(Sender: TObject);
begin
	{ vOXo[̍쐬 }
	FProgressBar := TProgressBar.Create(StatusBar);
	with FProgressBar do
	begin
		Parent := StatusBar;
		Top := 2;
		Height := StatusBar.Height - 2;
		Left := 0;
		Width := 100;
	end;

	(Application.MainForm as TKatfavMainForm).AddSubForm(Self);
end;

procedure TFullTextSearchForm.StartSearch(List: T2chIndexList;
  Config: TFullTextSearchConfig);
begin
	if List <> nil then FIndexList.Assign(List);
	if Config <> nil then FConfig.Assign(Config);

	ClearAll;
	Caption := Format('''%s''ŃXbhSc', [FConfig.SearchText]);

	TerminateThread;

	Show;

	FInSearching := True;
	FThread := TFullTextSearchThread.Create(Self, FIndexList, FConfig);
end;

procedure TFullTextSearchForm.TerminateThread;
begin
	if Assigned(FThread) then
	begin
		FThread.Terminate;
		if FThread.Suspended then FThread.Resume;
		FThread.WaitFor;
		FreeAndNil(FThread);
	end;
	FInSearching := False;
end;

procedure TFullTextSearchForm.FormDestroy(Sender: TObject);
begin
	(Application.MainForm as TKatfavMainForm).RemoveSubForm(Self);

	TerminateThread;
	ClearAll;
end;

procedure TFullTextSearchForm.ListViewClick(Sender: TObject);
var
	Text: string;
	PInfo: PFullTextSearchInfo;
	DatArray: array [0 .. 4] of string;
	List: TStringList;
	I, L: Integer;
begin
	if ListView.Selected = nil then Exit;
	List := TStringList.Create;
	try
		PInfo := PFullTextSearchInfo(ListView.Selected.Data);
		StringSplit(PInfo.Line, '<>', DatArray);
		Text := Format('%d O:%s [%s] %s',
			[PInfo.Number, HTML2String(DatArray[0]), DatArray[1], AWKGSub('<[^>]+>', '', DatArray[2])]);
		Text := Text + #13#10 + HTML2String(DatArray[3]);

		List.Text := Text;
		
		for I := 0 to List.Count - 1 do
		begin
			Text := List[I];
			L := Length(Text);
			{ 擪Ɩ̋󔒂̍폜 }
			if (L > 0) and (Text[1] = ' ') then
			begin
				Text := Copy(Text, 2, MaxInt);
				Dec(L);
			end;
			if (L > 0) and (Text[L] = ' ') then
				Text := Copy(Text, 1, L - 1);
			List[I] := Text;
		end;

		Memo.Lines.Assign(List);
	finally
		List.Free;
	end;
end;

procedure TFullTextSearchForm.NotifyProgress(Index, Max: Integer; const Text: string);
begin
	FProgressBar.Min := 1;
	FProgressBar.Max := Max;
	FProgressBar.Position := Index;
	StatusText := Text;
end;

procedure TFullTextSearchForm.SetStatusText(const Value: string);
begin
	StatusBar.Panels[1].Text := Value;
end;

procedure TFullTextSearchForm.SearchComplete(AMatchCount: Integer);
begin
	Caption := Format('''%s''̃XbhS', [FConfig.SearchText]);
	StatusText :=
		Format('XbhS܂(%dX/%dqbg)', [FIndexList.Count, AMatchCount]);
	FInSearching := False;
end;

procedure TFullTextSearchForm.RefreshColumn;
begin
	ListView.Height := ListView.Height - 1;
end;

procedure TFullTextSearchForm.ToolBarCloseExecute(Sender: TObject);
begin
	Close;
end;

procedure TFullTextSearchForm.ToolBarResearchExecute(Sender: TObject);
begin
	FConfig.TargetThread := fttFixed;
	if TFullTextSearchConfigDlg.Execute(Self, FConfig) then
		StartSearch;
end;

procedure TFullTextSearchForm.ToolBarCancelUpdate(Sender: TObject);
begin
	(Sender as TAction).Enabled := FInSearching;
end;

procedure TFullTextSearchForm.ToolBarCancelExecute(Sender: TObject);
begin
	TerminateThread;

	StatusText := Format('𒆎~܂(%dqbg)B',
		[ListView.Items.Count]);
	FProgressBar.Position := 1;
end;

procedure TFullTextSearchForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
	{ EBhEƃ }
	Action := caFree;
end;

{ TFullTextSearchThread }

procedure TFullTextSearchThread.Complete;
begin
	FOwner.SearchComplete(FMatchCount);
end;

constructor TFullTextSearchThread.Create(AOwner: TFullTextSearchForm; List: T2chIndexList; Config: TFullTextSearchConfig);
begin
	FConfig := Config;
	FIndexList := List;
	FOwner := AOwner;

	inherited Create(False);
end;

destructor TFullTextSearchThread.Destroy;
begin
	inherited;
end;

procedure TFullTextSearchThread.Execute;
var
	br, Tag: regex_t;
	DatList: T2chThreadDat;
	I, J: Integer;
	RIndex, RLength: Integer;
	SearchRegExp: TbmRExp;
	SearchText, Text: string;
	ThreadIdx: T2chThreadIndex;
begin
	DatList := nil;
	SearchRegExp := nil;

	try
		DatList := T2chThreadDat.Create;
		SearchRegExp := TbmRExp.Create;
		
		AWKCompilePattern(br, '<[Bb][Rr]>');
		AWKCompilePattern(Tag, '<[^>]+>');

		if FConfig.UseRegularExpression then
		begin
			SearchRegExp.AsciiIgnoreCase := not FConfig.CaseInsensitive;
			SearchRegExp.RegExp := FConfig.SearchText;
		end else
		begin
			SearchText := FConfig.SearchText;
			if not FConfig.CaseInsensitive then
				SearchText := AnsiLowerCase(SearchText);
		end;
			
		for I := 0 to FIndexList.Count - 1 do
		begin
			if Terminated then Exit;

			ThreadIdx := FIndexList[I];
			DatList.LoadFromFile(TKatjusha.GetKatjushaBaseDir + ChangeFileExt('log\' + ThreadIdx.LogPath, '.dat'));

			FProgressIndex := I + 1;
			FProgressMax := FIndexList.Count;
			FProgressText := Format('[%d/%d]''%s''c',
				[I + 1, FIndexList.Count, ThreadIdx.ThreadName]);
			Synchronize(Notify);

			for J := 0 to DatList.Count - 1 do
			begin
				if Terminated then Exit;

				case FConfig.Target of
					ftsAll: Text := DatList[J];
					ftsName: Text := DatList.Names[J];
					ftsMail: Text := DatList.Mails[J];
					ftsDate, ftsID: Text := DatList.Dates[J];
					ftsMessage: Text := DatList.Messages[J];
				end;

				if FConfig.IgnoreSpace then
				begin
					Text := StringReplace(Text, '@', '', [rfReplaceAll]);
					Text := StringReplace(Text, ' ', '', [rfReplaceAll]);
					Text := AWKGSub(br, '', Text);
				end;

				if not FConfig.IgnoreSpace then
					Text := AWKGSub(br, #13#10, Text);

				if FConfig.IgnoreTag then
					Text := AWKGSub(Tag, '', Text);

				if Terminated then Exit;

				if FConfig.UseRegularExpression then
				begin
					if bmREMatch(SearchRegExp, Text, 1, RIndex, RLength) <> 0 then
						Hit(Text, RIndex, J + 1, DatList[J], ThreadIdx);
				end else
				begin
					if not FConfig.CaseInsensitive
						then RIndex := AnsiPos(SearchText, AnsiLowerCase(Text))
						else RIndex := AnsiPos(SearchText, Text);
					if RIndex > 0 then
						Hit(Text, RIndex, J + 1, DatList[J], ThreadIdx);
				end;
			end;
		end;

		Synchronize(Complete);
	finally
		DatList.Free;
		SearchRegExp.Free;
	end;
end;

procedure TFullTextSearchThread.Hit(Text: string; Index, Number: Integer;
	const Line: string; ThreadIdx: T2chThreadIndex);
var
	SubStr: string;
	I: Integer;
{ RPos }
	function RPos(const SubStr: string; Str: string): Integer;
	var
		I: Integer;
	begin
		Result := 0;
		repeat
			I := Pos(SubStr, Str);
			if I > 0 then
			begin
				if Result > 0 then Result := Result + Length(SubStr) - 1;
				Result := Result + I;
				Str := Copy(Str, I + Length(SubStr), MaxInt);
			end;
		until I <= 0;
	end;
{ MinNotZero }
	function MinNotZero(const A: array of Integer): Integer;
	var
		I: Integer;
	begin
		Result := MaxInt;
		for I := 0 to High(A) do
			if A[I] <> 0 then Result := Min(Result, A[I]);
		if Result = MaxInt then Result := 0;
	end;
begin
  { qbg }
	Inc(FMatchCount);
	
	{ ʕ\p̕ }
	SubStr := Copy(Text, 1, Index);
	I := Max(RPos('<>', SubStr), RPos(#13#10, SubStr));
	if 0 < I then
		SubStr := Copy(SubStr, I + 2, MaxInt);
	Text := Copy(Text, Index + 1, MaxInt);
	I := MinNotZero([Pos('<>', Text), Pos(#13#10, Text)]);
	if 0 < I then
		Text := Copy(Text, 1, I - 1);
	Text := SubStr + Text;
	
	Text := HTML2String(Text);
	Text := AWKGSub('^[ ]+', '', Text);
	Text := AWKGSub('[ ]+$', '', Text);

	FInfo.Title := StringReplace(ThreadIdx.ThreadName, 'M', ',', [rfReplaceAll]);
	FInfo.Text := Text;
	FInfo.Line := Line;
	FInfo.Number := Number;

	Synchronize(SendResult);
end;

procedure TFullTextSearchThread.Notify;
begin
	FOwner.NotifyProgress(FProgressIndex, FProgressMax, FProgressText);
end;

procedure TFullTextSearchThread.SendResult;
begin
	FOwner.CatchResult(@FInfo);
end;

end.

