/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
 
package org.apache.jetspeed.util.servlet;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import javax.servlet.*;
import javax.servlet.http.*;
import java.util.Locale;

/**
 *
 * EcsServletResponse is an HttpServletResponse implementation that 
 * wraps an existing HttpServletResponse and redirects its output to
 * an alternate OutputStream.
 *
 * @author <a href="raphael@apache.org">Raphal Luta</a>
 */
public class EcsServletResponse implements HttpServletResponse
{
    /** The wrapped ServletResponse */
    private HttpServletResponse rsp;

    /** The new OutputStream to use */
    private EcsServletOutputStream sout; 

    /** The PrintWriter to use if defined */
    private PrintWriter writer; 
    
    /** state flag: has getOutputStream() already been called ? */
    private boolean outUsed = false;

    /** state flag: has getWriter() already been called ? */
    private boolean writerUsed = false;
   
    /** Build a Response
     *
     * @param rsp the base HttpServletResponse to wrap
     * @param out the OutputStream to be used for writing data
     */
    public EcsServletResponse(HttpServletResponse rsp, OutputStream out) {
        this.rsp = rsp;
        this.sout = new EcsServletOutputStream(out);
    }        

    /**
     * @see javax.servlet.ServletResponse#getCharacterEncoding
     */
    public String getCharacterEncoding()
    {
        return rsp.getCharacterEncoding();
    }
    
    /**
     * @see javax.servlet.ServletResponse#getOutputStream
     */
    public ServletOutputStream getOutputStream() throws IOException 
    {
        if (writerUsed)
        {
            throw new IllegalStateException("Writer already used");
        }

        this.outUsed = true;

        return this.sout;
    }
    
    /**
     * @see javax.servlet.ServletResponse#getWriter
     */
    public PrintWriter getWriter() throws IOException
    {
        if (outUsed) 
        {
            throw new IllegalStateException("OutputStream already in use");
        }
        
        if (writer==null) 
        {
            this.writer = new PrintWriter(this.sout);
        }
    
        this.writerUsed = true;
        
        return writer;
    }
    
    /**
     * @see javax.servlet.ServletResponse#setContentLength
     */
    public void setContentLength(int len)
    {
        // silently fail
    }
    
    /**
     * @see javax.servlet.ServletResponse#setContentType
     */
    public void setContentType(String type) 
    {
        // silently fail
    }
    
    /**
     * @see javax.servlet.ServletResponse#setBufferSize
     */
    public void setBufferSize(int size)
    {
        // silently fail
    }
    
    /**
     * @see javax.servlet.ServletResponse#getBufferSize
     */
    public int getBufferSize() 
    {
        return rsp.getBufferSize();
    }
    
    /**
     * @see javax.servlet.ServletResponse#flushBuffer
     */
    public void flushBuffer() throws IOException
    {
        this.sout.flush();
    }
    
    /**
     * @see javax.servlet.ServletResponse#isCommitted
     */
    public boolean isCommitted()
    {
        // we don't implement reset() so always claim 
        // the data is committed
        return true;
    }
    
    /**
     * @see javax.servlet.ServletResponse#reset
     */
    public void reset()
    {
        // silently fail
    }

    /**
     * @see javax.servlet.ServletResponse#setLocale
     */
    public void setLocale(Locale loc)
    {
        // we can safely ignore this
    }
    
    /**
     * @see javax.servlet.ServletResponse#getLocale
     */
    public Locale getLocale()
    {
        return rsp.getLocale();
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#addCookie
     */
    public void addCookie(Cookie cookie)
    {
        rsp.addCookie(cookie);
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#containsHeader
     */
    public boolean containsHeader(String name)
    {
        return rsp.containsHeader(name);
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#encodeURL
     */
    public String encodeURL(String url)
    {
        return rsp.encodeURL(url);
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#encodeRedirectURL
     */
    public String encodeRedirectURL(String url)
    {
        return rsp.encodeRedirectURL(url);
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#encodeUrl
     */
    public String encodeUrl(String url)
    {
        return rsp.encodeUrl(url);
    }
    
    /**
     * @see javax.servlet.http.HttpServletResponse#encodeRedirectUrl
     */
    public String encodeRedirectUrl(String url)
    {
        return rsp.encodeRedirectUrl(url);
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#sendError
     */
    public void sendError(int sc, String msg) throws IOException
    {
        // silently fail
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#sendError
     */
    public void sendError(int sc) throws IOException
    {
        // silently fail
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#sendRedirect
     */
    public void sendRedirect(String location) throws IOException
    {
        // silently fail
    }
    
    /**
     * @see javax.servlet.http.HttpServletResponse#setDateHeader
     */
    public void setDateHeader(String name, long date)
    {
        // silently fail
    }
    
    /**
     * @see javax.servlet.http.HttpServletResponse#addDateHeader
     */
    public void addDateHeader(String name, long date)
    {
        // silently fail
    }
    
    /**
     * @see javax.servlet.http.HttpServletResponse#setHeader
     */
    public void setHeader(String name, String value)
    {
        // silently fail
    }
    
    /**
     * @see javax.servlet.http.HttpServletResponse#addHeader
     */
    public void addHeader(String name, String value)
    {
        // silently fail
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#setIntHeader
     */
    public void setIntHeader(String name, int value)
    {
        // silently fail
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#addIntHeader
     */
    public void addIntHeader(String name, int value)
    {
        // silently fail
    }

    /**
     * @see javax.servlet.http.HttpServletResponse#setStatus
     */
    public void setStatus(int sc)
    {
        // silently fail
    }
  
    /**
     * @see javax.servlet.http.HttpServletResponse#setStatus
     */
    public void setStatus(int sc, String sm)
    {
        // silently fail
    }

    public void resetBuffer()
    {
        // silently fail
        //added by Ben Woodward
    }

}
