/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.util;


import java.util.Hashtable;
import java.util.Locale;
import java.util.Enumeration;
import java.util.Map;
import org.apache.jetspeed.services.resources.JetspeedResources;
import java.io.Serializable;

/**
A class for storing MetaData about an object.

@author <a href="mailto:raphael@apache.org">Raphal Luta</a>
@version $Id: MetaData.java,v 1.10 2002/10/21 22:31:15 taylor Exp $
*/
public class MetaData implements Serializable
{

    public static final String DEFAULT_TITLE = 
        JetspeedResources.getString(JetspeedResources.DEFAULT_TITLE_KEY);

    public static final String DEFAULT_DESCRIPTION = 
        JetspeedResources.getString(JetspeedResources.DEFAULT_DESCRIPTION_KEY);

    public static final String DEFAULT_IMAGE = 
        JetspeedResources.getString(JetspeedResources.DEFAULT_IMAGE_KEY);

    /**
    Hashtable to store all the properties
    */
    private Hashtable data = new Hashtable();

    /**
    Sets a title in the default locale
    */
    public void setTitle(String title)
    {
        setTitle(title, null);
    }

    /**
    Sets a title for the given locale
    */
    public void setTitle(String title, Locale locale)
    {
        setProperty("title", locale, title);
    }

    /**
    Returns the title for the default locale
    */
    public String getTitle()
    {
        return getTitle(null);
    }

    /**
    Returns the title for the given locale, if the title isn't defined
    in this locale or the locale, return the default title
    */
    public String getTitle(Locale locale)
    {
        String title = (String) getProperty("title", locale);

        if ((title == null) && (locale != null))
        {
            title = (String) getProperty("title", null);
        }

        if (title == null)
        {
            title = DEFAULT_TITLE;
        }

        return title;
    }

    /**
    Sets a descriptive image in the default locale
    */
    public void setImage(String image)
    {
        setImage(image, null);
    }

    /**
    Sets a descriptive image for the given locale
    */
    public void setImage(String image, Locale locale)
    {
        setProperty("image", locale, image);
    }

    /**
    Returns the descriptive image for the default locale
    */
    public String getImage()
    {
        return getImage(null);
    }

    /**
    Returns the descriptive image for the given locale, if the image isn't defined
    in this locale or the locale, return the default image
    */
    public String getImage(Locale locale)
    {
        String image = (String) getProperty("image", locale);

        if ((image == null) && (locale != null))
        {
            image = (String) getProperty("image", null);
        }

        if (image == null)
        {
            image = DEFAULT_IMAGE;
        }

        return image;
    }

    /**
    Sets a description in the default locale
    */
    public void setDescription(String description)
    {
        setDescription(description, null);
    }

    /**
    Sets a description for the given locale
    */
    public void setDescription(String description, Locale locale)
    {
        setProperty("description", locale, description);
    }

    /**
    Returns the description for the default locale
    */
    public String getDescription()
    {
        return getDescription(null);
    }

    /**
    Returns the title for the given locale, if the title isn't defined
    in this locale or the locale, return the default title
    */
    public String getDescription(Locale locale)
    {
        String desc = (String) getProperty("description", locale);

        if ((desc == null) && (locale != null))
        {
            desc = (String) getProperty ("description", null);
        }

        if (desc == null)
        {
            desc = DEFAULT_DESCRIPTION;
        }

        return desc;
    }

    /**
     Stores a property for later retrieval, uses the notation
     propertyName.localeName for locale distinction

    <p>For example, title.fr_FR will store the French title, while
    title.sp will keep the Spanich one. title will keep the value
    which can be used when defaulting because en entry is not present
    for the queried locale</p>
    */
    private void setProperty(String name, Locale locale, Object value)
    {
        if (name == null)
        {
            return;
        }

        if (locale != null)
        {
            name = name + "." + locale.toString();
        }

        if (value == null)
        {
            data.remove(name);
        }
        else
        {
            data.put(name, value);
        }
    }

    /**
    Retrieves a property by name for a given locale
    */
    private Object getProperty(String name, Locale locale)
    {
        if (name == null)
        {
            return null;
        }

        String extname = null;
        if (locale != null)
        {
            extname = name + "." + locale.toString();
        }

        Object obj = null;

        if (extname != null)
        {
            obj = data.get(extname);
        }
        if (obj == null)
        {
            obj = data.get(name);
        }

        return obj;
    }

    /**
    Retrieves a property by name for a given locale
    */
    private Map getProperties() 
    {
        return data;
    }

    /**
    Merges the properties defined in the param MetaData into the current one.
    If values are defined in both object for the same key,the one passed as 
    parameter updates the previous one
    */
    public void merge(MetaData meta)
    {
        Map map = meta.getProperties();
        Hashtable params = (Hashtable) map;
        Enumeration en = params.keys();

        while (en.hasMoreElements())
        {
            Object key = en.nextElement();
            String value = (String) params.get(key);
            
            if (value != null)
            {
                this.data.put(key, value);
            }
        }
    }
}
