/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security;

import java.util.Iterator;
import java.util.HashMap;

// Junit imports
import junit.framework.Test;
import junit.framework.TestSuite;

import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

// Jetspeed imports
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.jetspeed.om.security.Permission;
import org.apache.jetspeed.om.security.JetspeedPermissionFactory;

import org.apache.jetspeed.services.security.JetspeedSecurityCache;

/**
 * Unit test for PermissionManagement interface
 * 
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: TestPermissionManagement.java,v 1.6 2003/08/08 18:28:03 morciuch Exp $
 */

public class TestPermissionManagement extends JetspeedTestCase {    

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestPermissionManagement( String name ) {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[]) 
    {
        junit.awtui.TestRunner.main( new String[] { TestPermissionManagement.class.getName() } );
    }
 
    public void setup() 
    {
        //System.out.println("Setup: Testing Turbine Permission Management");         
    }

    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite() 
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestPermissionManagement.class );
    }

    /**
     * Tests getPermissions method
     * @throws Exception
     */

    public void testGetPermissions() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        HashMap map = new HashMap();
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            Iterator permissions = service.getPermissions();
            while (permissions.hasNext())
            {
                permission = (Permission)permissions.next();
                map.put(permission.getName(), permission);
                //System.out.println("permission = " + permission.getName());
                //System.out.println("id = " + permission.getId());
            }
            assertTrue(map.get("view") != null);
            assertTrue(map.get("customize") != null);
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getPermissions Test OK ");

    }

    /**
     * Tests getPermissions method
     * @throws Exception
     */

    public void testGetPermissionsForUser() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        HashMap map = new HashMap();
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            Iterator permissions = service.getPermissions("user");
            while (permissions.hasNext())
            {
                permission = (Permission)permissions.next();
                map.put(permission.getName(), permission);
            }
            assertTrue(map.get("view") != null);
            assertTrue(map.get("customize") != null);
            assertTrue(map.get("close") == null);

            map.clear();
            permissions = service.getPermissions("admin");            
            while (permissions.hasNext())
            {
                permission = (Permission)permissions.next();
                map.put(permission.getName(), permission);
            }
            assertTrue(map.get("view") != null);
            assertTrue(map.get("customize") != null);
            assertTrue(map.get("close") != null);

        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getPermissions Test OK ");

    }

    /**
     * Tests addPermission method 
     * @throws Exception
     */

    public void testAddPermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            permission = JetspeedPermissionFactory.getInstance();
            permission.setName("bogus");
            service.addPermission(permission);
            System.out.println("new permission id = " + permission.getId());
            assertTrue(permission.getId() != null);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            permission = JetspeedPermissionFactory.getInstance();
            permission.setName("bogus");
            service.addPermission(permission);
            fail("Should've thrown a dup key exception on permission");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);           
        }

        System.out.println("Completed addPermission Test OK ");

    }

    /**
     * Tests getRemovePermission method 
     * @throws Exception
     */

    public void testRemovePermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            service.removePermission("bogus");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.removePermission("catchmeifyoucan");
            fail("Should've thrown a not found exception on permission");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);
        }

        System.out.println("Completed addPermission Test OK ");

    }

    /**
     * Tests getPermission method
     * @throws Exception
     */

    public void testGetPermission() throws Exception 
    {
        PermissionManagement service = getService();
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            Permission permission = service.getPermission("view");
            System.out.println("*** permission id = " + permission.getId());
            assertTrue(permission.getName().equals("view"));
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getPermission Test OK ");

    }

    /**
     * Tests savePermission method 
     * @throws Exception
     */

    public void testSavePermission() throws Exception 
    {
        PermissionManagement service = getService();
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            Permission permission = service.getPermission("customize");
            service.savePermission(permission);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed savePermission Test OK ");

    }

    /**
     * Tests grantPermission method 
     * @throws Exception
     */
    public void testGrantPermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            service.grantPermission("user", "close");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.grantPermission("badrole", "close");
            fail("Should've thrown a bad role exception on grant");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);           
        }
        try
        {
            service.grantPermission("user", "badpermission");
            fail("Should've thrown a bad permission exception on grant");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);           
        }

        System.out.println("Completed grantPermission Test OK ");

    }

    /**
     * Tests revokePermission method 
     * @throws Exception
     */
    public void testRevokePermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            service.revokePermission("user", "close");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.revokePermission("badrole", "close");
            fail("Should've thrown a bad user exception on revoke");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof PermissionException);           
        }

        System.out.println("Completed revokePermission Test OK ");

    }

    /**
     * Tests hasPermission method 
     * @throws Exception
     */
    public void testHasPermission() throws Exception 
    {
        PermissionManagement service = getService();
        Permission permission = null;
        JetspeedSecurityCache.loadRolePermissions();

        try
        {
            boolean has = service.hasPermission("user", "view");
            assertTrue(true == has);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            boolean has = service.hasPermission("user", "close");
            assertTrue(false == has);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed hasPermission Test OK ");

    }

  /*
    Configuration object to run Turbine outside a servlet container
    ( uses turbine.properties )
    */
    private static TurbineConfig config = null;
    
    /**
    Sets up TurbineConfig using the system property:
    <pre>turbine.properties</pre>
    */
    static
    {
        try
        {
            config = new TurbineConfig( "../webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }

    private static PermissionManagement getService()
    {
        return (PermissionManagement)TurbineServices
                .getInstance()
                .getService(PermissionManagement.SERVICE_NAME);
    }

}






