package org.apache.jetspeed.services.logging;
/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *    "Apache Turbine" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// Log4J classes
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

/**
 * The implementation of loggers for Jetspeed.
 *
 * This class acts as a wrapper so that the underlying logging implementation
 * is hidden fromthe rest of Jetspeed
 * The current implementation uses Log4J.
 *
 * @author <a href="mailto:harald@ommang.com">Harald Ommang</a>
 */
public class JetspeedLogger
{
 
    /**
     * The Log4J logger that is wrapped
     */        
    private Logger logger;

    /**
     * Constructor. Initialises this class with a given logger.
     * If the logger is null, one is given named from this class.
     *
     * @param logger The logger to wrap
     */    
    public JetspeedLogger(Logger logger)
    {
        if(logger != null)
        {
            this.logger = logger;
        } else
        {
            this.logger = LogManager.getLogger(JetspeedLogger.class.getName());
        }
    }

    /**
     * Checks if the current logger is enabled for debug logging.
     *
     * @return true if debug is enabled
     */
    public boolean isDebugEnabled()
    {
        return logger.isDebugEnabled();
    }

    /**
     * Checks if the current logger is enabled for error logging.
     *
     * @return true if error is enabled
     */    
    public boolean isErrorEnabled()
    {
        return logger.isEnabledFor(Level.ERROR);
    }

    /**
     * Checks if the current logger is enabled for fatal logging.
     *
     * @return true if fatal is enabled
     */        
    public boolean isFatalEnabled()
    {
        return logger.isEnabledFor(Level.FATAL);
    }

    /**
     * Checks if the current logger is enabled for info logging.
     *
     * @return true if info is enabled
     */        
    public boolean isInfoEnabled()
    {
        return logger.isInfoEnabled();
    }

    /**
     * Checks if the current logger is enabled for trace logging.
     * Wtih log4J, this is the same as debug.
     *
     * @return true if trace is enabled
     */        
    public boolean isTraceEnabled()
    {
        return logger.isDebugEnabled();
    }

    /**
     * Checks if the current logger is enabled for warning logging.
     *
     * @return true if warning is enabled
     */
    public boolean isWarnEnabled()
    {
        return logger.isEnabledFor(Level.WARN);
    }

    /**
     * Logs the given object if debug is enabled
     *
     * @param obj Object to log
     */    
    public void debug(Object obj)
    {
        logger.debug(obj);
    }

    /**
     * Logs the given object and throwable if debug is enabled
     *
     * @param obj Object to log
     * @param throwable The underlying implementation may log stack trace for this
     */        
    public void debug(Object obj, Throwable throwable)
    {
        logger.debug(obj, throwable);
    }

    /**
     * Logs the given object if error is enabled
     *
     * @param obj Object to log
     */        
    public void error(Object obj)
    {
        logger.error(obj);
    }

    /**
     * Logs the given object and throwable if error is enabled
     *
     * @param obj Object to log
     * @param throwable The underlying implementation may log stack trace for this
     */            
    public void error(Object obj, Throwable throwable)
    {
        logger.error(obj, throwable);
    }

    /**
     * Logs the given object if fatal is enabled
     *
     * @param obj Object to log
     */        
    public void fatal(Object obj)
    {
        logger.fatal(obj);
    }

    /**
     * Logs the given object and throwable if fatal is enabled
     *
     * @param obj Object to log
     * @param throwable The underlying implementation may log stack trace for this
     */            
    public void fatal(Object obj, Throwable throwable)
    {
        logger.fatal(obj, throwable);
    }

    /**
     * Logs the given object if info is enabled
     *
     * @param obj Object to log
     */        
    public void info(Object obj)
    {
        logger.info(obj);
    }

    /**
     * Logs the given object and throwable if info is enabled
     *
     * @param obj Object to log
     * @param throwable The underlying implementation may log stack trace for this
     */            
    public void info(Object obj, Throwable throwable)
    {
        logger.info(obj, throwable);
    }

    /**
     * Logs the given object if trace is enabled
     * With Log4J, this is the same as debug
     *
     * @param obj Object to log
     */        
    public void trace(Object obj)
    {
        logger.debug(obj);
    }

    /**
     * Logs the given object and throwable if trace is enabled
     * With Log4J, this is the same as debug
     *
     * @param obj Object to log
     * @param throwable The underlying implementation may log stack trace for this
     */            
    public void trace(Object obj, Throwable throwable)
    {
        logger.debug(obj, throwable);
    }

    /**
     * Logs the given object if warning is enabled
     *
     * @param obj Object to log
     */        
    public void warn(Object obj)
    {
        logger.warn(obj);
    }

    /**
     * Logs the given object and throwable if warning is enabled
     *
     * @param obj Object to log
     * @param throwable The underlying implementation may log stack trace for this
     */            
    public void warn(Object obj, Throwable throwable)
    {
        logger.warn(obj, throwable);
    }
} // class JetspeedLogger

