/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.om.registry.base;

import org.apache.jetspeed.om.registry.ClientEntry;
import org.apache.jetspeed.om.registry.ClientRegistry;
import org.apache.jetspeed.om.registry.RegistryEntry;
import org.apache.jetspeed.om.registry.InvalidEntryException;
import org.apache.jetspeed.om.registry.RegistryException;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.Registry;
import java.util.Enumeration;

// regexp stuff
import org.apache.regexp.RE;

/**
 * Simple implementation of the ClientRegistry interface.
 * <p>Extends BaseRegistry implementation to override object creation
 * method and ensure Registry object is synchronized with its
 * persistence backend by delegating actual addition/deletion of objects
 * to the registry service.</p>
 * <p>To avoid loops, a RegistryService implementation using this class
 * nees to call the addLocalEntry/removeLocalEntry methods to modify
 * the in memory state of this Registry</p>
 *
 * @author <a href="shesmer@raleigh.ibm.com">Stephan Hesmer</a>
 * @author <a href="mailto:raphael@apache.org">Raphal Luta</a>
 * @version $Id: BaseClientRegistry.java,v 1.4 2003/07/23 19:50:15 morciuch Exp $
 */
public class BaseClientRegistry extends BaseOrderedRegistry implements ClientRegistry
{
    
    /**
     * Static initialization of the logger for this class
     */    
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(BaseClientRegistry.class.getName());    
    
    /**
    @see Registry#setEntry
    */
    public void setEntry( RegistryEntry entry ) throws InvalidEntryException
    {
        // Delegate to the RegistryService to ensure correct handling of
        // persistence if using file fragments

        try
        {
            Registry.addEntry(Registry.CLIENT, entry);
        }
        catch (RegistryException e)
        {
            logger.error("Exception", e);
        }
    }

    /**
    @see Registry#addEntry
    */
    public void addEntry( RegistryEntry entry ) throws InvalidEntryException
    {
        // Delegate to the RegistryService to ensure correct handling of
        // persistence if using file fragments

        try
        {
            Registry.addEntry(Registry.CLIENT, entry);
        }
        catch (RegistryException e)
        {
            logger.error("Exception", e);
        }
    }

    /**
    @see Registry#removeEntry
    */
    public void removeEntry( String name )
    {
        // Delegate to the RegistryService to ensure correct handling of
        // persistence if using file fragments

        Registry.removeEntry(Registry.CLIENT, name);
    }

    /**
    @see Registry#removeEntry
    */
    public void removeEntry( RegistryEntry entry )
    {
        // Delegate to the RegistryService to ensure correct handling of
        // persistence if using file fragments

        if (entry != null)
        {
            Registry.removeEntry(Registry.CLIENT, entry.getName());
        }
    }

    /**
     * Returns the client which matches the given useragent string.
     *
     * @param useragent     the useragent to match
     * @return the found client or null if the user-agent does not match any
     *  defined client
     */
    public ClientEntry findEntry(String useragent)
    {
        ClientEntry clientEntry = null;
        Enumeration clients = getEntries();

        if ( logger.isDebugEnabled() )
        {
            logger.debug( "ClientRegistry: Looking for client with useragent :" + useragent );
        }

        if (clients != null)
        {
            while (clients.hasMoreElements())
            {
                ClientEntry client = (ClientEntry)clients.nextElement();
                if (client.getUseragentpattern() != null)
                {
                    try
                    {
                        RE r = new RE(client.getUseragentpattern());
                        r.setMatchFlags(RE.MATCH_CASEINDEPENDENT);

                        if (r.match(useragent))
                        {

                            if ( logger.isDebugEnabled() )
                            {
                                logger.debug( "ClientRegistry: " + useragent + " matches " + client.getUseragentpattern() );
                            }

                            return client;
                        }
                        else
                        {
                            if ( logger.isDebugEnabled() )
                            {
                                logger.debug( "ClientRegistry: " + useragent + " does not match " + client.getUseragentpattern() );
                            }
                        }
                    }
                    catch (org.apache.regexp.RESyntaxException e)
                    {
                        String message = "ClientRegistryService: UserAgentPattern not valid : " + client.getUseragentpattern() + " : " + e.getMessage();
                        logger.error( message, e );
                    }
                }
            }
        }

        return clientEntry;
    }

    /**
     * Creates a new RegistryEntry instance compatible with the current
     * Registry instance implementation
     *
     * @return the newly created RegistryEntry
     */
    public RegistryEntry createEntry()
    {
        return new BaseClientEntry();
    }
}
