/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.modules.pages;

// Turbine Modules
import org.apache.turbine.modules.pages.DefaultPage;

// Turbine Utility Classes
import org.apache.turbine.util.RunData;
import org.apache.turbine.services.template.TurbineTemplate;

// Jetspeed Classes
import org.apache.jetspeed.util.MimeType;
import org.apache.jetspeed.om.registry.MediaTypeEntry;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.Registry;
import org.apache.jetspeed.capability.CapabilityMap;
import org.apache.jetspeed.services.TemplateLocator;
import org.apache.jetspeed.services.rundata.JetspeedRunData;
import org.apache.jetspeed.services.resources.JetspeedResources;

/**
 * When building sites using templates, Screens need only be defined
 * for templates which require dynamic (database or object) data.
 *
 * <p>
 *
 * This page can be used on sites where the number of Screens can be
 * much less than the number of templates.  The templates can be
 * grouped in directories with common layouts.  Screen modules are
 * then expected to be placed in packages corresponding with the
 * templates' directories and follow a specific naming scheme.
 *
 * <p>
 *
 * The template parameter is parsed and and a Screen whose package
 * matches the templates path and shares the same name minus any
 * extension and beginning with a capital letter is searched for.  If
 * not found, a Screen in a package matching the template's path with
 * name Default is searched for.  If still not found, a Screen with
 * name Default is looked for in packages corresponding to parent
 * directories in the template's path until a match is found.
 *
 * <p>
 *
 * For example if data.getParameters().getString("template") returns
 * /about_us/directions/driving.wm, the search follows
 * about_us.directions.Driving, about_us.directions.Default,
 * about_us.Default, Default, WebMacroSiteScreen (i.e. the default
 * screen set in TurbineResources).
 *
 * <p>
 *
 * Only one Layout module is used, since it is expected that any
 * dynamic content will be placed in navigations and screens.  The
 * layout template to be used is found in a similar way to the Screen.
 * For example the following paths will be searched in the layouts
 * subdirectory: /about_us/directions/driving.wm,
 * /about_us/directions/default.wm, /about_us/default.wm, /default.wm,
 * where wm is the value of the template.default.extension property.
 *
 * <p>
 *
 * This approach allows a site with largely static content to be
 * updated and added to regularly by those with little Java
 * experience.
 *
 * @author <a href="mailto:john.mcnally@clearink.com">John D. McNally</a>
 * @author <a href="mailto:mbryson@mont.mindspring.com">Dave Bryson</a>
 * @author <a href="mailto:paulsp@apache.org">Paul Spencer</a>
 * @version $Id: JetspeedTemplatePage.java,v 1.23 2003/07/23 19:50:14 morciuch Exp $
 */
public class JetspeedTemplatePage extends DefaultPage
{
    private static int httpLifetime = JetspeedResources.getInt("http.lifetime", -1);

    /**
     * Static initialization of the logger for this class
     */    
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(JetspeedTemplatePage.class.getName());     
    
    /**
     * Works with TemplateService to set up default templates and
     * corresponding class modules.
     *
     * @param data Turbine information.
     * @exception Exception, a generic exception.
     */
    protected void doBuildBeforeAction(RunData data) throws Exception
    {
        switch (httpLifetime)
        {
            case -1:
                break;
            case 0:
                data.getResponse().setHeader("Cache-Control", "no-cache");
                data.getResponse().setHeader("Pragma", "no-cache");
                data.getResponse().setDateHeader("Expires", 0);
                data.getResponse().setDateHeader("Last-Modified", System.currentTimeMillis());
                break;
            default:
                data.getResponse().setHeader("Cache-Control", "max-age=" + httpLifetime);
                data.getResponse().setDateHeader("Expires", System.currentTimeMillis() + (httpLifetime * 1000));
                data.getResponse().setDateHeader("Last-Modified", System.currentTimeMillis());
                break;
        }

        //Set the ContentType of the page
        CapabilityMap cm = ((JetspeedRunData)data).getCapability();
        MimeType mime = cm.getPreferredType();
        String characterSet = JetspeedResources.getString(JetspeedResources.CONTENT_ENCODING_KEY,"utf-8");
        data.setContentType( mime.getContentType());
        if ( mime != null )
        {
            MediaTypeEntry media = (MediaTypeEntry)Registry.getEntry(Registry.MEDIA_TYPE, mime.getCode());
            if ( media != null && media.getCharacterSet() != null)
            {
                characterSet = media.getCharacterSet();
            }
        }
        data.setCharSet( characterSet );

        if (logger.isDebugEnabled())
        {
            logger.debug( "JetspeedTemplatePage: Setting type to: " + cm.getPreferredType().getContentType()
                        + "; charset=" + JetspeedResources.getString(JetspeedResources.CONTENT_ENCODING_KEY,"utf-8")
                     );
        }

    }

    /**
     * Works with TemplateService to set up default templates and
     * corresponding class modules.
     *
     * @param data Turbine information.
     * @exception Exception, a generic exception.
     */
    protected void doBuildAfterAction(RunData data) throws Exception
    {
        // Either template or screen should be guaranteed by the SessionValidator
        // It is occasionally better to specify the screen instead of template
        // in cases where multiple Screens map to one template.  The template
        // is hardcoded into the Screen in this instance.  In this case this
        // action is skipped.

        if (!data.hasScreen())
        {

            // if only a screen but no template is specified, then we need to display
            // a legacy ecs screen --> screenTemplate = ECS
            if ( data.getTemplateInfo().getScreenTemplate() == null )
            {
                String screen = TurbineTemplate.getDefaultScreen();
                data.setScreenTemplate(screen);
            }

            String ext = TurbineTemplate.getDefaultExtension();

            String template = data.getTemplateInfo().getScreenTemplate();

            //save the initial requested template before mangling it
            ((JetspeedRunData)data).setRequestedTemplate(template);

            if (template.lastIndexOf('.')<0)
            {
                template=template+"."+ext;
            }
            if ( logger.isDebugEnabled() )
            {
                logger.debug("JetspeedTemplatePage: requested template = " + template);
            }

            // get real path now - this is a fix to get us thru 1.3a2
            // when the TurbineTemplateService can locate resources by NLS and mediatype,
            // then it can be removed

            String locatedScreen = TemplateLocator.locateScreenTemplate(data, template);
            data.setScreenTemplate( locatedScreen );
            if ( logger.isDebugEnabled() )
            {
                logger.debug("JetspeedTemplatePage: calculated template = " + locatedScreen);
            }

            String layout = TemplateLocator.locateLayoutTemplate(data, template);
            data.setLayoutTemplate(layout);
            if ( logger.isDebugEnabled() )
            {
                logger.debug("JetspeedTemplatePage: layoutTemplate is finally " + layout);
            }

            String screen = TurbineTemplate.getScreenName(template);
            if (screen == null)
            {
                throw new Exception("Screen could not be determined. \n" +
                    "No matches were found by TemplateService and the \n" +
                    "services.TurbineTemplateService.default.screen \n" +
                    "property was not set.");
            }
            data.setScreen(screen);
        }
    }

}
