/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.modules.actions.portlets;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.TreeMap;

import org.apache.jetspeed.om.registry.PortletEntry;
import org.apache.jetspeed.om.registry.base.BaseCategory;
import org.apache.jetspeed.services.Registry;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;

/**
 * An abstract class with helper methods for filtering Portlets.
 *
 * @author <a href="mailto:jford@apache.org">Jeremy Ford</a>
 * @version $Id: PortletFilter.java,v 1.2 2003/11/03 05:35:39 jford Exp $
 */
public abstract class PortletFilter
{
    /**
    * Static initialization of the logger for this class
    */
    private static final JetspeedLogger logger =
        JetspeedLogFactoryService.getLogger(PortletFilter.class.getName());

    /**
     * Method that filters a list of portlets based on a give filter name/value.
     * 
     * @param portlets List of portlets to filter
     * @param field The name of the filter
     * @param value The value of the filter
     * @return List of portlets that met the filter criteria
     */
    public static List filterPortlets(
        List portlets,
        String field,
        String value)
    {
        String[] fields = { field };
        String[] values = { value };

        return filterPortlets(portlets, fields, values);
    }

    /**
     * Method that filters a list of portlets based on certain criteria.
     * 
     * @param portlets The list of portlets to filter
     * @param fields The list of fields
     * @param values The list of values.  This should be in a 1:1 ratio with the fields.
     * @return List of portlets that met the filter criteria
     */
    public static List filterPortlets(
        List portlets,
        String[] fields,
        String[] values)
    {
        List filteredPortlets = new ArrayList();

        Iterator portletIter = portlets.iterator();
        while (portletIter.hasNext())
        {
            PortletEntry entry = (PortletEntry) portletIter.next();
            if (isFilteredIn(entry, fields, values))
            {
                filteredPortlets.add(entry);
            }
        }

        return filteredPortlets;
    }

    /**
     * Method that checks a portlet entry to see if it matches the given
     * filter criteria.
     * 
     * @param entry The entry to filter
     * @param fields The list of fields.
     * @param values The list of values.  This should be in a 1:1 ratio with the fields.
     * @return
     */
    private static boolean isFilteredIn(
        PortletEntry entry,
        String[] fields,
        String[] values)
    {
        boolean result = true;

        if (fields != null && values != null && fields.length == values.length)
        {
            for (int i = 0; i < fields.length && result; i++)
            {
                String field = fields[i];
                String value = values[i];

                if (field == null
                    || value == null
                    || field.length() == 0
                    || value.length() == 0)
                {
                    //skip and add to list
                }
                else if (field.equals("category"))
                {
                    result = result && entry.hasCategory(value);
                }
                else if (field.equals("media_type"))
                {
                    result = result && entry.hasMediaType(value);
                }
                else if (field.equals("parent"))
                {
                    if (entry.getParent() != null)
                    {
                        result = result && entry.getParent().equals(value);
                    }
                    else
                    {
                        result = false;
                    }
                }
                else if (field.equals("type"))
                {
                    if (entry.getType() != null)
                    {
                        result = result && entry.getType().equals(value);
                    }
                    else
                    {
                        result = false;
                    }
                }
                /*
                else if(field.equals("permission"))
                {
                    result = JetspeedSecurity.checkPermission((JetspeedUser) rundata.getUser(), 
                                                                     new PortalResource(entry), 
                                                                     value);
                }
                */
                else
                {
                    logger.warn("Invalid filter " + field + " attempted");
                }
            }
        }

        return result;
    }

    /**
     * Builds a list of all portlet categories
     * 
     * @param List portlets portlets to scan for categories
     * @return List of categories
     */
    public static List buildCategoryList(List portlets)
    {
        TreeMap catMap = new TreeMap();
        Iterator pItr = portlets.iterator();
        while (pItr.hasNext())
        {
            PortletEntry entry = (PortletEntry) pItr.next();

            Iterator cItr = entry.listCategories();
            while (cItr.hasNext())
            {
                BaseCategory cat = (BaseCategory) cItr.next();
                catMap.put(cat.getName(), cat);
            }
        }

        return new ArrayList(catMap.values());
    }

    /**
     * Method to return all portlets in the Portlet Registry
     * 
     * @return List of portlets
     */
    public static List getAllPortlets()
    {
        List regEntries = new ArrayList();

        Iterator iter = Registry.get(Registry.PORTLET).listEntryNames();
        while (iter.hasNext())
        {
            String entryName = (String) iter.next();
            regEntries.add(Registry.getEntry(Registry.PORTLET, entryName));
        }

        return regEntries;
    }

    /**
     * Method that returns a list of parents from the provided list of portlets.
     * 
     * @param portlets List of portlets to search for parents
     * @return List of portlets that are parents
     */
    public static List buildParentList(List portlets)
    {
        HashSet parentSet = new HashSet();

        Iterator portletIter = portlets.iterator();
        while (portletIter.hasNext())
        {
            PortletEntry regEntry = (PortletEntry) portletIter.next();

            String regType = regEntry.getType();
            if (regType.equalsIgnoreCase(PortletEntry.TYPE_ABSTRACT))
            {
                parentSet.add(regEntry);
            }
        }

        return new ArrayList(parentSet);
    }

}
