/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */


package org.apache.jetspeed.util.template;

// Java imports
import java.util.Stack;

// Cactus and Junit imports
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.awtui.TestRunner;

// Jetspeed imports
import org.apache.jetspeed.services.resources.JetspeedResources;
import org.apache.jetspeed.util.template.JetspeedLink;

// Turbine imports
import org.apache.turbine.services.pool.PoolService;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

/**
 * TestTurbineCache
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id: TestJetspeedLinkFactory.java,v 1.3 2003/03/04 00:05:16 sgala Exp $
 */
public class TestJetspeedLinkFactory extends TestCase
{
    
    /**
     * Configuration object to run Turbine outside a servlet container
     * ( uses turbine.properties )
     */
    private static TurbineConfig config = null;
    
    /**
     * Sets up TurbineConfig using the system property:
     * <pre>turbine.properties</pre>
     */
    static
    {
        try
        {
            config = new TurbineConfig( "../webapp",
            "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }
    
    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestJetspeedLinkFactory(String name)
    {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[])
    {
        TestRunner.main( new String[]
        { TestJetspeedLinkFactory.class.getName() } );
    }
    
    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite()
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestJetspeedLinkFactory.class );
    }
    
    /**
     * Simple test that get a JetspeedLink object
     *
     * @throws Exception
     */
    public void testSimpleGet() throws Exception
    {
        JetspeedLink jsLink = JetspeedLinkFactory.getInstance();
        System.out.println("Class return by JetspeedLinkFactory: " + jsLink.getClass().getName());
        assertNotNull( "Got JetspeedLink", jsLink);
        assertEquals( "Created class defined by tools.request.jslink"
        , JetspeedResources.getString("tool.request.jslink"
        , "org.apache.jetspeed.util.template.BaseJetspeedLink"), jsLink.getClass().getName());
        assertTrue( "Class instance of JetspeedLink", (jsLink instanceof JetspeedLink));
    }
    
    /**
     * Simple test that gets and put a JetspeedLink object
     *
     * @throws Exception
     */
    public void testGetandPut() throws Exception
    {
        PoolService poolService = (PoolService) TurbineServices.
        getInstance().getService(PoolService.SERVICE_NAME);
        JetspeedLink jsLink = null;
        int beforeSize;
        
        for (int counter = 0; counter < 10; counter++)
        {
            jsLink = JetspeedLinkFactory.getInstance();
            assertNotNull( "Get loop - Got JetspeedLink", jsLink);
            assertTrue( "Get loop - jsLink instance of JetspeedLink", (jsLink instanceof JetspeedLink));
        }
        String linkClassName = jsLink.getClass().getName();
        jsLink = null;
        
        for (int counter = 0; counter < 10; counter++)
        {
            jsLink = JetspeedLinkFactory.getInstance();
            assertNotNull( "Get/put loop - Got JetspeedLink", jsLink);
            assertTrue( "Get/put loop - jsLink instance of JetspeedLink", (jsLink instanceof JetspeedLink));
            beforeSize = poolService.getSize( linkClassName);
            JetspeedLinkFactory.putInstance(jsLink);
            assertTrue( "Class saved in pool", (beforeSize < poolService.getSize( linkClassName)));
            jsLink = null;
        }
    }
    public void testFillPool() throws Exception
    {
        Stack jsLinkStack = new Stack();
        JetspeedLink jsLink = null;

        PoolService poolService = (PoolService) TurbineServices.
        getInstance().getService(PoolService.SERVICE_NAME);
        int poolCapacity;

        jsLink = JetspeedLinkFactory.getInstance();
        String linkClassName = jsLink.getClass().getName();
        poolCapacity = poolService.getCapacity( linkClassName);

        System.out.println("Class Name  " + linkClassName);
        System.out.println("Pool Capacity " + poolCapacity);

        // Fill stack with objects
        for (int counter = 0; counter < poolCapacity; counter++)
        {
            jsLink = JetspeedLinkFactory.getInstance();
            assertNotNull( "Get loop - Got JetspeedLink", jsLink);
            assertTrue( "Get loop - jsLink instance of JetspeedLink", (jsLink instanceof JetspeedLink));
            jsLinkStack.push(jsLink);
        }
        
        // Fill up the pool
        while (jsLinkStack.empty() == false)
            JetspeedLinkFactory.putInstance( (JetspeedLink) jsLinkStack.pop());
        assertEquals( "Pool is full", poolCapacity, poolService.getSize(linkClassName));
        
        // Empty pool
        for (int counter = 0; counter < poolCapacity; counter++)
        {
            jsLink = JetspeedLinkFactory.getInstance();
            assertNotNull( "Get loop - Got JetspeedLink", jsLink);
            assertTrue( "Get loop - jsLink instance of JetspeedLink", (jsLink instanceof JetspeedLink));
        }
        assertEquals( "Pool is empty", 0, poolService.getSize(linkClassName));
    }
}
