/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security;

import java.util.Iterator;
import java.security.Principal;

import org.apache.turbine.services.Service;
import org.apache.jetspeed.om.security.JetspeedUser;

/**
 * <p> The <code>UserManagement</code> interface describes contract between 
 * the portal and security provider required for Jetspeed User Management.
 * This interface enables an application to be independent of the underlying 
 * user management technology.
 *
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: UserManagement.java,v 1.2 2002/06/28 05:37:34 taylor Exp $
 */

public interface UserManagement extends Service, CredentialsManagement  
{
    public String SERVICE_NAME = "UserManagement";

    /**
     * Retrieves a <code>JetspeedUser</code> given the primary principle.
     * The principal can be any valid Jetspeed Security Principal:
     *   <code>org.apache.jetspeed.om.security.UserNamePrincipal</code>
     *   <code>org.apache.jetspeed.om.security.UserIdPrincipal</code>
     *   
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param principal a principal identity to be retrieved.
     * @return a <code>JetspeedUser</code> associated to the principal identity.
     * @exception UserException when the security provider has a general failure retrieving a user.
     * @exception UnknownUserException when the security provider cannot match
     *            the principal identity to a user.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    JetspeedUser getUser(Principal principal)
        throws JetspeedSecurityException;

    /**
     * Retrieves a collection of all <code>JetspeedUser</code>s.
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @return a collection of <code>JetspeedUser</code> entities.
     * @exception UserException when the security provider has a general failure retrieving users.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    Iterator getUsers()
        throws JetspeedSecurityException;

    /**
     * Retrieves a collection of <code>JetspeedUser</code>s filtered by a security 
     * provider-specific query string. For example SQL, OQL, JDOQL.
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @return a collection of <code>JetspeedUser</code> entities.
     * @exception UserException when the security provider has a general failure retrieving users.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    Iterator getUsers(String filter)
        throws JetspeedSecurityException;

    /**
     * Saves a <code>JetspeedUser</code>'s attributes into permanent storage. 
     * The user's account is required to exist in the storage.
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception UserException when the security provider has a general failure retrieving users.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    void saveUser(JetspeedUser user)
        throws JetspeedSecurityException;

    /**
     * Adds a <code>JetspeedUser</code> into permanent storage. 
     * The security service can throw a <code>NotUniqueUserException</code> when the public
     * credentials fail to meet the security provider-specific unique constraints.
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @exception UserException when the security provider has a general failure retrieving users.
     * @exception NotUniqueUserException when the public credentials fail to meet 
     *                                   the security provider-specific unique constraints.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    void addUser(JetspeedUser user)
        throws JetspeedSecurityException;

    /**
     * Removes a <code>JetspeedUser</code> from the permanent store.
     * The security service may optionally check the current user context
     * to determine if the requestor has permission to perform this action.
     *
     * @param principal the principal identity to be retrieved.
     * @exception UserException when the security provider has a general failure retrieving a user.
     * @exception UnknownUserException when the security provider cannot match
     *            the principal identity to a user.
     * @exception InsufficientPrivilegeException when the requestor is denied due to insufficient privilege 
     */
    void removeUser(Principal principal)
        throws JetspeedSecurityException;

}
